/***************************************************************************
 *
 *  $Id: kzensearchcondition.cpp,v 1.5 2005/05/31 06:54:03 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include "kzentrack.h"
#include "kzensearchcondition.h"

KZenSearchCondition::KZenSearchCondition()
{
  setField(FArtist);
  setOperation(OEQUAL);
  setCaseSensitive(true);
  setValue("");
}

KZenSearchCondition::KZenSearchCondition(KZenSearchCondition * con)
{
  setField(con->getField());
  setOperation(con->getOperation());
  setCaseSensitive(con->getCaseSensitive());
  setValue(con->getValue());
}

KZenSearchCondition::KZenSearchCondition(FieldType field, 
					 Operation op, 
					 bool cs,
					 QVariant value)
{
  fField = field;
  fOperation = op;
  fCaseSensitive = cs;
  fValue = value;
}

KZenSearchCondition::~KZenSearchCondition()
{

}

void KZenSearchCondition::setValue(const QVariant & v)
{
  fValue = v;
}

bool KZenSearchCondition::checkTrack(KZenTrack * track)
{
  bool ret = false;
  QVariant field;
  switch (fField) {
  case FArtist:
    ret = checkOperation(QVariant(track->getArtist()));
    break;
  case FAlbum:
    ret = checkOperation(QVariant(track->getAlbum()));
    break;
  case FTitle:
    ret = checkOperation(QVariant(track->getTitle()));
    break;
  case FTracknumber:
    ret = checkOperation(QVariant(track->getTrackNumber()));
    break;
  case FGenre:
    ret = checkOperation(QVariant(track->getGenre()));
    break;
  case FYear:
    ret = checkOperation(QVariant(track->getYear()));
    break;
  case FRating:
    ret = checkOperation(QVariant(track->getRating()));
    break;
  case FCategory:
    {
      uint temp = fValue.toUInt();
      fValue = QVariant(true);
      if (track->getCategories().size()<=temp) {
	ret = checkOperation(QVariant(false));
      } else {
	ret = checkOperation(QVariant(track->getCategories()[temp]));
      }
      fValue = QVariant(temp);
    }
    break;
  default:
    ret = false;
  }
  return ret;
}

bool KZenSearchCondition::checkOperation(QVariant field)
{
  bool ret = false;
  switch (fOperation) {
  case OEQUAL:
    ret = (field == fValue);
    break;
  case ONOTEQUAL:
    ret = (field != fValue);
    break;
  case OCONTAINS:
    ret = field.toString().contains(fValue.toString(),fCaseSensitive);
    break;
  case ONOTCONTAINS:
    ret = !field.toString().contains(fValue.toString(),fCaseSensitive);
    break;
  case OLESS:
    if (field.canCast(QVariant::Int) && fValue.canCast(QVariant::Int))
      ret = (field.toInt() < fValue.toInt());
    else
      ret = false;
    break;
  case OLESSEQUAL:
    if (field.canCast(QVariant::Int) && fValue.canCast(QVariant::Int))
      ret = (field.toInt() <= fValue.toInt());
    else
      ret = false;
    break;
  case OMORE:
    if (field.canCast(QVariant::Int) && fValue.canCast(QVariant::Int))
      ret = (field.toInt() > fValue.toInt());
    else
      ret = false;
    break;
  case OMOREEQUAL:
    if (field.canCast(QVariant::Int) && fValue.canCast(QVariant::Int))
      ret = (field.toInt() >= fValue.toInt());
    else
      ret = false;
    break;
  default:
    ret = false;
  }
  return ret;
}

void KZenSearchCondition::assign(KZenSearchCondition * con)
{
  setField(con->getField());
  setOperation(con->getOperation());
  setCaseSensitive(con->getCaseSensitive());
  setValue(con->getValue());
}

QDataStream & operator<<(QDataStream & s, 
			 KZenSearchCondition & con)
{
  s << (int)con.getField();
  s << (int)con.getOperation();
  s << (int)con.getCaseSensitive();
  s << con.getValue();

  return s;
}

QDataStream & operator>>(QDataStream & s, 
			 KZenSearchCondition & con)
{
  int temp;
  s >> temp;
  con.setField((KZenSearchCondition::FieldType)temp);
  s >> temp;
  con.setOperation((KZenSearchCondition::Operation)temp);
  s >> temp;
  con.setCaseSensitive((bool)temp);  
  QVariant v;
  s >> v;
  con.setValue(v);

  return s;
}
