/***************************************************************************
 *
 *  $Id: kzenalbum.cpp,v 1.17 2005/06/02 07:09:01 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include <qpainter.h>
#include <qpixmap.h>

#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>

#include "kzenconfig.h"
#include "kzenalbum.h"

QStringList KZenAlbum::fAlbums;

KZenAlbum::KZenAlbum(const char * name)
{
  setName(name);

  fTracks = new QPtrList<KZenTrack>();
  fTracks->setAutoDelete(false);

  fItem = 0;
}

KZenAlbum::~KZenAlbum()
{
  delete fTracks;
}

void KZenAlbum::setName(const char * name)
{
  fName = name;

  if (!KZenAlbum::getAlbums().contains(name)) {
    KZenAlbum::getAlbums().append(name);
    KZenAlbum::getAlbums().sort();
  }
}

void KZenAlbum::addTrack(KZenTrack * track)
{
  fTracks->append(track);
}

QStringList & KZenAlbum::getAlbums()
{
  return fAlbums;
}
uint KZenAlbum::getTrackCount()
{
  return fTracks->count();
}

uint KZenAlbum::getLength()
{
  int length = 0;
  KZenTrack * track;
  for (uint tr=0;tr<fTracks->count();tr++) {
    track = fTracks->at(tr);
    length += track->getLength();
  }
  
  return length;
}

KZenAlbumItem::KZenAlbumItem(QListViewItem * parent, KZenAlbum * Album)
  :KZenItem(parent,KZenItem::ItemTypeAlbum)
{
  fAlbum = Album;

  setPixmap(0, UserIcon("album", 16));
  setText(0,fAlbum->getName());

  fSortColumn = KZen::FieldTrack;
  fSortOrder = Qt::Ascending;

  fIsCompilation = false;
}

KZenAlbumItem::~KZenAlbumItem()
{

}

void KZenAlbumItem::setIsCompilation(bool c)
{
  KZenTrack * track;
  for (uint t=0;t<fAlbum->getTrackList()->count();t++) {
    track = fAlbum->getTrackList()->at(t);
    track->setIsCompilation(c);
  }
  fIsCompilation = c;
}

void KZenAlbumItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
  bool sd = KZenConfig::get()->showDetailsAlbum();

  QFontMetrics fml(listView()->font());
  int margin = listView()->itemMargin()*2;
  int h = fml.lineSpacing();
  if (sd) h += fml.lineSpacing();
  if (pixmap(0)->height()>h) h = pixmap(0)->height();
  if (h % 2 > 0) h++;
  
  setHeight(h + margin + 2);

  KListView *lv = (KListView *)listView();

  static QPixmap buffer;
  buffer.resize(width, height());

  if(buffer.isNull()) {
    KListViewItem::paintCell(p, cg, column, width, align);
    return;
  }
  
  QPainter pBuf(&buffer, true);
  if (isSelected()) {
    pBuf.fillRect(buffer.rect(),cg.highlight());
  } else {
    pBuf.fillRect(buffer.rect(),KZen::backgroundColor(this));
  }

  QFont font(p->font());
  QFontMetrics fm(p->fontMetrics());

  int text_x = pixmap(0)->width() + 10;
  int text_y = 1;
  int textHeight = fm.lineSpacing() + lv->itemMargin() + 1;

  pBuf.drawPixmap(4, (height() - pixmap(0)->height())/2,
		  *pixmap(0));
  
  if (isSelected()) {
    pBuf.setPen(cg.highlightedText());
  } else {
    pBuf.setPen(cg.text());
  }

  if (sd) font.setBold(true);
  pBuf.setFont(font);
  QFontMetrics fmName(font);

  QString name = text(0);
  
  if(fmName.width(name) + text_x + lv->itemMargin()*2 > width ) {
    int ellWidth = fmName.width(i18n("..."));
    QString text = QString::fromLatin1("");
    int i = 0;
    int len = name.length();
    while ( i < len && fmName.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
      text += name[i];
      i++;
    }
    name = text + i18n("...");
  }

  pBuf.drawText(text_x, text_y, width, textHeight, AlignVCenter, name);

  if (sd) {
    font.setBold(false);
    font.setItalic(true);
    pBuf.setFont(font);
    QFontMetrics fmInfo(font);
    
    QString info;

    if (fAlbum) {
      info += i18n("1 Track", "%n Tracks", fAlbum->getTrackCount());
      info += QString(i18n(" - [%2]")).arg(KZen::formatTime(fAlbum->getLength()));
    }
    
    if(fmInfo.width(info) + text_x + lv->itemMargin()*2 > width ) {
      int ellWidth = fmInfo.width(i18n("..."));
      QString text = QString::fromLatin1("");
      int i = 0;
      int len = info.length();
      while ( i < len && fmInfo.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
	text += info[i];
	i++;
      }
      info = text + i18n("...");
    }    
    pBuf.drawText(text_x, text_y + textHeight, width, fm.lineSpacing(), AlignVCenter, info);
  }
  
  pBuf.end();
  p->drawPixmap(0, 0, buffer);
}

KZenAlbumItemTL::KZenAlbumItemTL(QListView * parent, const QString & name)
  :KZenItem(parent,KZenItem::ItemTypeAlbumTL)
{
  setPixmap(0, UserIcon("album", 16));
  setText(0,name);

  fSortColumn = KZen::FieldTrack;
  fSortOrder = Qt::Ascending;

  fTracks = new QPtrList<KZenTrack>(); 
  fTracks->setAutoDelete(false);

  fIsCompilation = false;
}

KZenAlbumItemTL::~KZenAlbumItemTL()
{
  delete fTracks;
}

void KZenAlbumItemTL::addTrack(KZenTrack * track)
{
  fTracks->append(track);
}

void KZenAlbumItemTL::setIsCompilation(bool c)
{
  KZenTrack * track;
  for (uint t=0;t<fTracks->count();t++) {
    track = fTracks->at(t);
    track->setIsCompilation(c);
  }
  fIsCompilation = c;
}

void KZenAlbumItemTL::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
  bool sd = KZenConfig::get()->showDetailsAlbum();

  QFontMetrics fml(listView()->font());
  int margin = listView()->itemMargin()*2;
  int h = fml.lineSpacing();
  if (sd) h += fml.lineSpacing();
  if (pixmap(0)->height()>h) h = pixmap(0)->height();
  if (h % 2 > 0) h++;
  
  setHeight(h + margin + 2);

  KListView *lv = (KListView *)listView();

  static QPixmap buffer;
  buffer.resize(width, height());

  if(buffer.isNull()) {
    KListViewItem::paintCell(p, cg, column, width, align);
    return;
  }
  
  QPainter pBuf(&buffer, true);
  if (isSelected()) {
    pBuf.fillRect(buffer.rect(),cg.highlight());
  } else {
    pBuf.fillRect(buffer.rect(),KZen::backgroundColor(this));
  }

  QFont font(p->font());
  QFontMetrics fm(p->fontMetrics());

  int text_x = pixmap(0)->width() + 10;
  int text_y = 1;
  int textHeight = fm.lineSpacing() + lv->itemMargin() + 1;

  pBuf.drawPixmap(4, (height() - pixmap(0)->height())/2,
		  *pixmap(0));
  
  if (isSelected()) {
    pBuf.setPen(cg.highlightedText());
  } else {
    pBuf.setPen(cg.text());
  }

  if (sd) font.setBold(true);
  pBuf.setFont(font);
  QFontMetrics fmName(font);

  QString name = text(0);
  if (!sd) {
    name += " (";
    if (fIsCompilation) {
      name += "Various Artists";
    } else {
      name += text(1);
    }
    name += ")";
  }
  
  if(fmName.width(name) + text_x + lv->itemMargin()*2 > width ) {
    int ellWidth = fmName.width(i18n("..."));
    QString text = QString::fromLatin1("");
    int i = 0;
    int len = name.length();
    while ( i < len && fmName.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
      text += name[i];
      i++;
    }
    name = text + i18n("...");
  }

  pBuf.drawText(text_x, text_y, width, textHeight, AlignVCenter, name);

  if (sd) {
    font.setBold(false);
    font.setItalic(true);
    pBuf.setFont(font);
    QFontMetrics fmInfo(font);
    
    QString info;

    info += i18n("1 Track", "%n Tracks", fTracks->count());
    int length = 0;
    KZenTrack * track;
    for (uint tr=0;tr<fTracks->count();tr++) {
      track = fTracks->at(tr);
      length += track->getLength();
    }
    info += QString(i18n(" - [%2]")).arg(KZen::formatTime(length));

    info += " (";
    if (fIsCompilation) {
      info += "Various Artists";
    } else {
      info += text(1);
    }
    info += ")";
    
    if(fmInfo.width(info) + text_x + lv->itemMargin()*2 > width ) {
      int ellWidth = fmInfo.width(i18n("..."));
      QString text = QString::fromLatin1("");
      int i = 0;
      int len = info.length();
      while ( i < len && fmInfo.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
	text += info[i];
	i++;
      }
      info = text + i18n("...");
    }    
    pBuf.drawText(text_x, text_y + textHeight, width, fm.lineSpacing(), AlignVCenter, info);
  }
  
  pBuf.end();
  p->drawPixmap(0, 0, buffer);
}
