#!/bin/sh

#
#   A script for extracting and installing the KVIrc documentation
#
#   27-04-2000 Szymon Stefanek
#
#
#   This program is FREE software. You can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your opinion) any later version.
#
#   This program is distributed in the HOPE that it will be USEFUL,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#   See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program. If not, write to the Free Software Foundation,
#   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#

HELP_INSTALL_DIR="$1"

if test -z "$HELP_INSTALL_DIR"; then
	echo "### Missing installation directory"
	echo "### Please run this script manually:"
	echo "### $0 directory_for_the_help_files"
	echo "### The directory is usually PREFIX/share/kvirc/help/en/"
fi

echo "################################################################################"
echo "### Generating help files for KVIrc"
echo "### This may take a while..."
echo "################################################################################"

kvidoc_process_document()
{
	# Kill the end of the comment
	RETURN=`echo "$1" | sed -e '/\*\// q'`
	RETURN=`echo "$RETURN" | sed -e '/\*\// d'`
	# Kill leading tabs and empty lines
	RETURN=`echo "$RETURN" | grep -h "[a-zA-Z0-9{}]"`
 	RETURN=`echo "$RETURN" | sed -e 's/^[	 ]*//'`
	# Add the links
	RETURN=`echo "$RETURN" | sed -e 's/\[cmd\]\([a-zA-Z0-9_]*\)\[\/cmd\]/<a href=\"\1.kvihelp">\1<\/a>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\[cmd:\([a-zA-Z0-9_]*\)\]\([a-zA-Z0-9_]*\)\[\/cmd\]/<a href=\"\1.kvihelp">\2<\/a>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\[fnc\]\$\([a-zA-Z0-9_]*\)\[\/fnc\]/<a href=\"s_\1.kvihelp">\$\1<\/a>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\[class\]\([a-zA-Z0-9_]*\)\[\/class\]/<a href=\"class_\1.kvihelp">\1<\/a>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\[classfnc:\([a-zA-Z0-9_]*\)\]\$\([a-zA-Z0-9_]*\)\[\/classfnc\]/<a href=\"class_\1.kvihelp#\2">\$\2<\/a>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\[classevent:\([a-zA-Z0-9_]*\)\]\([a-zA-Z0-9_]*\)\[\/classevent\]/<a href=\"class_\1.kvihelp#\2">\2<\/a>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\[classsignal:\([a-zA-Z0-9_]*\)\]\([a-zA-Z0-9_]*\)\[\/classsignal\]/<a href=\"class_\1.kvihelp#\2">\2<\/a>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\[event\]\([a-zA-Z0-9_]*\)\[\/event\]/<a href=\"\1.kvihelp">\1<\/a>/g'`
	echo "$RETURN"
}

kvidoc_process_examples()
{
	RETURN=`echo "$1" | grep -h -i -A 400 "@examples:"`
	RETURN=`echo "$RETURN" | sed -e '/@examples:/ d'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'`
	RETURN=`echo "$RETURN" | sed -e 's/	/\\&nbsp;\\&nbsp;\\&nbsp;\\&nbsp;/g'`
	RETURN=`echo "$RETURN" | sed -e 's/@tab@/    /g'`
	RETURN=`echo "$RETURN" | sed -e 's/->/-\&gt;/g'`
	echo "$RETURN"
}

kvidoc_process_syntax()
{
	RETURN=`echo "$1" | grep -h -i -A 100 "@syntax:"`
	RETURN=`echo "$RETURN" | sed -e '/@syntax:/ d'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'`
 	# Replace eventual < and > in the syntax
	RETURN=`echo "$RETURN" | sed -e 's/</\&lt;/g'`
	RETURN=`echo "$RETURN" | sed -e 's/>/\&gt;/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\&lt;br&gt;/<br>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/@tab@/    /g'`
	echo "$RETURN"
}

kvidoc_process_parameters()
{
	RETURN=`echo "$1" | grep -h -i -A 100 "@parameters:"`
	RETURN=`echo "$RETURN" | sed -e '/@parameters:/ d'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'`
 	# Replace eventual < and > in the syntax
	RETURN=`echo "$RETURN" | sed -e 's/</\&lt;/g'`
	RETURN=`echo "$RETURN" | sed -e 's/>/\&gt;/g'`
	RETURN=`echo "$RETURN" | sed -e 's/\&lt;br&gt;/<br>/g'`
	RETURN=`echo "$RETURN" | sed -e 's/@tab@/    /g'`
	echo "$RETURN"
}

kvidoc_process_seealso()
{
 	RETURN=`echo "$1" | grep -h -i -A 100 "@seealso:"`
 	RETURN=`echo "$RETURN" | sed -e '/@seealso:/ d'`
 	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
 	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'`
	echo "$RETURN"
}

kvidoc_process_description()
{
 	RETURN=`echo "$1" | grep -h -i -A 150 "@description:"`
 	RETURN=`echo "$RETURN" | sed -e '/@description:/ d'`
 	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
 	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'` 	
	echo "$RETURN"
}

kvidoc_process_return()
{
	RETURN=`echo "$1" | grep -h -i -A 20 "@return:"`
	RETURN=`echo "$RETURN" | sed -e '/@return:/ d'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'`
	echo "$RETURN"
}

kvidoc_process_short()
{
 	RETURN=`echo "$1" | grep -h -i -A 100 "@short:"`
	RETURN=`echo "$RETURN" | sed -e '/@short:/ d'`
 	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'` 	
	echo "$RETURN"
}


kvidoc_process_window()
{
 	RETURN=`echo "$1" | grep -h -i -A 50 "@window:"`
	RETURN=`echo "$RETURN" | sed -e '/@window:/ d'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ q'`
	RETURN=`echo "$RETURN" | sed -e '/@[a-zA-Z]*:/ d'` 	
	echo "$RETURN"
}

kvidoc_tolower()
{
	RETURN=`echo "$1" | sed -e 'y/ABCDEFGHIJKLMNOPQRSTUVWXYZ/abcdefghijklmnopqrstuvwxyz/'`
	echo "$RETURN"
}


echo "################################################################################"
echo "### Generating help for commands..."
echo "################################################################################"

# First get the list of all the command names
COMMAND_NAMES=`grep -h -i "@command:" src/kvirc/kvi_commands.cpp | sed -e 's/[	 ]*@command:[ ]*//g'`
COMMAND_LINKS=""

for a_name in $COMMAND_NAMES; do
	LOWER_NAME=`kvidoc_tolower "$a_name"`
	echo "### Generating $HELP_INSTALL_DIR/$LOWER_NAME.kvihelp"

	DOCUMENT_DATA=`grep -h -i -A 150 "@command:[ ]*$a_name[ 	]*\$" src/kvirc/kvi_commands.cpp`
	DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@command:[ ]*[a-zA-Z_]*/ d'`
	DOCUMENT_DATA=`kvidoc_process_document "$DOCUMENT_DATA"`

	SYNTAX_DATA=`kvidoc_process_syntax "$DOCUMENT_DATA"`
	EXAMPLES_DATA=`kvidoc_process_examples "$DOCUMENT_DATA"`
	SEEALSO_DATA=`kvidoc_process_seealso "$DOCUMENT_DATA"`
	DESCRIPTION_DATA=`kvidoc_process_description "$DOCUMENT_DATA"`
	RETURN_DATA=`kvidoc_process_return "$DOCUMENT_DATA"`
	SHORT_DATA=`kvidoc_process_short "$DOCUMENT_DATA"`

 	# Create the main document
 	DOCUMENT="<doctitle>$a_name command</doctitle>
<docbody>
<hr>
"
if test -n "$SYNTAX_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Syntax</docsubtitle>
$SYNTAX_DATA<br>
"
fi
if test -n "$RETURN_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Return value</docsubtitle>
$RETURN_DATA<br>
"
else
	DOCUMENT="$DOCUMENT<docsubtitle>Return value</docsubtitle>
Not modified<br>"
fi
if test -n "$DESCRIPTION_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Description</docsubtitle>
$DESCRIPTION_DATA<br>
"
fi
if test -n "$EXAMPLES_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Examples</docsubtitle>
$EXAMPLES_DATA<br>
"
fi
if test -n "$SEEALSO_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>See also</docsubtitle>
$SEEALSO_DATA<br>
"
fi
DOCUMENT="$DOCUMENT<hr>
<a href=\"index.kvihelp\">Main Index</a>, <a href=\"commands.kvihelp\">Command Index</a>
</docbody>
"


	echo "$DOCUMENT" > $HELP_INSTALL_DIR/$LOWER_NAME.kvihelp




	# Update the command links

	if test -n "$COMMAND_LINKS"; then
		COMMAND_LINKS="$COMMAND_LINKS
<tr><td><a href=\"$LOWER_NAME.kvihelp\">$a_name</a></td>"
	else
		COMMAND_LINKS="<tr><td><a href=\"$LOWER_NAME.kvihelp\">$a_name</a></td>"
	fi

	if test -n "$SHORT_DATA"; then
		COMMAND_LINKS="$COMMAND_LINKS : <td>$SHORT_DATA</td></tr>"
	else
		COMMAND_LINKS="$COMMAND_LINKS<td></td></tr>"
	fi
done



# Prepare the main command index

COMMAND_LINKS=`echo "$COMMAND_LINKS" | sort`

COMMAND_INDEX="<doctitle>Supported commands</doctitle>
<docbody>
<hr>
<table bgcolor="#999999">
$COMMAND_LINKS
</table>
<hr>
<a href=\"index.kvihelp\">Main Index</a>
</docbody>
"

echo "### Generating $HELP_INSTALL_DIR/commands.kvihelp"
echo "$COMMAND_INDEX" > $HELP_INSTALL_DIR/commands.kvihelp

echo "################################################################################"
echo "### Generating help for functions..."
echo "################################################################################"

# First get the list of all the function names
FUNCTION_NAMES=`grep -h -i "@function:" src/kvirc/kvi_functions.cpp | sed -e 's/[	 ]*@function:[ ]*//g'`
FUNCTION_LINKS=""

for a_name in $FUNCTION_NAMES; do
	LOWER_NAME=`kvidoc_tolower "$a_name"`
	echo "### Generating $HELP_INSTALL_DIR/s_$LOWER_NAME.kvihelp"

	DOCUMENT_DATA=`grep -h -i -A 200 "@function:[ ]*$a_name" src/kvirc/kvi_functions.cpp`
	DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@function:[ ]*[a-zA-Z_]*/ d'`
	DOCUMENT_DATA=`kvidoc_process_document "$DOCUMENT_DATA"`

	SYNTAX_DATA=`kvidoc_process_syntax "$DOCUMENT_DATA"`
	EXAMPLES_DATA=`kvidoc_process_examples "$DOCUMENT_DATA"`
	SEEALSO_DATA=`kvidoc_process_seealso "$DOCUMENT_DATA"`
	DESCRIPTION_DATA=`kvidoc_process_description "$DOCUMENT_DATA"`
	SHORT_DATA=`kvidoc_process_short "$DOCUMENT_DATA"`

 	# Create the main document
 	DOCUMENT="<doctitle>$a_name function</doctitle>
<docbody>
<hr>
"
if test -n "$SYNTAX_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Syntax</docsubtitle>
$SYNTAX_DATA<br>
"
fi
if test -n "$DESCRIPTION_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Description</docsubtitle>
$DESCRIPTION_DATA<br>
"
fi
if test -n "$EXAMPLES_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Examples</docsubtitle>
$EXAMPLES_DATA<br>
"
fi
if test -n "$SEEALSO_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>See also</docsubtitle>
$SEEALSO_DATA<br>
"
fi
DOCUMENT="$DOCUMENT<hr>
<a href=\"index.kvihelp\">Main Index</a>, <a href=\"functions.kvihelp\">Function Index</a>
</docbody>
"

echo "$DOCUMENT" > $HELP_INSTALL_DIR/s_$LOWER_NAME.kvihelp


if test -n "$FUNCTION_LINKS"; then
	FUNCTION_LINKS="$FUNCTION_LINKS
<a href=\"s_$LOWER_NAME.kvihelp\">$a_name</a>"
else
	FUNCTION_LINKS="<a href=\"s_$LOWER_NAME.kvihelp\">$a_name</a>"
fi

if test -n "$SHORT_DATA"; then
	FUNCTION_LINKS="$FUNCTION_LINKS : $SHORT_DATA<br>"
else
	FUNCTION_LINKS="$FUNCTION_LINKS<br>"
fi

done

FUNCTION_LINKS=`echo "$FUNCTION_LINKS" | sort`

# Prepare the main functions index
FUNCTION_INDEX="<doctitle>Supported functions</doctitle>
<docbody>
<hr>
$FUNCTION_LINKS
<hr>
<a href=\"index.kvihelp\">Main Index</a>
</docbody>
"
echo "### Generating $HELP_INSTALL_DIR/functions.kvihelp"
echo "$FUNCTION_INDEX" > $HELP_INSTALL_DIR/functions.kvihelp

echo "################################################################################"
echo "### Generating help for identifiers..."
echo "################################################################################"

# First get the list of all the identifier names
IDENTIFIER_NAMES=`grep -h -i "@identifier:" src/kvirc/kvi_identifiers.cpp | sed -e 's/[	 ]*@identifier:[ ]*//g'`
IDENTIFIER_LINKS=""

for a_name in $IDENTIFIER_NAMES; do
	LOWER_NAME=`kvidoc_tolower "$a_name"`
	echo "### Generating $HELP_INSTALL_DIR/s_$LOWER_NAME.kvihelp"

	DOCUMENT_DATA=`grep -h -i -A 100 "@identifier:[ ]*$a_name\([ ]\+\|$\)" src/kvirc/kvi_identifiers.cpp`
	DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@identifier:[ ]*[a-zA-Z_]*/ d'`
	DOCUMENT_DATA=`kvidoc_process_document "$DOCUMENT_DATA"`

	SYNTAX_DATA=`kvidoc_process_syntax "$DOCUMENT_DATA"`
	EXAMPLES_DATA=`kvidoc_process_examples "$DOCUMENT_DATA"`
	SEEALSO_DATA=`kvidoc_process_seealso "$DOCUMENT_DATA"`
	DESCRIPTION_DATA=`kvidoc_process_description "$DOCUMENT_DATA"`
	SHORT_DATA=`kvidoc_process_short "$DOCUMENT_DATA"`

 	# Create the main document
 	DOCUMENT="<doctitle>$a_name identifier</doctitle>
<docbody>
<hr>
"
if test -n "$SYNTAX_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Syntax</docsubtitle>
$SYNTAX_DATA<br>
"
fi
if test -n "$DESCRIPTION_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Description</docsubtitle>
$DESCRIPTION_DATA<br>
"
fi
if test -n "$EXAMPLES_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Examples</docsubtitle>
$EXAMPLES_DATA<br>
"
fi
if test -n "$SEEALSO_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>See also</docsubtitle>
$SEEALSO_DATA<br>
"
fi
DOCUMENT="$DOCUMENT<hr>
<a href=\"index.kvihelp\">Main Index</a>, <a href=\"identifiers.kvihelp\">Identifier Index</a>
</docbody>
"

echo "$DOCUMENT" > $HELP_INSTALL_DIR/s_$LOWER_NAME.kvihelp

if test -n "$IDENTIFIER_LINKS"; then
	IDENTIFIER_LINKS="$IDENTIFIER_LINKS
<a href=\"s_$LOWER_NAME.kvihelp\">$a_name</a>"
else
	IDENTIFIER_LINKS="<a href=\"s_$LOWER_NAME.kvihelp\">$a_name</a>"
fi

if test -n "$SHORT_DATA"; then
	IDENTIFIER_LINKS="$IDENTIFIER_LINKS : $SHORT_DATA<br>"
else
	IDENTIFIER_LINKS="$IDENTIFIER_LINKS<br>"
fi

done


# Prepare the main identifiers index

IDENTIFIER_LINKS=`echo "$IDENTIFIER_LINKS" | sort`

IDENTIFIER_INDEX="<doctitle>Supported identifiers</doctitle>
<docbody>
<hr>
$IDENTIFIER_LINKS
<hr>
<a href=\"index.kvihelp\">Main Index</a>
</docbody>
"
echo "### Generating $HELP_INSTALL_DIR/identifiers.kvihelp"
echo "$IDENTIFIER_INDEX" > $HELP_INSTALL_DIR/identifiers.kvihelp

echo "################################################################################"
echo "### Generating help for events..."
echo "################################################################################"

# First get the list of all the event names
EVENT_NAMES=`grep -h -i "@event:" src/kvirc/kvi_event.cpp | sed -e 's/[	 ]*@event:[ ]*//g'`
EVENT_LINKS=""

for a_name in $EVENT_NAMES; do
	LOWER_NAME=`kvidoc_tolower "$a_name"`
	echo "### Generating $HELP_INSTALL_DIR/$LOWER_NAME.kvihelp"

	DOCUMENT_DATA=`grep -h -i -A 100 "@event:[ ]*$a_name" src/kvirc/kvi_event.cpp`
	DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@event:[ ]*[a-zA-Z_]*/ d'`
	DOCUMENT_DATA=`kvidoc_process_document "$DOCUMENT_DATA"`

	PARAMETERS_DATA=`kvidoc_process_parameters "$DOCUMENT_DATA"`
	SHORT_DATA=`kvidoc_process_short "$DOCUMENT_DATA"`
	DESCRIPTION_DATA=`kvidoc_process_description "$DOCUMENT_DATA"`
	EXAMPLES_DATA=`kvidoc_process_examples "$DOCUMENT_DATA"`
	SEEALSO_DATA=`kvidoc_process_seealso "$DOCUMENT_DATA"`
	WINDOW_DATA=`kvidoc_process_window "$DOCUMENT_DATA"`

 	# Create the main document
 	DOCUMENT="<doctitle>$a_name event</doctitle>
<docbody>
<hr>
"
if test -n "$PARAMETERS_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Parameters</docsubtitle>
$PARAMETERS_DATA<br>
"
fi
if test -n "$WINDOW_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Window</docsubtitle>
$WINDOW_DATA<br>
"
fi
if test -n "$DESCRIPTION_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Description</docsubtitle>
$DESCRIPTION_DATA<br>
"
fi
if test -n "$EXAMPLES_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>Examples</docsubtitle>
$EXAMPLES_DATA<br>
"
fi
if test -n "$SEEALSO_DATA"; then
	DOCUMENT="$DOCUMENT<docsubtitle>See also</docsubtitle>
$SEEALSO_DATA<br>
"
fi
DOCUMENT="$DOCUMENT<hr>
<a href=\"index.kvihelp\">Main Index</a>, <a href=\"events.kvihelp\">Event Index</a>
</docbody>
"

echo "$DOCUMENT" > $HELP_INSTALL_DIR/$LOWER_NAME.kvihelp

if test -n "$EVENT_LINKS"; then
	EVENT_LINKS="$EVENT_LINKS
<a href=\"$LOWER_NAME.kvihelp\">$a_name</a>"
else
	EVENT_LINKS="<a href=\"$LOWER_NAME.kvihelp\">$a_name</a>"
fi

if test -n "$SHORT_DATA"; then
	EVENT_LINKS="$EVENT_LINKS : $SHORT_DATA<br>"
else
	EVENT_LINKS="$EVENT_LINKS<br>"
fi

done


# Prepare the main events index

EVENT_LINKS=`echo "$EVENT_LINKS" | sort`

EVENT_INDEX="<doctitle>Supported events</doctitle>
<docbody>
<hr>
$EVENT_LINKS
<hr>
<a href=\"index.kvihelp\">Main Index</a>
</docbody>
"
echo "### Generating $HELP_INSTALL_DIR/events.kvihelp"
echo "$EVENT_INDEX" > $HELP_INSTALL_DIR/events.kvihelp

echo "################################################################################"
echo "### Generating quick help..."
echo "################################################################################"


# First get the list of all the files containing @quickhelp
FILE_NAMES=`find src/kvirc/*.cpp`

QH_LINKS=""

for a_file in $FILE_NAMES; do
	QH_NAMES=`grep -h -i "@quickhelp:" $a_file | sed -e 's/[	 ]*@quickhelp:[ ]*//g'`
	if test -n "$QH_NAMES"; then
		for a_name in $QH_NAMES; do
			LOWER_NAME=`echo "$a_name" | sed -e 'y/ABCDEFGHIJKLMNOPQRSTUVWXYZ/abcdefghijklmnopqrstuvwxyz/'`
			echo "### Generating $HELP_INSTALL_DIR/qh_$LOWER_NAME.kvihelp"
			DOCUMENT_DATA=`grep -h -i -A 150 "@quickhelp:[ ]*$a_name" $a_file`
			# Now strip the first line and the lines after the end of the comment
			DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@quickhelp:[ ]*[a-zA-Z_]*/ d'`
			DOCUMENT_DATA=`kvidoc_process_document "$DOCUMENT_DATA"`
		 	WIDGET_NAME=`echo "$DOCUMENT_DATA" | grep -i "@widget:"`
		 	WIDGET_NAME=`echo "$WIDGET_NAME" | sed -e 's/[	 ]*@widget:[ ]*//g'`
			DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@widget:[ ]*[a-zA-Z_ ]*/ d'`
		 	# Create the main document
		 	DOCUMENT="<doctitle>$WIDGET_NAME</doctitle>
<docbody>
<hr>
$DOCUMENT_DATA
<hr>
<a href=\"index.kvihelp\">Main Index</a> <a href=\"quickhelp.kvihelp\">Widget Index</a>
</docbody>
"
			echo "$DOCUMENT" > $HELP_INSTALL_DIR/qh_$LOWER_NAME.kvihelp

			if test -n "$QH_LINKS"; then
				QH_LINKS="$QH_LINKS<br>
<a href=\"qh_$LOWER_NAME.kvihelp\">$WIDGET_NAME</a>"
			else
				QH_LINKS="<a href=\"qh_$LOWER_NAME.kvihelp\">$WIDGET_NAME</a>"
			fi

		done
	fi
done

echo "### Generating $HELP_INSTALL_DIR/quickhelp.kvihelp"

DOCUMENT="<doctitle>Widget index</doctitle>
<docbody>
<hr>
$QH_LINKS<br>
<hr>
<a href=\"index.kvihelp\">Main Index</a>
</docbody>"

echo "$DOCUMENT" > $HELP_INSTALL_DIR/quickhelp.kvihelp

echo "################################################################################"
echo "### Generating class documentation..."
echo "################################################################################"


# First get the list of all the files containing @class
FILE_NAMES_KVIRC=`find src/kvirc/*.cpp`
FILE_NAMES_CORE=`find src/kvicore/*.cpp`
FILE_NAMES_LIBRARY=`find src/kvilib/*.cpp`
FILE_NAMES_PLUGINS=`find src/plugins/*/*.cpp`

FILE_NAMES="$FILE_NAMES_KVIRC $FILE_NAMES_CORE $FILE_NAMES_LIBRARY $FILE_NAMES_PLUGINS"

DOC_LINKS=""

for a_file in $FILE_NAMES; do
	CLASS_NAMES=`grep -h -i "@class:" $a_file | sed -e 's/[	 ]*@class:[ ]*//g'`
	if test -n "$CLASS_NAMES"; then
		for a_name in $CLASS_NAMES; do

			SHORT_DATA=""
			DOCUMENT_DATA=""
			DESCRIPTION_DATA=""
			EXAMPLES_DATA=""
			SEEALSO_DATA=""
			EVENTS_DATA=""
			FUNCTIONS_DATA=""
			SIGNALS_DATA=""
			INHERITS_DATA=""

			LOWER_NAME=`kvidoc_tolower "$a_name"`

			echo "### Generating $HELP_INSTALL_DIR/class_$LOWER_NAME.kvihelp"

			DOCUMENT_DATA=`grep -h -i -A 500 "@class:[ ]*$a_name" $a_file`
			# Now strip the first line and the lines after the end of the comment
			DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@class:[ ]*[a-zA-Z_]*/ d'`
			DOCUMENT_DATA=`kvidoc_process_document "$DOCUMENT_DATA"`

			# Now DOCUMENT_DATA contains the whole class text
			SHORT_DATA=`kvidoc_process_short "$DOCUMENT_DATA"`
			DESCRIPTION_DATA=`kvidoc_process_description "$DOCUMENT_DATA"`

		 	# Extract the "@events:" part
		 	EVENTS_DATA=`echo "$DOCUMENT_DATA" | grep -h -i -A 150 "@events:"`
		 	EVENTS_DATA=`echo "$EVENTS_DATA" | sed -e '/@events:/ d'`
 			EVENTS_DATA=`echo "$EVENTS_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 			EVENTS_DATA=`echo "$EVENTS_DATA" | sed -e '/@[a-zA-Z]*:/ d'`


# Build the events table

EVENTS_TABLE=""

EVENT_NAMES=`echo "$EVENTS_DATA" | grep -h -i "!ev:" | sed -e 's/[	 ]*!ev:[ ]*//g' | sed -e 's/[ ]*([ a-zA-Z,_&;\$0123456789=	\-]*)[ ]*//g'`

if test -n "$EVENT_NAMES"; then
	for a_fn in $EVENT_NAMES; do
		EV_DATA=`echo "$EVENTS_DATA" | grep -h -i -A 50 "!ev:[ ]*$a_fn"`

		EV_DATA=`echo "$EV_DATA" | sed -e '1 d'`
		EV_DATA=`echo "$EV_DATA" | sed -e 's/^[	 ]*//'`
		EV_DATA=`echo "$EV_DATA" | grep -s -h "[a-zA-Z0-9{}]"`
 		EV_DATA=`echo "$EV_DATA" | sed -e '/![a-zA-Z]*:/ q'`
 		EV_DATA=`echo "$EV_DATA" | sed -e '/![a-zA-Z]*:/ d'`
 		EV_DATA=`echo "$EV_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 		EV_DATA=`echo "$EV_DATA" | sed -e '/@[a-zA-Z]*:/ d'`
		EV_DECLARATION=`echo "$EVENTS_DATA" | grep -h -i "!ev:[ ]*$a_fn[ ]*(" | sed -e 's/[	 ]*!ev:[ ]*//g'`
		if test -n "$EV_DATA"; then

EVENTS_TABLE="$EVENTS_TABLE
<tr><td>
<b>$EV_DECLARATION</b><br>
<blockquote>
$EV_DATA<br>
</blockquote>
</td></tr>"

		fi
	done
fi

if test -n "$EVENTS_TABLE"; then
	EVENTS_TABLE="<table bgcolor=\"#999999\">
$EVENTS_TABLE
</table>
"
fi





		 	# Extract the "@inherits:" part
		 	INHERITS_DATA=`echo "$DOCUMENT_DATA" | grep -h -i -A 50 "@inherits:"`
		 	INHERITS_DATA=`echo "$INHERITS_DATA" | sed -e '/@inherits:/ d'`
 			INHERITS_DATA=`echo "$INHERITS_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 			INHERITS_DATA=`echo "$INHERITS_DATA" | sed -e '/@[a-zA-Z]*:/ d'`
		 	# Extract the "@functions:" part
		 	FUNCTIONS_DATA=`echo "$DOCUMENT_DATA" | grep -h -i -A 150 "@functions:"`
		 	FUNCTIONS_DATA=`echo "$FUNCTIONS_DATA" | sed -e '/@functions:/ d'`
 			FUNCTIONS_DATA=`echo "$FUNCTIONS_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 			FUNCTIONS_DATA=`echo "$FUNCTIONS_DATA" | sed -e '/@[a-zA-Z]*:/ d'`

# Build the functions table

FUNCTIONS_TABLE="<table bgcolor="#999999">"

FUNCTION_NAMES=`echo "$FUNCTIONS_DATA" | grep -h -i "!fn:" | sed -e 's/[	 ]*!fn:[ ]*//g' | sed -e 's/[ ]*([ a-zA-Z,_&;\$0123456789=	\-]*)[ ]*//g'`

if test -n "$FUNCTION_NAMES"; then
	for a_fn in $FUNCTION_NAMES; do
		FN_DATA=`echo "$FUNCTIONS_DATA" | grep -h -i -A 50 "!fn:[ ]*$a_fn"`
		FN_DATA=`echo "$FN_DATA" | sed -e '1 d'`
		FN_DATA=`echo "$FN_DATA" | sed -e 's/^[	 ]*//'`
		FN_DATA=`echo "$FN_DATA" | grep -h "[a-zA-Z0-9{}]"`
 		FN_DATA=`echo "$FN_DATA" | sed -e '/![a-zA-Z]*:/ q'`
 		FN_DATA=`echo "$FN_DATA" | sed -e '/![a-zA-Z]*:/ d'`
 		FN_DATA=`echo "$FN_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 		FN_DATA=`echo "$FN_DATA" | sed -e '/@[a-zA-Z]*:/ d'`
		FN_DECLARATION=`echo "$FUNCTIONS_DATA" | grep -h -i "!fn:[ ]*$a_fn[ ]*(" | sed -e 's/[	 ]*!fn:[ ]*//g'`
		if test -n "$FN_DATA"; then
		FN_STRIPPED_NAME=`echo "$a_fn" | sed -e 's/^[\$]*//g' | sed -e 'y/QWERTYUIOPASDFGHJKLZXCVBNM/qwertyuiopasdfghjklzxcvbnm/'`
FUNCTIONS_TABLE="$FUNCTIONS_TABLE
<tr><td>
<b><a name=\"$FN_STRIPPED_NAME\">$FN_DECLARATION</a></b><br>
<blockquote>
$FN_DATA<br>
</blockquote>
</td></tr>"

		fi
	done
fi
FUNCTIONS_TABLE="$FUNCTIONS_TABLE
</table>"

		 	# Extract the "@signals:" part
		 	SIGNALS_DATA=`echo "$DOCUMENT_DATA" | grep -h -i -A 150 "@signals:"`
		 	SIGNALS_DATA=`echo "$SIGNALS_DATA" | sed -e '/@signals:/ d'`
 			SIGNALS_DATA=`echo "$SIGNALS_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 			SIGNALS_DATA=`echo "$SIGNALS_DATA" | sed -e '/@[a-zA-Z]*:/ d'`

# Build the signals table

SIGNALS_TABLE="<table bgcolor="#999999">"

SIGNAL_NAMES=`echo "$SIGNALS_DATA" | grep -h -i "!sg:" | sed -e 's/[	 ]*!sg:[ ]*//g' | sed -e 's/[ ]*([ a-zA-Z,_&;\$0123456789=	\-]*)[ ]*//g'`

if test -n "$SIGNAL_NAMES"; then
	for a_fn in $SIGNAL_NAMES; do
		SG_DATA=`echo "$SIGNALS_DATA" | grep -h -i -A 50 "!sg:[ ]*$a_fn"`
		SG_DATA=`echo "$SG_DATA" | sed -e '1 d'`
		SG_DATA=`echo "$SG_DATA" | sed -e 's/^[	 ]*//'`
		SG_DATA=`echo "$SG_DATA" | grep -h "[a-zA-Z0-9{}]"`
 		SG_DATA=`echo "$SG_DATA" | sed -e '/![a-zA-Z]*:/ q'`
 		SG_DATA=`echo "$SG_DATA" | sed -e '/![a-zA-Z]*:/ d'`
 		SG_DATA=`echo "$SG_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 		SG_DATA=`echo "$SG_DATA" | sed -e '/@[a-zA-Z]*:/ d'`
		SG_DECLARATION=`echo "$SIGNALS_DATA" | grep -h -i "!sg:[ ]*$a_fn[ ]*(" | sed -e 's/[	 ]*!sg:[ ]*//g'`
		if test -n "$SG_DATA"; then
		SG_STRIPPED_NAME=`echo "$a_fn" | sed -e 's/^[\$]*//g' | sed -e 'y/QWERTYUIOPASDFGHJKLZXCVBNM/qwertyuiopasdfghjklzxcvbnm/'`
SIGNALS_TABLE="$SIGNALS_TABLE
<tr><td>
<b><a name=\"$SG_STRIPPED_NAME\">$SG_DECLARATION</a></b><br>
<blockquote>
$SG_DATA<br>
</blockquote>
</td></tr>"

		fi
	done
fi
SIGNALS_TABLE="$SIGNALS_TABLE
</table>"


			EXAMPLES_DATA=`kvidoc_process_examples "$DOCUMENT_DATA"`

			# Extract the "@seealso:" part
		 	SEEALSO_DATA=`echo "$DOCUMENT_DATA" | grep -h -i -A 50 "@seealso:"`
		 	SEEALSO_DATA=`echo "$SEEALSO_DATA" | sed -e '/@seealso:/ d'`
		 	SEEALSO_DATA=`echo "$SEEALSO_DATA" | sed -e '/@[a-zA-Z]*:/ q'`
 			SEEALSO_DATA=`echo "$SEEALSO_DATA" | sed -e '/@[a-zA-Z]*:/ d'`

		 	# Create the main document
		 	DOCUMENT="<doctitle>Class $LOWER_NAME</doctitle>
<docbody>
<hr>
<docsubtitle>Inherits</docsubtitle>
<blockquote>
$INHERITS_DATA<br>
</blockquote>
<docsubtitle>Functions</docsubtitle>
<blockquote>
$FUNCTIONS_TABLE<br>
</blockquote>
"

if test -n "$EVENT_NAMES"; then
	DOCUMENT="$DOCUMENT
<docsubtitle>Events</docsubtitle>
<blockquote>
$EVENTS_TABLE<br>
</blockquote>
"
fi

if test -n "$SIGNAL_NAMES"; then
	DOCUMENT="$DOCUMENT
<docsubtitle>Signals</docsubtitle>
<blockquote>
$SIGNALS_TABLE<br>
</blockquote>
"
fi

DOCUMENT="$DOCUMENT
<docsubtitle>Description</docsubtitle>
<blockquote>
$DESCRIPTION_DATA<br>
</blockquote>
<docsubtitle>Examples</docsubtitle>
<blockquote>
$EXAMPLES_DATA<br>
</blockquote>
<docsubtitle>See also</docsubtitle>
<blockquote>
$SEEALSO_DATA<br>
</blockquote>
<hr>
<a href=\"index.kvihelp\">Main Index</a> <a href=\"classes.kvihelp\">Class Index</a>
</docbody>
"
			echo "$DOCUMENT" > $HELP_INSTALL_DIR/class_$LOWER_NAME.kvihelp

			if test -n "$DOC_LINKS"; then
				DOC_LINKS="$DOC_LINKS<br>
<tr><td><a href=\"class_$LOWER_NAME.kvihelp\">$LOWER_NAME</a></td><td>$SHORT_DATA</td></tr>"
			else
				DOC_LINKS="<tr><td><a href=\"class_$LOWER_NAME.kvihelp\">$LOWER_NAME</a></td><td>$SHORT_DATA</td></tr>"
			fi

		done
	fi
done

echo "### Generating $HELP_INSTALL_DIR/classes.kvihelp"

DOC_LINKS=`echo "$DOC_LINKS" | sort`

DOCUMENT="<doctitle>Classes index</doctitle>
<docbody>
<hr>
<table bgcolor=\"#999999\">
$DOC_LINKS
</table>
<hr>
<a href=\"index.kvihelp\">Main Index</a>, <a href=\"syntax_objects.kvihelp\">Objects documentation</a>
</docbody>"

echo "$DOCUMENT" > $HELP_INSTALL_DIR/classes.kvihelp

echo "################################################################################"
echo "### Generating misc docs..."
echo "################################################################################"

# First get the list of all the files containing @document
FILE_NAMES_KVIRC=`find src/kvirc/*.cpp`
FILE_NAMES_CORE=`find src/kvicore/*.cpp`
FILE_NAMES_LIBRARY=`find src/kvilib/*.cpp`
FILE_NAMES_PLUGINS=`find src/plugins/*/*.cpp`

FILE_NAMES="$FILE_NAMES_KVIRC $FILE_NAMES_CORE $FILE_NAMES_LIBRARY $FILE_NAMES_PLUGINS"

DOC_LINKS=""

for a_file in $FILE_NAMES; do
	DOC_NAMES=`grep -h -i "@document:" $a_file | sed -e 's/[	 ]*@document:[ ]*//g'`
	if test -n "$DOC_NAMES"; then
		for a_name in $DOC_NAMES; do
			LOWER_NAME=`echo "$a_name" | sed -e 'y/ABCDEFGHIJKLMNOPQRSTUVWXYZ/abcdefghijklmnopqrstuvwxyz/'`
			echo "### Generating $HELP_INSTALL_DIR/$LOWER_NAME"
			DOCUMENT_DATA=`grep -h -i -A 150 "@document:[ ]*$a_name" $a_file`
			# Now strip the first line and the lines after the end of the comment
			DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@document:[ ]*[a-zA-Z_]*/ d'`
			DOCUMENT_DATA=`kvidoc_process_document "$DOCUMENT_DATA"`
		 	DOCUMENT_TITLE=`echo "$DOCUMENT_DATA" | grep -i "@title:"`
		 	DOCUMENT_TITLE=`echo "$DOCUMENT_TITLE" | sed -e 's/[	 ]*@title:[ ]*//g'`
			DOCUMENT_DATA=`echo "$DOCUMENT_DATA" | sed -e '/@title:[ ]*[a-zA-Z_ ]*/ d'`
		 	# Create the main document
		 	DOCUMENT="<doctitle>$DOCUMENT_TITLE</doctitle>
<docbody>
<hr>
$DOCUMENT_DATA
<hr>
<a href=\"index.kvihelp\">Main Index</a> <a href=\"misc.kvihelp\">Misc. Documentation Index</a>
</docbody>
"
			echo "$DOCUMENT" > $HELP_INSTALL_DIR/$LOWER_NAME

			if test -n "$DOC_LINKS"; then
				DOC_LINKS="$DOC_LINKS<br>
<a href=\"$LOWER_NAME\">$DOCUMENT_TITLE</a>"
			else
				DOC_LINKS="<a href=\"$LOWER_NAME\">$DOCUMENT_TITLE</a>"
			fi

		done
	fi
done

echo "### Generating $HELP_INSTALL_DIR/misc.kvihelp"

DOCUMENT="<doctitle>Misc documentation index</doctitle>
<docbody>
<hr>
$DOC_LINKS<br>
<hr>
<a href=\"index.kvihelp\">Main Index</a>
</docbody>"

echo "$DOCUMENT" > $HELP_INSTALL_DIR/misc.kvihelp


echo "################################################################################"
echo "### KVIRC DOCUMENTATION INSTALLATION COMPLETE"
echo "################################################################################"
echo "###"
echo "### Mailing list and CVS information can be obtained from"
echo "###          http://www.kvirc.net/"
echo "###"
echo "### We hope that you will enjoy KVIrc."
echo "###"
echo "### - Szymon Stefanek and the KVIrc Development Team"
echo "###"
echo "################################################################################"
