//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007,2008 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2006 Martin Albrecht <malb@informatik.uni-bremen.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
//
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <cstdlib>
#include <iostream>
#include <assert.h>

#include "silcprotocol.h"
#include "silcaccount.h"
#include "silcbuddycontact.h"
#include "silcchannelcontact.h"
#include "silcbuddycontactinfowidget.h"
#include "silcfiletransfer.h"
#include "silcbuddyattributes.h" 
#include "silcbuddycliententry.h"
//#include "silcmessagemanager.h"

#include <kopetemetacontact.h>
#include <kopetechatsession.h>
#include <kopeteuiglobal.h>

#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>
#include <kaction.h>
#include <kactionmenu.h>
#include <ktoggleaction.h>
#include <kinputdialog.h>

// sorry for this hack, unfortunately we need it for
// the macros of recent libsilc to work ...
typedef SilcTK::SilcUInt8 SilcUInt8;


SilcBuddyContact::SilcBuddyContact(SilcAccount *account,
				   const QString &nick, 
                                   const QString &fingerprint,
				   Kopete::MetaContact *meta,
				   const QString &icon)
  : SilcContact(account, QString("@%1").arg(fingerprint), meta, icon),
    _fpTrusted(false), 
    _watched(false), 
    _allowRichText(false),
    _channels(), 
    _activeManager(NULL), 
    _actionIsOp(NULL),
    _actionIsQuiet(NULL),
    _actionKick(NULL),
    _actionClientSelectMenu(NULL)
{
  _attributes = new SilcBuddyAttributes(this);
  _clientEntry = new SilcBuddyClientEntry(this);

  setNickName(nick);
  setFileCapable(true);

  QObject::connect
    (this, SIGNAL(onlineStatusChanged(Kopete::Contact *,
				      const Kopete::OnlineStatus &,
				      const Kopete::OnlineStatus &)),
     this, SLOT(slotOnlineStatusChanged(Kopete::Contact *,
					const Kopete::OnlineStatus &,
					const Kopete::OnlineStatus &)));

  QObject::connect
    (this, SIGNAL(propertyChanged(Kopete::PropertyContainer *, const QString &,
				  const QVariant &, const QVariant &)),
     this, SLOT(slotPropertyChanged(Kopete::PropertyContainer *,
				    const QString &, const QVariant &,
				    const QVariant &)));
}


SilcBuddyContact::~SilcBuddyContact() {
  this->watchme(false);

  delete _attributes;
  delete _clientEntry;
}

void
SilcBuddyContact::slotSendMessage(Kopete::Message &msg,
				  Kopete::ChatSession *session)
{
  if(session != manager()) return;

  SilcAccount *account = static_cast<SilcAccount *>(this->account());

  if(! account->conn()) {
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, 
       i18n("Unable to send this message now. The protocol is currently "
	    "offline and does not support offline sending."),
       i18n( "User is Not Reachable"));
    return;
  }

  if(onlineStatus() == SilcProtocol::protocol()->statusDetached) {
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
       i18n("This message cannot be sent right now, since the remote client "
	    "is currently detached and the protocol does not support "
	    "offline sending."),
       i18n("User is Not Reachable"));
    return;
  }

  // get plain text message ...
  SilcTK::SilcMessageFlags flags = SILC_MESSAGE_FLAG_UTF8;
  unsigned char *buf = NULL;
  SilcTK::SilcUInt32 buflen = 0;
  QByteArray plaintext;

  if(account->signPrivateMessages())
    flags |= SILC_MESSAGE_FLAG_SIGNED;

  if(allowRichText()) {
    SilcTK::SilcMime mime = getMessageAsMime(msg);
    buf = SilcTK::silc_mime_encode(mime, &buflen);
    SilcTK::silc_mime_free(mime);

    flags |= SILC_MESSAGE_FLAG_DATA;
  }
  else {
    plaintext = msg.plainBody().toUtf8();
    buf = (unsigned char *) plaintext.constData();
    buflen = plaintext.length();

    // use of rich text is forbidden, reset message to plain
    // (so our channel log doesn't show any markup as well)
    msg.setPlainBody(msg.plainBody());
  }

  prettyPrintMessage(msg, flags);

  // pass message to libsilc ...
  SilcTK::silc_client_send_private_message
    (account->client(), account->conn(), clientEntries()(), flags,
     account->sha1hash, buf, buflen);     

  // append message locally ...
  session->appendMessage(msg);
  session->messageSucceeded();

  // reset idle time of ourself
  account->myself()->setIdleTime(1);
}



void 
SilcBuddyContact::silc_private_message(SilcTK::SilcClient client,
				       SilcTK::SilcClientConnection,
				       SilcTK::SilcClientEntry sender,
				       SilcTK::SilcMessagePayload payload,
				       SilcTK::SilcMessageFlags flags,
				       const unsigned char *message,
				       SilcTK::SilcUInt32 message_len)
{
  SilcAccount *account = static_cast<SilcAccount *>(client->application);
  SilcBuddyContact *buddy = (SilcBuddyContact *) sender->context;
  
  if(! buddy)
    buddy = account->contactManager()->createBuddy
      (sender->nickname, NULL, sender);

  if(! buddy) {
    std::cerr << "unable to allocate new buddy instance" << std::endl;
    return;
  }

  // reset idletime of sender
  buddy->setIdleTime(1);

  // If the messages is digitally signed, verify it, if possible.
  SignatureStatus sigstat = Unknown;
  if (flags & SILC_MESSAGE_FLAG_SIGNED)
    sigstat = buddy->verifySignature(payload);

  // make sure there's a ChatSession, then add ourself ...
  Kopete::ChatSession *session = buddy->manager(Kopete::Contact::CanCreate);
  session->addContact(buddy, Kopete::OnlineStatus::Online);

  // convert Utf8 stuff depending on whether SILC_MESSAGE_FLAG_UTF8 is set
  QString text;
  if(flags & SILC_MESSAGE_FLAG_UTF8)
    text = QString::fromUtf8((const char *) message, message_len);
  else if(flags & SILC_MESSAGE_FLAG_DATA);
  else
    text = QString::fromLatin1((const char *) message, message_len);

  Kopete::Message msg;
  if(flags & SILC_MESSAGE_FLAG_NOTICE) {
    msg = Kopete::Message(buddy, account->myself());
    msg.setPlainBody(QString("%1 -*- %2").arg(buddy->nickName()).arg(text));
    msg.setDirection(Kopete::Message::Internal);
    msg.setType(Kopete::Message::TypeAction);
  }


  else if(flags & SILC_MESSAGE_FLAG_DATA) {
    /* SilcMimeMessage */
    QStringList *filenames;
    SilcTK::SilcMime tmp = SilcTK::silc_mime_decode(NULL, message, message_len);
    /* Assemble mime partials */
    SilcTK::SilcMime mime = buddy->mime_asm(tmp);
    if (!mime) return;

    QString type = SilcTK::silc_mime_get_field(mime, "Content-Type");
    if(type.isEmpty()) goto mimeout;

    if (type.left(21).compare("multipart/alternative") == 0) {
      msg = Kopete::Message(buddy, account->myself());
      msg.setDirection(Kopete::Message::Inbound);
      msg.setType(Kopete::Message::TypeNormal);
      buddy->mimeAlternateToMsg(msg, mime, buddy->allowRichText());
      session->appendMessage(msg);
    }
    else {
      filenames = buddy->saveMime(mime);
      for(QStringList::Iterator it = filenames->begin(); 
	  it != filenames->end(); ++it ) {
        msg = Kopete::Message(buddy, account->myself());
	msg.setHtmlBody(buddy->mimeDisplayMessage(*it));
	msg.setDirection(Kopete::Message::Inbound);
	msg.setType(Kopete::Message::TypeNormal);

        prettyPrintMessage(msg, flags, sigstat);
        session->appendMessage(msg);
      }
      delete filenames;
    }

  mimeout:
    SilcTK::silc_mime_free(mime);
    goto maybecreateview;
  }
  else {
    msg = Kopete::Message(buddy, account->myself());
    msg.setPlainBody(text);
    msg.setDirection(Kopete::Message::Inbound);
    msg.setType((flags & SILC_MESSAGE_FLAG_ACTION)
		? Kopete::Message::TypeAction
		: Kopete::Message::TypeNormal);
  }

  prettyPrintMessage(msg, flags, sigstat);
  session->appendMessage(msg);

 maybecreateview:
  // make sure there's a view associated with this ChatSession ...
  buddy->view();

  // force online status online, in case buddy was marked offline - we just got
  // a message, therefore he or she needs to be online :-)
  //
  // FIXME, this shouldn't be necessary actually, implement some slotConnected
  if(buddy->onlineStatus() == SilcProtocol::protocol()->statusOffline)
    buddy->setOnlineStatus(SilcProtocol::protocol()->statusOnline);
}


void 
SilcBuddyContact::serialize(QMap<QString, QString> &serializedData,
			    QMap<QString, QString>&)
{
  serializedData["fpTrusted"] = fpTrusted() ? "yes" : "no";
  serializedData["allowRichText"] = allowRichText() ? "yes" : "no";
}

/**
 * note: if we trust some fingerprint we gotta have it's key
 */

void 
SilcBuddyContact::setFpTrusted(bool trust)
{
  if(trust) {
    if( !havePublicKey() ) {
      account()->sendSilcCommand( QString("GETKEY %1").arg(nickName()) );
    }
  }
  _fpTrusted = trust;
}

QString
SilcBuddyContact::convFingerprint(const char *fp)
{
  QString qfp = QString::null;
  //asserting 20 byte of valid data
  for(int i = 0; i<20 ; fp ++) {
    qfp += ((*fp >> 4) & 0xF) > 9 
      ? ((*fp >> 4) & 0xF) + 'A' - 10
      : ((*fp >> 4) & 0xF) + '0';
    qfp += (*fp & 0xF) > 9 ? (*fp & 0xF) + 'A' - 10 : (*fp & 0xF) + '0';
    if((++i % 2) == 0 && i!=20 ) qfp += ':'; //&& fp[1]
    if(i == 10) qfp += ':';
  }

  return qfp;
}


void
SilcBuddyContact::slotIsOp(void)
{
  if(! _activeManager) return;

  Kopete::ContactPtrList members = _activeManager->members();
  SilcChannelContact *ch = static_cast<SilcChannelContact *>(members.first());

  ch->setOp(this, _actionIsOp->isChecked());
}


void
SilcBuddyContact::slotIsQuiet(void)
{
  if(! _activeManager) return;

  Kopete::ContactPtrList members = _activeManager->members();
  SilcChannelContact *ch = static_cast<SilcChannelContact *>(members.first());

  ch->setQuiet(this, _actionIsQuiet->isChecked());
}


void
SilcBuddyContact::slotKick(void)
{

  if(! _activeManager) return;

  Kopete::ContactPtrList members = _activeManager->members();
  SilcChannelContact *ch = static_cast<SilcChannelContact *>(members.first());

  bool performKick = true;
  QString kickMsg = KInputDialog::getText
    (i18n("kick message for '%1'", nickName()),
     i18n("Please enter a kick message to send along with the kick of '%1'. "
	  "Click 'Cancel' if you don't want to kick the buddy.", nickName()),
     QString(), &performKick);
  
  if(performKick)
    ch->kick(this, kickMsg);
}

QList<KAction *> *
SilcBuddyContact::customContextMenuActions(Kopete::ChatSession *manager)
{
  _activeManager = manager; 

  QList<KAction*> *actions = new QList<KAction*>();

  if(! _actionClientSelectMenu)
    _actionClientSelectMenu =
      new KActionMenu(KIcon("connect_no"), i18n("Select Client Entry"), this);
  
  if(! _actionIsOp) {
    _actionIsOp = new KToggleAction(this);
    _actionIsOp->setText(i18n("&Operator"));
    connect(_actionIsOp, SIGNAL(triggered(bool)), SLOT(slotIsOp()));
  }

  if(! _actionIsQuiet) {
    _actionIsQuiet = new KToggleAction(this);
    _actionIsQuiet->setText(i18n("&Quiet"));
    connect(_actionIsQuiet, SIGNAL(triggered(bool)), SLOT(slotIsQuiet()));
  }
  
  if(! _actionKick) {
    _actionKick = new KAction(this);
    _actionKick->setText(i18n("&Kick from Channel"));
    connect(_actionKick, SIGNAL(triggered(bool)), SLOT(slotKick()));
  }

  SilcChannelContact *channel = NULL;

  if(manager) {
    Kopete::ContactPtrList members = manager->members();
    
    if(! strcmp(members.first()->metaObject()->className(),
		"SilcChannelContact"))
      channel = static_cast<SilcChannelContact *>(members.first());
  }

  SilcBuddyContact *me = static_cast<SilcBuddyContact *>(account()->myself());
  bool userIsOp = channel && channel->isOp(me);

  _actionIsOp->setEnabled(userIsOp
			  || (channel && channel->isFounder (me)));
  _actionIsOp->setChecked(channel && channel->isOp(this));

  _actionIsQuiet->setEnabled(userIsOp && me != this);
  _actionIsQuiet->setChecked(channel && channel->isQuiet(this));

  _actionKick->setEnabled(userIsOp && me != this);

  actions->append(_actionIsOp);
  actions->append(_actionIsQuiet);
  actions->append(_actionKick);

  /* The client entry select */
  foreach(KAction *action, _actionClientSelect) 
    _actionClientSelectMenu->remove(action);
  _actionClientSelect.clear();

  SilcTK::SilcClientEntry active_entry = clientEntries()();

  SilcTK::SilcClientEntry entry;
  int i;
  if (clientEntries().size() > 1) {
    for(i = 0; i < clientEntries().size(); i++) {
      entry = clientEntries()[i];

      QString status;
      if (entry->mode & (SILC_UMODE_DETACHED))
	status = i18n ("Detached");

      else if (entry->mode & (SILC_UMODE_GONE))
	status = i18n ("Gone");

      else if (entry->mode & (SILC_UMODE_BUSY))
	status = i18n ("Busy");

      else if (entry->mode & (SILC_UMODE_HYPER))
	status = i18n ("Hyper Active");

      else if (entry->mode & (SILC_UMODE_INDISPOSED))
	status = i18n ("Indisposed");

      else
	status = i18n ("Online");

      KToggleAction *action = new KToggleAction(this);
      action->setText(QString::fromLatin1("%1 (%2@%3, %4)")
		      .arg(entry->nickname)
		      .arg(entry->username).arg(entry->hostname)
		      .arg(status));
      action->setObjectName(QString(i));
      if(entry == active_entry)
	action->setChecked(true);

      connect(action, SIGNAL(triggered(bool)), SLOT(slotSelectClientEntry()));

      
      _actionClientSelect.append(action);
      _actionClientSelectMenu->addAction(action);
    }
    actions->append(_actionClientSelectMenu);
  }
  
  return actions;
}

void 
SilcBuddyContact::slotSelectClientEntry(void)
{

  kDebug() << "tooltip" << toolTip() << endl;
  int currentItem, i;
  for(i = 0; i < _actionClientSelect.size(); i++) 
    if (_actionClientSelect[i] == sender()) {
      currentItem = i;
      break;
    }
  kDebug() << "client entry selected:" << currentItem << endl;
  clientEntries().setActive(currentItem);
}


void
SilcBuddyContact::slotOnlineStatusChanged(Kopete::Contact *,
                                          const Kopete::OnlineStatus &status,
                                          const Kopete::OnlineStatus & /*old*/)
{
  if(status == SilcProtocol::protocol()->statusOffline
     || status.status() == Kopete::OnlineStatus::Unknown) {
    // assume we're offline, thus left all channels ...
    clientEntries().clear();
  }
}


void 
SilcBuddyContact::sendFile(const KUrl &sourceURL,
			   const QString & /* fileName */, uint /* fileSize */)
{
  QString filePath;

  if(! sourceURL.isValid())
    filePath = KFileDialog::getOpenFileName(KUrl(), "*", 0L,
					    i18n("Kopete File Transfer"));
  else
    filePath = sourceURL.path();

  QFile file(filePath);
  if(! file.exists())
    return;

  if(account()->useSilcMime()) {
    if (file.size() < (23 << (5 + (2 * 3)))) 
      sendFileAsMime(filePath);
    else {
        int answer = KMessageBox::questionYesNo
          (Kopete::UI::Global::mainWidget(),
           QString(i18n("You are trying to send a big file via Silc"
                        "MIME message. Do you want to send it via Silc"
                        "Filetransfer?")),
           i18n("Sending MIME message"));
        if (answer == KMessageBox::Yes) 
          new SilcFileTransfer(account(), this, filePath);
        else
          sendFileAsMime(filePath);
    }
  }
  else 
    new SilcFileTransfer(account(), this, filePath);
}


void 
SilcBuddyContact::slotUserInfo(void)
{
  new SilcBuddyContactInfoWidget(this);;
}

/**
 * @short verify the validity of the signature on the provided message
 * @todo we should have a locally stored public key cache
 */
SilcContact::SignatureStatus 
SilcBuddyContact::verifySignature(SilcTK::SilcMessagePayload message)
{
  // get public key from the signature payload and compare it with the
  // one stored in the client entry
  const unsigned char *pk_data;
  SilcTK::SilcUInt32 pk_datalen;
  SilcTK::SilcPublicKey pk = 
     silc_message_signed_get_public_key(message, &pk_data, &pk_datalen);

  SilcContact::SignatureStatus retval;
  QString fp;
  if(pk) {
    // extract the fingerprint from the signatures public key
    char *fpKey = SilcTK::silc_hash_fingerprint(NULL, pk_data, pk_datalen);
    fp = QString(fpKey).replace(QChar(' '), QChar(':'));
    free(fpKey);

    // compare the stored fingerprint with the received key's one
    if(fp.compare(fingerprint()) == 0)
      retval = fpTrusted() ? Trusted : Valid;
    else
      retval = Unknown;
  }
  
  else {
    fp = fingerprint();
    retval = fpTrusted() ? Trusted : Valid;
  }

  QString fn = publicKeyPath(fp);
  if(QFile::exists(fn)) {
    if(pk) SilcTK::silc_pkcs_public_key_free(pk);
    if (! SilcTK::silc_pkcs_load_public_key(fn.toLatin1().constData(), &pk) &&
	! SilcTK::silc_pkcs_load_public_key(fn.toLatin1().constData(), &pk))
      return Unknown;
  }
  
  // if we don't have a public key, we simply cannot verify ...
  if(! pk) return Unknown;
  
  // check whether the signature is valid
  if(SilcTK::silc_message_signed_verify
     (message, pk, account()->sha1hash) != SILC_AUTH_OK)
    retval = Failed;

  SilcTK::silc_pkcs_public_key_free(pk);
  return retval;
}

/**
 * returns the path to look for the clients public key.
 *
 * The returned filename is of the form 
 @verbatim
  $kopete_appdata/kopete_silc_clientkeys/clientkey_XXXX_XXXX_XXXX_XXXX_XXXX__XXXX_XXXX_XXXX_XXXX_XXXX.pub
 @endverbatim
 *
 * @param fp provide a fingerprint
 */

const QString 
SilcBuddyContact::publicKeyPath(const QString fp) 
{
  QString pkfile = QString("kopete_silc_clientkeys/clientkey_%1.pub")
    .arg(QString(fp).replace(":","_"));
  QString fn = KStandardDirs::locateLocal("appdata", pkfile);

  return fn;
}

/**
 * Behaves like the above function but receives fingerprint from this buddy.
 */

const QString 
SilcBuddyContact::publicKeyPath(void) const {
  return SilcBuddyContact::publicKeyPath(this->fingerprint());
}

/**
 * performs a WHOIS request for a buddy.
 *
 * If possible a known public key is used. 
 *
 * @returns true if has be called with public key and false otherwise
 */

bool 
SilcBuddyContact::whoami(void) 
{
  QString fp = publicKeyPath();
  if(QFile::exists(fp)) {
    account()->sendSilcCommand(QString("WHOIS -pubkey %1 -details").arg(fp));
    return true;
  }
  else {
    account()->sendSilcCommand(QString("WHOIS %1 -details").arg(nickName())); 
    return false;
  }
}

/**
 * @brief either add or remote the buddy from the watch list
 *
 * @return whether public keys are used for watching
 */
 /*
 * This seems to be wrong, pubkey watch of offline buddies and libsilc
 * 1.0.3 works 
 *
 *   We rely on always watching using the nickname, i.e. never use the 
 *   public key for watching purposes. This is simply because we cannot
 *   watch offline buddies using public key approach ...
 *
 *   ... and first testing whether the buddy is offline before setting the
 *   suitable watch is painful.
 *
 *   @return whether public keys are used for watching, i.e. always false
 */
bool
SilcBuddyContact::watchme(bool watch)
{ 
  QString fp = this->publicKeyPath();
  this->_watched = watch;

  if(! watch) {
    if(QFile::exists(fp)) {
      account()->sendSilcCommand(QString("WATCH -pubkey -%1").arg(fp),
                                 (SilcTK::SilcClientCommandReply) 
                                 watchme_callback, 
                                 this);
      return true;

    } else {
      // @fixme check whether there is another buddy with this
      // nickname, which we might disable watching for, ...
      account()->sendSilcCommand(QString("WATCH -del %1").arg(nickName()),
                                 (SilcTK::SilcClientCommandReply)
                                 watchme_callback, 
                                 this);
      return false;
    }
  } else {
    if(QFile::exists(fp)) {
     account()->sendSilcCommand(QString("WATCH -pubkey +%1").arg(fp),
                                (SilcTK::SilcClientCommandReply)
                                watchme_callback, 
                                this);
     return true;

    } else {
      account()->sendSilcCommand(QString("WATCH -add %1").arg(nickName()),
                                 (SilcTK::SilcClientCommandReply)
                                 watchme_callback, 
                                 this);
      return false;
    }
  }
}


/*
 * @brief callback for buddy->watchme to retry the watch, if it fails
 */
bool
SilcBuddyContact::watchme_callback(SilcTK::SilcClient,
                 SilcTK::SilcClientConnection,
                 SilcTK::SilcCommand,
                 SilcTK::SilcStatus status,
                 SilcTK::SilcStatus error,
                 void *context,
                 va_list)
{
  SilcBuddyContact *buddy = (SilcBuddyContact *) context;
  if(status != SILC_STATUS_OK && ( error == SILC_STATUS_ERR_NO_SUCH_CLIENT_ID
                                   || error == SILC_STATUS_ERR_TIMEDOUT))
    buddy->watchme(buddy->watched());
  return false;
}


/**
* Updates the local cache of WHOIS information which may be displayerd
* via the user information interface.
*
*/

void
SilcBuddyContact::updateWhois(QString username, QString realname) {
  this->_username=username;
  this->_realname=realname;
  this->_update=QDateTime::currentDateTime();
  emit signalWhois(this->nickName(),username,realname);
}


/**
 * returns true if a public key matching a given fingerprint is found locally.
 */
bool 
SilcBuddyContact::havePublicKey(QString fp) 
{
  return QFile::exists(SilcBuddyContact::publicKeyPath(fp));
}

/**
 * behaves essentially as the above function but receives the fingerprint from
 * this buddy.
 */

bool
SilcBuddyContact::havePublicKey(void) {
  return SilcBuddyContact::havePublicKey(this->fingerprint());
}


/**
 * make sure to update /watch status in case of nickname changes
 */
void
SilcBuddyContact::slotPropertyChanged(Kopete::PropertyContainer *contact,
				      const QString &key,
				      const QVariant &oldValue,
				      const QVariant &)
{
  if(contact != this) return;
  if(key.compare("nickName")) return;

  if(! _watched) return;

  // @fixme check whether there is another buddy with this nickname, which
  // we might disable watching for, ...
  if(! oldValue.toString().isNull())
    account()->sendSilcCommand(QString("WATCH -del %1")
			       .arg(oldValue.toString()));
  watchme(true);
}


void
SilcBuddyContact::sendFileAsMime(const QString &fileName) 
{
  int chunks = 0;
  SilcTK::SilcBuffer buffer;
  QFile file(fileName);
  Kopete::ChatSession *session = manager(Kopete::Contact::CanCreate);

  /* Sending Chunks */
  SilcTK::SilcDList parts = getFileAsMime(fileName);
  SilcTK::silc_dlist_start(parts);
  while ((buffer = (SilcTK::SilcBuffer)SilcTK::silc_dlist_get(parts)) != SILC_LIST_END) { 
    chunks++;
    SilcTK::silc_client_send_private_message
      (account()->client(), account()->conn(), clientEntries()(), 
       SILC_MESSAGE_FLAG_DATA, account()->sha1hash, 
       (unsigned char*)buffer->data, (buffer->end - buffer->data));
  }
  SilcTK::silc_mime_partial_free(parts);

  
  Kopete::Message msg = Kopete::Message(account()->myself(), this);
  msg.setHtmlBody(account()->myself()->mimeDisplayMessage(fileName, chunks));
  msg.setDirection(Kopete::Message::Outbound);
  msg.setType(Kopete::Message::TypeNormal);
  

  session->appendMessage(msg);
  session->messageSucceeded();
}

void 
SilcBuddyContact::mimeAlternateToMsg( Kopete::Message &msg, 
                                      SilcTK::SilcMime mime,
                                      bool allowRichText) const
{
  SilcTK::SilcDList parts = SilcTK::silc_mime_get_multiparts(mime, NULL);
  SilcTK::SilcMime part;
  QString type, text, html;

  SilcTK::silc_dlist_start(parts);
  while ((part = (SilcTK::SilcMime)SilcTK::silc_dlist_get(parts)) != SILC_LIST_END) {
    type = SilcTK::silc_mime_get_field(part, "Content-Type");
    if(type.left(10).compare("text/plain") == 0) {
      if (type.contains("utf-8"))
        text = QString::fromUtf8((char *)SilcTK::silc_mime_get_data(part, NULL));
      else
        text = QString::fromLatin1((char *)SilcTK::silc_mime_get_data(part, NULL));
    }
    else if(type.left(9).compare("text/html") == 0) {
      if (type.contains("utf-8"))
        html = QString::fromUtf8((char *)SilcTK::silc_mime_get_data(part, NULL));
      else
        html = QString::fromLatin1((char *)SilcTK::silc_mime_get_data(part, NULL));
    }
  }
  if(! allowRichText || html.isEmpty())
    /* Use text */
    msg.setPlainBody(text);
  else 
    msg.setHtmlBody(html);
}

SilcBuddyContactData::SilcBuddyContactData(SilcAccount * account, 
                                                QString nickName,
                                                QString finger,
                                                Kopete::MetaContact *meta)
 :nickname(nickName),finger(finger),meta(meta),account(account)
{
  return; 
}

#include "silcbuddycontact.moc"
