// --------------------------------------------------------------------
// Interface for Qt and Ipelib
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeq.h"
#include "ipestyle.h"

#include <QFile>

// --------------------------------------------------------------------

/*! \class IpeQStream
  \brief Stream writing into an open QIODevice.
*/

//! Constructor.
IpeQStream::IpeQStream(QIODevice *dev)
  : iDevice(dev), iBuffer(BUFSIZE), iPos(0)
{
  // nothing
}

void IpeQStream::PutChar(char ch)
{
  put(ch);
}

void IpeQStream::PutString(IpeString s)
{
  for (int i = 0; i < s.size(); ++i)
    put(s[i]);
}

void IpeQStream::PutCString(const char *s)
{
  while (*s)
    put(*s++);
}

void IpeQStream::PutRaw(const char *data, int size)
{
  while (size-- > 0)
    put(*data++);
}

int IpeQStream::Tell() const
{
  return iDevice->pos() + iPos;
}

void IpeQStream::flush()
{
  if (iPos > 0)
    iDevice->write(iBuffer.data(), iPos);
  iPos = 0;
}

// --------------------------------------------------------------------

/*! \class QStringDataSource
  \brief IpeDataSource reading from a QString.
*/

//! Construct XML source.
QStringDataSource::QStringDataSource(QString str)
{
  iData = IpeQ(str);
  iPos = 0;
}

//! Get one character.
int QStringDataSource::GetChar()
{
  if (iPos < iData.size())
    return iData[iPos++];
  else
    return EOF;
}

// --------------------------------------------------------------------

/*! \class XmlQSource
  \brief Data source for parsing XML from a QIODevice.
*/

QDataSource::QDataSource(QIODevice *dev, bool skipPercent)
  : iSkipPercent(skipPercent), iDev(dev), iBuffer(BUFSIZE),
    iPos(0), iBytes(0)
{
  // nothing
}

int QDataSource::get()
{
  if (iPos >= iBytes) {
    iBytes = iDev->read(iBuffer.data(), BUFSIZE);
    if (iBytes <= 0)
      return EOF;
    iPos = 0;
  }
  int ch = iBuffer[iPos++];
  return ch;
}

int QDataSource::GetChar()
{
  char ch = get();
  if (iSkipPercent && ch == '\n')
    ch = get();
  return ch;
}

void QDataSource::reset()
{
  iDev->reset();
  iPos = 0;
  iBytes = 0;
}

// --------------------------------------------------------------------
