/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gdk/gl/pixmap.h
//! @brief A GdkGLPixmap C++ wrapper interface.
//!
//! Provides Pixmap, an OpenGL pixmap which is maintained off-screen.

#ifndef INTI_GDK_GL_PIXMAP_H
#define INTI_GDK_GL_PIXMAP_H

#ifndef INTI_GDK_DRAWABLE_H
#include <inti/gdk/drawable.h>
#endif

#ifndef INTI_GDK_GL_DRAWABLE_H
#include <inti/gdk/gl/drawable.h>
#endif

#ifndef __GDK_GL_PIXMAP_H__
#include <gdk/gdkglpixmap.h>
#endif

namespace Inti {

namespace Gdk {

class Pixmap;

namespace GL {

class Config;
class PixmapClass;

//! @class Pixmap pixmap.h inti/gdk/gl/pixmap.h
//! @brief A GdkGLPixmap C++ wrapper class.
//!
//! Pixmap represents an OpenGL pixmap which is maintained off-screen.

class Pixmap : public Gdk::Drawable, public Gdk::GL::Drawable
{
	friend class G::Object;
	friend class PixmapClass;

	Pixmap(const Pixmap&);
	Pixmap& operator=(const Pixmap&);

protected:
//! @name Constructors
//! @{

	explicit Pixmap(GdkGLPixmap *pixmap, bool reference = true);
	//!< Construct a new Pixmap from an existing GdkGLPixmap.
	//!< @param pixmap A pointer to a GdkGLPixmap.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>pixmap</EM> can be a newly created GdkGLPixmap or an existing
	//!< GdkGLPixmap. (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	virtual ~Pixmap();
	//!< Destructor.

//! @}

	static Pointer<Pixmap> create(const Config& glconfig, const Gdk::Pixmap& pixmap, const int *attrib_list = 0);
	//!< Creates an off-screen rendering area.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param pixmap The Gdk::Pixmap to be used as the rendering area.
	//!< @param attrib_list This must be set to null or empty (first attribute of none).
	//!< @return The new Gdk::GL::Pixmap.
	//!<
	//!< <BR>The <EM>attrib_list</EM> argument is currently unused. This must be set to
	//!< null or empty (first attribute of none, see GLX 1.3 spec).

	static Pixmap* get_gl_pixmap(Gdk::Pixmap& pixmap);
	//!< Gets the Gdk:GL::Pixmap held by <EM>pixmap</EM>.
	//!< @param pixmap A Gdk::Pixmap.
	//!< @return The Gdk::GL::Pixmap.

	static Gdk::GL::Drawable* get_gl_drawable(Gdk::Pixmap& pixmap);
	//!< Gets the Gdk::GL::Drawable held by the pixmap.
	//!< @param pixmap A Gdk::GL::Pixmap.
	//!< @return The Gdk::GL::Drawable.
	//!<
	//!< <BR>This method returns the result of get_gl_pixmap() cast to a Gdk::GL::Drawable.

	static bool is_gl_capable(Gdk::Pixmap& pixmap);
	//!< Determines whether the pixmap is OpenGL-capable.
	//!< @param pixmap A Gdk::Pixmap.
	//!< @return <EM>true</EM> if the pixmap is OpenGL-capable, <EM>false</EM> otherwise.

	static Pointer<Pixmap> set_gl_capability(Gdk::Pixmap& pixmap, const Config& glconfig, const int *attrib_list = 0);
	//!< Set the OpenGL-capability for <EM>pixmap</EM>.
	//!< @param pixmap The Gdk::Pixmap to be used as the rendering area.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param attrib_list This must be set to null or empty (a first attribute of none).
	//!< @return The Gdk::GL::Pixmap used by the pixmap if it is successful, null otherwise.
	//!<
	//!< <BR>This method creates a new Gdk::GL::Pixmap held by <EM>pixmap</EM>. The <EM>attrib_list</EM>
	//!< argument is currently unused. This must be set to null or empty (a first attribute of none).

	static void unset_gl_capability(Gdk::Pixmap& pixmap);
	//!< Unset the OpenGL-capability for <EM>pixmap</EM>.
	//!< @param pixmap The Gdk::Pixmap that was used as the rendering area.
	//!<
	//!< <BR>This method destroys the Gdk::GL::Pixmap held by the pixmap.

//! @name Accessors
//! @{

	GdkGLPixmap* gdk_gl_pixmap() const { return (GdkGLPixmap*)instance; }
	//!< Get a pointer to the GdkGLPixmap structure.

	GdkGLPixmapClass* gdk_gl_pixmap_class() const;
	//!< Get a pointer to the GdkGLPixmapClass structure.

	operator GdkGLPixmap* () const;
	//!< Conversion operator; safely converts a Pixmap to a GdkGLPixmap pointer.

	Gdk::Pixmap* get_pixmap() const;
	//!< Gets the Gdk::Pixmap associated with the <EM>pixmap</EM>.
	//!< @return The Gdk::Pixmap associated with the pixmap.
	//!<
	//!< <BR>Notice that Gdk::GL::Pixmap is not Gdk::Pixmap, but another Gdk::Drawable
	//!< which has an associated Gdk::Pixmap.

//! @}
//! @name Methods
//! @{

	virtual void dispose();
	//!< Disposes of the OpenGL resources associated with the pixmap
	//!< and decrements pixmap's reference count.

//! @}
};

} // namespace GL

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_GL_PIXMAP_H

