/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gdk/gl/drawable.h
//! @brief A GdkGLDrawable C++ wrapper interface.
//!
//! Provides Drawable, an OpenGL rendering surface interface.

#ifndef INTI_GDK_GL_DRAWABLE_H
#define INTI_GDK_GL_DRAWABLE_H

#ifndef INTI_G_TYPE_H
#include <inti/glib/type.h>
#endif

#ifndef __GDK_GL_DRAWABLE_H__
#include <gdk/gdkgldrawable.h>
#endif

namespace Inti {

namespace Gdk {

namespace GL {

class Config;
class Context;
class DrawableClass;

//! @class Drawable drawable.h inti/gdk/gl/drawable.h
//! @brief A GdkGLDrawable C++ wrapper class.
//!
//! Drawable represents an OpenGL rendering surface interface.

class Drawable : public virtual G::TypeInterface
{
	friend class DrawableClass;

protected:
//! @name Constructors
//! @{

	virtual ~Drawable() = 0;
	//!< Destructor.

//! @}
//  Override these do_ methods when you want to change the default behaviour of the GdkGLDrawable.

	virtual GdkGLContext* do_create_new_context(GdkGLContext *share_list, bool direct, int render_type);

	virtual bool do_make_context_current(GdkGLDrawable *read, GdkGLContext *glcontext);

	virtual bool do_is_double_buffered();

	virtual void do_swap_buffers();

	virtual void do_wait_gl();

	virtual void do_wait_gdk();

	virtual bool do_gl_begin(GdkGLDrawable *read, GdkGLContext *glcontext);

	virtual void do_gl_end();

	virtual GdkGLConfig* do_get_gl_config();

	virtual void do_get_size(int *width, int *height);

public:
//! @name Accessors
//! @{

	GdkGLDrawable* gdk_gl_drawable() const { return (GdkGLDrawable*)instance; }
	//!< Get a pointer to the GdkGLDrawable structure.

	GdkGLDrawableClass* gdk_gl_drawable_class() const;
	//!< Get a pointer to the GdkGLDrawableClass structure.

	operator GdkGLDrawable* () const;
	//!< Conversion operator; safely converts an Drawable to a GdkGLDrawable pointer.

	bool is_double_buffered() const;
	//!< Determines whether the drawable supports the double-buffered visual.
	//!< @return <EM>true</EM> if the double-buffered visual is supported, <EM>false</EM> otherwise.

	Config* get_gl_config() const;
	//!< Gets the Gdk::GL::Config with which the drawable is configured.
	//!< @return The Gdk::GL::Config.

	void get_size(int *width, int *height) const;
	//!< Fills <EM>width</EM> and <EM>height</EM> with the size of the GL drawable.
	//!< @param width The location to store the drawable's width, or null to ignore.
	//!< @param height The location to store the drawable's height, or null to ignore.
	//!<
	//!< <BR>The width or height argument can be null if you only want the other one.

//! @}

	static Drawable* get_current();
	//!< Gets the current Gdk::GL::Drawable.
	//!< @return The current Gdk::GL::Drawable or null if there is no current drawable.

//! @name Methods
//! @{

	bool make_current(Context& glcontext);
	//!< Attach an OpenGL rendering context to a drawable.
	//!< @param glcontext A Gdk::GL::Context.
	//!< @return <EM>true</EM> if it is successful, <EM>false</EM> otherwise.

	void swap_buffers();
	//!< Exchange the front and back buffers.

	void wait_gl();
	//!< Complete OpenGL execution prior to subsequent GDK drawing calls.

	void wait_gdk();
	//!< Complete GDK drawing execution prior to subsequent OpenGL calls.

	bool gl_begin(Context& glcontext);
	//!< Delimits the begining of the OpenGL execution.
	//!< @param glcontext A Gdk::GL::Context.
	//!< @returns <EM>true</EM> if it is successful, <EM>false</EM> otherwise.

	void gl_end();
	//!<  Delimits the end of the OpenGL execution.

//! @}
};

} // namespace GL

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_GL_DRAWABLE_H
