package freenet.transport;
import freenet.*;
import java.net.Socket;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.io.IOException;
import java.util.StringTokenizer;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
                
  Requires Classes: ProtocolAddress

 */

/**
 * A class representing the address of an Adaptive Network client in the
 * network which uses the TCP/IP protocol.
 * @author <A HREF="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</A>
 */

final class tcpAddress extends Address {

    private InetAddress host;
    private int port;

    private String valname;

    private final tcpTransport transport;

    
    private tcpAddress(tcpTransport transport) {
        super(transport);
        this.transport = transport;
    }
    
    /**
     * Creates an address from an InetAddress object and port number.
     */
    tcpAddress(tcpTransport transport, InetAddress host, int port)
                                        throws BadAddressException {
        this(transport);
        setPort(port);
        this.host = host;
        valname = host.getHostAddress();
    }

    /**
     * Creates an address from a host name or IP string and port number.
     */
    tcpAddress(tcpTransport transport, String hostname, int port)
                                throws BadAddressException {
        this(transport);
        setPort(port);
        try {
            host = InetAddress.getByName(hostname);
        }
        catch (UnknownHostException e) {
            throw new BadAddressException(""+e);
        }
        valname = hostname;
    }

    /** 
     * Creates an address from a string in the format "a.b.c.d:p"
     * where a,b,c,d are the (decimal) octets of the IPv4 address
     * and p is the (decimal) port number.
     */
    tcpAddress(tcpTransport transport, String str) throws BadAddressException {
        this(transport);
        int colon = str.indexOf(':');
        if (colon == -1) {
            throw new BadAddressException("both the IP and port number must be given");
        }
        valname = str.substring(0, colon);
        try {
            host = InetAddress.getByName(valname);
        }
        catch (UnknownHostException e) {
            throw new BadAddressException(""+e);
        }
        setPort(Integer.parseInt(str.substring(colon + 1)));
    }

    public final Connection connect() throws ConnectFailedException {
        return new tcpConnection(transport, this);
    }

    public final String getValString() {
        return valname + ":" + port;
    }

    public final ListeningAddress listenPart() {
        return new tcpListeningAddress(transport, port);
    }

    public final boolean equalsHost(Object o) {
        return o instanceof tcpAddress &&
               ((tcpAddress) o).host.equals(host);
    }

    final void setPort(int port) throws BadAddressException {
        if (port < 0 || port > 65535) {
            throw new BadAddressException("port number "+port+" out of range");
        }
        this.port = port;
    }

    final InetAddress getHost() {
        return host;
    }

    final int getPort() {
        return port;
    }
}


