package freenet.thread;

import freenet.Core;

/**
 * A simple thread pool, with an option to force creation of overflow
 * threads when the number of active threads has reached the limit.
 * Records Freenet diagnostics.
 * @author tavin
 */
public final class PThreadFactory implements ThreadFactory {

    private final ThreadGroup tg;
    private final int limit;

    private int active = 0;

    private PThread h_thread = null;
    
    /**
     * @param tg     ThreadGroup all created threads will belong to
     * @param limit  nominal maximum number of active threads
     */
    public PThreadFactory(ThreadGroup tg, int limit) {
        this.tg = tg;
        this.limit = limit;
        synchronized (this) {
            while (active < limit)
                new PThread(++active);
        }
    }

    /**
     * @return  the number of currently executing threads
     */
    public final int activeThreads() {
        return active;
    }

    /**
     * @return  the instantaneous number of idle threads;
     *          may return a negative number if the force option
     *          was used with getThread()
     */
    public final int availableThreads() {
        return limit - active;
    }

    /**
     * @return  a thread from the pool that will execute the given job
     *          when its start() method is called
     */
    public synchronized final Thread getThread(Runnable job, boolean force) {
        if (h_thread != null) {
            PThread thread = h_thread;
            h_thread = h_thread.next;
            thread.next = null;
            thread.job = job;
            ++active;
            return thread;
        }
        else if (force) {
            Core.diagnostics.occurrenceCounting("overflowThreads", 1);
            ++active;
            return new OThread(job);
        }
        else {
            Core.diagnostics.occurrenceCounting("insufficientThreads", 1);
            return null;
        }
    }


    // re-queue
    private synchronized final void putThread(PThread thread) {
        thread.next = h_thread;
        h_thread = thread;
        --active;
    }
    

    private final class OThread extends Thread {
        
        final Runnable job;
        
        OThread(Runnable job) {
            super(tg, "OThread+" + (active - limit));
            this.job = job;
        }
        
        public final void run() {
            try {
                Core.diagnostics.occurrenceCounting("jobsExecuted", 1);
                job.run();
            }
            catch (Throwable e) {
                Core.logger.log(this, "Unhandled throw in job",
                                e, Core.logger.ERROR);
            }
            finally {
                synchronized (PThreadFactory.this) {
                    --active;
                }
            }
        }
    }

    private final class PThread extends Thread {

        PThread next;
    
        final int num;
        Runnable job = null;
    
        PThread(int num) {
            super(tg, "PThread-"+num);
            this.num = num;
            super.start();
        }

        public final synchronized void start() {
            this.notify();
        }

        public final void run() {
            while (true) {
                putThread(this);
                synchronized (this) {
                    while (job == null) {
                        try {
                            this.wait();
                        }
                        catch (InterruptedException e) {}
                    }
                }
                try {
                    Core.diagnostics.occurrenceCounting("jobsExecuted", 1);
                    job.run();
                }
                catch (Throwable e) {
                    Core.logger.log(this, "Unhandled throw in job",
                                    e, Core.logger.ERROR);
                }
                finally {
                    job = null;
                    setName("PThread-"+num);  // reset name if it was altered
                }
            }
        }
        
        protected final void finalize() {
            Core.logger.log(this, this+" died!  Respawning..",
                            Core.logger.ERROR);
            new PThread(num);
        }
    }
}




