package freenet.support;
import freenet.crypt.*;
import freenet.Core;
import java.io.*;
/**
 * This is a bucket that writes bytes to a file, encrypting them with
 * a random key that is only kept in memory. Note that the security of this
 * is debatable as Freenet is completely promiscuous about it's memory, which
 * may well end up getting swapped to disk.
 *
 * @author oskar
 **/
public class CryptBucket extends FileBucket {
    
    private byte[] key;
    private String cipher;
    private BlockCipher encrypt;

    /**
     * Creates a new Bycket in a given file encrypted with a given cipher.
     * @param file the file to write to
     * @param cipher  the name of the cipher to use.
     * @exception IllegalArgumentException - if the given cipher is not 
     *            supported
     **/
    public CryptBucket(File file, String cipher) {
	super(file);
	this.cipher = cipher;
	BlockCipher bc = Util.getCipherByName(cipher);
	if (bc == null)
	    throw new IllegalArgumentException();
	key = new byte[bc.getKeySize() >> 3];
	Core.randSource.nextBytes(key);
    }

    /**
     * Creates a new Bycket in a temporary file encrypted with a given cipher.
     * @param cipher  the name of the cipher to use.
     * @exception IllegalArgumentException - if the given cipher is not 
     *            supported
     **/
    public CryptBucket(String cipher) {
	super();
	this.cipher = cipher;
	BlockCipher bc = Util.getCipherByName(cipher);
	if (bc == null)
	    throw new IllegalArgumentException();
	key = new byte[bc.getKeySize() >> 3];
	Core.randSource.nextBytes(key);
    }

    public OutputStream getOutputStream() throws IOException {
	boolean res = restart;
	if (res) {
	    encrypt = Util.getCipherByName(cipher);
	    encrypt.initialize(key);
	}
	OutputStream out = new CipherOutputStream(encrypt,
						  super.getOutputStream());
	if (res) {
	    byte[] iv = new byte[encrypt.getBlockSize() >> 3];
	    byte[] tmp =  new byte[encrypt.getBlockSize() >> 3];
	    System.arraycopy(key,0,tmp,0,tmp.length);
	    Util.makeKey(tmp,iv,0,iv.length);
	    out.write(key);
	}
	return out;
    }

    public InputStream getInputStream() throws IOException {
	InputStream in = super.getInputStream();
	if (in == null)
	    return null;
	BlockCipher decrypt = Util.getCipherByName(cipher);
	decrypt.initialize(key);
	return new CipherInputStream(decrypt, in);
    }

    public long size() {
	// subtract iv
	return encrypt == null ? 0 : super.size() - (encrypt.getBlockSize() >> 3);
    }
}







