package freenet.session;

import freenet.node.NodeMessageObject;
import freenet.node.*;
import freenet.*;
import freenet.support.Logger;
import java.util.*;
import java.io.IOException;
import java.math.BigInteger;
import freenet.crypt.*;

/**
 * The LinkManager keeps track of inter-node cryptography. 
 *
 * Changed name to FnpLinkManager - Oskar
 *
 * @author Scott
 */
public class FnpLinkManager implements LinkConstants, LinkManager {
    
    public static final int DESIGNATOR=1;
    
    //Stores the keys that are active at any given time.
    private final Hashtable activeLinks, activePeers;
    
    /** Discard expired links.
      */
    public synchronized void cleanupLinks() {
        long now = System.currentTimeMillis();
        Enumeration e = activeLinks.keys();
        while (e.hasMoreElements()) {
            IdKey li       = (IdKey) e.nextElement();
            FnpLinkToken l = (FnpLinkToken) activeLinks.get(li);
            if (l != null && now > l.inboundExpiresAt()) {
                removeLink(l, li);
            }
        }
    }


    
    public FnpLinkManager() {
        activePeers = new Hashtable(5);
        activeLinks = new Hashtable(5);
        DiffieHellman.init();
    }

    
    public Link createOutgoing(Authentity privMe,
                               Identity pubMe, 
                               Identity bob,
                               Connection c) throws CommunicationException {

        if (!(privMe instanceof DSAAuthentity) ||
            !(pubMe instanceof DSAIdentity) ||
            !(bob instanceof DSAIdentity)) {
            Core.logger.log(this, "Cannot negoiate inbound with non DSA " 
                            + " keys.", Logger.ERROR);
            throw new RuntimeException("Keys not DSA");
        }
        
        long time = System.currentTimeMillis();
        FnpLink l = new FnpLink(this, c);        
        l.solicit((DSAAuthentity) privMe, 
                  (DSAIdentity) pubMe.getKey(), 
                  (DSAIdentity) bob.getKey(), true);
        Core.diagnostics.occurrenceContinuous("authorizeTime",
                                             System.currentTimeMillis()- time);
        return l;
    }

    public Link acceptIncoming(Authentity privMe,
                               Identity pubMe, 
                               Connection c) throws CommunicationException {

        if (!(privMe instanceof DSAAuthentity) ||
            !(pubMe instanceof DSAIdentity)) {
            Core.logger.log(this, "Cannot negoiate inbound with non DSA " 
                            + " keys.", Logger.ERROR);
            throw new RuntimeException("Keys not DSA");
        }

        long time = System.currentTimeMillis();
        FnpLink l = new FnpLink(this, c);
        l.accept((DSAAuthentity) privMe, 
                 (DSAIdentity) pubMe, LAX);
        Core.diagnostics.occurrenceContinuous("authorizeTime",
                                            System.currentTimeMillis()- time);
        return l;
    }

    /**
     * Adds a key to the link manager.  The key will be automatically
     * expired by the link manager when KEY_LIFETIME milliseconds pass.
     */
    public synchronized LinkToken addLink(Identity remotePK, Identity me, 
                                          byte[] k) {

        Digest ctx = new SHA1();
        byte[] hk  = new byte[ctx.digestSize() >> 3];
        ctx.update(k);
        ctx.digest(true, hk, 0);

        FnpLinkToken oldLt = (FnpLinkToken) activePeers.get(remotePK);
        if (oldLt != null) {
            removeLink(oldLt);
        }

        BigInteger linkIdentifier=Util.byteArrayToMPI(hk);

        LinkToken lt=new FnpLinkToken(remotePK, me, k, linkIdentifier);
        //Core.logger.log(this, "Adding link identified by " 
        //                + linkIdentifier.toString(16), Core.logger.DEBUG);
        Core.logger.log(this, "Adding link to peer " + remotePK,
                        Core.logger.DEBUG);
        Core.diagnostics.occurrenceCounting("liveLinks", 1);
        synchronized (this) {
            activeLinks.put(new IdKey(linkIdentifier), lt);
            activePeers.put(remotePK, lt);
        }
        return lt;
    }

    public synchronized void removeLink(FnpLinkToken l) {
        removeLink(l, new IdKey(l.getKeyHash()));
    }

    public synchronized void removeLink(FnpLinkToken l, IdKey li) {
        Core.logger.log(this, "removing link "+l, Logger.DEBUG);
        l.expire();     // FIXME ??? Nah, it just doesn't do much.
        activeLinks.remove(li);
        activePeers.remove(l.getPeerIdentity());
        Core.diagnostics.occurrenceCounting("liveLinks", -1); 
    }

    // this is a kaffe workaround...
    // ... this one seems easy, why aren't we just patching kaffe?
    // ... I have, and they took it, but the workaround is easy enough
    //     to keep this until they release.
    private class IdKey {
        public BigInteger id;
        public IdKey(BigInteger id) {
            this.id = id;
        }
        public int hashCode() { // our values are random...
            return id.intValue();
        }
        public boolean equals(Object o) {
            return (o instanceof IdKey) && id.equals(((IdKey) o).id);
        }
    }

    /**
     * Searches for the given hashed key in the active links.
     * (Note the minomer, this is used on inbound restarts
     * but covers all links)
     * @returns The link structure, if found, null otherwise.
     **/
    public synchronized FnpLinkToken searchInboundLinks(BigInteger linkIdentifier) {
        FnpLinkToken lt = (FnpLinkToken) activeLinks.get(new IdKey(linkIdentifier));
        return lt == null || System.currentTimeMillis() > lt.inboundExpiresAt() ? null : lt;
    }

    /**
     * Searches for the given remote public key in the active links.
     * (Note the minomer, this is used on inbound restarts
     * but covers all links). 
     * @returns The link structure, if found, null otherwise.
     */
    public synchronized FnpLinkToken searchOutboundLinks(Identity linkIdentifier) {
        FnpLinkToken lt = (FnpLinkToken) activePeers.get(linkIdentifier);
        return lt == null || System.currentTimeMillis() > lt.outboundExpiresAt() ? null : lt;
    }

    public final int designatorNum() {
        return DESIGNATOR;
    }
}

    
