package freenet.presentation;

import freenet.*;
import freenet.support.io.DiscontinueInputStream;
import freenet.support.Logger;
import freenet.crypt.*;
import freenet.crypt.ciphers.*;
import freenet.message.VoidMessage;
import java.io.*;

/**
 * This is the implementation of the standard protocol used by Freenet
 * to communicate messages. Freenet Protocol (FNP) uses text based messages
 * with a simple header and fieldname=fieldvalue format. FNP utilizes 
 * encryption, currently by doing a DH exchange.
 **/

public class FreenetProtocol extends Presentation {

    public static int DESIGNATOR = 0x0001;

    public int designatorNum() {
        return DESIGNATOR;
    }

    /**
     * Creates a new RawMessage of a connection by reading this presentation. This
     * method locks until an entire message has been read (excluding trailing)
     * @param c     The connectio to read
     * @return      A new raw message
     */
    public RawMessage readMessage(InputStream in) 
	throws InvalidMessageException, EOFException{

	return new FNPRawMessage(in);
    }


    /** 
     * Creates a new RawMessage of a given type that uses this presentation
     * @param messageType   The name of the message type
     * @param close         Whether to keep alive the connection after 
     *                      receiving or sending this message.
     * @param sustain       Whether to sustain the connection.
     * @param fs            A set of message specific fields.
     * @param trailingLength The length of the trailing data, or 0 if there 
     *                       is no trailing data
     * @param trailingName   The name of the trailing field, or null if there
     *                       is no trailing data
     * @param data           An inputstream containing the trailing data,
     *                       straight off the DataStore (decrypted).
     * @return     A new raw message
     */
    public  RawMessage newMessage(String messageType, boolean close, 
                                  boolean sustain, FieldSet fs, 
                                  long trailingLength, 
                                  String trailingName, 
                                  DiscontinueInputStream data) {
        return new FNPRawMessage(messageType, close, sustain, fs, 
                                 trailingLength, trailingName, data);
    }


    public int exptMessageSize() {
	return 5000;
    }

    public Message getCloseMessage() {
        return new VoidMessage(Core.randSource.nextLong(), true, null);
    }

    public Message getSustainMessage() {
        return new VoidMessage(Core.randSource.nextLong(), false, true, null);
    }

}
