package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.node.states.data.*;
import freenet.node.ds.*;
import freenet.message.*;
import freenet.support.Logger;
import java.io.*;
import freenet.support.io.*;

/**
 * This is the state pertaining to DataRequests that are pending.
 */

public class DataPending extends Pending {

    /** For new requests.
      */
    public DataPending(long id, int htl, Key key, Peer orig,
                       FeedbackToken ft, RequestInitiator ri) {
        super(id, htl, key, orig, ft, ri);
    }

    /** For restarts.
      */
    DataPending(RequestState ancestor) {
        super(ancestor);
    }

    public String getName() {
        return "DataRequest Pending";
    }

    final Request createRequest(FieldSet otherFields, NodeReference ref) {
        return new DataRequest(id, hopsToLive, searchKey, ref, otherFields);
    }

 
    //=== message handling ====================================================

    // these two are special because they cause the state to be recreated
    // (restart condition)

    public State receivedMessage(Node n, QueryRejected qr) throws StateException {
        try {
            checkFailureTable(n);
            super.receivedQueryRejected(n, qr);
        }
        catch (EndOfRouteException e) {
            dataNotFound(n);
            return new RequestDone(this);
        }
        catch (RequestAbortException rae) {
            return rae.state;
        }
        return new DataPending(this);
    }

    public State receivedMessage(Node n, RequestInitiator ri) throws StateException {
        try {
            checkFailureTable(n);
            super.receivedRequestInitiator(n, ri);
        }
        catch (EndOfRouteException e) {
            dataNotFound(n);
            return new RequestDone(this);
        }
        catch (RequestAbortException rae) {
            return rae.state;
        }
        return new DataPending(this);
    }
    
    // the rest don't...

    public State receivedMessage(Node n, DataRequest dr) {
        try {
            checkFailureTable(n);
            super.receivedRequest(n, dr);
            return this;
        } catch (RequestAbortException rae) {
            return rae.state;
        }
    }

    public State receivedMessage(Node n, QueryRestarted qr) throws StateException {
        try {
            super.receivedQueryRestarted(n, qr);
        }
        catch (RequestAbortException rae) {
            return rae.state;
        }
        return this;
    }

    public State receivedMessage(Node n, Accepted a) throws StateException {
        super.receivedAccepted(n, a);
        return this;
    }

    public State receivedMessage(Node n, DataReply dr) throws StateException {
        return super.receivedDataReply(n, dr);
    }
    
    public State receivedMessage(Node n, DataNotFound dnf) throws StateException { 
        if (routedTime >= 0) {
            Core.diagnostics.occurrenceContinuous("hopTime", 
                                                (System.currentTimeMillis() -
                                                 routedTime) / hopsToLive);
        }
        if (!fromLastPeer(dnf)) {
            throw new BadStateException("DataNotFound from the wrong peer!");
        }
        cancelRestart();
        
        long toq = Math.min(dnf.timeOfQuery(),  // not stupid...
                            System.currentTimeMillis());
                
        dataNotFound(n, toq);
        // Add this key to the FailureTable
        if (!n.ds.contains(searchKey)) // that sort of sucks...
            n.ft.failedToFind(searchKey, hopsToLive, toq);
        // well, technically they did their job...
        routes.routeSucceeded();
        return new RequestDone(this);
    }

    private final void checkFailureTable(Node n) throws RequestAbortException {
        long toq = n.ft.shouldFail(searchKey, hopsToLive);
        if (toq > 0) {
            dataNotFound(n, toq);
            throw new RequestAbortException(new RequestDone(this));
        }
    }

    private final void dataNotFound(Node n) {
        dataNotFound(n, System.currentTimeMillis());
    }

    private final void dataNotFound(Node n, long toq) {
        try {
            ft.dataNotFound(n, toq);
        } catch (CommunicationException e) {
            n.logger.log(this, "Failed to reply with DataNotFound: "+e,
                         Logger.MINOR);
        }
    }
}


