package freenet.node.states.data;

import freenet.*;
import freenet.node.*;
import freenet.node.ds.*;
import freenet.support.*;
import freenet.support.io.*;
import java.io.*;


/**
 * Sends data from the store. If the reading from the store fails, then
 * the sent data will be padded until the end of the next part, where
 * CB_RESTARTED will be written as the control byte. When the send is finished
 * (for better or worse) a DataSent message will be returned to the parent
 * chain indicating the state with a control byte.
 *
 * @author oskar 
 */

public class SendData extends DataState {

    private final OutputStream send;
    private final KeyInputStream in;
    private final long length, partSize;

    private volatile int result = -1;
    private boolean silent = false;
    
    
    public SendData(long id, long parent, OutputStream send,
                    KeyInputStream in, long length, long partSize) {
        super(id, parent);
        this.send     = send;
        this.in       = in;
        this.length   = length;
        this.partSize = partSize;
    }

    public final String getName() {
        return "Sending Data";
    }

    /** If sending upstream, you want CB_ABORTED.
      * If sending downstream, you want CB_RESTARTED.
      */
    public final void abort(int cb) {
        silent = true;
        result = cb;
    }

    public final int result() {
        return result;
    }

    /** Sheesh!  We're too overworked to even try to write CB_ABORTED.
      */
    public final void lost(Node n) {
        try {
            in.close();
        }
        catch (IOException e) {
            n.logger.log(this, "I/O error closing KeyInputStream",
                         e, Logger.ERROR);
        }
        try {
            send.close();
        }
        catch (IOException e) {
            n.logger.log(this, "I/O error closing data send stream",
                         e, Logger.MINOR);
        }
    }
    

    
    public State received(Node n, MessageObject mo) throws BadStateException {
        if (!(mo instanceof DataStateInitiator))
            throw new BadStateException("expecting DataStateInitiator");

        // if there is an IOException, this says
        // whether it was while writing to the store
        boolean inWrite = false;
        
        long moved = 0;

        try {
            byte[] buffer = new byte[Core.blockSize];
            while (moved < length) {
                inWrite = false;
                if (result != -1) throw new CancelledIOException();
                int m = in.read(buffer, 0, (int) Math.min(length - moved, buffer.length));
                if (m == -1) throw new IOException();
                inWrite = true;
                send.write(buffer, 0, m);
                moved += m;
            } 
            send.close();
            result = Presentation.CB_OK;
        }
        catch (CancelledIOException e) {}  // result already set
        //catch (BadDataException e) {
        //    result = e.getFailureCode();
        //}
        catch (IOException e) {
            //result = (inWrite ? Presentation.CB_SEND_CONN_DIED
            //                  : Presentation.CB_CACHE_FAILED);
            result = (inWrite ? Presentation.CB_SEND_CONN_DIED
                              : in.getFailureCode());
        }
        finally {
        
            try {
                in.close();
            }
            catch (IOException e) {
                n.logger.log(this, "I/O error closing KeyInputStream",
                             e, Logger.ERROR);
            }

            if (moved < length && !inWrite) {
                try {
                    // pad until end of part
                    long tmpLen = partSize + Key.getControlLength();
                    tmpLen = Math.min(tmpLen - moved % tmpLen, length - moved) - 1;
                    
                    byte[] zeroes = new byte[Core.blockSize];
                    while (tmpLen > 0) {
                        int m = (int) Math.min(tmpLen, zeroes.length);
                        send.write(zeroes, 0, m);
                        tmpLen -= m;
                    }
                    send.write(result == Presentation.CB_ABORTED ? Presentation.CB_ABORTED
                                                                 : Presentation.CB_RESTARTED);
                }
                catch (IOException e) {
                    // this is important b/c it lets the parent chain know that
                    // it shouldn't try to get back in touch with the upstream
                    result = Presentation.CB_SEND_CONN_DIED;
                }
            }
            
            if (result != Presentation.CB_OK) {
                try {
                    send.close();
                }
                catch (IOException e) {
                    n.logger.log(this, "I/O error closing data send stream",
                                 e, Logger.MINOR);
                }
            }
            
            // we could be exiting with an uncaught exception or something..
            if (result == -1) result = Presentation.CB_SEND_CONN_DIED;
            
            // had to wait around to see if we'd get a CB_SEND_CONN_DIED
            // when padding the write
            if (!silent) n.schedule(new DataSent(this));
        }
        
        return null;
    }
}



