package freenet.node.rt;

import freenet.Key;
import freenet.Identity;
import freenet.node.NodeReference;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.BinaryTree.Node;
import freenet.support.RedBlackTree.RBNodeImpl;
import java.io.IOException;
import java.util.Enumeration;


/**
 * The basic core of a routing table that routes by walking
 * through closest keys.
 * @author tavin
 */
public abstract class TreeRoutingTable implements RoutingTable {

    protected final RoutingStore routingStore;
    protected final int maxNodes,
                        maxRefsPerNode;
    

    protected final BinaryTree refTree = new RedBlackTree();


    public TreeRoutingTable(RoutingStore routingStore,
                            int maxNodes, int maxRefsPerNode) throws IOException {
        
        this.routingStore = routingStore;
        this.maxNodes = maxNodes;
        this.maxRefsPerNode = maxRefsPerNode;
        
        // build binary tree of references
        Enumeration rme = routingStore.elements();
        while (rme.hasMoreElements()) {
            RoutingMemory mem = (RoutingMemory) rme.nextElement();
            ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
            Enumeration refs = refSet.references();
            while (refs.hasMoreElements()) {
                Reference ref = (Reference) refs.nextElement();
                refTree.treeInsert(new RBNodeImpl(ref), true);
            }
        }
    }

    
    public synchronized void reference(Key k, NodeReference nr) {

        long now = System.currentTimeMillis();
        
        RoutingMemory mem = routingStore.getNode(nr.getIdentity());
        if (mem == null || nr.supersedes(mem.getNodeReference()))
            mem = routingStore.putNode(nr);
        
        ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
        
        Reference ref = new Reference(k, nr.getIdentity(), now);

        Node oldRef = refTree.treeInsert(new RBNodeImpl(ref), false);
        if (oldRef != null) {
            ref = (Reference) oldRef.getObject();
            ref.timestamp = now;
            refSet.remove(ref);
        }
        
        // enforce refs per node limit
        if (refSet.size() >= maxRefsPerNode) {
            refTree.treeRemove(refSet.pop());
        }

        refSet.append(ref);
        
        mem.setProperty("keys", refSet);
        
        // enforce max nodes limit
        if (routingStore.size() > maxNodes)
            discard(maxNodes / 20);
    }

    public synchronized boolean references(Identity id) {
        return routingStore.contains(id);
    }

    public synchronized Routing route(Key k) {
        return new TreeRouting(this,
                               new MetricWalk(refTree, new Reference(k), false));
    }

    public final RoutingStore getRoutingStore() {
        return routingStore;
    }

    public final Object semaphore() {
        return this;
    }

    /**
     * Discards information about some nodes to free up memory.  The
     * deletion algorithm is derived from the subclass implementation.
     * All that is required is a Comparable object for each node.
     * 
     * This routine requires building a heap, so it is called in a
     * high water / low water mark pattern.
     */
    protected synchronized void discard(int num) {
        Heap heap = new Heap(routingStore.size());
        Enumeration rme = routingStore.elements();
        while (rme.hasMoreElements()) {
            RoutingMemory mem = (RoutingMemory) rme.nextElement();
            Comparable c = getDiscardValue(mem);
            heap.put(new DiscardSort(c, mem.getIdentity()));
        }
        DiscardSort d;
        while (num-- > 0 && null != (d = (DiscardSort) heap.pop()))
            routingStore.remove(d.ident);
    }

    private static final class DiscardSort implements Comparable {
        final Comparable cmp;
        final Identity ident;
        DiscardSort(Comparable cmp, Identity ident) {
            this.cmp = cmp;
            this.ident = ident;
        }
        public final int compareTo(Object o) {
            return compareTo((DiscardSort) o);
        }
        public final int compareTo(DiscardSort o) {
            return cmp.compareTo(o.cmp);
        }
    }

    /**
     * @return  an object that can be used to decide what node to discard
     *          by picking the nodes with the highest sort-value according
     *          to the Comparable returned.
     */
    protected abstract Comparable getDiscardValue(RoutingMemory mem);
    
    
    public abstract RTDiagSnapshot getSnapshot();

    // For book keeping.
    //protected abstract void attempted(RoutingMemory mem);
    // This is not needed because either routeConnected()
    // or connectFailed() is always called.
    
    protected abstract boolean isRoutable(RoutingMemory mem);


    // always call one of these:
    
    protected abstract void routeConnected(RoutingMemory mem);

    protected abstract void connectFailed(RoutingMemory mem);
    

    // then this can be called at any time:

    protected abstract void timedOut(RoutingMemory mem);

    
    // then only one of these (or timedOut()):
    
    protected abstract void authFailed(RoutingMemory mem);

    protected abstract void routeAccepted(RoutingMemory mem);
    
    
    // then only one of these (or timedOut()):
    
    protected abstract void routeSucceeded(RoutingMemory mem);
    
    protected abstract void transferFailed(RoutingMemory mem);
    
    protected abstract void verityFailed(RoutingMemory mem);

    protected abstract void queryRejected(RoutingMemory mem, long attenuation);
}



