package freenet.node.rt;

import freenet.node.Node;
import freenet.support.Fields;
import freenet.support.Fields.ByteArrayComparator;
import freenet.support.sort.*;
import freenet.support.io.WriteOutputStream;
import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.Enumeration;
import java.text.DateFormat;
import java.util.Date;

/**
 * Routing table information and debugging console.
 * @author tavin
 */
public class RTConsole extends HttpServlet {

    Node node;
    
    public final void init() {
        this.node = (Node) getServletContext().getAttribute("freenet.node.Node");
    }
    
    
    public void doGet(HttpServletRequest req, HttpServletResponse resp)
                                                    throws IOException {
        resp.addHeader("Cache-control", "no-cache");
        resp.addHeader("Pragma", "no-cache");
        resp.addHeader("Expires", "Thu, 01 Jan 1970 00:00:00 GMT");
        listNodes(req, resp);
    }
    
    public void doPost(HttpServletRequest req, HttpServletResponse resp)
                                                    throws IOException {
        
        if (req.getParameter("reset") != null) {
            resetRT();
            doGet(req, resp);
        }
        else if (req.getParameter("delete") != null) {
            String[] nodes = req.getParameterValues("nodes");
            if (nodes != null)
                deleteNodes(getFingerprintsFor(nodes));
            doGet(req, resp);
        }
        else if (req.getParameter("download") != null) {
            String[] nodes = req.getParameterValues("nodes");
            if (nodes != null) {
                resp.setContentType("text/plain");
                downloadNodes(getFingerprintsFor(nodes),
                              new WriteOutputStream(resp.getOutputStream()));
            }
            resp.flushBuffer();
        }
        //resp.setStatus(resp.SC_MOVED_TEMPORARILY);
        //resp.addHeader("Location", req.getRequestURI());
    }
    

    private void resetRT() {
        RoutingTable rt = node.rt;
        synchronized (rt.semaphore()) {
            RoutingStore routingStore = rt.getRoutingStore();
            Enumeration rme = routingStore.elements();
            while (rme.hasMoreElements()) {
                RoutingMemory rtmem = (RoutingMemory) rme.nextElement();
                routingStore.remove(rtmem.getIdentity());
            }
        }
    }
    

    private static final ByteArrayComparator byteCmp = new ByteArrayComparator();
    
    private final byte[][] getFingerprintsFor(String[] identStr) {
        byte[][] ident = new byte[identStr.length][];
        for (int i=0; i<identStr.length; ++i)
            ident[i] = Fields.hexToBytes(identStr[i]);
        HeapSorter.heapSort(new ArraySorter(ident, byteCmp));
        return ident;
    }

    private final boolean containsFingerprint(byte[][] fps, byte[] fp) {
        int lo = 0, hi = fps.length - 1;
        while (lo <= hi) {
            int m = (lo + hi) >> 1;
            int cmp = byteCmp.compare(fp, fps[m]);
            if (cmp > 0)
                lo = m+1;
            else if (cmp < 0)
                hi = m-1;
            else
                return true;
        }
        return false;
    }
    
    
    private void deleteNodes(byte[][] ident) {
        RoutingTable rt = node.rt;
        synchronized (rt.semaphore()) {
            RoutingStore routingStore = rt.getRoutingStore();
            Enumeration rme = routingStore.elements();
            while (rme.hasMoreElements()) {
                RoutingMemory rtmem = (RoutingMemory) rme.nextElement();
                if (containsFingerprint(ident, rtmem.getIdentity().fingerprint()))
                    routingStore.remove(rtmem.getIdentity());
            }
        }
    }
    

    private void downloadNodes(byte[][] ident, WriteOutputStream out)
                                                    throws IOException {
        RoutingTable rt = node.rt;
        synchronized (rt.semaphore()) {
            RoutingStore routingStore = rt.getRoutingStore();
            Enumeration rme = routingStore.elements();
            while (rme.hasMoreElements()) {
                RoutingMemory rtmem = (RoutingMemory) rme.nextElement();
                if (containsFingerprint(ident, rtmem.getIdentity().fingerprint()))
                    rtmem.getNodeReference().getFieldSet().writeFields(out);
            }
        }
    }
    

    private void listNodes(HttpServletRequest req, HttpServletResponse resp)
                                                        throws IOException {
       
        resp.setContentType("text/html");
        PrintWriter out = resp.getWriter();
        
        out.println("<html>");
        out.println("<head><title>Routing Table Console: " + (new Date())
                                                           + "</title></head>");
        out.println("<body>");

        out.println("<form method=\"POST\">");
        
        RoutingTable rt = node.rt;
        synchronized (rt.semaphore()) {
            
            RoutingStore routingStore = rt.getRoutingStore();
            
            int backedOffRefs = 0,
                contactedRefs = 0,
                totalTrials = 0,
                totalSuccesses = 0;
            
            long now = System.currentTimeMillis();
            
            Enumeration rme;
            
            rme = routingStore.elements();
            while (rme.hasMoreElements()) {
                RoutingMemory rtmem = (RoutingMemory) rme.nextElement();
                CPAlgoData cpdata = CPAlgoData.getProperty(rtmem, "cpdata");
                totalTrials += cpdata.trials;
                totalSuccesses += cpdata.successes;
                if (backoffSecs(cpdata, now) != -1)
                    ++backedOffRefs;
                if (cpdata.successes > 0)
                    ++contactedRefs;
            }

            
            out.println("<p><code>");
            out.println("Number of node references: "
                        + routingStore.size() + "<br />");
            out.println("Contacted references: "
                        + contactedRefs + "<br />");
            out.println("Backed off node references: "
                        + backedOffRefs + "<br />");
            out.println("Total trials: "
                        + totalTrials + "<br />");
            out.println("Total successes: "
                        + totalSuccesses + "<br />");
            out.println("</code></p>");

            out.println("<p><input type=\"submit\" name=\"download\" value=\"download refs\" /></p>");
            
            out.println("<table border=\"1\" width=\"100%\">");
            out.println("<tr style=\"background-color: grey\">");
            out.println("<th align=\"left\" colspan=\"2\">Node</th>");
            out.println("<th align=\"left\">Refs</th>");
            out.println("<th align=\"left\">CP</th>");
            out.println("<th align=\"left\">Failure Intervals</th>");
            out.println("<th align=\"left\">Trials</th>");
            out.println("<th align=\"left\">Successes</th>");
            out.println("<th align=\"left\">Back-off Time</th>");
            out.println("<th align=\"left\">Last Attempt</th>");
            out.println("</tr>");


            rme = routingStore.elements();
            while (rme.hasMoreElements()) {
                
                RoutingMemory rtmem = (RoutingMemory) rme.nextElement();
                String identStr = Fields.bytesToHex(rtmem.getIdentity().fingerprint());
                ReferenceSet refs = ReferenceSet.getProperty(rtmem, "keys");
                CPAlgoData cpdata = CPAlgoData.getProperty(rtmem, "cpdata");
                
                out.println("<tr>");
                out.print("<td>");
                out.print("<input type=\"checkbox\" name=\"nodes\" value=\""+identStr+"\" />");
                out.println("</td>");
                out.print("<td><code>");
                out.print(rtmem.getNodeReference());
                out.println("</code></td>");
                out.println("<td><code>" + refs.size() + "</code></td>");
                out.println("<td><code>" + cpdata.effectiveCP(now) + "</code></td>");
                out.println("<td><code>" + cpdata.failureIntervals + "</code></td>");
                out.println("<td><code>" + cpdata.trials + "</code></td>");
                out.println("<td><code>" + cpdata.successes + "</code></td>");
                out.println("<td><code>" + backoffTime(cpdata, now) + "</code></td>");
                out.println("<td><code>" + lastAttemptedTime(cpdata) + "</code></td>");
                out.println("</tr>");
            }
            
            out.println("</table>");
        }

        out.println("<p>");
        out.println("<input type=\"submit\" name=\"delete\" value=\"delete nodes\" />");
        out.println("&nbsp; &nbsp; &nbsp;");
        out.println("<input type=\"submit\" name=\"reset\" value=\"clear table\" />");
        out.println("</p>");

        out.println("</form>");
        
        out.println("</body>");
        out.println("</html>");
        
        resp.flushBuffer();
    }

        
    private static final String lastAttemptedTime(CPAlgoData cpd) {
        return cpd.trials > 0
               ? DateFormat.getTimeInstance().format(new Date(cpd.lastRetryMs))
               : "never";
    }

    private static final String backoffTime(CPAlgoData cpd, long nowMs) {
        long b = backoffSecs(cpd, nowMs);
        return b == -1 ? "no" : b+" sec";
    }
    
    private static final long backoffSecs(CPAlgoData cpd, long nowMs) {
        if (((cpd.failureIntervals > 0) && (nowMs - cpd.lastRetryMs < CPAlgoData.SHORTBACKOFFMS)) ||
            (cpd.failuresThisInterval > CPAlgoData.MAXFAILURESPERINTERVAL)) {
            return Math.min(1, (cpd.intervalTimeoutMs - nowMs) / 1000);
        }
        else return -1;
    }
}












