package freenet.node.ds;

import freenet.*;
import freenet.fs.dir.*;
import freenet.fs.acct.Fragment;
import freenet.support.*;
import freenet.support.io.*;
import java.io.*;
import java.util.*;
import java.text.DateFormat;


/**
 * A DataStore implementation that uses a freenet.fs.LossyDirectory
 * to store the keys.  It automatically deletes least-recently-accessed
 * keys when it fills up.
 * 
 * @see freenet.fs.LossyDirectory
 * @author tavin
 */
public class FSDataStore implements DataStore {

    public static void dump(FSDataStore ds, PrintWriter pw) {
        synchronized (ds.dir.semaphore()) {
        
            pw.println("Free space: "+ds.dir.available());
            pw.println();

            pw.println("Committed keys");
            pw.println("--------------");
            dumpCommittedKeys(ds, pw);

            pw.println("LRU keys (oldest last)");
            pw.println("----------------------");
            dumpLRUKeys(ds, pw);
        }
    }

    public static void dumpCommittedKeys(FSDataStore ds, PrintWriter pw) {
        synchronized (ds.dir.semaphore()) {
            Enumeration keys = ds.dir.keys(true);
            while (keys.hasMoreElements()) {
                FileNumber fn = (FileNumber) keys.nextElement();
                Buffer buffer = ds.dir.fetch(fn);
                try {
                    pw.print(Fields.bytesToHex(fn.getByteArray()));
                    pw.print(" @ " + Fragment.rangeList(buffer.ticket().ranges));
                    pw.println();
                }
                finally {
                    buffer.release();
                }
            }
            pw.println();
        }
    }
    
    public static void dumpLRUKeys(FSDataStore ds, PrintWriter pw) {
        synchronized (ds.dir.semaphore()) {
            DateFormat df = DateFormat.getDateTimeInstance();
            Enumeration keys = ds.dir.lruKeys(false);
            while (keys.hasMoreElements()) {
                FileNumber fn = (FileNumber) keys.nextElement();
                Buffer buffer = ds.dir.fetch(fn);
                try {
                    pw.print(Fields.bytesToHex(fn.getByteArray()));
                    pw.print(" @ " + df.format(new Date(buffer.ticket().timestamp)));
                    pw.println();
                }
                finally {
                    buffer.release();
                }
            }
            pw.println();
        }
    }
    
    
    


    final LossyDirectory dir;

    final long maxDataSize;

    
    /**
     * @param dir          the backing storage
     * @param maxDataSize  largest key length that can be stored
     *                     in a non-circular buffer
     */
    public FSDataStore(LossyDirectory dir, long maxDataSize) {
        this.dir = dir;
        this.maxDataSize = maxDataSize;
    }


    /**
     * @return  a KeyOutputStream that transfers the written data
     *          into the store
     */
    public KeyOutputStream putData(Key k, long dataSize, FieldSet storables)
                                throws IOException, KeyCollisionException {

        // we need to know the byte-length of the Storables in advance
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        storables.writeFields(new WriteOutputStream(bout));
        dataSize += bout.size();

        FileNumber fn = new FileNumber(k.getVal());
        Buffer buffer;
            
        synchronized (dir.semaphore()) {
            
            if (dir.contains(fn))
                throw new KeyCollisionException();

            if (dataSize > maxDataSize) {
                // create circular buffer
                buffer = new CircularBuffer(dir.forceStore(maxDataSize, fn),
                                            dataSize);
                dataSize = maxDataSize;
            }
            else {
                // create normal buffer
                buffer = dir.forceStore(dataSize, fn);
            }
        }

        Core.logger.log(this, "storing key: "+k, Core.logger.DEBUG);
        
        FSDataStoreElement dse = new FSDataStoreElement(this, k, buffer, dataSize);
        KeyOutputStream kout = dse.getKeyOutputStream();
        bout.writeTo(kout);
        kout.flush();

        return kout;
    }

    
    /**
     * Retrieves the data for a key.
     * @return  a KeyInputStream from which the key data can be read,
     *          or null if not found
     */
    public KeyInputStream getData(Key k) throws IOException {
        
        Buffer buffer;
        synchronized (dir.semaphore()) {
            buffer = dir.fetch(new FileNumber(k.getVal()));
            if (buffer == null) {
                Core.logger.log(this, "key not found: "+k, Core.logger.DEBUG);
                return null;
            }
            buffer.touch();  // register access for LRU tracking
        }
        
        Core.logger.log(this, "key found: "+k, Core.logger.DEBUG);
        
        FSDataStoreElement dse = new FSDataStoreElement(this, k, buffer,
                                                        buffer.length());
        return dse.getKeyInputStream();
    }
    

    /**
     * Deletes a Key from the store.
     * @return true, if a Key was removed
     */
    public final boolean remove(Key k) {
        return dir.delete(new FileNumber(k.getVal()));
    }

    
    /**
     * @return true, if the Key is contained in the store
     */
    public final boolean contains(Key k) {
        return dir.contains(new FileNumber(k.getVal()));
    }
    

    private final Stack keyStack = new Stack();

    /**
     * @param k          the key to measure closeness to
     * @param inclusive  whether to include k, if found
     * @param limit      max size of returned array
     * 
     * @return  an array of Keys arranged in order of closeness
     */
    public Key[] findClosestKeys(Key k, boolean inclusive, int limit) {
        synchronized (keyStack) {
            try {
                synchronized (dir.semaphore()) {

                    FileNumber fnk = new FileNumber(k.getVal());
                    Enumeration e1 = dir.keys(new RangeFilePattern(fnk, false, false));
                    Enumeration e2 = dir.keys(new RangeFilePattern(fnk, inclusive, true));
                    
                    Walk w1 = new KeyWalk(new EnumerationWalk(e1));
                    Walk w2 = new KeyWalk(new EnumerationWalk(e2));
                
                    Key k1 = (Key) w1.getNext();
                    Key k2 = (Key) w2.getNext();

                    // push onto stack in order of closeness
                    while (limit-- > 0) {
                        if (k1 == null && k2 == null) {
                            break;
                        }
                        else if (k1 == null || (k2 != null && k.compareTo(k1, k2) > 0)) {
                            // use k2 instead of k1
                            keyStack.push(k2);
                            k2 = (Key) w2.getNext();
                        }
                        else {
                            // use k1 instead of k2
                            keyStack.push(k1);
                            k1 = (Key) w1.getNext();
                        }
                    }
                    
                    Key[] ret = new Key[keyStack.size()];
                    keyStack.copyInto(ret);

                    Core.logger.log(this,
                                    "returning "+ret.length+" closest keys to: "+k,
                                    Core.logger.DEBUG);
                    
                    return ret;
                }
            }
            finally {
                keyStack.removeAllElements();
            }
        }
    }

    private static final class KeyWalk implements Walk {
        private final Walk fnw;
        KeyWalk(Walk fnw) {
            this.fnw = fnw;
        }
        public final Object getNext() {
            FileNumber fn = (FileNumber) fnw.getNext();
            return fn == null ? null : new Key(fn.getByteArray());
        }
    }
}


