package freenet.interfaces.servlet;

import freenet.FieldSet;
import freenet.interfaces.ServiceException;
import freenet.node.Node;
import freenet.support.Logger;
import freenet.client.ClientFactory;
import freenet.config.*;
import freenet.support.sort.*;
import freenet.support.Comparable;
import javax.servlet.*;
import java.util.Enumeration;

/**
 * A container that can cope with multiple contexts
 * and multiple servlets within each context.
 * @author tavin
 */
public class MultipleHttpServletContainer extends HttpServletContainer {

    private static final Config config = new Config();
    static {
        // hm, there is really no good to come from defining these until
        // there is better support for FieldSets in the config lib
    }
    

    // data containers for looking up servlets by URI

    /**
     * For ordering URI path segments longest-first.
     */
    private static abstract class PathData implements Comparable {
        final String path;
        PathData(String path) {
            this.path = path;
        }
        public final int compareTo(Object o) {
            return compareTo((PathData) o);
        }
        public final int compareTo(PathData o) {
            return path.length() == o.path.length()
                   ? 0 : (path.length() < o.path.length() ? 1 : -1);
        }
    }
    
    private static final class ServletData extends PathData {
        final ServletPool pool;
        ServletData(String path, ServletPool pool) {
            super(path);
            this.pool = pool;
        }
    }

    private static final class ContextData extends PathData {
        final ServletData[] servletData;
        ContextData(String path, ServletData[] servletData) {
            super(path);
            this.servletData = servletData;
        }
    }

    
    // context data tuples ordered longest URI-path first
    private ContextData[] contextData;
    

    /**
     * In-process.
     */
    public MultipleHttpServletContainer(Node node) {
        super(node);
    }

    /**
     * Out-of-process.
     */
    public MultipleHttpServletContainer(Logger logger, ClientFactory factory) {
        super(logger, factory);
    }


    /**
     * @return  the Config options for initializing this Service
     *
     * NOTE:  it's empty right now, since support for sub-FieldSets
     *        is lacking in the config lib
     */
    public final Config getConfig() {
        return config;
    }

    
    /**
     * Sets up all contexts and servlets according to the config fields.
     * @throws ServiceException
     *         if the Class for any servlet was not specified,
     *         or could not be loaded,
     *         or if there is a URI collision
     */
    public void init(Params params) throws ServiceException {

        FieldSet contexts = params.getSet("context");
        if (contexts == null) {
            contexts = new FieldSet();
            if (params.getSet("servlet") != null) {
                FieldSet defaultContext = new FieldSet();
                defaultContext.put("uri", "");
                if (params.getSet("params") != null)
                    defaultContext.put("params", params.getSet("params"));
                defaultContext.put("servlet", params.getSet("servlet"));
                contexts.put("1", defaultContext);
            }
        }

        contextData = new ContextData[contexts.size()];
        Enumeration cte = contexts.elements();
        for (int i=0; i<contextData.length; ++i)
            contextData[i] = loadContext((FieldSet) cte.nextElement());

        QuickSorter.quickSort(new ArraySorter(contextData));
    }

    private ContextData loadContext(FieldSet context) throws ServiceException {
        
        String uri = context.get("uri");
        if (uri == null)
            uri = "";
        
        FieldSet params = context.getSet("params");
        if (params == null)
            params = new FieldSet();
        
        FieldSet servlets = context.getSet("servlet");
        if (servlets == null)
            servlets = new FieldSet();
        
        ServletContext servletContext = createServletContext(params);

        ServletData[] servletData = new ServletData[servlets.size()];
        Enumeration se = servlets.elements();
        for (int i=0; i<servletData.length; ++i) {
            servletData[i] = loadServlet(servletContext,
                                         (FieldSet) se.nextElement());
        }
        QuickSorter.quickSort(new ArraySorter(servletData));
        
        return new ContextData(uri, servletData);
    }

    private ServletData loadServlet(ServletContext context, FieldSet servlet)
                                                    throws ServiceException {

        String uri = servlet.get("uri");
        if (uri == null)
            uri = "/";

        String className = servlet.get("class");
        if (className == null)
            throw new ServiceException("Class not specified for servlet: " + 
                                       uri);

        Class cls;
        try {
            cls = Class.forName(className);
        }
        catch (ClassNotFoundException e) {
            throw new ServiceException(e.toString());
        }
        
        
        String servletName = servlet.get("name");
        if (servletName == null || servletName.trim().equals(""))
            servletName = cls.getName();
        
        FieldSet params = servlet.getSet("params");
        if (params == null)
            params = new FieldSet();
        
        ServletConfig config = createServletConfig(context, servletName, 
                                                   params);
        ServletPool pool = new SimpleServletPool(cls, context, config);

        return new ServletData(uri, pool);
    }
    

    public final String getContextPath(String uripath) {
        ContextData contextData = findContext(uripath);
        return contextData == null ? null : contextData.path;
    }
    
    public final String getServletPath(String uripath) {
        ContextData contextData = findContext(uripath);
        if (contextData != null) {
            ServletData servletData = findServlet(contextData, uripath);
            if (servletData != null)
                return servletData.path;
        }
        return null;
    }

    public final String getPathInfo(String uripath) {
        ContextData contextData = findContext(uripath);
        if (contextData != null) {
            ServletData servletData = findServlet(contextData, uripath);
            if (servletData != null) {
                int n = contextData.path.length() + servletData.path.length();
                if (uripath.length() > n)
                    return uripath.substring(n, uripath.length());
            }
        }
        return null;
    }
    

    public final ServletContext getContext(String uripath) {
        return getServletPool(uripath).getServletContext();
    }
    

    protected final ServletPool getServletPool(String uripath) {
        ContextData contextData = findContext(uripath);
        if (contextData != null) {
            ServletData servletData = findServlet(contextData, uripath);
            if (servletData != null)
                return servletData.pool;
        }
        return null;
    }



    private ContextData findContext(String uripath) {
        for (int i=0; i<contextData.length; ++i) {
            if (uripath.startsWith(contextData[i].path))
                return contextData[i];
        }
        return null;
    }

    private ServletData findServlet(ContextData contextData, String uripath) {
        uripath = uripath.substring(contextData.path.length(), 
                                    uripath.length());
        ServletData[] servletData = contextData.servletData;
        for (int i=0; i<servletData.length; ++i) {
            if (uripath.startsWith(servletData[i].path))
                return servletData[i];
        }
        return null;
    }
}




