package freenet.interfaces;

import freenet.*;
import freenet.support.Logger;
import freenet.session.*;
import java.io.*;

/**
 * Runs connections using standard Freenet
 * session and presentation negotiation.
 */
public class FreenetConnectionRunner implements ConnectionRunner {
    
    protected final Core core;
    protected SessionHandler sh;
    protected PresentationHandler ph;
    protected OpenConnectionManager ocm;
    protected int maxInvalid;

    /**
     * @param core        the Core containing the SessionHandler,
     *                    PresentationHandler, and the node's key pair
     * @param ocm         passed to the ConnectionHandler constructor
     * @param maxInvalid  passed to the ConnectionHandler constructor
     */
    public FreenetConnectionRunner(Core core,
                                   SessionHandler sh, PresentationHandler ph,
                                   OpenConnectionManager ocm, int maxInvalid) {
        this.core = core;
        this.sh = sh;
        this.ph = ph;
        this.ocm = ocm;
        this.maxInvalid = maxInvalid;
    }
    
    public void handle(Connection conn) {
        try {
            Link l;
            Presentation p;

            InputStream raw = conn.getIn();
            int i = raw.read() << 8 | raw.read();
            if (i < 0)
                throw new EOFException();
            LinkManager lm = sh.get(i);
            if (lm == null) {
                throw new NegotiationFailedException(conn.getPeerAddress(),
                                                     "Unsupported link 0x"
                                                     + Integer.toHexString(i));
            }

            l = lm.acceptIncoming(core.privateKey, 
                                  core.identity, 
                                  conn);

            InputStream crypt = l.getInputStream();
            i = crypt.read() << 8 | crypt.read();
            if (i < 0)
                throw new EOFException();
            p = ph.get(i);
            if (p == null) {
                throw new NegotiationFailedException(conn.getPeerAddress(),
                                                     l.getPeerIdentity(),
                                                     "Unsupported presentation 0x"
                                                     + Integer.toHexString(i));
            }

            ConnectionHandler ch = 
                new ConnectionHandler(ocm, p, l, core.ticker(),
                                      maxInvalid, core.maxPadding, false);
        
            ch.run(); // we already have our own thread
            
        } catch (IOException e) {
            core.logger.log(this, "Inbound connection failed: "+e, Logger.MINOR);
            conn.close();
        } catch (CommunicationException e) {
            core.logger.log(this, "Inbound connection failed: "+e, Logger.MINOR);
            conn.close();
        }
    }
}






