package freenet.fs.dir;

import freenet.fs.acct.AccountingProcess;
import freenet.fs.acct.sys.*;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.BinaryTree.Node;
import java.io.*;
import java.util.Enumeration;

/**
 * Orders FileNumber keys by timestamp.
 * @author tavin
 */
final class LRUMap extends AccountingTree {

    LRUMap(AccountingProcess proc, Cache cache) {
        super(proc, new LRUEntry.Marshal(), cache);
    }


    final boolean insert(long timestamp, FileNumber fn) {
        LRUEntry le = new LRUEntry(timestamp, fn);
        return null == treeInsert(new AccountingTreeNode(le), false);
    }

    final boolean remove(long timestamp, FileNumber fn) {
        LRUEntry le = new LRUEntry(timestamp, fn);
        return null != treeRemove(le);
    }


    /**
     * @return  FileNumber keys, in ascending order from
     *          the least recently used, or descending order
     */
    final Walk keys(boolean ascending) {
        return new LRUEntry.FileNumberWalk(treeWalk(ascending));
    }


    
    private static final class LRUEntry implements Comparable {

        private static final class FileNumberWalk implements Walk {
            private final Walk walk;
            FileNumberWalk(Walk walk) {
                this.walk = walk;
            }
            public final Object getNext() {
                Node n = (Node) walk.getNext();
                return n == null ? null : ((LRUEntry) n.getObject()).fn;
            }
        }
    
    
        private static final class Marshal implements AccountingTreeMarshal {
            public final Comparable readEntry(DataInput din, int len)
                                                throws IOException {
                int dirID = din.readUnsignedShort();
                long timestamp = din.readLong();
                byte[] key = new byte[len - 10];
                din.readFully(key);
                return new LRUEntry(timestamp, new FileNumber(dirID, key));
            }
            public final int getEntryLength(Comparable entry) {
                return 10 + ((LRUEntry) entry).fn.key.length;
            }
            public final void writeEntry(Comparable entry, DataOutput out)
                                                        throws IOException {
                LRUEntry le = (LRUEntry) entry;
                out.writeShort(le.fn.dirID);
                out.writeLong(le.timestamp);
                out.write(le.fn.key);
            }
        }
        

        final long timestamp;
        final FileNumber fn;
        

        LRUEntry(long timestamp, FileNumber fn) {
            this.timestamp = timestamp;
            this.fn = fn;
        }

        public final int compareTo(Object o) {
            return compareTo((LRUEntry) o);
        }

        public final int compareTo(LRUEntry le) {
            return timestamp == le.timestamp ? fn.compareTo(le.fn)
                                             : (timestamp > le.timestamp ? 1 : -1);
        }

        public final String toString() {
            return fn + " @ " + timestamp;
        }
    }
}


