package freenet.fs.dir;

import freenet.fs.acct.fsck.*;
import freenet.fs.acct.*;
import freenet.fs.acct.sys.*;
import freenet.support.*;
import freenet.support.BinaryTree.Node;
import java.util.Vector;
import java.io.IOException;

/**
 * Performs various syntactic and semantic consistency checks
 * on the data in the FSDirectory accounting tables.
 * @author tavin
 */
public abstract class FSDirectoryCheck implements FSDirectoryConst {

    private FSDirectoryCheck() {}


    /**
     * Sets up six AccountingTreeCheck instances for each of the six
     * accounting structs composing the FSDirectory, then runs the
     * accounting initialization process against them.
     * @return  the AccountingTreeCheck objects as a FaultAnalysis array
     */
    public static FaultAnalysis[] checkSyntax(AccountingTable acct)
                                                throws IOException {
        
        AccountingInitializer acctInit = new AccountingInitializer(acct);
        SingleAccountingProcess proc = new SingleAccountingProcess(acctInit);

        AccountingTreeCheck[] checks = new AccountingTreeCheck[6];
        checks[0] = new AccountingTreeCheck(proc.share(LIVE_TICKET_MAP),
                                            "0x0101: Live Ticket Map");
        checks[1] = new AccountingTreeCheck(proc.share(MAIN_TICKET_MAP),
                                            "0x0102: Main Ticket Map");
        checks[2] = new AccountingTreeCheck(proc.share(FRAGMENT_SIZE_MAP),
                                            "0x0201: Fragment Size Map");
        checks[3] = new AccountingTreeCheck(proc.share(FRAGMENT_POSITION_MAP),
                                            "0x0202: Fragment Position Map");
        checks[4] = new AccountingTreeCheck(proc.share(KEY_MAP),
                                            "0x0301: Key Map");
        checks[5] = new AccountingTreeCheck(proc.share(LRU_MAP),
                                            "0x0401: LRU Map");

        SharedAccountingInitializer shInit = new SharedAccountingInitializer();
        shInit.add(LIVE_TICKET_MAP, checks[0]);
        shInit.add(MAIN_TICKET_MAP, checks[1]);
        shInit.add(FRAGMENT_SIZE_MAP, checks[2]);
        shInit.add(FRAGMENT_POSITION_MAP, checks[3]);
        shInit.add(KEY_MAP, checks[4]);
        shInit.add(LRU_MAP, checks[5]);
        acctInit.initialize(shInit);

        return checks;
    }

    
    /**
     * Sets up four consistency checks and returns them
     * in post-processed state.
     * @return  the consistency checks as a FaultAnalysis array
     */
    public static FaultAnalysis[] checkSemantics(AccountingTable acct)
                                                    throws IOException {
        
        AccountingInitializer acctInit = new AccountingInitializer(acct);
        SingleAccountingProcess proc = new SingleAccountingProcess(acctInit);
        
        Cache cache = new LRUCache(100);

        TicketMap liveTicketMap, mainTicketMap;
        FragmentSizeMap fsizeMap;
        FragmentPositionMap fposMap;
        KeyMap keyMap;
        LRUMap lruMap;
    
        liveTicketMap = new TicketMap(proc.share(LIVE_TICKET_MAP), true);
        mainTicketMap = new TicketMap(proc.share(MAIN_TICKET_MAP), false);
        fsizeMap = new FragmentSizeMap(proc.share(FRAGMENT_SIZE_MAP), cache);
        fposMap = new FragmentPositionMap(proc.share(FRAGMENT_POSITION_MAP), cache);

        keyMap = new KeyMap(proc.share(KEY_MAP), cache);
        lruMap = new LRUMap(proc.share(LRU_MAP), cache);

        SharedAccountingInitializer shInit = new SharedAccountingInitializer();
        shInit.add(LIVE_TICKET_MAP, liveTicketMap);
        shInit.add(MAIN_TICKET_MAP, mainTicketMap);
        shInit.add(FRAGMENT_SIZE_MAP, fsizeMap);
        shInit.add(FRAGMENT_POSITION_MAP, fposMap);
        shInit.add(KEY_MAP, keyMap);
        shInit.add(LRU_MAP, lruMap);
        acctInit.initialize(shInit);
                                                    
        return new FaultAnalysis[] {
            //new TicketAnalysis(liveTicketMap, mainTicketMap, keyMap, fposMap),
            //new KeyAnalysis(liveTicketMap, mainTicketMap, keyMap),
            //new FragmentPositionAnalysis(liveTicketMap, mainTicketMap, fposMap),
            new FragmentSizeAnalysis(proc, fposMap, fsizeMap)
        };
    }



    private static final class TicketAnalysis implements FaultAnalysis {

        TicketAnalysis(TicketMap liveTicketMap, TicketMap mainTicketMap,
                       KeyMap keyMap, FragmentPositionMap fposMap) {
        }

        public final String getDescription() {
            return "Check that tickets are represented in the key map " +
                   "and fragment position map";
        }
        
        public final boolean hasFaults() {
            return false;
        }
        
        public final Fault getNextFault() {
            return null;
        }
        
        public final void commitFixes() throws IOException {
        }
    }


    private static final class KeyAnalysis implements FaultAnalysis {
        
        KeyAnalysis(TicketMap liveTicketMap, TicketMap mainTicketMap,
                    KeyMap keyMap) {
        }
        
        public final String getDescription() {
            return "Check that all tickets referenced in the key map " +
                   "are in fact contained in the ticket maps";
        }
        
        public final boolean hasFaults() {
            return false;
        }
        
        public final Fault getNextFault() {
            return null;
        }
        
        
        public final void commitFixes() throws IOException {
        }
    }


    private static final class FragmentPositionAnalysis implements FaultAnalysis {

        FragmentPositionAnalysis(TicketMap liveTicketMap,
                                 TicketMap mainTicketMap,
                                 FragmentPositionMap fposMap) {
        }
            
        public final String getDescription() {
            return "Check that all tickets referenced in the " +
                   "fragment position map are in fact contained " +
                   "in the ticket maps";
        }
        
        public final boolean hasFaults() {
            return false;
        }
        
        public final Fault getNextFault() {
            return null;
        }
        
        
        public final void commitFixes() throws IOException {
        }
    }


    /**
     * By the time we run this analysis, we believe the fragment
     * position map is 100% correct.
     */
    private static final class FragmentSizeAnalysis implements FaultAnalysis {

        private final SingleAccountingProcess proc;
        
        private final FragmentPositionMap fposMap;
        private final FragmentSizeMap fsizeMap;
        
        private final Vector faults = new Vector();
        private int faultsIndex = 0;
        
        FragmentSizeAnalysis(SingleAccountingProcess proc,
                             FragmentPositionMap fposMap,
                             FragmentSizeMap fsizeMap) {

            this.proc = proc;
            this.fposMap = fposMap;
            this.fsizeMap = fsizeMap;
            
            synchronized (faults) {
                
                Walk walk;
                Fragment f, f2 = null;
                
                walk = fposMap.fragments();
                while (null != (f = (Fragment) walk.getNext())) {
                    if (f2 != null && f2.getUpperBound() < f.getLowerBound() - 1) {
                        Fragment f3 = new Fragment(f2.getUpperBound() + 1,
                                                   f.getLowerBound() - 1);
                        if (!fsizeMap.contains(f3))
                            faults.addElement(new FragmentAbsentFault(f3));
                    }
                    f2 = f;
                }
                
                walk = fsizeMap.fragments();
                while (null != (f = (Fragment) walk.getNext())) {
                    f2 = fposMap.probeFree(f);
                    if (f2 == null || !f2.equals(f))
                        faults.addElement(new FragmentExtantFault(f));
                }
            }
        }

        public final String getDescription() {
            return "Check that the contents of the fragment size map " +
                   "mirror the contents of the fragment position map";
        }
        
        public final boolean hasFaults() {
            return !faults.isEmpty();
        }
        
        public final Fault getNextFault() {
            return faultsIndex < faults.size()
                   ? (Fault) faults.elementAt(faultsIndex++)
                   : null;
        }
        
        
        public final void commitFixes() throws IOException {
            proc.flush();
        }


        private abstract class FragmentFault implements Fault {
            final Fragment frag;
            FragmentFault(Fragment frag) {
                this.frag = frag;
            }
            public final int getSeverity() {
                return MINOR;
            }
        }
        
        private final class FragmentAbsentFault extends FragmentFault {
            FragmentAbsentFault(Fragment frag) {
                super(frag);
            }
            public final String getDescription() {
                return "Fragment is missing from the size-ordered free map: "+frag;
            }
            public final void fix() {
                fsizeMap.put(frag);
            }
        }

        private final class FragmentExtantFault extends FragmentFault {
            FragmentExtantFault(Fragment frag) {
                super(frag);
            }
            public final String getDescription() {
                return "Fragment should not be in the size-ordered free map: "+frag;
            }
            public final void fix() {
                fsizeMap.remove(frag);
            }
        }
    }
}



