package freenet.fs.acct.sys;

import freenet.fs.acct.*;
import freenet.support.*;
import freenet.support.BinaryTree.Node;
import freenet.support.Comparable;
import freenet.support.RedBlackTree.AbstractRBNode;
import java.util.Vector;
import java.io.*;


/**
 * An in-memory tree of nodes corresponding to slots in the accounting table,
 * each containing a sub-tree that is serialized and deserialized based on
 * demand and LRU-caching.
 * 
 * An AccountingTree is not thread-safe!
 * 
 * @author tavin
 */
public class AccountingTree implements AccountingStruct, BinaryTree {

    public static void dump(AccountingTree tree, PrintWriter out)
                                            throws IOException {
        Node n = tree.indexTree.treeMin();
        while (n.hasParent())
            n = n.getParent();
        dump((BlockNode) n, out);
    }

    private static void dump(BlockNode bn, PrintWriter out) throws IOException {
        out.println("DUMPING: "+bn);
        Node n2 = bn.subTree().treeMin();
        while (n2.hasParent())
            n2 = n2.getParent();
        AbstractBinaryTree.dump(n2, out);
        if (bn.hasLeftChild())
            dump((BlockNode) bn.getLeftChild(), out);
        if (bn.hasRightChild())
            dump((BlockNode) bn.getRightChild(), out);
    }
    

    private final AccountingProcess proc;
    private final AccountingTreeMarshal marshal;
    
    private final Cache cache;
    
    private final BinaryTree indexTree = new RedBlackTree();
    
    
    public AccountingTree(AccountingProcess proc,
                          AccountingTreeMarshal marshal,
                          Cache cache) {
        this.proc = proc;
        this.marshal = marshal;
        this.cache = cache;
    }


    public final void found(int bnum, DataInput din) throws IOException {
        BlockNode bn = new BlockNode(bnum, din);
        found(bn);
    }

    public final void found(int bnum, DataInput din, BlockTransaction btx) throws IOException {
        BlockNode bn = new BlockNode(bnum, din, btx);
        found(bn);
        if (bn.subTree().getSerialWidth() > proc.getDataWidth())
            found(bn.split());
    }

    public final void found(BlockTransaction btx) throws IOException {
        BlockNode bn = new BlockNode(btx);
        found(bn);
        if (bn.subTree().getSerialWidth() > proc.getDataWidth())
            found(bn.split());
    }

    private final void found(BlockNode bn) {
        if (null != indexTree.treeInsert(bn, false))
            throw new AccountingException("duplicate block: "+bn);
        found(bn.subTree());
    }

    
    /**
     * Override me.
     */
    protected void found(BinaryTree subTree) {}


    
    public Node treeInsert(Node n, boolean replace) {
        BlockNode bn = (BlockNode) indexTree.treeMatch(n.getObject(), -1);
        if (bn == null) {
            bn = new BlockNode();
            bn.subTree().treeInsert(n, false);
            indexTree.treeInsert(bn, false);
            return null;
        }
        else {
            SerialTree st = bn.subTree();
            Node old = st.treeInsert(n, replace);

            if ((old == null || replace) && st.getSerialWidth() > proc.getDataWidth()
                && null != indexTree.treeInsert(bn.split(), false))
                throw new AccountingException("split block is a duplicate!?");

            return old;
        }
    }

    public final boolean treeRemove(Node n) {
        return treeRemove((AccountingTreeNode) n);
    }

    public boolean treeRemove(AccountingTreeNode atn) {
        SerialTree st = atn.owner;
        if (st == null) {
            return false;
        }
        else if (st.retired) {
            atn = (AccountingTreeNode) treeSearch(atn.getObject());
            if (atn == null)
                return false;
            st = atn.owner;
        }
        try {
            return st.treeRemove(atn);
        }
        finally {
            if (st.getSerialWidth() == 0) {
                BlockNode bn = (BlockNode) st.superTreeNode();
                BlockTransaction bt = st.transaction();
                if (bt != null)
                    proc.abandon(bt);
                if (bn.bnum != -1)
                    proc.retire(bn.bnum);
                st.retire();
                indexTree.treeRemove(bn);
            }
        }
    }

    public final Node treeRemove(Comparable searchKey) {
        Node n = treeSearch(searchKey);
        if (n == null)
            return null;
        treeRemove(n);
        return n;
    }


    public Node treeSearch(Comparable searchKey) {
        BlockNode bn = (BlockNode) indexTree.treeMaxConstrained(searchKey, true);
        return bn == null ? null : bn.subTree().treeSearch(searchKey);
    }

    public Node treeMatch(Comparable searchKey, int cmpSense) {
        BlockNode bn = (BlockNode) indexTree.treeMatch(searchKey, -1);
        if (bn == null)
            return null;
        Node n = bn.subTree().treeMatch(searchKey, cmpSense);
        if (cmpSense > 0 && n.getObject().compareTo(searchKey) < 0) {
            bn = (BlockNode) indexTree.treeSuccessor(bn);
            if (bn != null) {
                Node n2 = bn.subTree().treeMin();
                if (n2 != null)
                     n = n2;
            }
        }
        return n;
    }
        

    public Node treeMin() {
        BlockNode bn = (BlockNode) indexTree.treeMin();
        return bn == null ? null : bn.subTree().treeMin();
    }

    public Node treeMinConstrained(Comparable searchKey, boolean inclusive) {
        BlockNode bn = (BlockNode) indexTree.treeMaxConstrained(searchKey, true);
        if (bn == null) {
            return treeMin();
        }
        else {
            Node n = bn.subTree().treeMinConstrained(searchKey, inclusive);
            if (n == null) {
                bn = (BlockNode) indexTree.treeSuccessor(bn);
                return bn == null ? null : bn.subTree().treeMin();
            }
            else return n;
        }
    }

    public Node treeMax() {
        BlockNode bn = (BlockNode) indexTree.treeMax();
        return bn == null ? null : bn.subTree().treeMax();
    }

    public Node treeMaxConstrained(Comparable searchKey, boolean inclusive) {
        BlockNode bn = (BlockNode) indexTree.treeMaxConstrained(searchKey, inclusive);
        return bn == null ? null : bn.subTree().treeMaxConstrained(searchKey, inclusive);
    }


    public final Node treeSuccessor(Node n) {
        return treeSuccessor((AccountingTreeNode) n);
    }

    public Node treeSuccessor(AccountingTreeNode atn) {
        SerialTree st = atn.owner;
        if (st.retired) {
            return treeMinConstrained(atn.getObject(), false);
        }
        atn = (AccountingTreeNode) st.treeSuccessor(atn);
        if (atn == null) {
            BlockNode bn = (BlockNode) indexTree.treeSuccessor(st.superTreeNode());
            return bn == null ? null : bn.subTree().treeMin();
        }
        else return atn;
    }


    public final Node treePredecessor(Node n) {
        return treePredecessor((AccountingTreeNode) n);
    }

    public Node treePredecessor(AccountingTreeNode atn) {
        SerialTree st = atn.owner;
        if (st.retired) {
            return treeMaxConstrained(atn.getObject(), false);
        }
        atn = (AccountingTreeNode) st.treePredecessor(atn);
        if (atn == null) {
            BlockNode bn = (BlockNode) indexTree.treePredecessor(st.superTreeNode());
            return bn == null ? null : bn.subTree().treeMax();
        }
        else return atn;
    }
    

    public Walk treeWalk(boolean ascending) {
        return AbstractBinaryTree.treeWalk(this, ascending);
    }

    public Walk treeWalk(Node node, boolean ascending) {
        return AbstractBinaryTree.treeWalk(this, node, ascending);
    }

    public Walk treeWalk(Comparable searchKey, boolean inclusive, boolean ascending) {
        return AbstractBinaryTree.treeWalk(this, searchKey, inclusive, ascending);
    }
    


    private final Vector serialTreeVec = new Vector();
    
    /**
     * A node in the indexTree of accounting blocks.
     */
    private final class BlockNode extends AccountingTreeBlock
                                  implements BlockEntry {

        int bnum;

        BlockData data;


        BlockNode(int bnum, DataInput din) throws IOException {
            this(bnum, din, null);
            //System.out.println("cache(): "+data);
            cache.cache(data);
        }
        
        BlockNode(int bnum, DataInput din, BlockTransaction btx) throws IOException {
            this.bnum = bnum;
            SerialTree subTree;
            synchronized (serialTreeVec) {
                try {
                    if (din != null) {
                        SerialTree.parseIntoVector(marshal, din, serialTreeVec, this);
                    }
                    if (btx != null) {
                        din = proc.resume(btx, this);
                        SerialTree.parseIntoVector(marshal, din, serialTreeVec, this);
                    }
                    subTree = new SerialTree(this, marshal, serialTreeVec, btx);
                }
                finally {
                    serialTreeVec.removeAllElements();
                }
            }
            data = new BlockData(subTree);
        }

        BlockNode(BlockTransaction btx) throws IOException {
            this(-1, null, btx);
        }
        
        BlockNode() {
            this.bnum = -1;
            SerialTree subTree = new SerialTree(this, marshal, proc.create(this));
            data = new BlockData(subTree);
        }


        public final String toString() {
            if (data == null || data.subTree.transaction() == null)
                return proc + " / " + bnum;
            else
                return proc + " / " + bnum + " tx "
                       + data.subTree.transaction().getTransactionID();
        }

        
        BlockNode split() {
            BlockNode bn = new BlockNode();
            Walk walk = data.subTree.treeWalk(false);
            AccountingTreeNode atn;
            while (data.subTree.getSerialWidth() > proc.getDataWidth() >> 1
                   && null != (atn = (AccountingTreeNode) walk.getNext())) {
                data.subTree.treeRemove(atn);
                bn.data.subTree.treeInsert(new AccountingTreeNode(atn), false);
            }
            return bn;
        }
        

        SerialTree subTree() throws AccountingIOException {
            if (data == null) {
                SerialTree subTree;
                try {
                    DataInput din = proc.load(bnum);
                    if (din == null) {
                        throw new AccountingException("block disappeared from disk: "+this);
                    }
                    synchronized (serialTreeVec) {
                        try {
                            SerialTree.parseIntoVector(marshal, din, serialTreeVec, this);
                            subTree = new SerialTree(this, marshal, serialTreeVec, null);
                        }
                        finally {
                            serialTreeVec.removeAllElements();
                        }
                    }
                }
                catch (IOException e) {
                    throw new AccountingIOException(e);
                }
                data = new BlockData(subTree);
            }
            //System.out.println("promote(): "+data);
            cache.promote(data);
            return data.subTree;
        }


        public final void freeze(int dest, DataOutput out) throws IOException {
            bnum = dest;
            data.subTree.freeze(out);
            //System.out.println("cache(): "+data);
            cache.cache(data);
        }

        final BlockTransaction transact() {
            //System.out.println("uncache(): "+data);
            cache.uncache(data);
            return proc.modify(bnum, this);
        }
        

        private final class BlockData extends DoublyLinkedListImpl.Item
                                      implements Cacheable {

            SerialTree subTree;

            BlockData(SerialTree subTree) {
                this.subTree = subTree;
            }

            public final void drop() {
                data = null;
                subTree.retire();
            }
        }
    }
}



