package freenet.fs.acct;

import freenet.support.Comparator;
import freenet.support.Fields;
import java.io.*;

/**
 * A book-keeping device for the creation or modification of a block.
 * Provides an additional ordering based on transaction ID number,
 * via BlockTransaction.ComparatorByID.
 * @author tavin
 */
public class BlockTransaction extends BlockElement {

    private final int txid;
    private byte[] annot = new byte[0];

    
    BlockTransaction(int bnum, int txid) {
        super(bnum);
        this.txid = txid;
    }

    public String toString() {
        return super.toString() + " (transaction) ID: " + txid
               + "  " + Fields.bytesToHex(annot);
    }
    
    public final void annotate(byte[] annot) {
        byte[] a = new byte[this.annot.length + annot.length];
        System.arraycopy(this.annot, 0, a, 0, this.annot.length);
        System.arraycopy(annot, 0, a, this.annot.length, annot.length);
        this.annot = a;
    }
    
    public final int getAnnotationLength() {
        return annot.length;
    }

    public final byte[] getAnnotation() {
        return annot;
    }

    public final DataInput readAnnotation() {
        return new DataInputStream(new ByteArrayInputStream(annot));
    }
    
    public final int getTransactionID() {
        return txid;
    }

    public final int getRecordLength() {
        return AnnotationBlock.getRecordLength(this);
    }

    final byte[] extractAnnotation() {
        if (annot.length > 0) {
            try {
                return getAnnotation();
            }
            finally {
                resetAnnotation();
            }
        }
        else return null;
    }

    final void resetAnnotation() {
        annot = new byte[0];
    }

    static final class ComparatorByID implements Comparator {
        public final int compare(Object o1, Object o2) {
            return compare((BlockTransaction) o1, (BlockTransaction) o2);
        }
        public final int compare(BlockTransaction bt1, BlockTransaction bt2) {
            return bt1.txid == bt2.txid ? 0 : (bt1.txid > bt2.txid ? 1 : -1);
        }
    }

    static final class ComparatorByAnnotationLength implements Comparator {
        public final int compare(Object o1, Object o2) {
            return compare((BlockTransaction) o1, (BlockTransaction) o2);
        }
        public final int compare(BlockTransaction bt1, BlockTransaction bt2) {
            return bt1.getAnnotationLength() == bt2.getAnnotationLength()
                   ? 0 : (bt1.getAnnotationLength() > bt2.getAnnotationLength() ? 1 : -1);
        }
    }
}


