package freenet.fs;

import freenet.fs.acct.Fragment;
import java.io.*;

/**
 * A WriteLock protects against reading or writing 
 * to a byte range while the lock is in place.
 *
 * @author Scott
 * @author tavin
 */
public class WriteLock extends Lock {

    public WriteLock(LockGrantor lg, long lo, long hi) {
        super(lg, lo, hi);
    }

    public final int getTypeMask() {
        return WRITE;
    }

    public final int getProtectionMask() {
        return ALL;
    }

    public final String getTypeName() {
        return "WRITE";
    }

    public final OutputStream getStream() throws IOException {
        return new WriteLockOutputStream(ticket.getOutputStream());
    }

    /**
     * Writes to the locked range, staying behind conflicting locks.
     */
    private class WriteLockOutputStream extends FilterOutputStream {
        
        WriteLockOutputStream(OutputStream out) {
            super(out);
        }

        public void write(int b) throws IOException {
            try {
                waitAvailable(1);
                out.write(b);
                move(1);
            }
            catch (LockException e) {
                throw new EOFException(""+e);
            }
        }

        public void write(byte[] buf, int off, int len) throws IOException {
            try {
                waitAvailable(len);
                out.write(buf, off, len);
                move(len);
            }
            catch (LockException e) {
                throw new EOFException(""+e);
            }
        }

        public void close() throws IOException {
            try {
                out.close();
            }
            finally {
                unlock();
            }
        }
        
        protected void finalize() throws Throwable {
            close();
        }
    }

    /**
     * Creates a stream that invisibly handles the locking and unlocking.
     */
    public static OutputStream getOutputStream(LockGrantor lg, long lo, long hi)
                                                            throws IOException {
        WriteLock wl = new WriteLock(lg, lo, hi);
        boolean worked = false;
        try {
            OutputStream ret = wl.getStream();
            worked = true;
            return ret;
        }
        finally {
            if (!worked)
                wl.unlock();
        }
    }

    public static OutputStream getOutputStream(LockGrantor lg, Fragment[] ranges)
                                                            throws IOException {
        WriteLock[] locks = new WriteLock[ranges.length];
        synchronized (lg.semaphore()) {
            for (int i=0; i<ranges.length; ++i) {
                locks[i] = new WriteLock(lg,
                                         ranges[i].getLowerBound(),
                                         ranges[i].getUpperBound());
            }
        }
        boolean worked = false;
        try {
            OutputStream ret = new LockedOutputStream(locks);
            worked = true;
            return ret;
        }
        finally {
            if (!worked) {
                for (int i=0; i<locks.length; ++i)
                    locks[i].unlock();
            }
        }
    }
}


