package freenet.fs;

import java.io.*;

/**
 * Similar to the RAFStorage, but only uses one file, and
 * allows the file size to be increased without limit.
 * @see RAFStorage
 * @author tavin
 */
public class UnlimitedRAFStorage implements Storage {

    private final File file;

    
    public void initialize(LockGrantor lg) {}
    
    
    /**
     * @param file  the file to use
     */
    public UnlimitedRAFStorage(File file) {
        this.file = file;
    }

    public final long size() {
        return file.length();
    }

    public final long truncation() {
        return file.length();
    }

    
    public final InputStream getInputStream(long start, long end) throws IOException {
        return new FSInputStream(start);
    }

    private class FSInputStream extends InputStream {

        private final RandomAccessFile raf;
        
        private FSInputStream(long pos) throws IOException {
            raf = new RandomAccessFile(file, "r");
            raf.seek(pos);
        }

        public final int read() throws IOException {
            return raf.read();
        }

        public final int read(byte[] buf, int off, int len) throws IOException {
            return raf.read(buf, off, len);
        }
        
        public final void close() throws IOException {
            raf.close();
        }

        protected void finalize() throws IOException {
            close();
        }
    }

    
    public final OutputStream getOutputStream(long start, long end) throws IOException {
        return new FSOutputStream(start);
    }
    
    private class FSOutputStream extends OutputStream {

        private final RandomAccessFile raf;
        private boolean closed = false;
        
        private FSOutputStream(long pos) throws IOException {
            raf = new RandomAccessFile(file, "rw");
            raf.seek(pos);
        }

        public void write(int b) throws IOException {
            raf.write(b);
        }

        public void write(byte[] buf, int off, int len) throws IOException {
            raf.write(buf, off, len);
        }

        public void flush() throws IOException {
            raf.getFD().sync();
        }

        public void close() throws IOException {
            if (!closed) {
                try {
                    closed = true;
                    raf.getFD().sync();
                }
                finally {
                    raf.close();
                }
            }
        }

        protected void finalize() throws IOException {
            close();
        }
    }
}


