package freenet.fs;

import freenet.support.Fields;

/**
 * Locks provide access to a given range of bytes on a first-come,
 * first-served basis where the byte pointer of each lock must stay
 * behind the byte pointers of all (conflicting) locks that were
 * granted earlier -- i.e., "sliding locks".
 *
 * @author Scott
 * @author tavin
 *
 * (I changed this from an interface to an abstract class
 *  and added the more generalized sliding code.)
 */
public abstract class Lock {

    /** typemask constants */
    public static final int NONE  = 0x00,
                            READ  = 0x01,
                            WRITE = 0x02,
                            ALL   = 0xff;


    /** to give notification when we move */
    final LockTicket ticket;
    
    /** tracks the locks we are blocking on */
    final LockSlide slide;

    /** last known number of available bytes */
    private long available;


    /**
     * @param lg  lock granting authority
     * @param lo  lower bound of locked range
     * @param hi  upper bound of locked range
     */
    Lock(LockGrantor lg, long lo, long hi) {
        ticket = new LockTicket(getTypeMask(), lo, hi);
        slide = lg.grant(ticket, getProtectionMask());
        getAvailable();
        // this seems to slow things down a lot (too much output anyway)
        //freenet.Core.logger.log(this,
        //                        ticket + " / " + slide + " / "
        //                        + Fields.longToHex(available) + " available",
        //                        freenet.Core.logger.DEBUG);
    }

    /**
     * @return  total length of locked range
     */
    public final long getLength() {
        return ticket.hi + 1 - ticket.lo;
    }

    /**
     * @return  the lower bound of the locked range
     */
    public final long getLowerBound() {
        return ticket.lo;
    }

    /**
     * @return  the upper bound of the locked range
     */
    public final long getUpperBound() {
        return ticket.hi;
    }

    /**
     * @return  the current byte-pointer within the locked range
     */
    public final long getPos() {
        return ticket.pos;
    }

    /**
     * @return  number of bytes left in the range
     *          (0 if already unlocked)
     */
    public final long getBytesLeft() {
        return ticket.pos == -1 ? 0 : ticket.hi + 1 - ticket.pos;
    }

    /**
     * Advance the byte-pointer and wake up any blocking threads,
     * if warranted.
     * @throws LockException
     *         if the lock's already unlocked,
     *         or we're not going forwards
     */
    public final void move(long n) throws LockException {
        if (n <= 0 || n > available) {
            throw new LockException("Lock.move(): n = " + n +
                                    ", available = " + available +
                                    ", ticket = " + ticket);
        }
        ticket.move(n);
        if (ticket.pos > ticket.hi)
            unlock();
        else
            available -= n;
    }

    /**
     * @return  the number of bytes that may be read/written
     *          without blocking
     */
    public final long getAvailable() {
        long slidePos = slide.getSlidePos();
        available = slidePos == -1
                    ? getBytesLeft()
                    : Math.min(slidePos - ticket.pos, getBytesLeft());
        return available;
    }

    /**
     * Blocks until some requested number of bytes are available.
     * @throws LockException
     *         if the requested number exceeds the lock's upper limit
     */
    public final long waitAvailable(long a) throws LockException {
        if (a > getBytesLeft()) {
            throw new LockException("Lock.waitAvailable(): a = " + a +
                                    ", ticket = " + ticket);
        }
        while (a > available) {
            long slidePos = slide.waitForSlide(ticket.pos + a);
            available = slidePos == -1
                        ? getBytesLeft()
                        : Math.min(slidePos - ticket.pos, getBytesLeft());
        }
        return available;
    }

    /**
     * Unlocks the lock, waking up any threads blocking on it.
     */
    public final void unlock() {
        ticket.unlock();
        available = 0;
    }

    /**
     * Try to make sure locks get unlocked..
     */
    protected final void finalize() {
        ticket.unlock();
    }

    /**
     * @return  type name, e.g., "READ" or "WRITE"
     */
    public abstract String getTypeName();

    /** 
     * @return  the lock type, e.g. READ or WRITE
     */
    public abstract int getTypeMask();
    
    /**
     * @return  the type masks that block this lock
     */
    public abstract int getProtectionMask();
    
    /**
     * @return  string description of the lock
     * example: "(READ: 10->1f)"
     */
    public final String toString() {
        return '('+getTypeName()+": "
               +Fields.longToHex(ticket.lo)+"->"+Fields.longToHex(ticket.hi)
               +')';
    }        
}


