package freenet.client.metadata;
import freenet.client.*;
import freenet.FieldSet;
import freenet.config.PropertySet;
import freenet.support.io.ReadInputStream;
import freenet.support.io.WriteOutputStream;
import freenet.support.Fields;
import freenet.support.Bucket;
import freenet.support.ArrayBucket;
import freenet.support.BucketFactory;
import freenet.support.Loader;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.lang.reflect.InvocationTargetException;
import java.io.*;
/**
 * Parses a metadata stream and produces a series of parts.
 * @author oskar
 */
public class Metadata {

    public static final int VERSION = 1;
    
    private final Hashtable commands = new Hashtable();
    private VersionCommand version;
    private ArrayBucket trailing;

    /**
     * Create a new Metadata object.
     * @param props  Properties to use when handeling metadata.
     */
    public Metadata(VersionCommand v) {
        this.version = v;
    }

    /**
     * Create a new Metadata object.
     */
    public Metadata() {
        this.version = new VersionCommand(this);
    }


    /**
     * Create a new Metadata object from a wire.
     * @param props  Properties to use when heandeling metadata.
     * @param in     InputStream to read.
     */
    public Metadata(InputStream metadata) 
        throws InvalidPartException, IOException {

        parse(metadata);
    }

    /**
     * Returns whatever is left on the trailing field after all the 
     * standard metadata is read.
     */
    public InputStream getTrailing() {
        return trailing.getInputStream();
    }

    /**
     * Add a DocumentCommand. Note that there can only be one that contains
     * a trailing field - if this does, any previous such will be overwritten.
     * (Which also mean, do NOT add a trailing later - Bad things will happen).
     */
    public void addCommand(DocumentCommand dc) {
        commands.put(dc.getName(), dc);
    }

    public void addDocument(DocumentCommand dc) {
        addCommand(dc);
    }

    public static int revision() {
        return VERSION;
    }

    public VersionCommand getVersion() {
        return version;
    }

    protected void parse(InputStream metadata) 
        throws InvalidPartException, IOException {

        ReadInputStream in = new ReadInputStream(metadata);

        String name;
        try {
            name = in.readTo('\n', '\r');
        } catch (EOFException e) {
            version = new VersionCommand(this);
            return; // I'm tolerating completely empty metadata 
        }

        if (!name.equals("Version"))
            throw new InvalidPartException("Must start with version");
     
        FieldSet fs = new FieldSet();
        String end = fs.parseFields(in);
        version = new VersionCommand(this, fs);

        while (end.equals("EndPart")) {
            fs = new FieldSet();
            name = in.readTo('\n','\r');

            if (!"Document".equals(name))
                throw new InvalidPartException("Document command expected. "
                                               + " Got: " + name);
            end = fs.parseFields(in);
            DocumentCommand d = new DocumentCommand(fs);
            commands.put(d.getName(), d);
        }

        if (!end.equals("End")) {
            throw new InvalidPartException("Malformed endstring: " + end);
        }

        ArrayBucket ab = new ArrayBucket();
        ab.read(in);
        if (ab.size() > 0)
            this.trailing = ab;
    }

    public void writeTo(OutputStream rout) throws IOException {
        WriteOutputStream out = new WriteOutputStream(rout);
        //System.err.println("LALA got to writeto");
        out.writeUTF("Version", '\n');
        FieldSet fs = version.toFieldSet();
        Enumeration e = commands.elements();
        fs.writeFields(out, e.hasMoreElements() ? "EndPart" : "End"); 
        while(e.hasMoreElements()) {
            fs = ((DocumentCommand) e.nextElement()).toFieldSet();
            //System.err.println("LALA WRITING: " + fs.toString());
            out.writeUTF("Document",'\n');
            fs.writeFields(out, e.hasMoreElements() ? "EndPart" : "End"); 
        }
        // copy trailing
        byte[] b = new byte[0xffff];
        int i;

        if (trailing != null) {
            InputStream in = trailing.getInputStream();
            while ((i = in.read(b)) != -1) {
                out.write(b, 0, i);
            }
        }
        out.flush();
    }

    public DocumentCommand getDocument(String name) {
        return (DocumentCommand) commands.get(name);
    }

    public DocumentCommand getDefaultDocument() {
        return (DocumentCommand) commands.get("");
    }

    public Enumeration getDocumentNames() {
        return commands.keys();
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        for (Enumeration e = commands.elements() ; e.hasMoreElements();) {
            sb.append(e.nextElement());
            if (e.hasMoreElements())
                sb.append(", ");
        }
        return sb.toString();
    }



    public String getMimeType(String defaultMimeType) {
	for (Enumeration e = commands.elements() ; e.hasMoreElements();) {
            DocumentCommand command = (DocumentCommand)(e.nextElement());
	    MetadataPart[] parts = command.getNonControlParts();
	    int i;
	    for (i=0; i< parts.length; i++) {
		if (parts[i] instanceof InfoPart) {
		    String format = ((InfoPart)parts[i]).format();
		    if (format != null) {
			return format;
		    }
		}
	    }
        }

	return defaultMimeType;
    }

    // hmmmm... This may falsely return a SplitFile for 
    // complex metadata.  Good enough for now.
    public SplitFile getSplitFile() {
	for (Enumeration e = commands.elements() ; e.hasMoreElements();) {
            DocumentCommand command = (DocumentCommand)(e.nextElement());
	    MetadataPart part = command.getControlPart();
	    if (part instanceof SplitFile) {
		return (SplitFile)part;
	    }
        }
	return null;
    }
}


















