/*
  This code is part of fproxy, an HTTP proxy server for Freenet.
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

package freenet.client.http;

import freenet.keys.SVK;
import freenet.Core;
import freenet.support.*;
import freenet.support.mime.*;
import freenet.client.*;
import freenet.client.metadata.*;
import freenet.client.events.StateReachedEvent;
import freenet.client.events.TransferEvent;
import freenet.client.events.TransferCompletedEvent;
import freenet.client.listeners.ClientCollisionListener; // hmmm build issue gone?

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.CharArrayWriter;

import javax.servlet.*;
import javax.servlet.http.*;

// REDFLAG: Cleanly factor SplitFile inserting into a 
//          separate component.  LATER.

/**
 * Implementation helper class runs insert requests 
 * in a separate thread.
 **/
class InsertContext implements Reapable, Runnable {
    public final int LIFETIME_MS = 1800000;
    public final String CIPHER = "Twofish";

    final static String PREFIX = "/__INTERNAL__00/";
    private final static String CMD_CANCEL = "cancel";

    String contextID = null;

    StringBuffer statusMsg = new StringBuffer("");
    long deathTime = -1;
    volatile boolean done = false;
    volatile boolean success = false;

    // Args
    String key = null;
    MIME_binary dataPart = null;
    int htl = -1;
    int nThreads = -1;
    int retries = 2;
    String contentType = null;
    String encoderName = null;
    BucketFactory bucketFactory = null;
    ClientFactory clientFactory = null;
    FECFactory fecFactory = null;
    ContextManager contextManager = null;

    AutoRequester requester = null;
    BlockInserter blockInserter = null;
    boolean canceling = false;
    Thread thread = null;

    // Dumps progress messages into statusMsg.
    ClientEventListener listener = new InsertContextEventLog();
 
    ////////////////////////////////////////////////////////////
    final static void handle(InsertContext context, String urlValue,
                             HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {

        //  if (!urlValue.startsWith(PREFIX)) {
        //     throw new RuntimeException("Assertion Failure: Bad Request");
        //  }

        if (context != null) {
                // Let it render the page.
            context.renderPage(urlValue, resp);
            return;
        }
        
        // We can get here if the Reaper released the context.
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        PrintWriter pw = resp.getWriter();
        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Insert Status");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body bgcolor=\"#ffffff\">");
        pw.println("<h1>Unknown, finished, or canceled insert request!</h1>");
        pw.println("<p><a href=\"/\">Done</a>.</p> \n");
        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer();
    }

    private final void renderPage(String url, HttpServletResponse resp)
        throws IOException {
        
        String baseUrl = "/"; // REDFLAG: do better...
        
        if (url.endsWith(CMD_CANCEL)) {
            cancel();
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.setContentType("text/html");
            PrintWriter pw = resp.getWriter();
            pw.println("<html>");
            pw.println("<head>");
            pw.println("<title>");
            pw.println("Canceled Insert");
            pw.println("</title>");
            pw.println("</head>");
            pw.println("<body bgcolor=\"#ffffff\">");
            pw.println("<h1>Canceled Insert</h1>");
            pw.println("<p><a href=\"" + baseUrl + "\">Done</a>.</p> \n");
            pw.println("</body>");
            pw.println("</html>");
            resp.flushBuffer();
            return;
        }
        
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        PrintWriter pw = resp.getWriter();
        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Insert Status");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body bgcolor=\"#ffffff\">");
        
        if (blockInserter != null) {
            // Special case for split file insertion.
            pw.println("<hr>");
            pw.println("Inserting a split file can consume a lot of resources.  Use the link");
            pw.println("below if you want to stop the insert cleanly. <p>");
            pw.println("<p><a href=\"" + PREFIX + contextID +
                       "/" + CMD_CANCEL +
                       "/\">Cancel Insert</a> </p> \n");
        }
        
        pw.println("<hr> <pre>");
        pw.flush();
        int pos = 0;
        
        try {
            String msg = null;
            synchronized (this) {
                while (!getDone()) {
                    msg = getStatusMsg();
                    if (msg.length() > pos) {
                        pw.write(msg.substring(pos));
                        pw.flush();
                        pos = msg.length(); 
                    }
                    wait();
                } 
                
                // REDFLAG: butt ugly, but what if we are already done?
                msg = getStatusMsg();
                if (msg.length() > pos) {
                    pw.write(msg.substring(pos));
                    pw.flush();
                    pos = msg.length(); 
                }
            }
        }
        catch (InterruptedException ie) {
            // REDFLAG
        }
        
        pw.println("</pre><hr>");
        
        String exitStatus = "<h1> Insert Succeeded. </h1>";
        if (!getSucceeded()) {
            exitStatus = "<h1> Insert Failed. </h1>";
        }
        
        pw.println(exitStatus);
        
        pw.println("<p><a href=\"" + baseUrl + "\">Done</a>.</p> \n");
        pw.println("</body></html>");

        pw.flush();
        resp.flushBuffer();
        
        // Release context immediately, instead of 
        // waiting for Reaper to kill it.
        reap();
    }
    ////////////////////////////////////////////////////////////

    // Setting encoderName != null causes
    // the data to be inserted as a redundant splitfile.
    public InsertContext(Reaper reaper,
			 ClientFactory clientFactory,
                         BucketFactory bucketFactory,
                         FECFactory fecFactory,
                         ContextManager contextManager,
			 String key,
			 MIME_binary dataPart,
			 int htl,
                         int nThreads,
			 String contentType,
                         String encoderName) {

        if (encoderName != null) {
            this.blockInserter = new BlockInserter(clientFactory);
            blockInserter.setBlockListener( new BlockInserterEventLog() );
            this.statusMsg.append("Inserting SplitFile: " + key +"\n\n");
        }
        else {
            this.requester = new AutoRequester(clientFactory);
            this.statusMsg.append("Inserting: " + key +"\n\n");
        }
        this.clientFactory = clientFactory;
        this.bucketFactory = bucketFactory;
        this.fecFactory = fecFactory;
        this.contextManager = contextManager;

	this.key = key;
	this.dataPart = dataPart;
	this.htl = htl;
        this.nThreads = nThreads;
	this.contentType = contentType;
        this.encoderName = encoderName;

	this.contextID = contextManager.add(this);
        touch();
	reaper.add(this);
    }

    public final synchronized String getStatusMsg() { return statusMsg.toString(); }
    public final String getId() { return contextID; }
    public final synchronized boolean getDone() { return done; }
    public final synchronized boolean getSucceeded() { return success; }

    // REDFLAG: suspect. think this through...
    public final void cancel() {
        synchronized (this) {
            canceling = true;
        }

        if (blockInserter != null) {
            // This is a synchronized call.
            blockInserter.cancel();
        }

        synchronized (this) {
            if ((thread != null) && 
            (thread != Thread.currentThread())) {
                thread.interrupt();
            }
        }
    }

    private final Bucket zeroPad(Bucket b, int blockSize) throws IOException {
        // Copy even when we don't need to to make
        // cleanup simpler.  
        //
        //if (b.size() == blockSize) {
        //    return b;
        //}
        if (b.size() > blockSize) {
            throw new IllegalArgumentException("Bucket too big.");
        }

        final int BUFSIZE = 4096;

        Bucket ret = null;
        Bucket newBucket = null;
        InputStream in = null;
        OutputStream out = null; 
        byte[] buf = new byte[BUFSIZE];
        try {
            newBucket = bucketFactory.makeBucket(blockSize);
            in = b.getInputStream();
            out = newBucket.getOutputStream();
            int bytes = 0;
            // Copy
            while ((bytes = in.read(buf)) > 0) {
                out.write(buf, 0, bytes);
            }
            
            // Zero pad.
            for(int i = 0; i < buf.length; i++) {
                buf[i] = 0;
            }
            bytes = blockSize - (int)b.size();
            while (bytes > 0) {
                int nWrite = buf.length;
                if (bytes < nWrite) {
                    nWrite = bytes;
                }
                out.write(buf, 0, nWrite);
                bytes -= nWrite;
            }
            out.close(); out = null;
            in.close(); in = null;
            ret = newBucket; newBucket = null;
        }
        finally {
            if (out != null) {
                try { out.close(); } catch (Exception e) {}
            }
            if (in != null) {
                try { in.close(); } catch (Exception e) {}
            }
            if (newBucket != null) {
                try { bucketFactory.freeBucket(newBucket); } catch (Exception e) {}
            }
        }
        return ret;
    }

    // How do we handle failure???
    private String[] insertBlocks(Bucket[] blocks, int htl, int retries)
        throws IOException, InterruptedException {

        retries++;
        String[] ret = new String[blocks.length];
        while (!canceling) {
            int count = 0;
            int i = 0;
            for (i = 0; i < ret.length; i++) {
                if (ret[i] == null) {
                    count++;
                }
            }
            if (count == 0) {
                return ret;
            }
            
            retries--;
            if (retries < 0) {
                throw new IOException("One or more blocks failed to insert"); 
            }

            Bucket[] working = new Bucket[count];
            int[] indices = new int[count];
            count = 0;
            for (i = 0; i < ret.length; i++) {
                if (ret[i] == null) {
                    working[count] = blocks[i];
                    indices[count] = i;
                    count++;
                }
            }

            println("Inserting " + working.length + " blocks into Freenet.");

            synchronized (blockInserter) {
                System.err.println("InsertContext.insertBlocks -- nThreads: " + nThreads);
                if (!canceling) {
                    blockInserter.start(working, htl, nThreads, false);
                    while (blockInserter.isRunning()) {
                        blockInserter.wait();
                    }
                }
            }

            String[] uris = blockInserter.getURIs();
            
            for (i = 0; i < uris.length; i++) {
                if (uris[i] != null) {
                    ret[indices[i]] = uris[i];
                }
            }
        }
        
        // Canceling gets us here.
        throw new IOException("One or more blocks failed to insert"); 
    }

    private final String[] insertSplitFileSegment(FileBucket data, FECEncoder encoder, 
                                                  int segment, int htl) 
        throws IOException, InterruptedException {
        
        Bucket[] dataBlocks = null;
        Bucket[] checkBlocks = null;
        String[] ret = null;
        try {
            int blockSize = encoder.getBlockSize();
            int n = encoder.getN(segment);
            int k = encoder.getK(segment);
            int segLen = encoder.getSegmentSize();

            // Segment file.
            dataBlocks = RandomAccessFileBucket.segment(data.getFile(), blockSize,
                                                        segLen * segment, k, true);

            if (dataBlocks.length != k) {
                throw new RuntimeException("Assertion Failure: dataBlocks.length != k");
            }
        
            // Zero pad last block.
            dataBlocks[dataBlocks.length - 1] = zeroPad(dataBlocks[dataBlocks.length - 1], blockSize);

            // Do magic.
            println("Creating " + (n-k) + " " + blockSize + " byte check blocks.");
            println("Be patient. This can take a long time...");
            checkBlocks = encoder.encode(segment, dataBlocks);
            println("Done FEC encoding.");
            Bucket[] allBlocks = new Bucket[dataBlocks.length + checkBlocks.length];
            System.arraycopy(dataBlocks, 0, allBlocks, 0, dataBlocks.length);
            System.arraycopy(checkBlocks, 0, allBlocks, dataBlocks.length, checkBlocks.length);

            // Handles retrying, throws on failure after retry.
            ret = insertBlocks(allBlocks, htl, retries);
        }
        finally {
            // We only need to free the Bucket allocated by zeroPad.
            // The the buckets created by RandomAccessFile.segment
            // don't own the file they reference.
            if (dataBlocks != null) {
                try { bucketFactory.freeBucket(dataBlocks[dataBlocks.length - 1]); } 
                catch (Exception e) {}
            }
            if (checkBlocks != null) {
                for (int i = 0; i < checkBlocks.length; i++) {
                    if (checkBlocks[i] == null) {
                        continue;
                    }
                    try { bucketFactory.freeBucket(checkBlocks[i]); } 
                    catch (Exception e) {}
                }
            }
        }

        return ret;
    } 

    // Caller is responsible for deleting the data bucket.
    private SplitFile insertSplitFileBlocks(FileBucket data, int htl) 
        throws IOException, InterruptedException {
        
        // Full speed ahead and damn the torpedos!
        // REDFLAG: 0) experiments on max n, 1) striping.
        //if (data.size() > 1024 * 1024 * 32) {
        //    throw new IOException("File too big for current FEC implementation.");
        //}

        FECEncoder encoder = null;
        SplitFile ret = null;
        try {
            // Get FEC encoder.
            encoder = fecFactory.getEncoder(encoderName);
            if (encoder == null) {
                throw new IOException("Couldn't get FEC encoder: " + encoderName);
            }
            
            if (!encoder.init((int)data.size(), bucketFactory)) {
                throw new IOException("Couldn't initialize FEC encoder: " + encoderName);
            }
            
            int nDataBlocks = 0;
            int nCheckBlocks = 0;
            int i;
            for (i = 0; i < encoder.getSegmentCount();  i++) {
               final int n = encoder.getN(i);
               final int k = encoder.getK(i);
               nDataBlocks += k;
               nCheckBlocks += n - k;
            }

            int dataOffset = 0;
            int checkOffset = 0;
            String[] dataURIs = new String[nDataBlocks];
            String[] checkURIs = new String[nCheckBlocks];

            for (i = 0; i < encoder.getSegmentCount(); i++) {
                println("\nProcessing segment " + (i + 1) + " of " + 
                        encoder.getSegmentCount() + ".");
                String[] uris = insertSplitFileSegment(data, encoder, i, htl);
                final int n = encoder.getN(i);
                final int k = encoder.getK(i);
                if (uris.length != n) {
                    throw new RuntimeException("Assertion Failure: uris.length != n");
                }
                
                System.arraycopy(uris, 0, dataURIs, dataOffset, k);
                dataOffset += k;
                System.arraycopy(uris, k, checkURIs, checkOffset, n - k);
                checkOffset += n - k;
                println("Segment " + (i + 1) + " inserted. \n");
            }

            ret = encoder.makeMetadata(dataURIs, checkURIs);
        }
        finally {
            if (encoder != null) {
                try {
                    encoder.release();
                }
                catch (Exception e) {
                    // This should never happen, but people will
                    // write crappy FECEncoder implementations...
                }
            }
        }

        return ret;
    }

    // Caller must delete bucket.
    private final Bucket makeMetaData(SplitFile sf, String mimeType) throws IOException {
        
        Bucket ret = null;
        Bucket meta = null;
        OutputStream out = null;
        try {
            DocumentCommand doc = new DocumentCommand();
            doc.addPart(new InfoPart("Redundant SplitFile created by FProxy",
                                     mimeType));
            doc.addPart(sf);
            
            Metadata md = new Metadata();
            md.addDocument(doc);
        
            meta = bucketFactory.makeBucket(-1); // REDFLAG: ???
            out = meta.getOutputStream();
            md.writeTo(out);
            ret = meta;
            meta = null;
        }
        catch (InvalidPartException ivp) {
            throw new IOException(ivp.toString());
        }
        finally {
            if (out != null) {
                try { out.close(); } catch(Exception e) {}
            }
            if (meta != null) {
                try { bucketFactory.freeBucket(meta); } catch (Exception e) {}
            }
        }

        return ret;
    }

    
    // PESTER: Official DoneListener doesn't keep success
    //         status ????
    private class MyDoneListener implements ClientEventListener {
        public void receive(ClientEvent ce) {
            if (!(ce instanceof StateReachedEvent)) {
                return; 
            }

            StateReachedEvent sr = (StateReachedEvent) ce;
            
            if (sr.getState() == Request.FAILED || 
                sr.getState() == Request.DONE ||
                sr.getState() == Request.CANCELLED) {
                synchronized (MyDoneListener.this) {
                    done = true;
                    success = sr.getState() == Request.DONE;
                    MyDoneListener.this.notifyAll();  
                }
            }
        }
        
        public final synchronized boolean isDone() { return done; }
        public final synchronized boolean getSucceeded() { return success; }

        private boolean done = false;
        private boolean success = false;
    }


    private final String doInsert(Bucket meta, String uri) 
        throws IOException, InterruptedException, InsertSizeException {

        if (canceling) {
            return null;
        }

        PutRequest request = new PutRequest(htl, uri, CIPHER, meta, new NullBucket());
        ClientCollisionListener collision = new ClientCollisionListener();
        MyDoneListener done = new MyDoneListener();
        request.addEventListener(collision);
        request.addEventListener(done);

        Client c = clientFactory.getClient(request);

        // PESTER: blockingRun eats interrupted exception.
        c.start();
        synchronized (done) {
            while (!done.isDone() && !canceling) {
                done.wait();
            }
        }
        
        if (!done.isDone()) {
            // Handle asynchronous canceling.
            c.cancel();
        }

        if (!done.getSucceeded()) {
            if ((uri.indexOf("CHK@") != -1) &&
                collision.collisionHappened()) {
                println("CHK already existed in Freenet.");
            }
            else {
                return null;
            }
        }
        return request.getURI().toString();
    }

    private final String insertMetadata(Bucket meta, String uri) 
        throws IOException, InterruptedException, InsertSizeException {
        
        if (canceling) {
            return null;
        }

        FreenetURI finalURI = new FreenetURI(uri);
        if (!finalURI.getKeyType().equals("CHK") &&
            meta.size()  > SVK.SVK_MAXSIZE) {

            // Do redirect for really huge SplitFiles.
            String chkURI = doInsert(meta, "freenet:CHK@");
            if (chkURI == null) {
                return null;
            }

            println("\nFinished inserting CHK:  " + chkURI);
            println("Inserting redirect...\n");

            Redirect redirect = new Redirect(new FreenetURI(chkURI));
            
            Bucket tmpBucket = null;
            OutputStream out = null;
            String insertedURI = null;
            try {
                DocumentCommand doc = new DocumentCommand();
                doc.addPart(redirect);
                
                Metadata md = new Metadata();
                md.addDocument(doc);
                
                tmpBucket = bucketFactory.makeBucket(-1); // REDFLAG: ???
                out = tmpBucket.getOutputStream();
                md.writeTo(out);
                out.close();
                out = null;
                insertedURI = doInsert(tmpBucket, uri);
            }
            catch (InvalidPartException ivp) {
                throw new IOException(ivp.toString());
            }
            finally {
                if (out != null) {
                    try { out.close(); } catch(Exception e) {}
                }
                if (tmpBucket != null) {
                    try { bucketFactory.freeBucket(tmpBucket); } catch (Exception e) {}
                }
            }
            
            return insertedURI;
        }

        return doInsert(meta, uri);
    }

    public void run() {

        synchronized (InsertContext.this) {
            thread = Thread.currentThread();
        }

	Core.logger.log(this, "InsertContext.run -- started.",
			Logger.DEBUGGING);
        Bucket meta = null;
	try {
            // Insert redundant SplitFile
            if (encoderName != null) {
                // hmmm... Ugly cast required so that we can do in place file segmentation.
                meta = makeMetaData(insertSplitFileBlocks((FileBucket)dataPart.getBody(), htl),
                                    contentType);
                
                println("\nData and check blocks inserted. inserting SplitFile metadata...\n");

                // Force URI into canonical form.
                String uri = insertMetadata(meta, (new FreenetURI(key)).toString());
                if (uri != null) {
                    println("INSERTED SPLITFILE: " + uri);
                    success = true;
                    synchronized (this) {
                        notifyAll();
                    }
                }
                else {
                    if (!canceling) {
                        println("INSERT FAILED: Couldn't insert splitfile metadata.");
                    }
                    else {
                        println("INSERT CANCELED.");
                    }

                    synchronized (this) {
                        notifyAll();
                    }
                }
            }
            else {
                // Handle non-redundant insert.
                requester.addEventListener(listener);
                if (dataPart.getBody().size() < SVK.SVK_MAXSIZE - 2048) {
                    // 2048 is an arbitrary fudge factor to take the size
                    // of the metadata into account.
                    // PESTER: Why doesn't AutoRequester default to non-redirected
                    //         insertion of SVK keys which are small enough? FIX.
                    requester.doRedirect(false);
                }

                if (requester.doPut(new FreenetURI(key) , dataPart.getBody(), htl, contentType)) {
                    // hmmm... need to collect stuff like final uri.
                    println("INSERTED: " + requester.getKey());
                    success = true;
                    synchronized (this) {
                        notifyAll();
                    }
                }
                else {
                    //System.err.println("failed: " + requester.getError());
                    println("INSERT FAILED:");
                    //System.err.println("failed (1): " + requester.getError());
                    
                    if (requester.getError() != null) {
                        println(requester.getError());
                    }
                }
            }
	}
	catch (Exception e) {
            e.printStackTrace();
            // Don't log spurious errors.
            if (!canceling) {
                println("UNEXPECTED EXCEPTION: " + e.toString());
                CharArrayWriter stackTrace = new CharArrayWriter();
                e.printStackTrace(new PrintWriter(stackTrace));
                println(stackTrace.toString());
            }
        }
	finally {
	    try { 
		if(dataPart != null) {
                    dataPart.freeBody();
                    dataPart = null;
                }
	    }
	    catch (IOException ioe) {
		Core.logger.log(this, "Couldn't release insert data bucket:" + ioe,
				Logger.DEBUGGING);
	    }

            if (meta != null) {
                try { bucketFactory.freeBucket(meta); } catch (Exception e) {}
            }

            if (requester != null) {
                requester.removeEventListener(listener);
            }

	    synchronized (this) {
		done = true;
                thread = null;
		notifyAll();
	    }
	}
	Core.logger.log(this, "InsertContext.run -- Done: " + done,
			Logger.DEBUGGING);

    }
	
    public synchronized boolean reap() {
        if (!done) {
            Core.logger.log(this, "InsertContext.reap -- reaping running request! ",
                            Logger.DEBUGGING);
            println("TIMED OUT: fproxy gave up on the request because it took too long.");
            done = true;
            // REDFLAG: test this code path
            // REDFLAG: interrupt thread.
        }

	contextManager.remove(contextID);

	if (dataPart != null) {
	    try { 
		dataPart.freeBody();
	    }
	    catch (IOException ioe) {
		Core.logger.log(this, "ERROR: reap() couldn't release insert data bucket:" + ioe,
				Logger.DEBUGGING);
	    }
	}
	dataPart = null;
	requester = null;
	// statusMsg = null; 
        notifyAll();
	return true;
    }

    public synchronized boolean isExpired() { 
	return System.currentTimeMillis() > deathTime;
    }

    public synchronized final void touch() {
        System.err.println("InsertContext.touch -- called.");
	deathTime = System.currentTimeMillis() + LIFETIME_MS;
    }

    protected final synchronized void println(String text) {
	//System.err.println("*** " + text);
	statusMsg.append(text + "\n");
	notifyAll();
    }

    protected void finalize() throws Throwable {
	reap();
	super.finalize();
    }

    // Writes text log messages int the statusMsg buffer.
    class InsertContextEventLog implements ClientEventListener {
	public void receive(ClientEvent ce) {
	    String msg = ce.getDescription();
	    if (msg != null) {
		println(msg.trim());
	    }
            touch();
	}
    }

    class BlockInserterEventLog implements BlockInserter.BlockEventListener {
        public void receive(int blockNum, boolean succeeded, ClientEvent evt) {
            String msg = evt.getDescription();
	    if (msg != null) {
                if (evt instanceof StateReachedEvent) {
                    StateReachedEvent sr = (StateReachedEvent) evt;
                    if ((sr.getState() == Request.FAILED) && succeeded) {
                        // Don't freak out end users with FAILED messages
                        // on key collisions.
                        msg = "State DONE reached.";
                    }
                }
                else if (evt instanceof TransferEvent) {
                    msg = null; // Too voiciferous.
                }
                else if (evt instanceof TransferCompletedEvent) {
                    msg = null; // Too voiciferous.
                }

                if ( msg != null) {
                    println("block " + blockNum + ": " + msg.trim());
                }
	    }

            touch(); 
        }
    }
}









