package freenet.client;

import freenet.node.Node;
import freenet.Identity;
import freenet.Key;
import freenet.KeyException;
import freenet.client.ClientKey;
import freenet.client.AbstractClientKey;
import freenet.client.events.*;
import freenet.client.listeners.CollectingEventListener;
import freenet.crypt.*;
import freenet.support.*;
import java.io.*;
import java.util.Random;
import java.util.Enumeration;
import java.lang.reflect.Method;

/** VirtualClient knows how to deal with client Requests that don't need
  * to go over the wire, such as ComputeCHKRequest and ComputeSVKRequest.
  */
final class VirtualClient implements ClientFactory {

    protected Node node;
    protected Random randSource;
    protected BucketFactory bucketFactory;
    
    VirtualClient(Node n, Random r, BucketFactory bf) {
        node = n;
        randSource = r;
        bucketFactory = bf;
    }

    public final Client getClient(Request req) throws UnsupportedRequestException {
        return new VirtualClientInstance(req);
    }

    public boolean supportsRequest(Class req) {
        try {
            VirtualClientInstance.class.getMethod("execute", new Class[] {req});
        }
        catch (Exception e) {
            return false;
        }
        return true;
    }

    private class VirtualClientInstance implements Client {
    
        Request req;
        CollectingEventListener cel;
        Method exec;

        private VirtualClientInstance(Request req)
                                throws UnsupportedRequestException {
            try {
                exec = this.getClass().getMethod("execute",
                                                 new Class[] {req.getClass()});
            }
            catch (Exception e) {
                e.printStackTrace();
                throw new UnsupportedRequestException();
            }
            req.addEventListener(cel = new CollectingEventListener());
            this.req = req;
            state(Request.PREPARED);
        }
        
        public final void start() {
            blockingRun();
        }

        public final boolean cancel() {
            return false;
        }

        public int blockingRun() {
            try {
                state(Request.REQUESTING);
                exec.invoke(this, new Object[] {req});
                state(Request.DONE);
            }
            catch (Exception e) {
                req.produceEvent(new ExceptionEvent(e));
                state(Request.FAILED);
            }
            return req.state();
        }

        public final Enumeration getEvents() {
            return cel.events();
        }
    
        private void state(int state) {
            if (state >= Request.FAILED && state <= Request.DONE) {
                req.state(state);
                req.produceEvent(new StateReachedEvent(state));
            }
            else throw new IllegalArgumentException(
                "State value was not between FAILED and DONE"
            );
        }

        // Tavin, your execute methods have to be public if you
        // want to use Class.getMethod(String, Class[]) above.
        // see http://java.sun.com/j2se/1.3/docs/api/java/lang/Class.html
        //           #getMethod(java.lang.String, java.lang.Class[])
        // --gj
        
        public void execute(ComputeCHKRequest req) throws Exception {
            ClientCHK chk = new ClientCHK();
            chk.setCipher(req.cipherName);
            Bucket ctBucket = bucketFactory.makeBucket(
                chk.getTotalLength(req.meta.size() + req.data.size())
            );
            try {
                chk.encode(req.meta, req.data, ctBucket);
            }
            finally {
                bucketFactory.freeBucket(ctBucket);
            }
            req.clientKey = chk;
            req.produceEvent(new GeneratedURIEvent("Generated CHK",
                                                   chk.getURI()));
        }
    
        public void execute(ComputeSVKPairRequest req) throws Exception {
            ClientSVK svk = new ClientSVK(randSource);
            req.clientKey = svk;
            req.produceEvent(new GeneratedKeyPairEvent(
                svk.getPrivateKey(),
                svk.getPublicKeyFingerPrint())
            );
        }
    }
}







