#import "InstallerController.h"
#import "InstallerSDL.h"
#import "InstallerMedia.h"
#import "InstallerMusic.h"
#import "InstallerFrameworks.h"
#import "InstallerTools.h"
#import "InstallerPrefs.h"
#import "InstallerDownload.h"

BOOL lockedComponents;

@implementation InstallerController

- (NSApplicationTerminateReply)applicationShouldTerminate:(NSApplication *)sender
{
	if (canExit)
		return NSTerminateNow;
	
	return NSTerminateCancel;
}

- (void)awakeFromNib
{
	NSPoint windowPos;
	NSString *versionString;
	
	// Check the writability of the components path
	lockedComponents = ![fileManager isWritableFileAtPath:componentsPath];

	// Centre the window
	windowPos.x = [[window screen] frame].size.width / 2 - [window frame].size.width / 2;
	windowPos.y = [[window screen] frame].size.height / 2 - [window frame].size.height / 2;
	[window setFrameOrigin:windowPos];	
	
	// Write out version of FreeCraft being installed
	versionString = [[NSDictionary dictionaryWithContentsOfFile:[resourcesPath stringByAppendingString:@"FreeCraft.app/Contents/Info.plist"]] objectForKey:@"CFBundleShortVersionString"]; 
	if (versionString)
		[freecraftLabel setStringValue:[NSString stringWithFormat:@"FreeCraft %@ Installer", versionString]];
	
	// Setup all installer components
	[installerSDL setup];
	[installerMedia setup];
	[installerMusic setup];
	[installerPrefs setup];
	
	// Update the total to be downloaded given standard options
	[self updateTotalDownload];
}

- (void)dealloc
{
	[tools autorelease];
	[downloader autorelease];
}

- (NSString *)destinationPath
{
	return [installLocationLabel stringValue];
}

- (void)downloadProgress:(id)sender value:(double)value
{	
	// Increment the progress bar
	[progress setDoubleValue:value];
	[progress displayIfNeeded];
}

- (void)downloadFailed:(id)sender reason:(NSString *)reason
{
	NSRunAlertPanel(@"Download Failed", [NSString stringWithFormat:@"The download failed with the system giving this reason:\n\n%@\n\nYou will now be returned to the installer's main window. FreeCraft has not been installed but previously complete downloads should have been saved.", reason], @"Ok", NULL, NULL);
	[self stop:NULL];
}

- (id)init
{
	// Initalization stuff
	[super init];
	canExit = YES;
	
	// Specify ourselves as the NSApp delegate
	[NSApp setDelegate:self];
	
	// Create the components path (if it does not exist)
	if (![fileManager fileExistsAtPath:componentsPath])
		[fileManager createDirectoryAtPath:componentsPath attributes:nil];

	// Create the tools and downloader objects for use by others
	tools = [[InstallerTools alloc] initWithClient:self];
	downloader = [[InstallerDownload alloc] initWithClient:self];
	
	return self;
}

- (IBAction)launchWebsite:(id)sender
{
	[[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:kFreeCraftSite]];
}

- (IBAction)selectLocation:(id)sender
{
	NSOpenPanel *openPanel = [NSOpenPanel openPanel];

	// Show the selection panel
	[openPanel setCanChooseDirectories:YES];
	[openPanel setCanChooseFiles:NO];
	if ([openPanel runModalForTypes:NULL] == NSOKButton) {
		
		// Make changes to the install path
		[installLocationLabel setStringValue:[[openPanel filenames] objectAtIndex:0]];
		
	}
}

- (BOOL)prepare
{
	// Check that our install location is free
	if ([fileManager fileExistsAtPath:[[installLocationLabel stringValue] stringByAppendingString:@"/FreeCraft.app"]]) {
		NSRunAlertPanel(@"FreeCraft Already Exists", @"A file named \"FreeCraft.app\" already exists at the given install location, please remove this file manually before installing FreeCraft.", @"Ok", NULL, NULL);
		return NO;
	}
	
	return YES;
}


- (IBAction)start:(id)sender
{
	BOOL installOk;
	
	// Check if installation is ok
	installOk = [self prepare] && [installerSDL prepare] && [installerMedia prepare] && [installerMusic prepare];
	if (!installOk)
		return;
	
	// Close the options window and start the progress panel
	[window orderOut:self];
	[installationPanel center];
	[installationPanel makeKeyAndOrderFront:self];
	
	// Setup the progress bar and label
	[progress setMaxValue:kProgressUnit * 20.5];
	[downloader setProgressSize:kProgressUnit * 15.0];
	[progress setIndeterminate:NO];
	[progressLabel setStringValue:@"Downloading components..."];
	
	// Now based on the above either commence the downloads or skip ahead
	[downloader commenceDownloads:[NSArray arrayWithObjects:installerSDL, installerMedia, installerMusic, nil] withCallback:@selector(downloadsFinished:)];
}

- (void)downloadsFinished:(id)sender
{
	// Stop exiting
	canExit = NO;
	
	// Expand SDL
	[progressLabel setStringValue:@"Expanding SDL..."];
	[progressLabel displayIfNeeded];
	[installerSDL expand];
	[progress incrementBy:kProgressUnit];
	
	// Move Frameworks into place
	[progressLabel setStringValue:@"Installing FreeCraft..."];
	[progressLabel displayIfNeeded];
	[installerFrameworks install];
	[progress incrementBy:kProgressUnit];
	
	// Move FreeCraft into place
	[progressLabel setStringValue:@"Installing FreeCraft..."];
	[progressLabel displayIfNeeded];
	[fileManager copyPath:[resourcesPath stringByAppendingString:@"FreeCraft.app"] toPath:[[self destinationPath] stringByAppendingString:@"/FreeCraft.app"] handler:NULL];
	[progress incrementBy:kProgressUnit];
	
	// Expand the media
	[progressLabel setStringValue:@"Installing Media..."];
	[progressLabel displayIfNeeded];
	if ([installerMedia expand] == NO) {
		[fileManager removeFileAtPath:[[self destinationPath] stringByAppendingString:@"/FreeCraft.app"] handler:NULL];
		canExit = YES;
		[NSApp terminate:self];
	}
	[progress incrementBy:kProgressUnit];
	
	// Expand the music
	[progressLabel setStringValue:@"Installing Music..."];
	[progressLabel displayIfNeeded];
	[installerMusic expand];
	[installerMusic configure];
	[progress incrementBy:kProgressUnit];
	
	// Make the preferences file
	[installerPrefs make];
	[progressLabel setStringValue:@"Installation Complete!"];
	[progressLabel displayIfNeeded];
	[progress setDoubleValue:[progress maxValue]];
	
	// Congratulate
	if ([installerSDL exitReminder])
		NSRunAlertPanel(@"Installation Successful", @"The installation of FreeCraft has been successful however you still need to install the SDL runtime libraries. Depending on your installation an installer for these libraries may appear after the installer quits. This installer will now quit.", @"Ok", NULL, NULL);
	else
		NSRunAlertPanel(@"Installation Successful", @"The installation of FreeCraft has been successful. This installer will now quit.", @"Ok", NULL, NULL);
	
	// Launch SDL
	[installerSDL launch];
	
	// Now we can exit
	canExit = YES;
	[NSApp terminate:self];
}

- (IBAction)stop:(id)sender
{
	if (canExit || sender == NULL) {
		
		// Close the progress panel and open the options window again
		if (sender != NULL) {
			if (NSRunAlertPanel(@"Really Abort Installation?", @"Are you sure you really want to abort this installation, doing so may cause you to lose data that you have already partially downloaded?", @"Yes", @"No", NULL) == NSAlertAlternateReturn)
				return;
		}
		[installationPanel orderOut:self];
		[window makeKeyAndOrderFront:self];
		if ([downloader isDownloading])
			[downloader reset];
		[progress setIndeterminate:YES];
		[progressLabel setStringValue:@"Preparing to Install..."];
		
		// Setup all installer components again
		[installerSDL setup];
		[installerMedia setup];
		[installerMusic setup];
		[installerPrefs setup];
		
	}
}

- (void)toolFailed:(id)sender named:(NSString *)name code:(int)code
{
	NSRunAlertPanel(@"Tool Failed", [NSString stringWithFormat:@"The %@ tool has unexpectedly failed with termination status %d. You will now be returned to the installer's main window.", name, code], @"Ok", NULL, NULL);
	[self stop:NULL];
}

- (id)tools
{
	return tools;
}

- (void)updateTotalDownload
{
	unsigned int totalDownload;
	
	// Display the total download with current settings (use 1 MB = 1,024 ^ 2 bytes) or explain that downloads are not available
	if (lockedComponents)
		[totalDownloadLabel setStringValue:@"Downloads not available when installing from locked volumes"];
	else {
		totalDownload = [installerSDL downloadSize] + [installerMedia downloadSize] + [installerMusic downloadSize];
		[totalDownloadLabel setStringValue:[NSString stringWithFormat:@"Total Download: %.1f MB", (totalDownload / (1024.0 * 1024.0))]];
	}
}

@end
