/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*************************************************************
*
*  File: hessian.c 
*
*  Purpose: Main file for hessian commands.
*
*/

#define GDEBUG 
int hess_debug;  /* debugging flag */
/* use ysmp */

#include "include.h"
#include "f2c.h"

/* for constructing sparse matrix */
int total_rows;
struct hess_entry *entry_start;
          
int hmode; /* mode of motion */
struct hess_index *array;
struct hess_verlist *vhead = NULL;  /* main vertex list */
int vcount;          /* total number of vertices */
static  REAL **volrows = NULL;        /* volume constraint rows of matrix */
#ifdef XXX
REAL *rhs = NULL;                 /* right side of augmented matrix  */
REAL *X = NULL;                 /* solution to augmented matrix  */
#endif
struct hess_index *congrads = NULL; /* constraint grads on left */
REAL *conrhs = NULL;    /* right side of augmented matrix for constraints */
int total_entries; /* of hessian */
int A_total; /* entries for A matrix */
int A_rows;  /* rows for A matrix */
int vertex_rows; /* number of rows for vertex degrees of freedom */

static REAL **vproj_base; /* vertex projection matrix row pointers */
static REAL *vproj_space; /* vertex projection matrix arena */
static REAL ***conhess_base; /* vertex constraint hessian arena */
REAL rhs_norm;

static int ritz_done_flag;
static REAL **ritzvecs;
static REAL *ritzev;
static int ritzdim;
void optparamhess ARGS((REAL *));  /* right side */

/**********************************************************************
    Hash list routines for Hessian matrix entries.
    Key is (row,col).
**********************************************************************/

#define PRIME 99991
#define hash(row,col)  (abs((row)*97+(col)*PRIME))
static int max_fill;     /* max number of entries until enlarge */
static int hashcount;    /* current number of entries */
static int hash_extraprobes; /* for measuring efficiency */

/********************************************************************
* 
* function: hess_hash_init()
*
* purpose: Initialize hash table.
*/
void hess_hash_init()
{ int i;

  table_size = hash_per_row*A_rows+10;
  max_fill = 4*table_size/5;  /* leave enough empty for decent hash */
  if ( !hessian_quiet_flag )
  { sprintf(msg,"Hess init alloc: %d\n",table_size);
     outstring(msg);
  }
  hashcount = 0;
  if ( hashtable ) temp_free((char*)hashtable);
  hashtable = 
     (struct hess_entry *)mycalloc(table_size,sizeof(struct hess_entry));
  for ( i = 0 ; i < table_size ; i++ ) hashtable[i].row = HASHEMPTY;
  hash_extraprobes = 0;
}

/********************************************************************
* 
* function: hess_hash_expand()
*
* purpose: Expands hash table
*/

void hess_hash_expand()
{ struct hess_entry *newtable,*oldtable;
  int i;
  struct hess_entry *e;
  int newsize; 
  int oldsize = table_size;

  if ( !hashtable ) hess_hash_init();
  newsize = table_size*2; 
  oldtable = hashtable;
  newtable = 
     (struct hess_entry *)mycalloc(newsize,sizeof(struct hess_entry));
  for ( i = 0 ; i < newsize ; i++ ) newtable[i].row = HASHEMPTY;
  table_size =  newsize;
  max_fill = 4*table_size/5;
  hashtable = newtable;

  /* reinsert */
  hashcount = 0;
  for ( i = 0, e = oldtable ; i < oldsize ; i++,e++ )
     if ( e->row != HASHEMPTY )
        hess_hash_search(e->col,e->row,e->value);
  myfree((char*)oldtable);

  if ( !hessian_quiet_flag )
  { sprintf(msg,"Expanded hashtable size: %d.\n",max_fill);
    outstring(msg);
  }
}

/********************************************************************
* 
* function: hess_hash_search()
*
* purpose: Finds existing entry or allocates entry.
*             Installs key values, and adds hessian value.
* return:  Pointer to entry.
*/
void hess_hash_search(col,row,value)
int row,col;  /* meant to do upper triangle */
REAL value;  /* value to add */
{
  struct hess_entry *e;
  int spot;

  if ( row > col ) return;
  if ( value == 0.0 ) return;

  if ( col >=  A_rows ) /* a constraint grad */
  { if ( !array[col].u.congrad ) 
        array[col].u.congrad = (REAL *)mycalloc(A_rows,sizeof(REAL));
     e = (struct hess_entry *)(array[col].u.congrad+row);
     e->value += value;
     return;
  }
  if ( hashcount >= max_fill ) hess_hash_expand();

  /* search hash table */
  spot = hash(row,col) % table_size;
  e = hashtable + spot;
  while ( e->row != HASHEMPTY )
  { if ( (e->row == row) && (e->col == col) )
     { e->value += value; return; 
     }
     spot++;
     if ( spot >= table_size ) spot -= table_size;
     e = hashtable + spot;
     hash_extraprobes++;
  }
  /* if here, then have empty slot and need to insert */
  e->col = col; e->row = row;  hashcount++; 
  e->value = value;
}

/***********************************************************************
* 
* function: fill_grad()
*
* purpose: process gradient of function at constraint 
*/

void fill_grad(v,grad,rhs)
struct hess_verlist *v;
REAL *grad;
REAL *rhs;
{ REAL g[MAXCOORD];
  int k,a,b;

  if ( rhs_flag )
  { if ( v->proj )
     { vec_mat_mul(grad,v->proj,g,SDIM,v->freedom);
        for ( k = 0 ; k < v->freedom ; k++ )
            rhs[v->rownum+k] -= g[k];
     }
     else
        for ( k = 0 ; k < v->freedom ; k++ )
            rhs[v->rownum+k] -= grad[k];
  }

  if ( hess_flag && v->conhess )
  { 
     for ( a = 0 ; a < v->freedom ; a++ )
      for ( b = 0 ; b <= a ; b++ )
        hess_hash_search(v->rownum+a,v->rownum+b,SDIM_dot(grad,v->conhess[a][b]));
  }
}

/*************************************************************************
*
* function: fill_self_entry()
*
* purpose: fill proper hessian matrix spot for single vertex second derivs
*/

void fill_self_entry(v_id,self)
vertex_id v_id;
REAL **self; /* full dim values */
{ int j,k;
  struct hess_verlist *v;

  v = vhead + ordinal(v_id);

  if ( v->proj )
     { MAT2D(temp_mat,MAXCOORD,MAXCOORD);
        MAT2D(temp_mat2,MAXCOORD,MAXCOORD);
        tr_mat_mul(v->proj,self,temp_mat,SDIM,v->freedom,SDIM);
        mat_mult(temp_mat,v->proj,temp_mat2,v->freedom,SDIM,v->freedom);
        for ( j = 0 ; j < v->freedom ; j++ )
          for ( k = 0 ; k <= j ; k++ )
             hess_hash_search(v->rownum+j,v->rownum+k, temp_mat2[j][k]);
     }
  else 
  { for ( j = 0 ; j < v->freedom ; j++ )
        for ( k = 0 ; k <= j ; k++ )
          hess_hash_search(v->rownum+j,v->rownum+k,self[j][k]);
  }
}


/*************************************************************************
*
* function: fill_mixed_entry()
*
* purpose: fill proper hessian matrix spot for mixed vertex second derivs
*/

void fill_mixed_entry(v_id1,v_id2,mixed)
vertex_id v_id1,v_id2;
REAL **mixed; /* full dim values */
{ int k,j;
  REAL **oo;
  MAT2D(temp_mat,MAXCOORD,MAXCOORD);
  MAT2D(temp_mat2,MAXCOORD,MAXCOORD);
  struct hess_verlist *v1,*v2;
  
  v1 = vhead + ordinal(v_id1); 
  v2 = vhead + ordinal(v_id2);
  if ( v1->proj )
     { tr_mat_mul(v1->proj,mixed,temp_mat,SDIM,v1->freedom,SDIM);
        oo = temp_mat;
     }
  else oo = mixed;
  if ( v2->proj )
     { mat_mult(oo,v2->proj,temp_mat2,v1->freedom,SDIM,v2->freedom);
        oo = temp_mat2;
     }

  if ( v1->rownum > v2->rownum )
    for ( j = 0 ; j < v1->freedom ; j++ )
     for ( k = 0 ; k < v2->freedom ; k++ )
        hess_hash_search(v1->rownum+j,v2->rownum+k,oo[j][k]);
  else 
    for ( j = 0 ; j < v1->freedom ; j++ )
     for ( k = 0 ; k < v2->freedom ; k++ )
        hess_hash_search(v2->rownum+k,v1->rownum+j,oo[j][k]);

}


/********************************************************************
* 
* function: hess_hash_sort()
*
* purpose: Create sorted list of hessian entries, row major order
*             Does 2-digit radix sort on row,col with count sort on each.
* input:    total_rows - upper bound for row,col.
* output:  *nptr - number of entries
*             **listptr - allocated list (temp_calloc)
*/
void hess_hash_sort(hessian_rows,nptr,listptr)
int hessian_rows;  /* rows in hessian */
int *nptr; /* for return of total entries */
struct hess_entry **listptr; /* list in, for return of sorted list */
{ int i;
  struct hess_entry *e;
  int *counts;
  int *starts;
  int sum,oldsum;

  counts = (int *)temp_calloc(2*hessian_rows+1,sizeof(int));
  starts = counts + hessian_rows;

  /* count entries in row */
  for ( i = 0, e = hashtable ; i < table_size ; i++, e++ )
     if ( e->row != HASHEMPTY ) counts[e->row]++;
  for ( i = 0 ; i < hessian_rows ; i++ )
     array[i].count = counts[i];

  /* get starting points of each row */
  for ( i = 0, sum = 0 ; i < hessian_rows  ; i++ )
  { oldsum = sum;
     sum += counts[i];
     counts[i] = starts[i] = oldsum;
  } 
  starts[hessian_rows] = sum;

  /* sort into row bins */
  for ( i = 0, e = hashtable ; i < table_size ; i++,e++ )
  { 
     struct hess_entry eorig;
     struct hess_entry etmp;
     if ( e->row == HASHEMPTY ) continue;
     if ( i < counts[e->row] ) continue;
     eorig = *e;
     e->row = HASHEMPTY;
     do
        { /* follow chain of replacements */
          int spot;
          int k;
          spot = counts[eorig.row];
          etmp = hashtable[spot];
          /* insertion sort into row */
          for ( k = spot ; k > starts[eorig.row] ; k-- )
          { if ( hashtable[k-1].col > eorig.col )
                hashtable[k] = hashtable[k-1];
             else break;
          }
          hashtable[k] = eorig;
          counts[eorig.row]++;
          eorig = etmp;
        }
        while ( eorig.row != HASHEMPTY ); /* stop when get back to empty */ 
    }
  *listptr = hashtable;
  *nptr = hashcount;
  A_total = hashcount;
  temp_free((char*)counts);
  hash_per_row = 1 + (5*hashcount)/(4*A_rows+1); /* for next time */

  if ( !hessian_quiet_flag )
  { sprintf(msg,"Hessian entries: %d  Final hashtable size: %d next hash_per_row: %d.\n",hashcount, table_size,hash_per_row);
     outstring(msg);
     sprintf(msg,"Hash extra probes: %d\n",hash_extraprobes);
     outstring(msg);
  }
}


/********************************************************************
* 
* function: hess_hash_end()
*
* purpose: Deallocate hessian hash table.
*/
void hess_hash_end()
{
  if ( hashtable )  myfree((char*)hashtable);
  hashtable = NULL;
}
/********************************************************************
     End Hessian hash routines.
********************************************************************/


/**************************************************************
*
*  Function: hessian_auto()
*
*  Purpose:  Global minimization of energy by finding
*            critical point of quadratic approximation
*            to total energy function.
*            Using only lower left triangle of symmetric matrix.
*            Uses whatever sparse matrix solver is in effect at the moment.
*
*/

void hessian_auto() /* automatic hessian */
{ REAL old_energy = web.total_energy;
  static struct linsys S;
  REAL *rhs;
  REAL *X;

  free_system(&S); /* in case anything left over from error */

  hmode = hessian_normal_flag;
  hessian_cleanup(); /* cleanup from previous round */
  hessian_init(&rhs,&X);
  hess_flag = 1; rhs_flag = 1; hessian_fill(rhs); 
  sp_Hessian_solver(&S,rhs,X);
  hessian_move((REAL)1.0,ACTUAL_MOVE,X);

  if ( check_increase_flag && (web.total_energy > (1+1e-13)*old_energy) )
     { kb_error(1596,"Hessian move would have raised energy.  Restoring coords.\n",
          WARNING);
        restore_coords(&saved);
     }
  hessian_exit(X);
  free_system(&S);
  temp_free((char*)rhs);
  temp_free((char*)X);
}

/**************************************************************
*
*  Function: hessian_seek()
*
*  Purpose: Same as hessian_auto(), but uses direction of minimum
*           as seek direction, and moves down it.
*
*  Return: best stepsize
*/

REAL hessian_seek(maxscale) /* automatic hessian */
REAL maxscale;
{ REAL best_scale;
  static struct linsys S;
  REAL *rhs,*X;

  free_system(&S); /* in case anything left over from error */

  hmode = hessian_normal_flag;
  hessian_init(&rhs,&X);
  hess_flag = 1; rhs_flag = 1; hessian_fill(rhs); 
  if ( hessian_linear_metric_flag )
     { free_system(&Met);
       linear_metric_setup(A_rows,&Met);
     }
  else if ( web.area_norm_flag )
     { free_system(&Met);
       star_metric_setup(A_rows,&Met);
     }
  sp_Hessian_solver(&S,rhs,X);
  best_scale = hessian_line_seek(maxscale,X);
  hessian_exit(X);
  free_system(&S);
  temp_free((char*)rhs);
  temp_free((char*)X);
  return best_scale;
}

/*****************************************************************************
*
* function: square_grad()
*
* purpose: calculates gradients for rhs and returns square of gradient 
*
*/

REAL square_grad_old(rhs)
REAL *rhs;
{
  hess_flag = 0; rhs_flag = 1; hessian_fill(rhs); 
  /* should rather apply inverse metric to rhs */
  if ( hessian_linear_metric_flag ) return sparse_metric_dot(rhs,rhs,&Met);
  else return dot(rhs,rhs,total_rows);
}
REAL square_grad()
{ REAL sum;
  vertex_id v_id;
  REAL *f;
  int i;
 
  calc_all_grads(CALC_VOLGRADS);
  volume_restore(1.,ACTUAL_MOVE);
  calc_force();
  pressure_forces();
  fix_vertices(); /* project forces to constraints, etc */
  calc_lagrange();
  lagrange_adjust();
  vgrad_end();
  sum = 0.0;
  FOR_ALL_VERTICES(v_id)
  { REAL star;
     f = get_force(v_id);
     if ( hessian_linear_metric_flag )
        star = Met.apinv[vhead[ordinal(v_id)].rownum];
     else star = 1.0;
     if ( hessian_normal_flag )
     { struct hess_verlist *v;              /* current  vertex */
        REAL r[MAXCOORD];
        REAL *rr = r;
        v = vhead + ordinal(v_id);
        vec_mat_mul(f,v->proj,rr,SDIM,v->freedom);
        sum += dot(r,r,v->freedom)*star;
     }
     else sum += SDIM_dot(f,f)*star;
  }
  for ( i = 0 ; i < optparamcount ; i++ )
     sum += optparam[i].grad*optparam[i].grad;
  return sum;
}
/*****************************************************************************
*
* function: hessian_line_seek()
*
* purpose: seek optimum motion in downhill direction, after downhill 
*  found.
*  save_coords() assumed to have been done in hessian_init();
*  step size bounded by web.max_scale.
*  If input parameter maxscale is nonzero, will obey.
*
* return: optimal stepsize
*/

REAL hessian_line_seek(maxscale,X)
REAL maxscale;
REAL *X;
{ REAL scale0=0.0,scale1,scale2=0.0;
  REAL energy0,energy1,energy2=0.0;
  int seekcount = 0;
  int dirflag = 1; /* for positive or negative direction */
  REAL stepsize;
  REAL typescale;
  
  if ( maxscale <= 0.0 ) return 0.0;

  typescale = maxscale/100;

  /* figure which way to go */
  energy0 = min_square_grad_flag ? square_grad(): web.total_energy;

  hessian_move(typescale*1e-3,TEST_MOVE,X);
  energy1 = min_square_grad_flag ? square_grad(): web.total_energy;
  restore_coords(&saved);

  hessian_move(-typescale*1e-3,TEST_MOVE,X);
  energy2 = min_square_grad_flag ? square_grad(): web.total_energy;
  restore_coords(&saved);

  if ( (energy1 >= energy0) && (energy2 >= energy0) )
     return 0.0;
  if ( energy1 > energy2 ) dirflag = -1;
  else dirflag = 1;

  stepsize = scale1 = 0.499999999*typescale;
  hessian_move(dirflag*stepsize,TEST_MOVE,X);
  energy1 = min_square_grad_flag ? square_grad(): web.total_energy;
  restore_coords(&saved);
  if ( energy1 < energy0 )
  { do
     { stepsize *= 2;
        if ( stepsize > maxscale ) { stepsize = maxscale; break; }
        hessian_move(dirflag*stepsize,TEST_MOVE,X);
        energy2 = min_square_grad_flag ? square_grad(): web.total_energy;
        scale2 = stepsize;
        restore_coords(&saved);
        if ( energy2 > energy1 )
        { stepsize /= 2; break; }
        else 
        { energy0 = energy1; scale0 = scale1;
          energy1 = energy2; scale1 = scale2;
        }
     } while ( stepsize < maxscale );
  }
  else /* energy1 >= energy0 */
  { seekcount = 0;
     do
     { if ( seekcount++ > 20 ) { stepsize = 0.0; break; }
        energy2 = energy1; scale2 = scale1;
        stepsize = scale2/2;
        hessian_move(dirflag*stepsize,TEST_MOVE,X);
        energy1 = min_square_grad_flag ? square_grad(): web.total_energy;
        scale1 = stepsize;
        restore_coords(&saved);
     } while ( energy1 > energy0 );
  }
  if ( (stepsize > 0.0) && (stepsize < maxscale) )
  { REAL denom;
    /* now quadratic interpolation for minimum energy */
    denom = energy0*(scale1-scale2)+energy1*(scale2-scale0)
                 + energy2*(scale0 - scale1);
    if ( denom == 0.0 ) stepsize = 0.0;
    else
    { stepsize = ((energy0-energy2)*scale1*scale1
            +(energy1-energy0)*scale2*scale2
            +(energy2-energy1)*scale0*scale0)/2/denom;
    }
    if ( stepsize < scale0 ) stepsize = scale0;
    if ( stepsize > scale2 ) stepsize = scale2;
  }
  else if ( stepsize < 0.0 ) stepsize = 0.0;
  else if ( stepsize > maxscale ) stepsize = maxscale;
  hessian_move(dirflag*stepsize,ACTUAL_MOVE,X);

  if ( min_square_grad_flag )
  { energy1 =  square_grad();
#ifdef LONGDOUBLE
     sprintf(msg,"square gradient: %3.*Lg\n",DPREC,energy1);
#else
     sprintf(msg,"square gradient: %3.15g\n",energy1);
#endif 
     outstring(msg);
  }

  return dirflag*stepsize;
} /* end hessian_line_seek() */

/***************************************************************************
*
* function: hessian_saddle()
*
* purpose: move downhill optimally if at saddle point
*/

void hessian_saddle(maxscale)
REAL maxscale;
{ REAL low;
  struct linsys S;
  REAL stepsize;
  REAL *rhs,*X;

  memset((char*)&S,0,sizeof(struct linsys));
  hmode = hessian_normal_flag;
  hessian_init(&rhs,&X);
  hess_flag = 1; rhs_flag = 1; hessian_fill(rhs); 
  S.P = (int *)mycalloc(A_rows,sizeof(int));
  (*sp_AIJ_setup_func)(array,A_rows,&S);
  if ( sp_ordering_func ) (*sp_ordering_func)(&S);
  (*sp_constraint_setup_func)
                     (array+bodyrowstart,web.bodycount + gen_quant_count,&S);
  if ( hessian_linear_metric_flag )
     { free_system(&Met);
       linear_metric_setup(A_rows,&Met);
     }
  else if ( web.area_norm_flag )
     { free_system(&Met);
       star_metric_setup(A_rows,&Met);
     }

  low =  lowest_eigenpair(&S,X);
  if ( low >= 0.0 )
     { 
#ifdef LONGDOUBLE
        sprintf(msg,"Lowest eigenvalue %2.*Lg. Not a saddle point.\n",DPREC,low); 
#else
        sprintf(msg,"Lowest eigenvalue %2.15g. Not a saddle point.\n",low); 
#endif 
        outstring(msg);
        goto saddle_exit;
     }
#ifdef LONGDOUBLE
  sprintf(msg,"Lowest eigenvalue %2.*Lg\n",DPREC,low);
#else
  sprintf(msg,"Lowest eigenvalue %2.15g\n",low);
#endif 
  outstring(msg);
  stepsize = hessian_line_seek(maxscale,X);
  sprintf(msg,"stepsize %g\n",(DOUBLE)stepsize);
  outstring(msg);
#ifdef LONGDOUBLE
  sprintf(msg,"1.  energy: %17.*Lg \n", DPREC,web.total_energy);
#else
  sprintf(msg,"1. %s: %17.15g energy: %17.15g \n",
                areaname,web.total_area,web.total_energy);
#endif 
  outstring(msg);
  update_display();
  reset_conj_grad();
saddle_exit:
  hessian_exit(X); /* cleanup */
  free_system(&S);
  temp_free((char*)rhs);
  temp_free((char*)X);
} /* end hessian_saddle() */

/***************************************************************************
* 
* function: print_hessian_menu()
*
* purpose: Prints full menu in hessian_menu command.
*/
void print_hessian_menu()
{
  outstring("\n");
  outstring("1. Fill in hessian matrix.\n");
  outstring("2. Fill in right side. (Do 1 first)\n");
  outstring("3. Solve. (Do 2 first)\n");
  outstring("4. Move. (Do 3,V,E,F,C,B or X  first)\n");
  outstring("7. Restore original coordinates.\n");
  outstring("9. Toggle debugging. (Don't do this!)\n");
  outstring("B. Chebychev (For Hessian solution ).\n");
  outstring("C. Chebychev (For most negative eigenvalue eigenvector).\n");
  outstring("E. Lowest eigenvalue. (By factoring. Do 1 first)\n");
  outstring("F. Lowest eigenvalue. (By conjugate gradient. Do 1 first)\n");
  outstring("L. Lanczos. (Finds eigenvalues near probe value. )\n");
  outstring("R. Lanczos with selective reorthogonalization.\n");
  outstring("Z. Ritz subspace iteration for eigenvalues. (Do 1 first)\n");
  outstring("X. Pick Ritz vector for motion. (Do Z first)\n");
  outstring("P. Eigenvalue probe. (By factoring. Do 1 first)\n");
  outstring("V. Eigenvalue probe with eigenvector. (By factoring. Do 1 first)\n");
  outstring("S. Seek along direction. (Do 3,V,E,F,C,B or X first)\n");
  outstring("Y. Toggle YSMP/alternate minimal degree factoring.\n");
  outstring("U. Toggle Bunch-Kaufman version of min deg.\n");
  outstring("M. Toggle projecting to global constraints in move. (Leave this alone!)\n");
  outstring("G. Toggle minimizing square gradient in seek.\n");
  outstring("0. Exit hessian.\n");
}

/***********************************************************************
*
*  function: hessian_menu()
*
*  purpose: give user more interactive control over Hessian move
*     and debugging.
*/

void hessian_menu()
{   char response[30];
    int left_flag = 0; /* whether hessian filled */
    int right_flag = 0; /* whether rhs filled in */
    int dir_flag = 0 ; /* whether direction filled in */
    struct linsys S;
    REAL stepsize;
    REAL *rhs = NULL,*X = NULL;
    
    hessian_legal();
    save_coords(&saved);

    memset((char*)&S,0,sizeof(struct linsys));
    hess_debug = 0;
    hmode = hessian_normal_flag;
    for(;;)
     {
        prompt("Choice(?,1,2,3,4,7,9,B,C,E,F,L,R,Z,X,P,V,S,Y,U,M,G,0,q): ",
            response,sizeof(response));
        switch ( toupper(response[0]) )
          { case '?': print_hessian_menu(); break;
             case '1': 
                 hessian_cleanup(); /* cleanup from previous round */
                 free_system(&S);
                 hessian_init(&rhs,&X);
                 hess_flag = 1; rhs_flag = 1; hessian_fill(rhs); 
                 S.P = (int *)mycalloc(A_rows,sizeof(int));
                 (*sp_AIJ_setup_func)(array,A_rows,&S);
                 if ( sp_ordering_func ) (*sp_ordering_func)(&S);
                 (*sp_constraint_setup_func)
                     (array+bodyrowstart,web.bodycount + gen_quant_count,&S);
                 if ( hessian_linear_metric_flag )
                 { free_system(&Met);
                    linear_metric_setup(A_rows,&Met);
                 }
                 else if ( web.area_norm_flag )
                 { free_system(&Met);
                    star_metric_setup(A_rows,&Met);
                 }
                 left_flag = 1;
                 break;

             case '2': 
                  if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                  rhs_norm = dot(rhs,rhs,total_rows);
#ifdef LONGDOUBLE
                  sprintf(msg,"RHS norm: %20.*Lg\n",DPREC, rhs_norm);
#else
                  sprintf(msg,"RHS norm: %20.15g\n", rhs_norm);
#endif 
                  outstring(msg);
                  right_flag = 1;
                  break;
                
             case '3': 
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 if ( right_flag == 0 )
                     { outstring("Error: Must do step 2 first.\n"); break; }
                 if ( !(S.flags & S_FACTORED) ) (*sp_factor_func)(&S);
                 if ( !(S.flags & S_PROJECTED)) (*sp_hess_project_setup_func)(&S);
                 sp_hessian_solve(&S,rhs,X,SET_PRESSURE);
                 outstring("Motion found.\n");
                 dir_flag = 1;
                 break;

             case '4': 
                  if ( dir_flag == 0 )
                     { outstring("Error: Must do step 3,V,E,F,C,B or X first.\n"); 
                        break; 
                        }
                 prompt("Step size: ",response,sizeof(response));
                 stepsize = atof(response);
                 hessian_move(stepsize,ACTUAL_MOVE,X);
#ifdef LONGDOUBLE
                 sprintf(msg,"1. energy: %*.*Lg \n",DWIDTH,DPREC,web.total_energy);
#else
                 sprintf(msg,"1. %s: %17.15f energy: %17.15f \n",
                        areaname,web.total_area,web.total_energy);
#endif 
                 outstring(msg);
                 update_display();
                 break;

             case '7': restore_coords(&saved); update_display(); break;

             case '8': 
                       if ( left_flag == 0 )
                         { outstring("Error: Must do step 1 first.\n"); break; }
                       write_hessian(&S); break;

             case 'Q': case '0': case 0: hessian_exit(X); 
                          free_system(&Met);
                            free_system(&S);
                          if ( rhs ) temp_free((char*)rhs);
                          if ( X ) temp_free((char*)X);
                          return;
             case '9': hess_debug = !hess_debug;
                       if ( hess_debug ) 
                             outstring("Hessian debug ON.\n");
                       else outstring("Hessian debug OFF.\n");
                       break;
             case 'Z': /* ritz */
              { 
                 REAL probe;
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 prompt("Ritz probe value: ",response,sizeof(response));
                 probe = atof(response);
                 prompt("Number of eigenvalues: ",response,sizeof(response));
                 ritzdim = atoi(response);
                 if ( ritzvecs ) free_matrix(ritzvecs);
                 ritzvecs = dmatrix(0,ritzdim-1,0,S.N+S.concount);
                 do_ritz(&S,probe,ritzdim,ritzvecs);
                 ritz_done_flag = 1;
                 break;
              }
             case 'A':  /* new lowest eigenvalues, by conjugate gradient */ 
              { if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 prompt("Number of eigenvalues: ",response,sizeof(response));
                 ritzdim = atoi(response);
                 if ( ritzvecs ) free_matrix(ritzvecs);
                 ritzvecs = dmatrix(0,ritzdim-1,0,S.N+S.concount);
                 ritzev = (REAL*)temp_calloc(ritzdim,sizeof(REAL));
                 cg_ritz(&S,ritzdim,ritzvecs,ritzev); 
                 ritz_done_flag = 1;
                 break;
              }
             case 'X': /* pick ritz vector for motion */
              { int ritznum;
                 int i;
                 if ( ritz_done_flag == 0 )
                     { outstring("Error: Must do step Z first.\n"); break; }
                 sprintf(msg,"Eigenvector number (1 to %d): ",ritzdim);
                 prompt(msg,response,sizeof(response));
                 ritznum = atoi(response);
                 if ( (ritznum < 0) || (ritznum > ritzdim) )
                     { outstring("Error: Invalid eigenvector number.\n"); break; }
                 ritznum--;
                 for ( i = 0 ; i < S.N ; i++ ) X[i] = ritzvecs[ritznum][i];
                 dir_flag = 1;
                 break;
              }
             case 'L': 
             { int nprint,i;
                REAL evalues[NPRINT];
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 if ( !(S.flags & S_FACTORED) ) (*sp_factor_func)(&S);
                 if ( !(S.flags & S_PROJECTED)) (*sp_hess_project_setup_func)(&S);
                 nprint = lanczos(&S,KRYLOVDIM,evalues,NPRINT);
                 /* list, ones near probe value */
                 for ( i = 0 ; i < nprint ; i++ )
                 {
#ifdef LONGDOUBLE
                    sprintf(msg,"%d  %*.*Lf\n",i+1,DWIDTH,DPREC,evalues[i]);
#else
                    sprintf(msg,"%d  %20.15f\n",i+1,evalues[i]);
#endif 
                    outstring(msg);
                 }
              }
                 break;

             case 'R': 
             { int nprint,i;
                REAL evalues[NPRINT];
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 if ( !(S.flags & S_FACTORED) ) (*sp_factor_func)(&S);
                 if ( !(S.flags & S_PROJECTED)) (*sp_hess_project_setup_func)(&S);
                 nprint = selective_lanczos(&S,KRYLOVDIM,evalues,NPRINT);
                 /* list, ones near probe value */
                 for ( i = 0 ; i < nprint ; i++ )
                 { 
#ifdef LONGDOUBLE
                    sprintf(msg,"%d  %*.*Lf\n",i+1,DWIDTH,DPREC,evalues[i]);
#else
                    sprintf(msg,"%d  %20.15f\n",i+1,evalues[i]);
#endif 
                    outstring(msg);
                 }
              }
              break;

             case 'C': 
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 chebychev_ev(&S);
                 dir_flag = 1;
                 break;

             case 'B': 
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 chebychev_hess(&S);
                 dir_flag = 1;
                 break;

             case 'P':  /* probe for eigenvalues */
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 bk_eigenprobe(&S);
                 break;

             case 'V':  /* probe for eigenvalues, with eigenvector */
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 bk_inverse_it(&S,X);
                 dir_flag = 1;
                 break;

             case 'E':  /* lowest eigenvalue */ 
              { REAL low;
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 low = lowest_eigenpair(&S,X); 
#ifdef LONGDOUBLE
                 sprintf(msg,"Lowest eigenvalue %2.*Lg\n",DPREC,low);
#else
                 sprintf(msg,"Lowest eigenvalue %2.15g\n",low);
#endif 
                 outstring(msg);
                 dir_flag = 1;
                 break;
              }
             case 'D':  /* lowest eigenvalue, by old conjugate gradient */ 
              { REAL low;
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 low = old_cg_lowest_eigenpair(&S,X); 
#ifdef LONGDOUBLE
                 sprintf(msg,"Lowest eigenvalue %2.*Lg\n",DPREC,low);
#else
                 sprintf(msg,"Lowest eigenvalue %2.15g\n",low);
#endif 
                 outstring(msg);
                 dir_flag = 1;
                 break;
              }
             case 'F':  /* lowest eigenvalue, by conjugate gradient */ 
              { REAL low;
                 if ( left_flag == 0 )
                     { outstring("Error: Must do step 1 first.\n"); break; }
                 low = cg_lowest_eigenpair(&S,X); 
#ifdef LONGDOUBLE
                 sprintf(msg,"Lowest eigenvalue %2.*Lg\n",DPREC,low);
#else
                 sprintf(msg,"Lowest eigenvalue %2.15g\n",low);
#endif 
                 outstring(msg);
                 dir_flag = 1;
                 break;
              }
             case 'S':
                 if ( dir_flag == 0 )
                     { outstring("Error: Must get direction first.\n"); break; }
                 stepsize = hessian_line_seek(1.0,X);
                 sprintf(msg,"stepsize %g\n",(DOUBLE)stepsize);
                 outstring(msg);
#ifdef LONGDOUBLE
                 sprintf(msg,"1.  energy: %*.*Lg \n",DWIDTH,DPREC,web.total_energy);
#else
                 sprintf(msg,"1. %s: %17.15g energy: %17.15g \n",
                        areaname,web.total_area,web.total_energy);
#endif 
                 outstring(msg);
                 update_display();
                 break;

             case 'Y': /* toggle YSMP */
                 if ( ysmp_flag == YSMP_FACTORING )
                 { ysmp_flag = MINDEG_FACTORING;
                    sp_mul_func = bk_mul;
                    sp_AIJ_setup_func= bk_AIJ_setup;
                    sp_constraint_setup_func = bk_constraint_setup;
                    sp_hess_project_setup_func= BK_hess_project_setup;
                    sp_factor_func = xmd_factor;
                    sp_solve_func = xmd_solve;
                    sp_solve_multi_func = xmd_solve_multi;
                    sp_ordering_func = NULL;
                    outstring("Using alternate minimal degree factoring.\n");
                    if ( left_flag ) outstring("Must do step 1 again.\n");
                    dir_flag = left_flag = right_flag = 0;
                 }
                 else
                 { ysmp_flag = YSMP_FACTORING;
                    sp_mul_func = bk_mul;
                    sp_AIJ_setup_func= bk_AIJ_setup;
                    sp_constraint_setup_func = bk_constraint_setup;
                    sp_hess_project_setup_func= BK_hess_project_setup;
                    sp_factor_func = ysmp_factor;
                    sp_solve_func = ysmp_solve;
                    sp_solve_multi_func = ysmp_solve_multi;
                    sp_ordering_func = NULL;
                    outstring("Using YSMP.\n");
                 }
                 break;

             case 'U': /* toggle BK version of min degree */
                 BK_flag = !BK_flag;
                 if ( BK_flag )
                    outstring("Using Bunch-Kauffman version of minimal degree.\n");
                 else
                 outstring("Using non-Bunch-Kauffman version of minimal degree.\n");
                 break;

             case 'M':
                 hess_move_con_flag = !hess_move_con_flag;
                 if ( hess_move_con_flag )
                    outstring("Projecting global constraints in move.\n");
                 else
                 outstring("Not projecting global constraints in move.\n");
                 break;
             case 'G':
                 min_square_grad_flag = !min_square_grad_flag;
                 if ( min_square_grad_flag )
                    outstring("Minimizing square gradient in seek.\n");
                 else
                    outstring("Minimizing regular energy in seek.\n");
                 break;
             default : outstring("Illegal choice.\n");
          }
     }
} /* end hessian_menu() */

/**************************************************************************
*
*  function: hessian_legal()
*
*  purpose: see if Hessian is legal for this surface.
*/
void hessian_legal()
{ int i;
  body_id b_id;

  if ( !quantities_only_flag )
  {
     if ( optparamcount > 0 ) 
     if ( auto_convert_flag ) convert_to_quantities(); else
     kb_error(2012,"Hessian for optimizing parameters requires convert_to_quantities.\n",
        RECOVERABLE);

     if ( web.representation == SIMPLEX ) 
        if ( auto_convert_flag ) convert_to_quantities(); else
        kb_error(1597,"Hessian for simplex model requires convert_to_quantities.\n", RECOVERABLE);

    for ( i = 0 ; i < web.concount ; i++ )
     if ( get_constraint(i)->attr & CON_CONTENT )
     { if ( auto_convert_flag ) { convert_to_quantities(); break; }
        else kb_error(1843,"Hessian for constraint content requires convert_to_quantities.\n",RECOVERABLE);
     }
 
    if ( web.symmetry_flag )
    FOR_ALL_BODIES(b_id)
    { if ( get_battr(b_id) & FIXEDVOL )
      { if ( auto_convert_flag ) { convert_to_quantities(); break; }
         else 
            kb_error(1844,"Cannot do torus body volumes with hessian. Do convert_to_quantities.\n",RECOVERABLE);
      }
    }

     if ( web.modeltype != LINEAR )
     { 
          if ( auto_convert_flag ) convert_to_quantities(); else
          kb_error(1598,"Hessian for non-LINEAR models requires convert_to_quantities.\n",
            RECOVERABLE);
     }
     if ( web.representation == STRING ) 
        if ( auto_convert_flag ) convert_to_quantities(); else
        kb_error(1599,"Hessian for string model requires convert_to_quantities.\n", RECOVERABLE);

     if ( web.surfen_count )
        if ( auto_convert_flag ) convert_to_quantities(); else
        kb_error(1600,"Cannot do Hessian with surface_energy. Do convert_to_quantities.\n",

         RECOVERABLE);
     if ( web.gravflag && (web.grav_const != 0.0))
        if ( auto_convert_flag ) convert_to_quantities(); else
        kb_error(1601,"Hessian method for gravity requires convert_to_quantities.\n",

          RECOVERABLE);
     if ( web.pressflag ||  web.pressure_flag )
        if ( auto_convert_flag ) convert_to_quantities(); else
        kb_error(1602,"Hessian with pressure in energy requires convert_to_quantities.\n", RECOVERABLE);

     if ( web.wulff_flag )
     kb_error(1603,"Cannot do Hessian method with crystalline integrand.\n", RECOVERABLE);

     if ( web.convex_flag )
        kb_error(1604,"Cannot do gap energy Hessian, since it would need third derivatives.\n", RECOVERABLE);

     if ( web.metric_flag )
        if ( auto_convert_flag ) convert_to_quantities(); else
        kb_error(1605,"Hessian with Riemannian metric requires convert_to_quantities.\n", RECOVERABLE);
    }

  if ( count_fixed_vol() && !pressure_set_flag && !web.pressure_flag )
      kb_error(1606,"Pressures invalid. Do regular iteration before Hessian to set pressures.\n",
        RECOVERABLE);

}

/*****************************************************************************
*
* function: hessian_init()
*
* purpose: Allocate and fill in Hessian
*/

void hessian_init(rhs,X)
REAL **rhs;  /* right side, allocated by hessian_init */
REAL **X;  /* solution, allocated by hessian_init */
{
  vertex_id v_id;
  int j,i,gam,a,b,m,n;
  struct hess_verlist *v;              /* current  vertex */
  int currow = 0;
  MAT2D(grad,MAXCOORD,MAXCOORD);
  MAT2D(gradcopy,MAXCOORD,MAXCOORD);
  MAT2D(cc,MAXCOORD,MAXCOORD);
  MAT2D(pp,MAXCOORD,MAXCOORD);
  MAT2D(qq,MAXCOORD,MAXCOORD);
  MAT2D(gg,MAXCOORD,MAXCOORD);
  MAT3D(second,MAXCOORD,MAXCOORD,MAXCOORD);
  REAL dummy;
  int total_proj,vproj_count,total_conhess,vproj_alloc,conhess_alloc;
  int vproj_spot;
  REAL ***v_normal = NULL;

  hessian_legal();
  if ( saved.coord == NULL ) save_coords(&saved);  /* in case this doesn't work */
  total_entries = 0;
  vcount = web.skel[VERTEX].max_ord+1;
  vhead = (struct hess_verlist *)temp_calloc(vcount,sizeof(struct hess_verlist));
  if ( hmode == NORMAL_MOTION ) v_normal = dmatrix3(vcount,SDIM,SDIM);

  if  ( vedge_timestamp < top_timestamp ) make_vedge_lists(); 

  /* populate vertex list and count degrees of freedom */
  total_proj = 0;    /* count columns needed */
  vproj_count = 0;  /* number of vertices needing projection */
  total_conhess = 0;    /* count rows needed */
  FOR_ALL_VERTICES(v_id)
  { conmap_t * conmap;
    int oncount;
    int kk;

    if ( get_vattr(v_id) & FIXED ) continue;
    v = vhead + ordinal(v_id);
    v->v_id = v_id;

    if ( hmode == SINGLE_DEGREE )
        v->freedom = 1;
    else if ( get_vattr(v_id) & BOUNDARY )
    { struct boundary *bdry = get_boundary(v_id);
      v->freedom = bdry->pcount;
      total_proj += v->freedom;
      vproj_count++;
      total_conhess += v->freedom;
    }
    else /* possible level set constraints */
    { 
      if ( hmode == NORMAL_MOTION ) 
      {
        kk = new_calc_vertex_normal(v_id,v_normal[ordinal(v_id)]);
        if ( kk < SDIM )
        { v->freedom = kk;
          total_proj += kk;
          vproj_count++;
        }
        else v->freedom = SDIM;
      }
      else /* v-proj still identity */
        v->freedom = SDIM;

      conmap = get_v_constraint_map(v_id);
      if ( conmap[0] )
      { /* calculate size of projection matrix to fewer degrees of freedom */
        for ( j = 1, oncount = 0; j <= (int)conmap[0] ; j++ )
        { if ( !(conmap[j] & CON_HIT_BIT) ) continue;
          oncount++;
        }

        if ( v->freedom + oncount > SDIM ) 
        { 
          total_proj += v->freedom;
          vproj_count++;
        }
        /* now constraint hessian */
        if ( oncount ) total_conhess += v->freedom;
      }
    }
  } /* end FOR_ALL_VERTICES first time through */

  if ( vproj_base ) myfree((char*)vproj_base);
  vproj_base = (REAL**)mycalloc(vproj_count*SDIM,sizeof(REAL*));
  if ( vproj_space ) myfree((char*)vproj_space);
  vproj_space = (REAL*)mycalloc(total_proj*SDIM,sizeof(REAL));;
  if ( conhess_base ) free_matrix3(conhess_base);
  conhess_base = dmatrix3(total_conhess,SDIM-1,SDIM);
  vproj_spot = vproj_alloc = conhess_alloc = 0;

  /* populate vertex list vproj and conhess */
  FOR_ALL_VERTICES(v_id)
  { conmap_t * conmap;
    int oncount;

    v = vhead + ordinal(v_id);
/*    v->rownum  = currow; */
    v->slant = 1.0;
    if ( v->freedom == 0 ) continue;
    if ( hmode == SINGLE_DEGREE )
     {} 
    else if ( get_vattr(v_id) & BOUNDARY )
    { struct boundary *bdry = get_boundary(v_id);
      v->proj = vproj_base + vproj_alloc; vproj_alloc += SDIM;
      for ( j = 0 ; j < SDIM ; j++ ) 
      { v->proj[j] = vproj_space + vproj_spot;
        vproj_spot+= v->freedom;
      }
      v->conhess = conhess_base + conhess_alloc;
      conhess_alloc += v->freedom;
      for ( j = 0 ; j < SDIM ; j++ )
      { eval_second(bdry->coordf[j],get_param(v_id),bdry->pcount,&dummy,
            v->proj[j],second[j],v_id);
        for ( m = 0 ; m < bdry->pcount ; m++ )
          for ( n = 0 ; n < bdry->pcount ; n++ )
             v->conhess[m][n][j] = second[j][m][n];
      }
    }
    else /* possible level set constraints */
    { 
      if (v->freedom < SDIM)  /* copy over normal basis */
      { 
        v->proj = vproj_base + vproj_alloc; vproj_alloc += SDIM;
        for ( j = 0 ; j < SDIM ; j++ ) 
        { v->proj[j] = vproj_space + vproj_spot;
          vproj_spot += v->freedom;
        }
        for ( j = 0 ; j < SDIM ; j++ )
          for ( i = 0 ; i < v->freedom ; i++ )
            v->proj[j][i] = v_normal[ordinal(v->v_id)][i][j];
      }

    conmap = get_v_constraint_map(v_id);     
    oncount = 0;
    if ( conmap[0] )
    { /* calculate projection matrix to fewer degrees of freedom */
      for ( j = 1; j <= (int)conmap[0] ; j++ )
      { struct constraint *constr;
        if ( !(conmap[j] & CON_HIT_BIT) ) continue;
        constr = get_constraint(conmap[j]);
        eval_second(constr->formula,get_coord(v_id),SDIM,
          &dummy, grad[oncount],second[oncount],v_id);  
           /* constraint value and derivs and hessian */
        oncount++;
      }
      if ( (v->freedom < SDIM) && (oncount > 0) ) /* normal or something */
      { 
        /* project basis to constraints */
        mat_mul_tr(grad,grad,gg,oncount,SDIM,oncount);
        mat_inv(gg,oncount);
        mat_mult(grad,v->proj,pp,oncount,SDIM,v->freedom);
        mat_mult(gg,pp,qq,oncount,oncount,v->freedom);
        tr_mat_mul(grad,qq,pp,oncount,SDIM,v->freedom);
        for ( i = 0 ; i < SDIM ; i++ )
          for ( j = 0 ; j < v->freedom ; j++ )
             v->proj[i][j] -= pp[i][j];
        
      }
      else if ( oncount > 0 )
      { v->freedom = SDIM - oncount;
        if ( v->freedom <= 0 ) continue;
        v->proj = vproj_base + vproj_alloc; vproj_alloc += SDIM;
        for ( j = 0 ; j < SDIM ; j++ ) 
        { v->proj[j] = vproj_space + vproj_spot;
          vproj_spot += v->freedom;
        }
        for ( m = 0 ; m < oncount ; m++ )
          for ( n = 0 ; n < SDIM ; n++ ) gradcopy[m][n] = grad[m][n];
        kernel_basis(gradcopy,v->proj,oncount,SDIM); 
      }
    }
    /* orthonormalize proj */
    if ( v->proj )
    { for ( i = 0 ; i < v->freedom ; i++ )
      { /* subtract previous components */
        REAL sum;
        for ( j = 0 ; j < i ; j++ ) 
        { for ( n = 0, sum = 0.0 ; n < SDIM ; n++ ) 
             sum += v->proj[n][j]*v->proj[n][i];
          for ( n = 0 ; n < SDIM ; n++ ) 
             v->proj[n][i] -= sum*v->proj[n][j];
        }
        /* normalize */
        for ( n = 0, sum = 0.0 ; n < SDIM ; n++ ) 
           sum += v->proj[n][i]*v->proj[n][i];
        sum = sqrt(sum);
        if ( sum > hessian_epsilon )
           for ( n = 0 ; n < SDIM ; n++ ) v->proj[n][i] /= sum;
        else /* have to skip this direction */
        { for ( n = 0 ; n < SDIM ; n++ )
            v->proj[n][i] = v->proj[n][v->freedom-1];
          v->freedom--;
          i--;
        }
      }
      if ( (hessian_normal_flag && (v->freedom == 1)) || hessian_double_normal_flag)
      { /* get cosine of normal and degree of freedom */
        REAL *nor = v_normal[ordinal(v->v_id)][0];
        v->slant = 0.0;
        for ( i = 0 ; i < SDIM ; i++ )
           v->slant += v->proj[i][0]*nor[i];
        v->slant /= sqrt(SDIM_dot(nor,nor));
        if ( v->slant < hessian_slant_cutoff ) 
              v->freedom = 0;
      }
    }
        
      /* now constraint hessian */
      if ( oncount )
      { v->conhess = conhess_base + conhess_alloc;
        conhess_alloc += v->freedom;
        mat_tsquare(grad,cc,oncount,SDIM);
        mat_inv(cc,oncount);
        for ( gam = 0 ; gam < SDIM ; gam++ )
         for ( a = 0 ; a < v->freedom ; a++ )
          for ( b = 0 ; b < v->freedom ; b++ )
          { REAL sum;
             for ( sum = 0.0, i = 0 ; i < oncount ; i++ )
              for ( j = 0 ; j < oncount ; j++ )
                for ( m = 0 ; m < SDIM ; m++ )
                 for ( n = 0 ; n < SDIM ; n++ )
                  sum += grad[j][gam]*cc[j][i]*v->proj[m][a]*second[i][m][n]
                              *v->proj[n][b];
             v->conhess[a][b][gam] = -sum;
          }
       }
    }
  
    if ( hessian_double_normal_flag && v->proj )
    { REAL sum;
      int k;
      /* special kludge to give double-dimension perturbations the
         same normal space as regular dimensions. See spin4d.fe
         for example of use. */
      for ( i =  0 ; i < v->freedom ; i++ )
      { /* assume low dim normals are first */
        for ( j = 0, sum = 0.0 ; j < SDIM/2 ; j++ )
           sum += v->proj[j][i]*v->proj[j][i];
        if ( sum < 1e-6 ) break;
      }
      /* copy over low dim basis to high dim, with transform */
      for ( k = 0 ; k < i ; k++ )
         for ( j = 0 ; j < SDIM/2 ; j++ ) 
         { v->proj[j][i+k] = v->proj[j+SDIM/2][k];
           v->proj[j+SDIM/2][i+k] = v->proj[j][k];
         } 
      v->freedom = 2*i;  
    } 

    if ( hess_debug && v->proj )
    { printf("v_id %d proj\n",ordinal(v_id)+1);
      for ( j = 0 ; j < v->freedom ; j++ )
      { for ( m = 0 ; m < SDIM ; m++ ) printf("%f ",(DOUBLE)v->proj[m][j]);
        printf("\n");
      }
    }

/*    currow += v->freedom;  */

  } /* end FOR_ALL_VERTICES second time through */

  if ( v_normal ) free_matrix3(v_normal); 

  /* assign row numbers */
  for ( i = 0, v = vhead, currow = 0 ; i <= web.skel[VERTEX].max_ord ; i++,v++ )
  { v->rownum = currow;
    currow += v->freedom;
  }

  /* check */
  if ( (vproj_alloc > vproj_count*SDIM) || (vproj_spot > total_proj*SDIM) ||
      (conhess_alloc > total_conhess) )
     kb_error(1608,"Internal error, hessian_init(): Overflow of vproj or conhess arena.\n",RECOVERABLE);

  /* rows for optimizing parameters */
  vertex_rows = currow;
  for ( i = 0 ; i < optparamcount ; i++)
  { optparam[i].rownum = currow++;
  }

  /* allocate matrix */
  A_rows = bodyrowstart = currow;
  quanrowstart = currow + web.bodycount;
  total_rows = quanrowstart + gen_quant_count;
  array = (struct hess_index *)temp_calloc(total_rows+1,sizeof(struct hess_index));
  pressures = (REAL*)temp_calloc(web.bodycount+gen_quant_count,sizeof(REAL));
  conrhs = (REAL*)temp_calloc(web.bodycount+gen_quant_count,sizeof(REAL));

    /* allocate right hand side and solution of augmented matrix */
  if ( rhs ) *rhs = (REAL *)temp_calloc(total_rows,sizeof(REAL));
  if ( X ) *X = (REAL *)temp_calloc(total_rows,sizeof(REAL));

  /* initialize hash table */
  hess_hash_init();

} /* end hessian_init */

/*************************************************************************
*
* function: hessian_fill()
*
* purpose: call routines that calculate hessian.
*/
void hessian_fill(rhs)
REAL *rhs;
{
  /* clear entries */
  if ( rhs_flag ) memset((char*)rhs,0,total_rows*sizeof(REAL));

  /* fill in hessian */
  if ( hessian_by_diff_flag )
     { /* hessian by crude numeric difference formulas */
        difference_hessian(vhead,rhs);
        if ( optparamcount ) optparamhess(rhs);
     }
  else /* hessian by explicit formulas */
     {
        if ( !quantities_only_flag )
        { 
          if ( web.symmetry_flag && !web.torus_flag )
          if ( auto_convert_flag ) {convert_to_quantities(); goto quantplace;}
          else
             kb_error(2717,
                "Must do convert_to_quantities to do Hessian with symmetry group.\n",
                    RECOVERABLE);
          if ( web.representation == SIMPLEX )
             simplex_hessian(rhs);
          else 
          { if ( web.representation == SOAPFILM ) area_hessian(rhs);
             edge_energy_hessian(rhs);
          }

          if ( web.surfen_count ) 
              surfen_hess();
        }
quantplace:
        if ( gen_quant_count )
          calc_quant_hess(rhs_flag,hess_flag,rhs);
        
        if ( !quantities_only_flag )
          if ( web.bodycount > web.full_flag  )
             body_hessian(rhs);

        if ( optparamcount ) optparamhess(rhs);
     }

  if ( hess_flag )
  { /* extract stuff from hash table */
     hess_hash_sort(A_rows,&total_entries,&entry_start);
     if ( !everything_quantities_flag )
      if ( web.full_flag )  /* pretend one body info only */
      { body_id b_id;
         FOR_ALL_BODIES(b_id)
         { if ( get_battr(b_id) & FIXEDVOL )
            { myfree((char*)array[bodyrowstart+ordinal(b_id)].u.congrad); 
              array[bodyrowstart+ordinal(b_id)].u.congrad = NULL;
             break;
            }
         }
      }
  }
}  /* end hessian_fill() */


/***********************************************************************
*
* function: hessian_move()
*
* purpose: move vertices by given stepsize times current vector
*/

void hessian_move(stepsize,mode,X)
REAL stepsize;
int mode; /* ACTUAL_MOVE or TEST_MOVE */
REAL *X;  /* direction */
{
  int i,j,n;
  struct hess_verlist *vc;              /* current  vertex */

  last_hessian_scale = stepsize;

  /* change optimizing parameters */
  for ( i = 0 ; i < optparamcount ; i++ )
  { optparam[i].velocity = X[optparam[i].rownum];
    globals[optparam[i].pnum].value.real += stepsize*X[optparam[i].rownum];
  }
  /* move vertices */
  for ( n = 0, vc = vhead ; n < vcount ; n++, vc++ )
     {
        REAL *coord;
        REAL *velocity;

        if ( vc->freedom <= 0 ) continue;
        coord = get_coord(vc->v_id);
        velocity = get_velocity(vc->v_id);
        if ( get_vattr(vc->v_id) & BOUNDARY )
        { struct boundary *boundary = get_boundary(vc->v_id);
          int pcount = boundary->pcount;
          REAL *param = get_param(vc->v_id);

          for ( i = 0 ; i < pcount ; i++ )
          { velocity[i] = X[vc->rownum + i];
             param[i] += stepsize*X[vc->rownum + i];
          }
          for ( i = 0 ; i < SDIM ; i++ )
                coord[i] = eval(boundary->coordf[i],param,vc->v_id);
        }
        else
        { for ( j = 0 ; j < SDIM ; j++ )
            if ( vc->proj )
            { velocity[j] = dot(vc->proj[j],X+vc->rownum,vc->freedom);
              coord[j] += stepsize*dot(vc->proj[j],X+vc->rownum,vc->freedom);
            }
            else
            { velocity[j] = X[vc->rownum + j]; 
              coord[j] += stepsize*X[vc->rownum + j];
            }
        }
     }
  if ( hess_move_con_flag )
  {
     calc_all_grads(CALC_VOLGRADS);
     project_all(hess_move_con_flag /* all constraints */,mode);
     vgrad_end();
  }
  else
     project_all(hess_move_con_flag /* all constraints ? */,mode);

  global_timestamp++;
  calc_energy();  /* energy after motion */
}  /* end hessian_move() */


/**************************************************************************
*
* function: hessian_exit()
* 
* purpose: clean up after hessian stuff.
*/

void hessian_exit(X)
REAL *X; /* direction, if any */
{ int j,n;
  struct hess_verlist *vc;              /* current  vertex */

  hess_debug = 0;

  /* store move in vertex force */
  if ( X && vhead )
     for ( n = 0, vc = vhead ; n < vcount ; n++, vc++ )
     {
        REAL *f;

        if ( vc->freedom <= 0 ) continue;
        f = get_force(vc->v_id);
        if ( get_vattr(vc->v_id) & BOUNDARY )
        { struct boundary *boundary = get_boundary(vc->v_id);
          int pcount = boundary->pcount;

          for ( j = 0 ; j < pcount ; j++ )
                f[j] = X[vc->rownum + j];
        }
        else
        { for ( j = 0 ; j < SDIM ; j++ )
            if ( vc->proj )
              f[j] = dot(vc->proj[j],X+vc->rownum,vc->freedom);
            else
              f[j] = X[vc->rownum + j];
        }
     }
  hessian_cleanup();
}

/*********************************************************************
*
* function: hessian_cleanup()
*
* purpose: free memory allocated for hessian stuff.
*
*/

void hessian_cleanup()
{
  hess_hash_end();
  if ( vhead ) temp_free((char *)vhead); vhead = NULL;
  if ( vproj_base ) myfree((char*)vproj_base); vproj_base = NULL;
  if ( vproj_space ) myfree((char*)vproj_space); vproj_space = NULL;
  if ( conhess_base ) free_matrix3(conhess_base); conhess_base = NULL;
  unsave_coords(&saved);
  /* if ( entry_start ) temp_free((char *)entry_start); */
  if ( array ) temp_free((char *)array);
  if ( volrows ) free_matrix(volrows);
  if ( pressures ) temp_free((char*)pressures);
  if ( congrads ) temp_free((char*)congrads);
  if ( conrhs ) temp_free((char*)conrhs);
  if ( ritzvecs ) free_matrix(ritzvecs); ritz_done_flag = 0; ritzvecs = NULL;
  vhead = NULL;
  entry_start = NULL;
  array = NULL;
  volrows = NULL;
  pressures = NULL;
  congrads = NULL;
  conrhs = NULL;
  return;

}

/********************************************************************
*
* function: write_hessian()
*
* purpose: Write hessian to file in format suitable for
*             wet cone investigations.
*/

void write_hessian(S)
struct linsys *S;
{
  FILE *fd;
  char name[100];
  int i,j,n,row;
  struct hess_verlist *v;

  prompt("Enter hessian file name: ",name,sizeof(name));
  fd = fopen(name,"w");
  if ( fd == NULL )
  { perror(name); return; }

  /* size info */
  fprintf(fd,"%d space dimension\n",SDIM);
  fprintf(fd,"%d rows\n",S->N);
  fprintf(fd,"%d entries\n",S->IA[S->N]-A_OFF);
  if ( hessian_linear_metric_flag )
  {  fprintf(fd,"%d metric rows\n",Met.N);
      fprintf(fd,"%d metric entries\n",Met.IA[S->N]-A_OFF);
  }
  else
  {  fprintf(fd,"%d metric rows\n",0);
      fprintf(fd,"%d metric entries\n",0);
  }
  fprintf(fd,"%d constraints\n",S->CN);

  /* per row info:
      point coords, basis vectors for each degree of freedom,
    */
  
  for ( i = 0, v = vhead ; i <= web.skel[VERTEX].max_ord ; i++,v++ )
  { REAL *x = get_coord(v->v_id);
     for ( j = 0 ; j < v->freedom ; j++ )
     { fprintf(fd,"%d ",v->rownum+j);
        for ( n = 0 ; n < SDIM ; n++ ) fprintf(fd," %21.15e",(DOUBLE)x[n]);
        if ( v->proj )
         for ( n = 0 ; n < SDIM ; n++ ) fprintf(fd," %21.15e",(DOUBLE)v->proj[n][j]);
        else
         for ( n = 0 ; n < SDIM ; n++ ) fprintf(fd," %d",n==j);
        fprintf(fd,"\n");
      }
  }

  /* the Hessian */
  for ( i = 0, v = vhead ; i <= web.skel[VERTEX].max_ord ; i++,v++ )
  { 
     if ( v->freedom == 0 ) continue;
     for ( row = v->rownum ; row < v->rownum + v->freedom ; row++ )
     { int start = S->IA[row]-A_OFF;
        int end = S->IA[row+1] - A_OFF;
        for ( j = start ; j < end ; j++ )
          fprintf(fd,"%d %d %21.15e\n",row,S->JA[j]-A_OFF,(DOUBLE)S->A[j]);
     }
  }
  /* the metric */
  if (hessian_linear_metric_flag)
  for ( i = 0, v = vhead ; i <= web.skel[VERTEX].max_ord ; i++,v++ )
  { 
     if ( v->freedom == 0 ) continue;
     for ( row = v->rownum ; row < v->rownum + v->freedom ; row++ )
     { int start = Met.IA[row]-A_OFF;
        int end = Met.IA[row+1] - A_OFF;
        for ( j = start ; j < end ; j++ )
          fprintf(fd,"%d %d %21.15e\n",row,Met.JA[j]-A_OFF,(DOUBLE)Met.A[j]);
     }
  }

  /* constraints */
  for ( i = 0 ; i < S->CN ; i++ )
     for ( j = 0 ; j < S->N ; j++ )
        fprintf(fd,"%d %d %21.15e\n",i,j,(DOUBLE)S->C[i][j]);
  fclose(fd);
}

/**********************************************************************
*
* function: optparamhess()
*
* purpose: fill in hessian and right side for optimizing parameter rows.
*
* method: finite differences
*/

void optparamhess(rhs)
REAL *rhs;  /* right side */
{ int i,j,m;
  struct oldcoord csaved;
  REAL *fake_rhs;    /* for grad differences */
  REAL *temprow;     /* for accumulating row values */
  struct gen_quant *q;

  if ( optparamcount <= 0 ) return;
  if ( !everything_quantities_flag ) 
     kb_error(2012,"Hessian for optimizing parameters requires convert_to_quantities.\n",
        RECOVERABLE);

  fake_rhs = (REAL*)temp_calloc(total_rows,sizeof(REAL));
  temprow  = (REAL*)temp_calloc(total_rows,sizeof(REAL));

  csaved.coord = NULL;
  save_coords(&csaved);
  for ( i = 0 ; i < optparamcount ; i++ )
      { REAL dp;
        REAL emid = web.total_energy;
        REAL eleft,eright;
        volgrad *vgptri;
        REAL g[MAXCOORD];
        REAL *ggg;
        struct hess_verlist *va;
        vertex_id v_id;

        restore_coords(&csaved);
        calc_energy(); calc_content(Q_FIXED);
        emid = web.total_energy;

        dp = globals[optparam[i].pnum].delta;

        memset((char*)fake_rhs,0,total_rows*sizeof(REAL));
        memset((char*)temprow ,0,total_rows*sizeof(REAL));

        /* for optimizing parameter self hessian */
        for ( m = 0, q = GEN_QUANTS ; m < gen_quant_count ; m++,q++ )
          if ( q->flags & Q_FIXED )
          { hess_hash_search(optparam[i].rownum,optparam[i].rownum,
                -q->pressure*( - 2*q->value )/dp/dp);
          }             

        /* right difference */
        globals[optparam[i].pnum].value.real += dp;
        project_all(0, TEST_MOVE);
        if ( fixed_constraint_flag || web.pressure_flag || web.pressflag )
            calc_content(Q_FIXED);
        calc_energy();  /* energy after motion */
        calc_quant_hess(1,0,temprow);
        eright = web.total_energy;
        calc_volgrads(NO_OPTS); /* global constraint hessians */
        FOR_ALL_VERTICES(v_id)
        { va = vhead + ordinal(v_id);
          vgptri = get_vertex_vgrad(v_id);
          while ( vgptri )
          { REAL ccoeff = -GEN_QUANTS[vgptri->qnum].pressure/2/dp;
             if ( va->proj )
                { vec_mat_mul(vgptri->grad,va->proj,g,SDIM,va->freedom);
                  ggg = g;
                }
             else ggg = vgptri->grad;
             for ( m = 0 ; m < va->freedom ; m++ )
                hess_hash_search(optparam[i].rownum,va->rownum+m,ccoeff*ggg[m]);
             vgptri = vgptri->chain;
          }
        }
        /* for global constraints */
        for ( m = 0, q = GEN_QUANTS ; m < gen_quant_count ; m++,q++ )
          if ( q->flags & Q_FIXED )
          { hess_hash_search(optparam[i].rownum,optparam[i].rownum,
                -q->pressure*( q->value )/dp/dp);
             hess_hash_search(quanrowstart+m,optparam[i].rownum,
                 q->value/dp/2);
             if ( rhs ) rhs[optparam[i].rownum] += q->pressure*q->value/dp/2;
          } 
        restore_coords(&csaved);  /* also restores opt params */

        /* left difference */
        globals[optparam[i].pnum].value.real -= dp;
        project_all(0, TEST_MOVE);
        if ( fixed_constraint_flag || web.pressure_flag || web.pressflag )
            calc_content(Q_FIXED);
        calc_energy();  /* energy after motion */
        calc_quant_hess(1,0,fake_rhs);
        eleft = web.total_energy;
        calc_volgrads(NO_OPTS); /* global constraint hessians */
        FOR_ALL_VERTICES(v_id)
        { 
          va = vhead + ordinal(v_id);
          vgptri = get_vertex_vgrad(v_id);
          while ( vgptri )
          { REAL ccoeff = GEN_QUANTS[vgptri->qnum].pressure/2/dp;
             if ( va->proj )
                { vec_mat_mul(vgptri->grad,va->proj,g,SDIM,va->freedom);
                  ggg = g;
                }
             else ggg = vgptri->grad;
             for ( m = 0 ; m < va->freedom ; m++ )
                hess_hash_search(optparam[i].rownum,va->rownum+m,ccoeff*ggg[m]);
             vgptri = vgptri->chain;
          }
        }
        /* for global constraints */
        for ( m = 0, q = GEN_QUANTS ; m < gen_quant_count ; m++,q++ )
          if ( q->flags & Q_FIXED )
          { hess_hash_search(optparam[i].rownum,optparam[i].rownum,
                -q->pressure*( q->value )/dp/dp);
             hess_hash_search(quanrowstart+m,optparam[i].rownum,
                 -q->value/dp/2);
             if ( rhs )  rhs[optparam[i].rownum] -= q->pressure*q->value/dp/2;
          } 
             

        vgrad_end();
        restore_coords(&csaved);  /* also restores opt params */
        web.total_energy = emid; /* restore */

        if ( rhs )
          rhs[optparam[i].rownum] += -(eright - eleft)/2/dp; /* rhs is neg grad */

        for ( j = 0 ; j < vertex_rows ; j++ )
          hess_hash_search(optparam[i].rownum,j,(fake_rhs[j]-temprow[j])/2/dp);
        hess_hash_search(optparam[i].rownum,optparam[i].rownum,
                (eright - 2*emid + eleft)/dp/dp);
     }
  temp_free((char*)fake_rhs);
  temp_free((char*)temprow);

  /* optimizing parameter mixed terms */
  for ( i = 0 ; i < optparamcount ; i++ )
  for ( j = 0 ; j < i ; j++ )
      { REAL dp;
        REAL vmid = globals[optparam[i].pnum].value.real;
        REAL emid = web.total_energy;
        REAL ehihi,ehilo,elohi,elolo;

        if ( fabs(vmid) > 1. ) dp = fabs(vmid)*1e-3;
        else dp = 1e-3;

        /* + + difference */
        globals[optparam[i].pnum].value.real += dp;
        globals[optparam[j].pnum].value.real += dp;
        project_all(0, TEST_MOVE);
        if ( fixed_constraint_flag || web.pressure_flag || web.pressflag )
            calc_content(Q_FIXED);
        calc_energy();  /* energy after motion */
        ehihi = web.total_energy;
        for ( m = 0, q = GEN_QUANTS ; m < gen_quant_count ; m++,q++ )
          if ( q->flags & Q_FIXED )
             hess_hash_search(optparam[i].rownum,optparam[j].rownum,
                -q->pressure*( q->value )/dp/dp/4);
             
        restore_coords(&csaved);  /* also restores opt params */

        /* + - difference */
        globals[optparam[i].pnum].value.real += dp;
        globals[optparam[j].pnum].value.real -= dp;
        project_all(0, TEST_MOVE);
        if ( fixed_constraint_flag || web.pressure_flag || web.pressflag )
            calc_content(Q_FIXED);
        calc_energy();  /* energy after motion */
        ehilo = web.total_energy;
        for ( m = 0, q = GEN_QUANTS ; m < gen_quant_count ; m++,q++ )
          if ( q->flags & Q_FIXED )
             hess_hash_search(optparam[i].rownum,optparam[j].rownum,
                -q->pressure*(-q->value )/dp/dp/4);
        restore_coords(&csaved);  /* also restores opt params */

        /* - + difference */
        globals[optparam[i].pnum].value.real -= dp;
        globals[optparam[j].pnum].value.real += dp;
        project_all(0, TEST_MOVE);
        if ( fixed_constraint_flag || web.pressure_flag || web.pressflag )
            calc_content(Q_FIXED);
        calc_energy();  /* energy after motion */
        elohi = web.total_energy;
        for ( m = 0, q = GEN_QUANTS ; m < gen_quant_count ; m++,q++ )
          if ( q->flags & Q_FIXED )
             hess_hash_search(optparam[i].rownum,optparam[j].rownum,
                -q->pressure*(-q->value )/dp/dp/4);
        restore_coords(&csaved);  /* also restores opt params */

        /* - - difference */
        globals[optparam[i].pnum].value.real -= dp;
        globals[optparam[j].pnum].value.real -= dp;
        project_all(0, TEST_MOVE);
        if ( fixed_constraint_flag || web.pressure_flag || web.pressflag )
            calc_content(Q_FIXED);
        calc_energy();  /* energy after motion */
        elolo = web.total_energy;
        for ( m = 0, q = GEN_QUANTS ; m < gen_quant_count ; m++,q++ )
          if ( q->flags & Q_FIXED )
             hess_hash_search(optparam[i].rownum,optparam[j].rownum,
                -q->pressure*( q->value )/dp/dp/4);
        restore_coords(&csaved);  /* also restores opt params */

        web.total_energy = emid; /* restore */

        hess_hash_search(optparam[i].rownum,optparam[j].rownum,
                (ehihi-ehilo-elohi+elolo)/dp/dp/4);
     }

  unsave_coords(&csaved);
} /* end optparamhess() */
