/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com */
using System;
using System.Text;
using System.IO;
using System.Collections;
using Db4objects.Db4o;
using Db4objects.Db4o.TA;
using Db4objects.Db4o.Config;
using Db4objects.Db4o.Activation;
using Db4objects.Db4o.IO;
using Db4objects.Db4o.Query;

namespace Db4objects.Db4odoc.Performance
{
    class UpdatePerformanceBenchmark
    {

        private static int _count = 100000;

        private static int _commitInterval = 1000;

        private static int _depth = 3;

        private static bool _isClientServer = false;

        private static bool TCP = true;

        private static string _filePath = "performance.db4o";

        private static string _host = "localhost";

        private static int PORT = 4477;


        private IObjectContainer objectContainer;

        private IObjectServer objectServer;


        private long _startTime;


        public static void Main(string[] arguments)
        {
            //new UpdatePerformanceBenchmark().RunConfigurationTest();
            //new UpdatePerformanceBenchmark().RunDifferentObjectsTest();
            //new UpdatePerformanceBenchmark().RunCommitTest();
            //new UpdatePerformanceBenchmark().RunHardDriveTest();
            //new UpdatePerformanceBenchmark().RunRamDiskTest();
            //new UpdatePerformanceBenchmark().RunClientServerTest();
            //new UpdatePerformanceBenchmark().RunInheritanceTest();
            new UpdatePerformanceBenchmark().RunIndexTest();
        }
        // end Main

        private void RunCommitTest()
        {
            System.Console.WriteLine("Update test with different Commit frequency");

            InitForCommitTest();

            Clean();
            System.Console.WriteLine("Test Update all:");
            Open(ConfigureForCommitTest());
            Store();
            UpdateItems(_count);
            Close();



            Clean();
            System.Console.WriteLine("Test Update all with Commit after each " + _commitInterval + " objects:");
            Open(ConfigureForCommitTest());
            Store();
            UpdateWithCommit(_count);
            Close();


        }
        // end RunCommitTest


        private void RunRamDiskTest()
        {
            System.Console.WriteLine("Update test: RAM disk");
            int objectsToUpdate = 90;
            InitForRamDriveTest();
            Clean();
            Open(ConfigureDriveTest());
            Store();
            System.Console.WriteLine("Updating  " + objectsToUpdate + " objects on a RAM drive:");
            UpdateItems(objectsToUpdate);
            Close();
        }
        // end RunRamDiskTest

        private void RunHardDriveTest()
        {
            System.Console.WriteLine("Update test: hard drive");
            int objectsToUpdate = 90;

            InitForHardDriveTest();
            Clean();
            Open(ConfigureDriveTest());
            Store();
            System.Console.WriteLine("Updating  " + objectsToUpdate + " objects on a hard drive:");
            UpdateItems(objectsToUpdate);
            Close();
        }
        // end RunHardDriveTest

        private void RunClientServerTest()
        {
            System.Console.WriteLine("Update test: Client/Server environment");
            int objectsToUpdate = 30;


            Init();
            Clean();
            Open(ConfigureClientServer());
            Store();
            System.Console.WriteLine("Update " + objectsToUpdate + " of " + _count + " objects locally:");
            UpdateItems(objectsToUpdate);
            Close();

            InitForClientServer();
            Clean();
            Open(ConfigureClientServer());
            Store();
            System.Console.WriteLine("Update " + objectsToUpdate + " of " + _count + " objects remotely:");
            UpdateItems(objectsToUpdate);
            Close();
        }
        // end RunClientServerTest

        private void RunInheritanceTest()
        {
            System.Console.WriteLine("Update test: objects with deep inheritance");

            int objectsToUpdate = 30;
            Init();
            Clean();
            Open(Configure());
            Store();
            System.Console.WriteLine("Updating " + objectsToUpdate + " objects");
            UpdateItems(objectsToUpdate);
            Close();

            Clean();
            Open(Configure());
            StoreInherited();
            System.Console.WriteLine("Updating " + objectsToUpdate + " inherited objects");
            UpdateItems(objectsToUpdate);
            Close();

        }
        // end RunInheritanceTest

        private void RunConfigurationTest()
        {
            System.Console.WriteLine("Update test with different configurations");

            //
            Clean();
            Init();
            System.Console.WriteLine("Update test: default configurations");
            Open(Db4oFactory.NewConfiguration());
            Store();
            UpdateItems(90);
            Close();
            //

            Clean();
            System.Console.WriteLine("Update test: memory IO adapter");
            Open(Configure());
            Store();
            UpdateItems(90);
            Close();
            //
            Clean();
            System.Console.WriteLine("Update test: cascade on Update");
            Open(ConfigureCascade());
            Store();
            UpdateTopLevelItems(90);
            Close();

            //
            Clean();
            System.Console.WriteLine("Update test: Transparent Persistence");
            Open(ConfigureTP());
            StoreActivatableItems();
            UpdateActivatableItems(90);
            Close();

        }
        // end RunConfigurationTest

        private void UpdateItems(int count)
        {
            StartTimer();
            IObjectSet result = objectContainer.QueryByExample(null);

            for (int i = 0; i < count; i++)
            {
                if (result.HasNext())
                {
                    Item item = (Item)result.Next();
                    item._name = "Updated";
                    Update(item);
                }
                else
                {
                    count = i;
                    break;
                }
            }
            StopTimer("Updated " + count + " items");
        }
        // end UpdateItems

        private void UpdateWithCommit(int count)
        {
            StartTimer();
            IObjectSet result = objectContainer.QueryByExample(null);
            int j = 0;
            for (int i = 0; i < count; i++)
            {
                if (result.HasNext())
                {
                    Item item = (Item)result.Next();
                    item._name = "Updated";
                    Update(item);
                    if (j >= _commitInterval)
                    {
                        j = 0;
                        objectContainer.Commit();
                    }
                    else
                    {
                        j++;
                    }
                }
                else
                {
                    count = i;
                    break;
                }
            }
            StopTimer("Updated " + count + " items ");
        }
        // end UpdateWithCommit

        private void UpdateTopLevelItems(int count)
        {
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level0").StartsWith(true);
            IObjectSet result = query.Execute();

            for (int i = 0; i < count; i++)
            {
                if (result.HasNext())
                {
                    Item item = (Item)result.Next();
                    item._name = "Updated";
                    Update(item);
                }
                else
                {
                    count = i;
                    break;
                }
            }
            StopTimer("Updated " + count + " items");
        }
        // end UpdateTopLevelItems

        private void UpdateActivatableItems(int count)
        {
            StartTimer();
            IQuery Query = objectContainer.Query();
            Query.Constrain(typeof(ActivatableItem));
            Query.Descend("_name").Constrain("level0").StartsWith(true);
            IObjectSet result = Query.Execute();

            for (int i = 0; i < count; i++)
            {
                if (result.HasNext())
                {
                    ActivatableItem item = (ActivatableItem)result.Next();
                    item.Name = "Updated";
                    Update(item);
                }
                else
                {
                    count = i;
                    break;
                }
            }
            StopTimer("Updated " + count + " items");
        }
        // end UpdateActivatableItems

        private void Update(Object item)
        {
            objectContainer.Store(item);
        }
        // end Update


        private void RunDifferentObjectsTest()
        {
            System.Console.WriteLine("Update test with different objects");
            int objectsToUpdate = 90;
            int updated = objectsToUpdate;

            InitDifferentObjectsTest();

            Clean();
            System.Console.WriteLine(" - primitive object with int field");
            Open(Configure());
            StoreSimplest();

            IObjectSet result = objectContainer.QueryByExample(null);
            StartTimer();
            for (int i = 0; i < objectsToUpdate; i++)
            {
                if (result.HasNext())
                {
                    SimplestItem item = (SimplestItem)result.Next();
                    item._id = 1;
                    Update(item);
                }
                else
                {
                    updated = i;
                    break;
                }
            }
            StopTimer("Updated " + updated + " items");
            Close();

            Clean();
            Open(Configure());
            System.Console.WriteLine(" - object with string field");
            Store();
            updated = objectsToUpdate;
            result = objectContainer.QueryByExample(null);
            StartTimer();
            for (int i = 0; i < objectsToUpdate; i++)
            {
                if (result.HasNext())
                {
                    Item item = (Item)result.Next();
                    item._name = "Updated";
                    Update(item);
                }
                else
                {
                    updated = i;
                    break;
                }
            }
            StopTimer("Updated " + updated + " items");
            Close();

            Clean();
            Open(Configure());
            System.Console.WriteLine(" - object with StringBuilder field");
            StoreWithStringBuilder();

            updated = objectsToUpdate;
            result = objectContainer.QueryByExample(null);
            StartTimer();
            for (int i = 0; i < objectsToUpdate; i++)
            {
                if (result.HasNext())
                {
                    ItemWithStringBuilder item = (ItemWithStringBuilder)result.Next();
                    item._name = new StringBuilder("Updated");
                    Update(item);
                }
                else
                {
                    updated = i;
                    break;
                }
            }
            StopTimer("Updated " + updated + " items");
            Close();

            Clean();
            Open(Configure());
            System.Console.WriteLine(" - object with int array field");
            StoreWithArray();
            updated = objectsToUpdate;
            result = objectContainer.QueryByExample(null);
            StartTimer();
            for (int i = 0; i < objectsToUpdate; i++)
            {
                if (result.HasNext())
                {
                    ItemWithArray item = (ItemWithArray)result.Next();
                    item._id = new int[] { 1, 2, 3 };
                    Update(item);
                }
                else
                {
                    updated = i;
                    break;
                }
            }
            StopTimer("Updated " + updated + " items");
            Close();

            Clean();
            Open(Configure());
            System.Console.WriteLine(" - object with ArrayList field");
            StoreWithArrayList();
            updated = objectsToUpdate;
            result = objectContainer.QueryByExample(null);
            StartTimer();
            for (int i = 0; i < objectsToUpdate; i++)
            {
                if (result.HasNext())
                {
                    ItemWithArrayList item = (ItemWithArrayList)result.Next();
                    item._ids = new ArrayList();
                    Update(item);
                }
                else
                {
                    updated = i;
                    break;
                }
            }
            StopTimer("Updated " + updated + " items");
            Close();
        }
        // end RunDifferentObjectsTest


        private void RunIndexTest()
        {
            System.Console.WriteLine("Update test for objects with and without indexed fields");

            int objectsToUpdate = 100;
            Init();
            System.Console.WriteLine("Updating " + objectsToUpdate + " of " + _count + " objects");
            Clean();
            Open(Configure());
            Store();
            UpdateItems(objectsToUpdate);
            Close();

            Clean();
            Init();
            System.Console.WriteLine("Updating " + objectsToUpdate + " of " + _count + " objects with indexed field");
            Open(ConfigureIndexTest());
            Store();
            UpdateItems(objectsToUpdate);
            Close();
        }
        // end RunIndexTest


        private void Init()
        {
            _count = 1000;
            _depth = 90;
            _isClientServer = false;

        }
        // end Init

        private void InitDifferentObjectsTest()
        {
            _count = 1000;
            _depth = 1;
            _isClientServer = false;

        }
        // end InitDifferentObjectsTest


        private void InitForClientServer()
        {
            _count = 1000;
            _depth = 90;
            _isClientServer = true;
            _host = "localhost";
        }
        // end InitForClientServer

        private void InitForRamDriveTest()
        {
            _count = 30000;
            _depth = 1;
            _filePath = "r:\\performance.db4o";
            _isClientServer = false;

        }
        // end InitForRamDriveTest

        private void InitForHardDriveTest()
        {
            _count = 10000;
            _depth = 3;
            _filePath = "performance.db4o";
            _isClientServer = false;
        }
        // end InitForHardDriveTest

        private void InitForCommitTest()
        {
            _count = 10000;
            _commitInterval = 1000;
            _depth = 3;
            _isClientServer = false;
        }
        // end InitForCommitTest

        private void Clean()
        {
            File.Delete(_filePath);
        }
        // end Clean

        private IConfiguration Configure()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            // using MemoryIoAdapter improves the performance 
            // by replacing the costly disk IO operations with 
            // memory access
            config.Io(new MemoryIoAdapter());
            return config;
        }
        // end Configure

        private IConfiguration ConfigureTP()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            // With Transparent Persistence enabled only modified
            // objects are written to disk. This allows to achieve 
            // better performance
            config.ObjectClass(typeof(Item)).CascadeOnUpdate(true);
            return config;
        }
        // end ConfigureTP

        private IConfiguration ConfigureCascade()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            // CascadeOnUpdate can be a performance-killer for 
            // deep object hierarchies
            config.ObjectClass(typeof(Item)).CascadeOnUpdate(true);
            return config;
        }
        // end ConfigureCascade

        private IConfiguration ConfigureIndexTest()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            config.Io(new MemoryIoAdapter());
            config.ObjectClass(typeof(Item)).ObjectField("_name").Indexed(true);
            return config;
        }
        // end ConfigureIndexTest

        private IConfiguration ConfigureForCommitTest()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            // the Commit information is physically written 
            // and in the correct order
            config.FlushFileBuffers(true);
            return config;
        }
        // end ConfigureForCommitTest

        private IConfiguration ConfigureClientServer()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            config.ClientServer().SingleThreadedClient(true);
            return config;
        }
        // end ConfigureClientServer

        private IConfiguration ConfigureDriveTest()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            config.FlushFileBuffers(true);
            return config;
        }
        // end ConfigureDriveTest

        private void Store()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                Item item = new Item("level" + i, null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new Item("level" + i + "/" + j, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }
        // end Store

        private void StoreActivatableItems()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                ActivatableItem item = new ActivatableItem("level" + i, null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new ActivatableItem("level" + i + "/" + j, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }
        // end StoreActivatableItems

        private void StoreInherited()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                ItemDerived item = new ItemDerived("level" + i, null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new ItemDerived("level" + i + "/" + j, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }
        // end StoreInherited


        private void StoreWithStringBuilder()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                ItemWithStringBuilder item = new ItemWithStringBuilder(new StringBuilder("level" + i), null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new ItemWithStringBuilder(new StringBuilder("level" + i + "/" + j), item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }
        // end StoreWithStringBuilder

        private void StoreSimplest()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                SimplestItem item = new SimplestItem(i, null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new SimplestItem(i, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }
        // end StoreSimplest

        private void StoreWithArray()
        {
            StartTimer();
            int[] array = new int[] { 1, 2, 3, 4 };
            for (int i = 0; i < _count; i++)
            {
                int[] id = new int[] { 1, 2, 3, 4 };
                ItemWithArray item = new ItemWithArray(id, null);
                for (int j = 1; j < _depth; j++)
                {
                    int[] id1 = new int[] { 1, 2, 3, 4 };
                    item = new ItemWithArray(id1, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }
        // end StoreWithArray

        private void StoreWithArrayList()
        {
            StartTimer();
            ArrayList idList = new ArrayList();
            idList.Add(1);
            idList.Add(2);
            idList.Add(3);
            idList.Add(4);
            for (int i = 0; i < _count; i++)
            {
                ArrayList ids = new ArrayList();
                ids.AddRange(idList);
                ItemWithArrayList item = new ItemWithArrayList(ids, null);
                for (int j = 1; j < _depth; j++)
                {
                    ArrayList ids1 = new ArrayList();
                    ids1.AddRange(idList);
                    item = new ItemWithArrayList(ids1, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }
        // end StoreWithArrayList

        private int TotalObjects()
        {
            return _count * _depth;
        }
        // end TotalObjects

        private void Open(IConfiguration config)
        {
            if (_isClientServer)
            {
                int port = TCP ? PORT : 0;
                string user = "db4o";
                string password = user;
                objectServer = Db4oFactory.OpenServer(_filePath, port);
                objectServer.GrantAccess(user, password);
                objectContainer = TCP ? Db4oFactory.OpenClient(_host, port, user,
                        password) : objectServer.OpenClient();
            }
            else
            {
                objectContainer = Db4oFactory.OpenFile(config, _filePath);
            }
        }
        // end Open

        private void Close()
        {
            objectContainer.Close();
            if (_isClientServer)
            {
                objectServer.Close();
            }
        }
        //end Close

        private void StartTimer()
        {
            _startTime = DateTime.Now.Ticks;
        }
        // end StartTimer

        private void StopTimer(string message)
        {
            long stop = DateTime.Now.Ticks;
            long duration = stop - _startTime;
            System.Console.WriteLine(message + ": " + duration + " ticks");
        }
        // end StopTimer

        public class Item
        {

            public string _name;
            public Item _child;

            public Item()
            {

            }

            public Item(string name, Item child)
            {
                _name = name;
                _child = child;
            }
        }
        // end Item

        public class ActivatableItem : IActivatable
        {

            private string _name;
            public ActivatableItem _child;

            [System.NonSerialized]
            IActivator _activator;

            public void Bind(IActivator activator)
            {
                if (_activator == activator)
                {
                    return;
                }
                if (activator != null && _activator != null)
                {
                    throw new System.InvalidOperationException();
                }
                _activator = activator;
            }

            public void Activate(ActivationPurpose purpose)
            {
                if (_activator == null) return;
                _activator.Activate(purpose);
            }


            public ActivatableItem()
            {

            }

            public ActivatableItem(string name, ActivatableItem child)
            {
                Name = name;
                _child = child;
            }

            public string Name
            {
                get
                {
                    return _name;
                }
                set
                {
                    _name = value;
                }
            }


        }
        // end ActivatableItem

        
        public class ItemDerived : Item
        {

            public ItemDerived(string name, ItemDerived child)
                : base(name, child)
            {
                
            }
        }
        // end ItemDerived

        public class ItemWithStringBuilder
        {

            public StringBuilder _name;
            public ItemWithStringBuilder _child;

            public ItemWithStringBuilder()
            {
            }

            public ItemWithStringBuilder(StringBuilder name, ItemWithStringBuilder child)
            {
                _name = name;
                _child = child;
            }
        }
        // end ItemWithStringBuilder

        public class SimplestItem
        {

            public int _id;
            public SimplestItem _child;

            public SimplestItem()
            {
            }

            public SimplestItem(int id, SimplestItem child)
            {
                _id = id;
                _child = child;
            }
        }
        // end SimplestItem

        public class ItemWithArray
        {

            public int[] _id;
            public ItemWithArray _child;

            public ItemWithArray()
            {
            }

            public ItemWithArray(int[] id, ItemWithArray child)
            {
                _id = id;
                _child = child;
            }
        }
        // end ItemWithArray

        public class ItemWithArrayList
        {

            public ArrayList _ids;
            public ItemWithArrayList _child;

            public ItemWithArrayList()
            {
            }

            public ItemWithArrayList(ArrayList ids, ItemWithArrayList child)
            {
                _ids = ids;
                _child = child;
            }
        }
        // end ItemWithArrayList
    }
}
