package buoy.widget;

import buoy.event.*;
import java.awt.*;
import javax.swing.JComponent;

/**
 * A CustomWidget is used for defining new graphical objects.  It occupies a rectangular region of a
 * window, and its preferred, minimum, and maximum sizes can all be modified.
 * <p>
 * In addition to the event types generated by all Widgets, CustomWidgets generate the following
 * event types:
 * <ul>
 * <li>{@link buoy.event.RepaintEvent RepaintEvent}</li>
 * </ul>
 *
 * @author Peter Eastman
 */

public class CustomWidget extends Widget
{
  private boolean opaque;
  
  /**
   * Create a new CustomWidget.
   */
  
  public CustomWidget()
  {
    component = new CustomWidgetComponent();
    opaque = true;
  }

  /**
   * Set the smallest size at which this Widget can reasonably be drawn.  When a WidgetContainer lays out
   * its contents, it will attempt never to make this Widget smaller than its minimum size.
   */
  
  public void setMinimumSize(Dimension size)
  {
    getComponent().setMinimumSize(size);
    invalidateSize();
  }
  
  /**
   * Set the largest size at which this Widget can reasonably be drawn.  When a WidgetContainer lays out
   * its contents, it will attempt never to make this Widget larger than its maximum size.
   */
  
  public void setMaximumSize(Dimension size)
  {
    getComponent().setMaximumSize(size);
    invalidateSize();
  }

  /**
   * Set the preferred size at which this Widget will look best.  When a WidgetContainer lays out
   * its contents, it will attempt to make this Widget as close as possible to its preferred size.
   */
  
  public void setPreferredSize(Dimension size)
  {
    getComponent().setPreferredSize(size);
    invalidateSize();
  }
  
  /**
   * Determine whether this Widget is set to be opaque.  If true, then it will be filled with its
   * background color before RepaintEvents are sent out.  If false, the parent container will show through.
   */
  
  public boolean isOpaque()
  {
    return opaque;
  }

  /**
   * Set whether this Widget should be opaque.  If true, then it will be filled with its background color
   * before RepaintEvents are sent out.  If false, the parent container will show through.
   */
  
  public void setOpaque(boolean opaque)
  {
    this.opaque = opaque;
  }
  
  /**
   * This inner class is the Component used to implement CustomWidgets.
   */
  
  private class CustomWidgetComponent extends JComponent
  {
    public void paintComponent(Graphics g)
    {
      if (opaque)
      {
        Dimension size = getSize();
        g.setColor(getBackground());
        g.fillRect(0, 0, size.width, size.height);
        g.setColor(getForeground());
      }
      CustomWidget.this.dispatchEvent(new RepaintEvent(CustomWidget.this, (Graphics2D) g));
    }
    
    public boolean isOpaque()
    {
      return opaque;
    }
  }
}