#!/usr/bin/perl

# This is a Tk interface for Bastille-Linux.  It is derived from
# the Bastille-1.1.1 InteractiveBastille.pl script, with the logic
# warped into the events-and-callbacks programming model typical
# of X programming.  It requires the Tk.pm perl module and at 
# least the X client runtime libraries.  The new version of the &Ask
# function is called from the callback attached to the Next button
# to display the next question.  Hopefully the logic to choose the
# next question is identical to the Curses version, but this would
# be the most likely place to find bugs.
#
# Control flow works like this:
#
# User interface implementation modules for Bastille have one
# externally-callable entry point: the do_Bastille function.  The
# purpose of do_Bastille is to initialize the interface, populate
# the Questions database, and start asking questions.
#
# In this Tk implementation, the contents of the config file is
# used to populated the answers in the Questions database, if there
# is a config file.
#
# After the Questions database is populated, &Ask is called to
# show the title page.  On this first call, &Ask creates the main
# window and all the widgets.  It then fills them in and returns.
# The caller then calls Tk's MainLoop routine, which never returns.
#
# Callback functions are attached to the Listbox and to the Back,
# Next, and Detail buttons.  The buttons are intended to function
# identically to the Curses version.  The Listbox shows all the
# modules in the database and allows the user to jump to the beginning
# of any module at any time.  (The callback basically looks up the
# index of the first question in the selected module and calls
# &Ask on it.)
#
# This version uses the underlying Bastille modules and questions
# database with no changes.
#
# The portions of this file derived from the original Bastille 1.1.1
# InteractiveBastille.pl script retain this copyright notice:
#
# Copyright (C) 2000 Jay Beale
# Licensed under the GNU General Public License    
#
# Tk additions copyright (C) 2000 by Paul L. Allen and licensed
# under the GNU General Public License.
#

#use Data::Dumper;

# The Mandrake 7.1 perl-Tk RPM installs and runs on RedHat 6.1 systems, even though
# the latter has a perl version prior to 5.6.  If your perl is 5.6 or you build your
# perl/Tk from source, , you can probably comment-out the "use lib" line below.  If 
# you don't have the Tk perl module installed, the Mandrake 7.1 RPM can be found here:
#
#    http://rpmfind.net/linux/Mandrake/7.1/Mandrake/RPMS/perl-Tk-800.018-2mdk.i586.rpm
#
# The sources that were current when this code was written are at:
#
#    http://www.perl.com/CPAN-local/modules/by-module/Tk/Tk800.022.tar.gz
#
if ( -d "/usr/lib/perl5/site_perl/5.6.0/i386-linux" ) {
	use lib "/usr/lib/perl5/site_perl/5.6.0/i386-linux";
}

use Tk;
use Tk::LabFrame;
use Tk::DialogBox;

sub do_Bastille {

#	True if we have not yet asked our first question.
#
	$first_time = 1;

#	Number_Modules is the number of modules loaded in by Load_Questions
#
	$Number_Modules=0;

#	These variables contain the state of and text for the detail button.
#
	$detail_toggle="high";
	$detail_button{"low"}="Explain More";
	$detail_button{"high"}="Explain Less";

#	Load in the questions database
#
	$Number_Modules=&Load_Questions;

#	Replace the title screen with appropriate verbiage and give it
#	a module name so it'll behave consistently in the module listbox.
#
	$Question{"Title_Screen"}{"short_exp"}=<<END_TITLESCREEN;
                             (Tk User Interface)
 
                                   v1.2.0
 
 
       Please answer all the questions to build a more secure system.
 
       The Next and Back buttons navigate forward and backward in the 
       questions database.  Changes made in the Answer field are *only*
       saved when you push the Next button!  The "modules" in the
       questions database are listed to the left.  You can jump to
       the start of any module simply by clicking on its name.

       Some questions have two levels of explanatory text, which you
       can adjust with the Explain Less/More button. 

       Please address bug reports and suggestions to jay\@bastille-linux.org
       Bugs in the Tk user interface are the fault of allenp\@nwlink.com.
END_TITLESCREEN
	$Question{"Title_Screen"}{'module'} = "Title Screen";

#	Adjust the no_epilogue text on the End Screen and give it a module name
#	for consistency.
#
    $Question{"End_Screen"}{"no_epilogue"}= <<END_BACK_NEXT;
Please choose a module from the list at left, and use Back/Next buttons 
to move among the questions you wish to change.


      Choose YES on this question later to implement your choices.
END_BACK_NEXT
	$Question{"End_Screen"}{'module'} = "End";

#	If there is an existing config file, snarf all the answers.
#
	&Populate_Answers;

	$next_question_index="Title_Screen";

	&Ask($next_question_index);
	$first_time = 0;

	MainLoop;

}

sub Ask {
# sub Ask (index to Question{} record)

##############################################################################
#
# More comments coming here, from Design Doc...
#
# WEIRD CASES:
#
#  1 If a record has no short_exp explanation, none is shown.  This is bad?
#  2 If a record has no question, no question is asked, but the explanation
#    is still displayed.  If this is the case, the default_answer is still
#    entered into the configuration, if it exists.
#  3 If a question has no answer, it doesn't create any blank lines or such
#    in the output file, as it will be skipped in &Output_Config_Files.  For
#    this reason, &Prompt_Input, which is only called when the record contains
#    a user-answerable question, pads a space to the end of any 0-length input.
#    Not to worry: Output_Config_Files replaces said space with 0-length input
#    NOTE: we couldn't just only print the answer field when a real question
#      existed -- this would improperly handle case 2.
#    
###############################################################################

    $in_epilogue = 0;

    my $frameopts  = ['-side' => 'top', '-padx' => 5, '-pady' => 5, '-ipadx' => 5, '-ipady' => 5, '-fill' => 'x'];
    my $lframeopts  = ['-side' => 'left', '-padx' => 5, '-pady' => 5, '-ipadx' => 5, '-ipady' => 5, '-fill' => 'y'];

#
#	Load question into local variables
#
    $index=$_[0];
#
#	Skip null records.  
#
    if (($Question{$index}{"short_exp"} eq "") &&
	($Question{$index}{"long_exp"} eq "") &&
	($Question{$index}{"question"} eq "")) {
#	    print "Skipping null record: $index\n";
#	    print Dumper($Question{$index}) . "\n";
	    $index = $Question{$index}{"yes_child"};
    }

    my $short_exp      =$Question{$index}{"short_exp"};
    my $long_exp       =$Question{$index}{"long_exp"};
    my $question       =$Question{$index}{"question"};
    my $toggle_confirm = $Question{$index}{"toggle_confirm"};
    my $toggle_yn      =$Question{$index}{"toggle_yn"};
    my $yes_epilogue   =$Question{$index}{"yes_epilogue"};
    my $no_epilogue    =$Question{$index}{"no_epilogue"};
    my $yes_child      =$Question{$index}{"yes_child"};
    my $no_child       =$Question{$index}{"no_child"};
    my $proper_parent  =$Question{$index}{"proper_parent"};
    my $answer     =$Question{$index}{"answer"};
    my $module     =$Question{$index}{"module"};
    
    if ($first_time) {
#
#	The main window
#
	$mw = MainWindow->new();
	$mw->title("Bastille Linux");

#	Frames to hold the modules listbox, question, explanation,
#	answer, and buttons.
#
	$lframe = $mw->LabFrame(
	    '-label' => "Modules",
	    '-labelside' => "acrosstop")->pack(@$lframeopts);
	$qframe = $mw->LabFrame(
	    '-label' => "Question", 
	    '-labelside' => "acrosstop")->pack(@$frameopts);
	$eframe = $mw->LabFrame(
	    '-label' => "Explanation", 
	    '-labelside' => "acrosstop")->pack(@$frameopts);
	$aframe = $mw->LabFrame(
	    '-label' => "Answer", 
	    '-labelside' => "acrosstop")->pack(@$frameopts);
	$bframe = $mw->Frame()->pack(
	    @$frameopts);

#	A scrolled listbox to hold module names
#
	$list = $lframe->Scrolled('Listbox',
	    '-scrollbars' => 'e',
	    '-width' => 20,
	    '-height' => 20)->pack();
	$list->bind("<1>", \&listbox_callback);

#	The question
#
	$qe  = $qframe->Entry(
	    '-width' => 80)->pack();

#	A scrolled Text widget for the explanation
#
	$tw = $eframe->Scrolled('Text',
	    '-wrap' => 'word',
	    '-scrollbars' => 'e',
	    '-width' => 80,
	    '-height' => 20)->pack();

#	The answer.  Note that there are three widgets defined here,
#	but their pack() methods have not been called.  This allows
#	us to show the text entry widget, the two yes/no radio
#	buttons, or neither, depending on where we are in the Questions
#	database.  (A widget can only appear when it has been packed, 
#	and it can be made to disappear by calling its pack method with
#	"forget" as an argument.)
#
	$answer_text = "";
	$ae = $aframe->Entry(
	    '-width' => 80,
	    '-textvariable' => \$answer_text);
	$ab1 = $aframe->Radiobutton(
	    '-text' => "Yes",
	    '-value' => "Y",
	    '-variable' => \$answer_text);
	$ab2 = $aframe->Radiobutton(
	    '-text' => "No",
	    '-value' => "N",
	    '-variable' => \$answer_text);

#	The Next, Back, and Detail buttons
#
	$bframe->Button(
	    '-text' => "Back",
	    '-command' => \&back_button)->pack(
	    '-expand' => 1,
	    '-side' => 'left');
	$bframe->Button(
	    '-text' => "Next",
	    '-command' => \&next_button)->pack(
	    '-expand' => 1,
	    '-side' => 'left');
	$detail_toggle = "high";
	$db = $bframe->Button(
	    '-command' => \&expn_button)->pack(
	    '-expand' => 1,
	    '-side' => 'left');

#	Build the list of modules.  The @module_index array
#	translates a module number into the first index for
#	that module in the %Question hash.  This allows us to
#	look up the first question of a module based on a mouse
#	click in the modules listbox.  The %reverse_module_index
#	hash converts a module name into a module number, so we
#	can highlight the module we're currently in.
#
	my $mod = "";
	my $ndx = 0;
	$nq = $next_question_index;
	while ($nq ne "RUN_SCRIPT") {
	    if ($mod ne $Question{$nq}{'module'}) {
	    	unless ($nq eq "End_Screen") {
	    		$mod = $Question{$nq}{'module'};
	    		($shortmod, $rest) = split(/.pm/, $mod);
	    		$list->insert($ndx, $shortmod);
	    		$module_index[$ndx]{'index'} = $nq;
	    		$reverse_module_index{$shortmod} = $ndx;
	    		$module_index[$ndx]{'done'} = 0;
	    		$ndx++;
	    	}
	    }
	    $nq = $Question{$nq}{'yes_child'};
	}
#
#	Fudge things so the end screen gets indexed properly
#
	$list->insert($ndx, "End Screen");
	$module_index[$ndx]{'index'} = "End_Screen";
	$reverse_module_index{'End'} = $ndx;
	$module_index[$ndx]{'done'} = 0;
    }

    ($modulename, $rest)  = split(/.pm/, $module);
    $moduleindex = $index;
    $db->configure('-text' => $detail_button{$detail_toggle});
    $list->selectionClear('0', 'end');
    unless ($modulename eq "") {
    	$list->selectionSet($reverse_module_index{$modulename});
    }
#
#	Explanation to display in routines -- we use short explanation if 
#	long is empty, since long explanation is optional
    
    my $explanation;
    if (($detail_toggle eq "high") and ($long_exp)) {
    	$explanation="$long_exp\n";
    } elsif ($short_exp) {
    	$explanation="$short_exp\n";
    } else {
#	Special case: when there is no explanation, this is a question
#	used either to show an epilogue or to (more likely) get some
#	default answer into the data stream

        $explanation="";
    }

#	Now, clear the screen fields and insert the new values.
#
    $tw->delete('0.0', 'end');
    $tw->insert('0.0', $explanation);

#	Trim the default answer from the end of the question, since
#	it might differ from the one we found in the config file.
#
    $question =~ s/\[.*\]$//;

    $qe->delete('0.0', 'end');
    $qe->insert('0.0', $question);
    
    $answer_text = $answer;
    if ($question eq "") {
#
#	If there is no question, don't show any answer widgets.
#
	$ae->pack('forget');
	$ab1->pack('forget');
	$ab2->pack('forget');
    } else {
#
#	Else, show either the Entry or the RadioButtons.
#
	if ($toggle_yn) {
	    $ae->pack('forget');
	    $ab1->pack('-side' => 'left');
	    $ab2->pack('-side' => 'left');
	} else {
	    $ab1->pack('forget');
	    $ab2->pack('forget');
	    $ae->pack();
	}
    }
}

# This is the callback for the Next button
#
sub next_button {
    my $answer  = $answer_text;

    unless ($in_epilogue) {
	if ($Question{$index}{'toggle_yn'}) {
	    if ($answer =~ /[Yy]/)  {
		$Question{$index}{'answer'} = "Y";
		$next_index = $Question{$index}{'yes_child'};
		if ($Question{$index}{'yes_epilogue'} and not $in_epilogue) {
		    $in_epilogue = 1;
		    &show_epilogue ("yes_epilogue");
		    return;
		}
	    } elsif ($answer =~ /[Nn]/)  {
		$Question{$index}{'answer'} = "N";
		$next_index = $Question{$index}{'no_child'};
		if ($Question{$index}{'no_epilogue'} and not $in_epilogue) {
		    $in_epilogue = 1;
		    &show_epilogue ("no_epilogue");
		    return;
		}
	    } else {
		$mw->bell();
		return;
	    }
	} else {
	    $Question{$index}{'answer'} = $answer;
	    $next_index = $Question{$index}{'yes_child'};
	    if ($Question{$index}{'yes_epilogue'} and not $in_epilogue) {
		$in_epilogue = 1;
		&show_epilogue ("yes_epilogue");
		return;
	    }
	}
    }

#    Show the credits if we've reached the end.
#
    if ($next_index eq "RUN_SCRIPT") {
	&finish_up;
	return;
    }

    &Ask($next_index);
}

sub show_epilogue {
    my $field = $_[0];

#    print "epilogue $field on record $index:\n";
#    print Dumper($Question{$index}) . "\n";

    $ae->pack('forget');
    $ab1->pack('forget');
    $ab2->pack('forget');
    $qe->delete('0.0', 'end');
    $answer_text = "";
    $tw->delete('0.0', 'end');
    $tw->insert('0.0', $Question{$index}{$field});
}

#	Display the credits and wait for the user to say OK
#
sub finish_up {
    my @tw_opts = ('-padx' => 5, '-pady' => 5, '-side' => 'top');
    my @but_opts = ('-padx' => 5, '-pady' => 5, '-side' => 'left', '-expand' => 1);
    &Output_Config_Files;
    $finishwin = $mw->Toplevel;
    $finishwin->title("Finishing Up");
    $finishwin->Label(
        '-text' => "Configuration file has been saved.\n\nWhat would you like to do now?")->pack(@tw_opts);
    $finishwin->Button(
        '-text' => "Apply Config to System",
        '-command' => \&apply_config)->pack(@but_opts);
    $finishwin->Button(
        '-text' => "Go Back and Change Config",
        '-command' => \&go_back)->pack(@but_opts);
    $finishwin->Button(
        '-text' => "Exit Without Changing System",
        '-command' => \&show_credits)->pack(@but_opts);
    $finishwin->grab;
}

sub apply_config {
    &Run_Bastille_with_Config;
    &show_credits;
}

sub go_back {
    $finishwin->destroy;
}

sub show_credits {
    my @tw_opts = ('-padx' => 5, '-pady' => 5, '-side' => 'top');
    $finishwin->destroy;
    $credwin = $mw->Toplevel;
    $credwin->title("Bastille Credits");
    
    open CREDITS, $GLOBAL_BFILE{"credits"};
    @creditsarray = <CREDITS>;
    close CREDITS;
    $creditstext = join(/ /, @creditsarray);
    $credwin->Text(
        '-relief' => 'flat',
        '-wrap' => 'word',
        '-height' => 24,
        '-width' => 80)->pack(@tw_opts)->insert('end', $creditstext);
    $credwin->Button('-text' => "Exit", '-command' => [sub {exit;}])->pack(@tw_opts);
    $credwin->grab;
}

# This is the callback for the Back button
#
sub back_button {
    if ($in_epilogue) {
        &Ask($index);
    } else {
        &Ask($Question{$index}->{'proper_parent'});
    }
}

# This is the callback for the Details toggle button.
#
sub expn_button {
    my $explanation;

    if ($detail_toggle eq "low") {
        $detail_toggle = "high";
    } else {
        $detail_toggle = "low";
    }
    
    $db->configure('-text' => $detail_button{$detail_toggle});

    if ($in_epilogue) {
        return;
    }

    if (($detail_toggle eq "high") and ($Question{$index}{'long_exp'})) {
        $explanation = $Question{$index}{'long_exp'};
    } elsif ($Question{$index}{'short_exp'}) {
        $explanation = $Question{$index}{'short_exp'};
    } else  {
        $explanation = "";
    }
    $tw->delete('0.0', 'end');
    $tw->insert('0.0', $explanation);

}

# This is the listbox callback
#
sub listbox_callback {
    my $sel = $list->curselection;

    my @tw_opts = ('-padx' => 5, '-pady' => 5, '-side' => 'top');
    my @but_opts = ('-padx' => 5, '-pady' => 5, '-side' => 'left', '-expand' => 1);

    $warnwin = $mw->DialogBox;
    $warnwin->title("Warning");
    $warnwin->Label(
        '-text' => "If you have made any modifications to the current module,\n" .
                 "Please be sure to complete the module by clicking on \"Next\"\n" .
                 "until you have finished the module before skipping to another \n" .
                 "module.  If you don't, you might leave the answers in an \n" .
                 "inconsistent state.")->pack(@tw_opts);
    $warnwin->Button(
        '-text' => "Continue anyway",
        '-command' => [sub { &Ask ($module_index[$sel]{'index'}); $warnwin->destroy; }])->pack(@but_opts);
    $warnwin->Button(
        '-text' => "Cancel",
        '-command' => [sub { $warnwin->destroy; $list->selectionClear('0', 'end'); }])->pack(@but_opts);
    my $answer = $warnwin->Show();

}

1;
