# Copyright (C) 1999, 2000 Jay Beale
# Copyright (C) 2001, Hewlett Packard Corporation
# Licensed under the GNU General Public License

package Bastille::API;

## TO DO:
#
#
#   1) Consider making the backup directory structure use the same
#      permissions on the directories that the original directories do, so
#      that undo is as easy as  tar -cpf - $backup_dir | ( cd /;  tar -xpf - )
#   2) Look for more places to insert error handling... 
#
#   3) Document this sucker bigtime...
#
#

## TO DO LATER:
#
#   - Use peterw's suggestion about storing diffs of original and Bastille'd
#     files...



##########################################################################
#
# This module forms the basis for the v1.1 API.
# 
 ##########################################################################

# 
# This module forms the initial basis for the Bastille Engine, implemented
# presently via a Perl API for Perl modules.
#
# This is still under construction -- it is very useable, but not very well
# documented, yet.
#

##########################################################################
#
#                          API Function Listing                               
#
##########################################################################
# The routines which should be called by Bastille modules are listed here,
# though they are better documented throughout this file.
#
# Distro Specific Stuff:
#
#  &GetDistro     - figures out what distro we're running, if it knows it...
#  &ConfigureForDistro - sets global variables based on the distro
#  &GetGlobal - returns hash values defined in ConfigureForDistro
#
#
# TUI Integration Stuff:
#
#  &ReadConfig    - reads the config file generated by the TUI
#  &GetGlobalConfig - returns value of hash set up by ReadConfig
#
# Logging Specific Stuff:
#
#  &ActionLog    -- logs a message to the bastille-action-log file
#  &ErrorLog     -- logs a message to the bastille-error-log file
#
# Input functions for the old input method...
#
#  &GetYN      * -- accepts a Y/N answer from console, ignoring hashed comments
#  &GetString  * -- accepts a line from console, ignoring commented lines
#
# File open/close/backup functions
#
#  &B_open     * -- opens a file handle and logs the action/error (OLD WAY!)
#  &B_open_plus  -- opens a pair of file handles for the old and new version
#                   of a file; respects logonly flag.  (NEW WAY)
#  &B_close    * -- closes a file handle and logs the action/error (OLD WAY!)
#  &B_close_plus -- closes a pair of file handles opened by B_open_plus,
#                   backing up one file and renaming the new file to the
#                   old one's name, logging actions/errors.  Respects the 
#                   logonly flag -- needs B_backup file.  Finally, sets
#                   new file's mode,uid,gid to old file's...  (NEW WAY)
#  &B_backup_file - backs up a file that is being changed/deleted into the
#                   $GLOBAL_BDIR{"backup"} directory.
#
# Content (Line-based) file modification functions
#
#  &B_blank_file  - blanks a file if it isn't ours, so that we can rewrite it
#  &B_append_line - opens a file, appends a line to it, and closes file
#  &B_insert_line - opens a file, inserts a line after a specified line, closes
#  &B_prepend_line- opens a file, prepends a line to it, and closes file
#  &B_replace_line- replaces one or more lines in a file using regep matching
#  &B_hash_comment_line
#                 - hash - comments one or more lines in a file
#  &B_hash_uncomment_line
#                 - hash - uncomments one or more lines in a file
#  &B_delete_line - deletes one or more lines in a file using regep matching
#
# Content (multiple-line-based) file mod functions (for PeterW's ipchains.pm)
#
#  &B_chunk_append
#
#  &B_chunk_replace
#
#
# Non-content file modification functions
#
#  &B_delete_file - deletes the named file, backing up a copy
#  &B_create_file - creates the named file, if it doesn't exist
#
#  &B_print       	- print a line to the named filehandle, logging the action
#  &B_chmod       	- chmod a file, logging the action
#  &B_chmod_if_exists  	- chmod a file, logging the action only if the file exists
#  &B_chown       	- change only the owner on a file, logging the action
#  &B_chgrp       	- change the group owner on a file, logging the action
#  &B_remove_suid       - remove suid bit from a given file.
#  &B_symlink     	- create a symlink to a file, creating the undo link
#
# More stuff
#
#  &B_chkconfig_off - removed all S links from rcX.d directory
#  &B_chkconfig_on  - make all S links from rcX.d directory
#  &B_createdir     - make a directory, if it doesn't exist, w/ undo link
#  &B_cp            - copy a file, respecting LOGONLY and PREFIX and undo func.
#  &B_mknod         - wrap mknod with undo and logonly and prefix functionality
#
#  &B_undo_log_entry - create entry in shell script, executed later by UndoBastille
#  &showDisclaimer  - Print the disclaimer and wait for 2 minutes for acceptance
###########################################################################
# Note:  GLOBAL_VERBOSE
# 
# All logging functions now check GLOBAL_VERBOSE and, if set, will print
# all the information sent to log files to STDOUT/STDERR as well.
#

#
# Note:  GLOBAL_LOGONLY
# 
# All Bastille API functions now check for the existence of a GLOBAL_LOGONLY 
# variable.  When said variable is set, no function actually modifies the 
# system.
#

# Note: GLOBAL_PREFIX (deprecated!)
#
#  GLOBAL_PREFIX IS GOING AWAY! 
#	While supported now, this variable complicates things considerably
#	and, as far as we can tell, is not widely used (in some cases it's been
#	misconstrued), so we plan to stop using it.
#
# All Bastille API functions now prepend, directly or indirectly, the variable
# $GLOBAL_PREFIX to the beginning of all filenames.  By "indirectly," I mean
# that some of the API routines use B_open_plus and B_backup_file, which
# prepend the $GLOBAL_PREFIX for them.
#
# For a module to work safely with the $GLOBAL_PREFIX, without said prefix
# being prepended twice, it must not append the prefix to any filename that
# it passes to the API.


# Libraries for the Backup_file routine: Cwd and File::Path
use Cwd;
use Bastille::HP_API;    
use File::Path;

# Export the API functions listed below for use by the modules.

use Exporter;
@ISA = qw ( Exporter );
@EXPORT = qw( PrepareToRun GetDistro ConfigureForDistro StartLogging ActionLog ErrorLog 
	      GetYN GetString SanitizeEnv
              B_open B_close B_print B_symlink StopLogging 
	      B_Display B_open_plus B_close_plus B_blank_file B_append_line
	      B_insert_line
	      B_prepend_line B_replace_line 
	      B_hash_comment_line B_hash_uncomment_line 
	      B_delete_line
	      B_create_file 
	      B_create_dir B_remove_suid
	      B_delete_file B_chmod B_chmod_if_exists B_chown B_chgrp 
	      B_chkconfig_off B_chkconfig_on B_cp B_place B_mknod
	      B_ch_rc B_System B_TODO B_install_jail B_chperm showDisclaimer
	      $GLOBAL_PREFIX $GLOBAL_LOGONLY $GLOBAL_VERBOSE
	      %GLOBAL_BIN %GLOBAL_DIR %GLOBAL_FILE
	      $GLOBAL_DISTRO
	      %GLOBAL_BDIR %GLOBAL_BFILE
	      %GLOBAL_CONFIG

              getGlobal getGlobalConfig getGlobalSwlist
	       );

&SanitizeEnv;

unless ($GLOBAL_DISTRO) { 
#   &GetDistro;  # Not needed since ConfigureForDistro already calls this 
    &ConfigureForDistro;
}

##############################################################################
#
#                         Bastille v1.1 directory structure
#
##############################################################################
#
#  /root/Bastille  -- parent directory for Bastille Linux
#
#  /root/Bastille/undo -- directory holding all undo scripts and backups...
#  /root/Bastille/undo/backup -- all the original files that Bastille modifies,
#                                with permissions intact
#
#  /root/Bastille/logs -- directory holding all logs
#
# Each of these directories can be prepended by $GLOBAL_PREFIX, allowing
# you to make no changes to the / filesystem while running Bastille on a 
# target filesystem...
#
##############################################################################

##############################################################################
#
#                         Bastille v1.2 directory structure
#
##############################################################################
#
#  /usr/sbin/          -- location of Bastille binaries
#  /usr/lib/Bastille   -- location of Bastille modules
#  /usr/share/Bastille -- location of Bastille data files
#  /etc/Bastille       -- location of Bastille config files
#
#  /var/log/Bastille      -- location of Bastille log files
#  /var/log/Bastille/undo -- directory holding all Bastille-created undo scripts
#  /var/log/Bastille/undo/backup -- directory holding the original files that
#                                   Bastille modifies, with permissions intact
#
# Each of these directories can be prepended by $GLOBAL_PREFIX, allowing
# you to make no changes to the / filesystem while running Bastille on a 
# target filesystem...
#
##############################################################################






##############################################################################
##############################################################################
##################  Actual functions start here... ###########################
##############################################################################
##############################################################################

###########################################################################
# 
# SanitizeEnv load a proper environment so Bastille cannot be tricked
# and Perl modules work correctly.
# 
###########################################################################
sub SanitizeEnv {
	 delete @ENV{'IFS','CDPATH','ENV','BASH_ENV'};
	 $ENV{CDPATH}=".";
	 $ENV{BASH_ENV}= "";
	 # Bin is needed here or else  /usr/lib/perl5/5.005/Cwd.pm 
	 # will not find `pwd`
	 # Detected while testing with -w, jfs
	 $ENV{PATH} = "/bin:/usr/bin";
	 # Giorgi, is /usr/local/bin needed? (jfs)
}

###########################################################################
# 
# PrepareToRun sets up Bastille to run.  It checks the ARGV array for
# special options and runs ConfigureForDistro to set necessary file
# locations and other global variables.
#
###########################################################################

sub PrepareToRun {

#     Make sure we're root!
    if ( $> != 0 ) {
	die "Bastille Back End must run as root!";
    }


    # Process command-line arguments...

    my $arg;

    $GLOBAL_PREFIX="";
    $GLOBAL_LOGONLY=0;
    $GLOBAL_VERBOSE=0;

	my @temp_argv = @ARGV;

#  IMHO it's better to have -l for logonly mode
#  and -v for verbose mode (added here) since admin's tend
#  to associate 'v' with verbose or version in command lines.. (jfs)
    while ( $arg = shift @temp_argv ) {
	if ( $arg =~ /-l/ ) {
	    $GLOBAL_LOGONLY=1;
	    &ActionLog("#\n# Setting LOGONLY (-l) mode On.\n#\n");
	}
	elsif ( $arg =~ /-r/ ) {
	    $GLOBAL_PREFIX= shift @temp_argv;
	    &ActionLog("#\n# Setting PREFIX to $GLOBAL_PREFIX\n#\n");
	}
	elsif ( $arg =~ /-v/ ) {
		$GLOBAL_VERBOSE=1;
		&ActionLog("Printing actions to STDOUT\n");
	}

    }

    # Make any directories that don't exist...
    foreach my $dir ( &getGlobal('BDIR', "undo") , &getGlobal('BDIR', "backup"), &getGlobal('BDIR', "log"),&getGlobal('BDIR', "config") ) {
	mkpath ($dir,0,0700);
    }

    # Make the TODO list file for HP-UX Distro
    &B_create_file(&getGlobal('BFILE', "TODO"));
    &B_blank_file(&getGlobal('BFILE', "TODO"),'a$b This pattern should not match anything in the file because $ means the end of the line 3.1415926');
    &B_append_line(&getGlobal('BFILE', "TODO"),"","Please take the following steps to make your system more secure:\n\n");

  # Configure for this distro...
  # &ConfigureForDistro;

  # Load in the user's configuration file...
  &ReadConfig;   
}

###########################################################################
#
# GetDistro checks to see if the target is a known distribution and reports
# said distribution.
#
# This is used through the script, but also by ConfigureForDistro.
# 
#
###########################################################################

sub GetDistro() {

    my ($release,$distro);

    # Only read files for the distro once.

    unless ($GLOBAL_DISTRO) {

	if ( -e "$GLOBAL_PREFIX/etc/mandrake-release" ) {
	    open(MANDRAKE_RELEASE,"$GLOBAL_PREFIX/etc/mandrake-release");
	    $release=<MANDRAKE_RELEASE>;
	    unless ( ($release =~ /^Mandrake Linux release (\d+\.\d+\w*)/) or ($release =~ /^Linux Mandrake release (\d+\.\d+\w*)/) ) {
			print "Couldn't determine Mandrake version! Setting to 6.0!\n";
			$distro="MN6.0";
	    }
	    else {
		$distro="MN$1";
	    }
	    close(MANDRAKE_RELEASE);
	}
	elsif ( -e "$GLOBAL_PREFIX/etc/immunix-release" ) {
	    open(IMMUNIX_RELEASE,"$GLOBAL_PREFIX/etc/immunix-release");
	    $release=<IMMUNIX_RELEASE>;
	    unless ($release =~ /^Immunix Linux release (\d+\.\d+\w*)/) {
			print "Couldn't determine Immunix version! Setting to 6.2!\n";
			$distro="RH6.2";
	    }
	    else {
		$distro="RH$1";
	    }
	    close(*IMMUNIX_RELEASE);
	}
	elsif ( -e "$GLOBAL_PREFIX/etc/redhat-release" ) {
	    open(*REDHAT_RELEASE,"$GLOBAL_PREFIX/etc/redhat-release");
	    $release=<REDHAT_RELEASE>;
	    unless ($release =~ /^Red Hat Linux release (\d+\.\d+\w*)/) {
			print "Couldn't determine Red Hat version! Setting to 6.2!\n";
			$distro="RH6.2";
	    }
	    else {
			$distro="RH$1";
	    }      
	    close(REDHAT_RELEASE);   
	}
	elsif ( -e "$GLOBAL_PREFIX/etc/debian_version" ) {
		$stable="2.2"; #Change this when Debian stable changes
		open(*DEBIAN_RELEASE,"$GLOBAL_PREFIX/etc/debian_version");
		$release=<DEBIAN_RELEASE>;
		unless ($release =~ /^(\d+\.\d+\w*)/) {
			print STDERR "System is not running a stable Debian GNU/Linux version. Setting to $stable.\n";
			$distro="DB$stable";
		}
		else {
			$distro="DB$1";
		}      
		close(DEBIAN_RELEASE);
       }
 	elsif ( -e "$GLOBAL_PREFIX/etc/SuSE-release" ) {
            open(*SUSE_RELEASE,"$GLOBAL_PREFIX/etc/SuSE-release");
            $release=<SUSE_RELEASE>;
            unless ($release =~ /^SuSE Linux (\d+\.\d+\w*)/) {
                        print "Couldn't deterine SuSE version! Setting to 7.2!\n";
                        $distro="SE7.2";
            } 
            else {      
                        $distro="SE$1";
            }
            close(SUSE_RELEASE);
        }   
 	elsif ( -e "$GLOBAL_PREFIX/etc/turbolinux-release") {
            open(*TURBOLINUX_RELEASE,"$GLOBAL_PREFIX/etc/turbolinux-release");
            $release=<TURBOLINUX_RELEASE>;
            unless ($release =~ /^Turbolinux Workstation (\d+\.\d+\w*)/) {
                        print "Couldn't determine TurboLinux version! Setting to 7.0!\n";
                        $distro="TB7.0";
            }
            else {
                        $distro="TB$1";
            }
            close(TURBOLINUX_RELEASE);
        }

        elsif ($release=`/bin/uname -sr` ) {
	    unless ( $release =~ /(^HP-UX)\s*B\.(\d+\.\d+)/ ) {
			print "Could not determine HP-UX version!\n";
			$distro="unknown";
	    }
	    else {
			$distro="$1$2";
	    }      
	}
	else {
	    print "Couldn't determine distro!\n";
	    print "We can't run on this distribution yet -- it is not yet
supported by Bastille Linux\n\n";
	    die "This distribution is as yet unsupported\n";
	}

	# Die if this distribution is unsupported
	my $supported=0;
	foreach $supported_distro ( "DB2.2", "RH6.0","RH6.1","RH6.2","RH7.0","RH7.1","RH7.2","MN6.0","MN6.1","MN7.0","MN7.1","MN7.2","MN8.0","MN8.1","HP-UX11.00","HP-UX11.11","SE7.2","TB7.0" ) {
		if ( $supported_distro eq $distro ) {
			$supported=1;
		}
	}
	die "This distribution version: $distro is not yet supported!\n" unless $supported;

    }
    else {
	$distro = $GLOBAL_DISTRO;
    }

    $GLOBAL_DISTRO=$distro;
    $distro;
}


###########################################################################
#
# &ConfigureForDistro configures the API for a given distribution.  This
# includes setting global variables that tell the Bastille API about
# given binaries and directories.
#
# WARNING: If a distro is not covered here, Bastille may not be 100% 
#          compatible with it, though 1.1 is written to be much smarter
#          about unknown distros...
#
###########################################################################

sub ConfigureForDistro {

    my $distro = &GetDistro;
    my $return=1;

    # Directories, as explained in Bastille directory structure...

    if ( ($distro =~ "^RH" ) or ($distro =~ "^MN") or ($distro =~ "^DB") or ($distro =~ "^SE") or ($distro =~ "^TB")){

	$GLOBAL_BDIR{"home"}= $GLOBAL_PREFIX . "/root/Bastille";

	$GLOBAL_BDIR{"log"} = $GLOBAL_PREFIX . "/var/log/Bastille";
	$GLOBAL_BFILE{"action-log"} = &getGlobal('BDIR', "log") . "/action-log";
	$GLOBAL_BFILE{"error-log"} = &getGlobal('BDIR', "log") . "/error-log";

	$GLOBAL_BDIR{"undo"}= $GLOBAL_PREFIX . "/var/log/Bastille" . "/undo";
	$GLOBAL_BFILE{"created-files"}= &getGlobal('BDIR', "undo") . "/undo-created-files";
	$GLOBAL_BFILE{"created-dirs"}= &getGlobal('BDIR', "undo") . "/undo-created-dirs";
	$GLOBAL_BFILE{"created-symlinks"} = &getGlobal('BDIR', "undo") . "/undo-created-symlinks";
	$GLOBAL_BFILE{"removed-symlinks"} = &getGlobal('BDIR', "undo") . "/undo-removed-symlinks";
	$GLOBAL_BFILE{"executed-commands"}= &getGlobal('BDIR', "undo") . "/undo-executed-commands";
	# Required files for new undo process...
	$GLOBAL_BFILE{"undo-actions"}= &getGlobal('BDIR', "undo") . "/undo-actions";
        $GLOBAL_BFILE{"undo-directory-perms.sh"}=&getGlobal('BDIR', "undo") . "/undo-directory-perms.sh";


	$GLOBAL_BFILE{"credits"}="/usr/share/Bastille/Credits";
	$GLOBAL_BDIR{"backup"}= &getGlobal('BDIR', "undo") . "/backup";
    
	$GLOBAL_BDIR{"config"} = $GLOBAL_PREFIX . "/etc/Bastille";
	$GLOBAL_BFILE{"config"} = &getGlobal('BDIR', "config") . "/config";

	$GLOBAL_BFILE{"TODO"}= $GLOBAL_BDIR{"log"} . "/TODO";

	#
	# Set necessary binaries
	#

	if (($distro =~ "^RH" ) or ($distro =~ "^MN") or ($distro =~ "^SE") or ($distro =~ "^TB") ) {
	    $GLOBAL_BIN{"accton"} = "/usr/sbin/accton";
	    $GLOBAL_FILE{"accton"} = "/usr/sbin/accton";
	}
	if ($distro =~ "^RH6.2" ) {
	    $GLOBAL_BIN{"accton"} = "/sbin/accton";
	    $GLOBAL_FILE{"accton"} = "/sbin/accton";
	}

	$GLOBAL_BIN{"bash"} = "/bin/bash";
        $GLOBAL_BIN{"chattr"}="/usr/bin/chattr";
	$GLOBAL_BIN{"chgrp"}="/bin/chgrp";
	$GLOBAL_BIN{"chmod"}="/bin/chmod";
        $GLOBAL_BIN{"chown"}="/bin/chown";
        $GLOBAL_BIN{"cp"}="/bin/cp";
        $GLOBAL_BIN{"crontab"}="/usr/bin/crontab";
        $GLOBAL_BIN{"diff"}="/usr/bin/diff";
        $GLOBAL_BIN{"groupadd"}="/usr/sbin/groupadd";
        $GLOBAL_BIN{"killall"}="/usr/bin/killall";
	$GLOBAL_BIN{"lilo"}="/sbin/lilo";
        $GLOBAL_BIN{"lpq"}="/usr/bin/lpq";
        $GLOBAL_BIN{"lprm"}="/usr/bin/lprm";
        $GLOBAL_BIN{"lpr"}="/usr/bin/lpr";
        $GLOBAL_BIN{"mknod"}="/bin/mknod";
        $GLOBAL_BIN{"mount"}="/bin/mount";
	$GLOBAL_BIN{"smbmnt"}='/usr/bin/smbmnt';
        $GLOBAL_BIN{"mv"}="/bin/mv";
	$GLOBAL_BIN{"rpm"}="/bin/rpm";
        $GLOBAL_BIN{"touch"}="/bin/touch";
         if ( $distro =~ /^DB/ ) {
		$GLOBAL_BIN{"dpkg"}="/usr/bin/dpkg";
		 $GLOBAL_BIN{"apt-get"}="/usr/sbin/apt-get";
	}
        $GLOBAL_BIN{"sulogin"}="/sbin/sulogin";
        $GLOBAL_BIN{"umount"}="/bin/umount";
        $GLOBAL_BIN{"useradd"}="/usr/sbin/useradd";
	# Added the next few for 1.2.0.pre12
	$GLOBAL_BIN{"ping"}="/bin/ping";
	$GLOBAL_BIN{"dump"}="/sbin/dump";
	$GLOBAL_BIN{"restore"}="/sbin/restore";
	$GLOBAL_BIN{"cardctl"}="/sbin/cardctl";
	$GLOBAL_BIN{"at"}="/usr/bin/at";
	$GLOBAL_BIN{"dos"}="/usr/bin/dos";
	$GLOBAL_BIN{"inndstart"}="/usr/bin/inndstart";
	$GLOBAL_BIN{"startinnfeed"}="/usr/bin/startinnfeed";
	$GLOBAL_BIN{"rsh"}="/usr/bin/rsh";
	$GLOBAL_BIN{"rcp"}="/usr/bin/rcp";
	$GLOBAL_BIN{"rlogin"}="/usr/bin/rlogin";
	$GLOBAL_BIN{"usernetctl"}="/usr/sbin/usernetctl";
	$GLOBAL_BIN{"traceroute"}="/usr/sbin/traceroute";
	$GLOBAL_BIN{"rlogind"}="/usr/sbin/in.rlogind";
	$GLOBAL_BIN{"rshd"}="/usr/sbin/in.rshd";
	$GLOBAL_BIN{"rexecd"}="/usr/sbin/in.rexecd";
	$GLOBAL_BIN{"rm"}="/bin/rm";
	$GLOBAL_BIN{"rmdir"}="/bin/rmdir";
	$GLOBAL_BIN{"ln"}="/bin/ln";
	$GLOBAL_BIN{"md5sum"}="/usr/bin/md5sum";

	#
	# Set necessary directories
	#

	$GLOBAL_DIR{"home"}="/home";
	$GLOBAL_DIR{"initd"}="/etc/rc.d/init.d";
        if ( ($distro =~ /^DB/) or ($distro =~ /^SE7.2/)) {
	         $GLOBAL_DIR{"initd"} = "/etc/init.d";
	}
        $GLOBAL_DIR{"log"}="/var/log";
        $GLOBAL_DIR{"pamd"}="/etc/pam.d";
        $GLOBAL_DIR{"rcd"}="/etc/rc.d";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_DIR{"rcd"} = "/etc";
	}
	$GLOBAL_DIR{"sbin"}="/sbin";
	$GLOBAL_BDIR{"share"}= $GLOBAL_PREFIX . "/usr/share/Bastille";


	#
	# Set necessary config/misc files
	#

        $GLOBAL_DIR{"floppy"}="/mnt/floppy";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_DIR{"floppy"}="/floppy";
	}
        $GLOBAL_FILE{"group"}="/etc/group";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_FILE{"httpd.conf"}="/etc/apache/httpd.conf";
	}
	elsif ($distro =~ /^SE7.2/ ) {
		$GLOBAL_FILE{"httpd.conf"}="/etc/httpd/httpd.conf";
	}
	else {
		$GLOBAL_FILE{"httpd.conf"}="/etc/httpd/conf/httpd.conf";
	}
	$GLOBAL_FILE{"Questions"} = &getGlobal('BDIR', "share") . "/Questions.txt";
	$GLOBAL_BFILE{"nodisclaimer"}= &getGlobal('BDIR', "share") . "/.nodisclaim";
	

	if ( $distro =~ /^DB2/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/apache/access.conf";
	} elsif ( $distro =~ /^DB3/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/apache/httpd.conf";
	} elsif ( $distro =~ /^RH6.[01]/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/httpd/conf/access.conf";
	} elsif ($distro =~ /^SE7.2/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/httpd/httpd.conf";
	} else {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/httpd/conf/httpd.conf";
	}

        $GLOBAL_FILE{"inittab"}="/etc/inittab";
        $GLOBAL_FILE{"lilo.conf"}="/etc/lilo.conf";
	$GLOBAL_FILE{"grub.conf"}="/etc/grub.conf";
        $GLOBAL_FILE{"limits.conf"}="/etc/security/limits.conf";
        $GLOBAL_FILE{"mtab"}="/etc/mtab";
        $GLOBAL_FILE{"pam_access.conf"}="/etc/security/access.conf";
	$GLOBAL_FILE{"hosts.allow"}="/etc/hosts.allow";
	$GLOBAL_FILE{"inetd.conf"}="/etc/inetd.conf";
	$GLOBAL_FILE{"xinetd.conf"}="/etc/xinetd.conf";
	$GLOBAL_DIR{"xinetd.d"}="/etc/xinetd.d";

	# Added for 1.2.0.pre12 to implement more requires_ tags...
	$GLOBAL_FILE{"rsh"}="/usr/bin/rsh";
	$GLOBAL_FILE{"gcc"}="/usr/bin/gcc";
	$GLOBAL_FILE{"chkconfig_apmd"}=&getGlobal('DIR', "rcd")."/rc3.d/S26apmd";
	$GLOBAL_FILE{"chkconfig_nfs"}=&getGlobal('DIR', "rcd")."/rc3.d/S60nfs";
	$GLOBAL_FILE{"chkconfig_pcmcia"}=&getGlobal('DIR', "rcd")."/rc3.d/S45pcmcia";
	$GLOBAL_FILE{"chkconfig_dhcpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S65dhcpd";
	$GLOBAL_FILE{"chkconfig_gpm"}=&getGlobal('DIR', "rcd")."/rc3.d/S85gpm";
	$GLOBAL_FILE{"chkconfig_innd"}=&getGlobal('DIR', "rcd")."/rc3.d/S95innd";
	$GLOBAL_FILE{"chkconfig_gated"}=&getGlobal('DIR', "rcd")."/rc3.d/S32gated";
	$GLOBAL_FILE{"chkconfig_ypbind"}=&getGlobal('DIR', "rcd")."/rc3.d/S17ypbind";
	$GLOBAL_FILE{"chkconfig_snmpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S50snmpd";
	$GLOBAL_FILE{"rc.config"}="/etc/rc.config"; 
	if ($distro =~ /^SE7.2/ ) {
	$GLOBAL_FILE{"chkconfig_apmd"}=&getGlobal('FILE', "rc.config");
        $GLOBAL_FILE{"chkconfig_nfs"}=&getGlobal('FILE', "rc.config");
        $GLOBAL_FILE{"chkconfig_pcmcia"}=&getGlobal('FILE', "rc.config");
        $GLOBAL_FILE{"chkconfig_dhcpd"}=&getGlobal('FILE', "rc.config");
        $GLOBAL_FILE{"chkconfig_gpm"}=&getGlobal('FILE', "rc.config");
        $GLOBAL_FILE{"chkconfig_ypbind"}=&getGlobal('FILE', "rc.config");
        }
	$GLOBAL_FILE{"sendmail.cf"}="/etc/sendmail.cf"; 
	$GLOBAL_FILE{"sysconfig_sendmail"}="/etc/sysconfig/sendmail";
	$GLOBAL_FILE{"named"}="/usr/sbin/named";
        $GLOBAL_BIN{'named-xfer'}='/usr/sbin/named-xfer';
	$GLOBAL_FILE{"chkconfig_named"}=&getGlobal('DIR', "rcd")."/rc3.d/S55named";
	$GLOBAL_FILE{"chkconfig_httpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S85httpd";
	$GLOBAL_FILE{"httpd"}="/usr/sbin/httpd";
	$GLOBAL_FILE{"lpr"}="/usr/bin/lpr";
	$GLOBAL_FILE{"ftpaccess"}="/etc/ftpaccess";
	$GLOBAL_FILE{"tcpd"}="/usr/sbin/tcpd";
	if ($distro eq 'RH7.2') {
	    $GLOBAL_FILE{"banners_makefile"}="/usr/share/doc/tcp_wrappers-7.6/Banners.Makefile";
	}
	elsif ($distro eq 'TB7.0') {
	    $GLOBAL_FILE{"banners_makefile"}="/usr/share/doc/packages/tcp_wrappers-7.6/Banners.Makefile";}
	else {
	    $GLOBAL_FILE{"banners_makefile"}="/usr/share/doc/tcp_wrappers-7.5/Banners.Makefile";
	}
	$GLOBAL_FILE{"profile"}="/etc/profile";
	$GLOBAL_FILE{"csh.login"}="/etc/csh.login";
	$GLOBAL_FILE{"zprofile"}="/etc/zprofile";
	$GLOBAL_FILE{"motd"}="/etc/motd";
	$GLOBAL_FILE{"issue"}="/etc/issue";

    }
    elsif ($distro =~ "^HP-UX" ) {
	&HP_ConfigureForDistro;
    }
    else {
	$return=0;
    }

    $return;
   
}


###########################################################################
# &ReadConfig reads in the user's choices from the TUI, stored in the file
# $GLOBAL_BFILE{"config"}.  We were using AppConfig here at first, but it was
# just such a pain in the, ermm, keyboard...
#
###########################################################################

sub ReadConfig {

    open CONFIG, &getGlobal('BFILE', "config") or die "Couldn't open config file!\n";
    while (my $line = <CONFIG>) {

	# Skip commented lines...
	unless ($line =~ /^\s*#/) {
	    if ($line =~ /^\s*(\w+).(\w+)\s*=\s*\"(.*)\"/ ) {
		$GLOBAL_CONFIG{$1}{$2}=$3;
	    }
	}
    }
    close CONFIG;
}


##############################################################################
# &ActionLog ($Log_items) prints $log_items to the action-log.  If said log
# doesn't exist, it is created.
##############################################################################

sub ActionLog {
	my @Actions = @_; # List of parameters passed to ActionLog
	# Just in case
	if ( defined $GLOBAL_BFILE{"action-log"} ) {
		unless ( -e &getGlobal('BFILE', "action-log") ) {
			open ACTIONLOG,">" . &getGlobal('BFILE', "action-log");
			close ACTIONLOG;
		}

		open ACTIONLOG,">>" . &getGlobal('BFILE', "action-log") or print "Couldn't open action-log file in ".&getGlobal('BFILE', "action-log").": $!\n";
		foreach my $thing ( @Actions ) {
			print ACTIONLOG $thing;
		}

		close ACTIONLOG;
	}
	if ( $GLOBAL_VERBOSE ) {
		foreach my $thing ( @Actions ) {
			print STDERR "$thing";
		}
	}

}

##############################################################################
# &ErrorLog ($log_items) prints $log_items to the error-log.  If said log
# doesn't exist, it is created.
#
# The idea for this log was Mike Rash's (mbr).
##############################################################################

sub ErrorLog {
    my @Errors = @_; # List of errors passed to ErrorLog
    # Write something to our ongoing "errorlog" file

    unless ( -e &getGlobal('BFILE', "error-log") ) {
	open ERRORLOG,">" .  &getGlobal('BFILE', "error-log");
	close ERRORLOG;
    }

    open ERRORLOG,">>" . &getGlobal('BFILE', "error-log") or print "Couldn't open error-log: $!\n";
    
    foreach $thing ( @Errors ) {
	print ERRORLOG $thing;
	print STDERR $thing;  
	# IMHO if the user is not warned of stuff  he might not be aware of
	# the logs (jfs)
    }
    
    close ERRORLOG;	


}


###########################################################################
# GetYN and GetString  are the two original v1.0.0-v1.0.3 input functions #
# used to read in (and log) input.                                        #
# Both functions ignore hash-commented lines.                             #
#                                                                         #
# Under the new architecture, these functions should fall out of use      #
# entirely , as we move to a TUI that writes AppConfig files.             #
#                                                                         #
###########################################################################

# &GetYN grabs a Y or an N from stdin.  It 
sub GetYN {
    # enhanced by Don Wilder and Peter W.
    my ( $prefix ) = @_;        # optional prefix
    my ($line, $ok, $warn);
    
    print "Press <Shift><Page Up>/<Page Down> to see previously scrolled text.\n";
    $ok = 0;
    while ( $ok == 0 ) {
       print "$warn$prefix(Y or N): ";
       $line=<STDIN>;
       print INPUTLOG $line;   # log answer
       if ( ($line =~ /^#/) or ($line !~ /^[YN]/i ) ) {
          # If we're in this loop, we've either read a comment, or the
	  # input did not begin with Y or N.  Ask again.
	  $warn = "Please respond with Y or N. ";
       } 
       else {
          $ok = 1;
       }
    }
    return(uc(substr($line,0,1)));

}

sub GetString {

   my $line;
		      
   #  Take the first line that isn't a comment, that is, the first line
   #  not beginning with:      
   #                   whitespace #
   while ( ($line=<STDIN>) =~ /^(\s*)#/) { 
      print INPUTLOG $line;
   } 

   print INPUTLOG $line;
   # Now, strip out the comment at the end...

   #
   # Note the strange structure (via the while statement).  This is to
   # prevent weird comments, like:
 
   #                
   #           #  this is a comment # with pound's in the # middle...
   # or 
   #           ##  this is a better example: 
   #      
   #           #  dial  #,#,1,2,3
   # 
    
   while ( $line =~ /^(.*[^\\])#/ ) {
      $pre_pound=$1;
      $pre_pound =~ s/(.*[^\s])\s+/$1/;
      $line = $pre_pound;
   }
   
   # Finally, change all escaped #'s to #'s:   
   $line =~ s/\\#/#/g;

   # Return the found line
   $line;
			      
}


###########################################################################
###########################################################################
#                                                                         #
# The B_<perl_function> file utilities are replacements for their Perl    #
# counterparts.  These replacements log their actions and their errors,   #
# but are very similar to said counterparts.                              #
#                                                                         #
###########################################################################
###########################################################################


###########################################################################
# B_open was the v1.0 open command.  It is still used in places in the
# code, though it should fall out of use quickly in v1.1, as it has been
# replaced by B_open_plus, which implements a new, smarter, backup scheme.
#
# B_open opens the given filehandle, associated with the given filename
# and logs appropriately.
#
# WARNING: B_open does not respect the $GLOBAL_PREFIX -- use this with care!
#
###########################################################################

sub B_open {
   my $return=1;
   my ($handle,$filename)=@_;

   unless ($GLOBAL_LOGONLY) {
       $return = open $handle,$filename;
   }

   ($handle) = "$_[0]" =~ /[^:]+::[^:]+::([^:]+)/;
   &ActionLog("open $handle,\"$filename\";\n");
   unless ($return) {
      &ActionLog("#open $handle , $filename failed...\n");
      &ErrorLog("#open $handle, $filename failed...\n");
   }
   
   $return;
}   

###########################################################################
# B_open_plus is the v1.1 open command.
# 
# &B_open_plus($handle_file,$handle_original,$file) opens the file $file
# for reading and opens the file ${file}.bastille for writing.  It is the
# counterpart to B_close_plus, which will move the original file to
# $GLOBAL_BDIR{"backup"} and will place the new file ${file}.bastille in its
# place.
#
# &B_open_plus makes the appropriate log entries in the action and error
# logs.
###########################################################################

sub B_open_plus {

    my ($handle_file,$handle_original,$file)=@_;
    my $return=1;
    my $return_file=1;
    my $return_old=1;
  
    my $original_file = $file;
    $file = $GLOBAL_PREFIX . $file;

    # Open the original file and open a copy for writing.
    unless ($GLOBAL_LOGONLY) {
	$return_old = open $handle_original,"$file";
	$return_file = open $handle_file,("> $file.bastille");
    }
    
    # Error handling/logging here...
    #&ActionLog("# Modifying file $original_file via temporary file $original_file.bastille\n");
    unless ($return_file) {
	$return=0;
	&ErrorLog("#open $original_file.bastille failed...\n");
    }
    unless ($return_old) {
	$return=0;
	&ErrorLog("#open $original_file failed.\n");
    }

    $return;
        
}

###########################################################################
# B_close was the v1.0 close command.  It is still used in places in the
# code, though it should fall out of use quickly in v1.1, as it has been
# replaced by B_close_plus, which implements a new, smarter, backup scheme.
#
# B_close closes the given filehandle, associated with the given filename
# and logs appropriately.
###########################################################################


sub B_close {
   my $return=1;

   unless ($GLOBAL_LOGONLY) {
       $return = close $_[0];
   }

   &ActionLog( "close $_[0];\n");
   unless ($return) {
      &ActionLog("#ERROR: close $_[0] failed...\n");
      &ErrorLog( "#ERROR: close $_[0] failed...\n");
   }

   $return;
}


###########################################################################
# B_close_plus is the v1.1 close command.
# 
# &B_close_plus($handle_file,$handle_original,$file) closes the files
# $file and ${file}.bastille, backs up $file to $GLOBAL_BDIR{"backup"} and
# renames ${file}.bastille to $file.  This backup is made using the
# internal API function &B_backup_file.  Further, it sets the new file's
# permissions and uid/gid to the same as the old file.
#
# B_close_plus is the counterpart to B_open_plus, which opened $file and 
# $file.bastille with the file handles $handle_original and $handle_file, 
# respectively.
#
# &B_close_plus makes the appropriate log entries in the action and error
# logs.
###########################################################################

sub B_close_plus {
    my ($handle_file,$handle_original,$file)=@_;
    my ($mode,$uid,$gid);
    my @junk;

    my $original_file;

    my $return=1;
    my $return_file=1;
    my $return_old=1;

    # Append the global prefix, but save the original for B_backup_file b/c
    # it appends the prefix on its own...

    $original_file=$file;
    $file = $GLOBAL_PREFIX . $file;

    #
    # Close the files and prepare for the rename
    #

    unless ($GLOBAL_LOGONLY) {
	$return_file = close $handle_file;
	$return_old = close $handle_original;
    }

    # Error handling/logging here...
    #&ActionLog("#Closing $original_file and backing up to " . &getGlobal('BDIR', "backup"));
    #&ActionLog("/$original_file\n");

    unless ($return_file) {
	$return=0;
	&ErrorLog("#close $original_file failed...\n");
    }
    unless ($return_old) {
	$return=0;
	&ErrorLog("#close $original_file.bastille failed.\n");
    }

    #
    # If we've had no errors, backup the old file and put the new one
    # in its place, with the Right permissions.
    #

    unless ( ($return == 0) or $GLOBAL_LOGONLY) {

	# Read the permissions/owners on the old file
	
	@junk=stat ($file);
	$mode=$junk[2];
	$uid=$junk[4];
	$gid=$junk[5];

	# Set the permissions/owners on the new file

	chmod $mode, "$file.bastille" or &ErrorLog("B_close_plus: Not able to retain permissions on $original_file!!!\n");
	chown $uid, $gid, "$file.bastille" or &ErrorLog("B_close_plus: Not able to retain owners on $original_file!!!\n");

	# Backup the old file and put a new one in place.
	
	&B_backup_file($original_file);
	rename "$file.bastille", $file or &ErrorLog("B_close_plus: not able to move $original_file.bastille to $original_file!!! BAD!!!\n");
    }

    $return;
}

###########################################################################
# &B_backup_file ($file) makes a backup copy of the file $file in 
# &getGlobal('BDIR', "backup").  Note that this routine is intended for internal
# use only -- only Bastille API functions should call B_backup_file.
#
# Note that this routine most _carefully_ handle the GLOBAL_PREFIX variable.
# All routines which call B_backup_file will pass the filename to back up,
# without prepending the prefix.
#
###########################################################################

sub B_backup_file {

    my $file=$_[0];
    my $directory=cwd;
    my $complain = 1;
    my $original_file = $file;
    $file = $GLOBAL_PREFIX . $file;

    my $backup_dir = &getGlobal('BDIR', "backup");
    my $backup_file = $backup_dir . $original_file;

    # We don't use this -- this is mostly here for commenting.
    #my $relative_file;

    my $return=1;

    # First, separate the file into the directory and the relative filename

    if ($file =~ /^(.*)\/([^\/]+)$/) {
	#$relative_file=$2;
	$directory = $1;
    }
    

    # Now, if the directory does not exist, create it.
    # Later:
    #   Try to set the same permissions on the patch directory that the
    #   original had...?

    unless ( -d ($backup_dir . $directory) ) {
	mkpath(( $backup_dir . $directory),0,0700);

    }

    # Now we backup the file.  If there is already a backup file there,
    # we will leave it alone, since it exists from a previous run and
    # should be the _original_ (possibly user-modified) distro's version 
    # of the file.

    if ( -e $file ) {
	unless ( -e $backup_file ) {
	    $command=&getGlobal("BIN","cp");
            `$command -p $file $backup_file`;
	    &B_undo_log_entry (&getGlobal("BIN","mv"). " $backup_file $file");
	}

	# Later, following PeterW's suggestion, place a diff of the file
	# in a second backup directory

    } else {
	# The file we were trying to backup doesn't exist.

	$return=0;
	# This is a non-fatal error, not worth complaining about
	$complain = 0;
	#&ErrorLog ("# Failed trying to backup file $file -- it doesn't exist!\n");
    }

    # Check to make sure that the file does exist in the backup location.
    
    unless ( -e $backup_file ) {
	$return=0;
	if ( $complain == 1 ) { 
	    &ErrorLog("# Failed trying to backup $file -- the copy was not created.\n"); 
	}
    }

    $return;
}


###########################################################################
# &B_blank_file ($filename,$pattern) blanks the file $filename, unless the
# pattern $pattern is present in the file.  This lets us completely redo
# a file, if it isn't the one we put in place on a previous run...
#
# B_blank_file respects $GLOBAL_LOGONLY and uses B_open_plus and B_close_plus
# so that it makes backups and only modifies files when we're not in "-v"
# mode...
#
# If the file does not exist, the function does nothing, and gives an error 
# to the Error Log
#
###########################################################################

sub B_blank_file {
    
    my ($filename,$pattern) = @_;
    my $return;

    # If this variable is true, we won't blank the file...

    my $found_pattern=0;

    if ($return=&B_open_plus (*BLANK_NEW,*BLANK_OLD,$filename) ) {

	my @lines;
	
	while (my $line = <BLANK_OLD>) {

	    push @lines,$line;
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	}

	# Only copy the old file if the new one didn't match.
	if ($found_pattern) {
	    while ($line = shift @lines ) {
		&B_print(*BLANK_NEW,$line);
	    }
	}
	else {
	    &ActionLog("Blanked file $filename\n");
	}
	&B_close_plus(*BLANK_NEW,*BLANK_OLD,$filename);
    }
    else {
	&ErrorLog("Couldn't blank file $filename since we couldn't open it or its replacement\n");
    }

}


###########################################################################
# &B_append_line ($filename,$pattern,$line_to_append)  modifies $filename,
# appending $line_to_append unless one or more lines in the file matches
# $pattern.  This is an enhancement to the append_line_if_no_such_line_exists
# idea.
#
# Additionally, if $pattern is set equal to "", the line is always appended.  
#
# B_append_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here's examples of where you might use this:
#
# You'd like to add a   root   line to /etc/ftpusers if none exists.
# You'd like to add a   Options Indexes  line to Apache's config. file,
# after you delete all Options lines from said config file.
#
###########################################################################

sub B_append_line {
   
    my ($filename,$pattern,$line_to_append) = @_;

    my $found_pattern=0;
    my $return=1;

    if ( &B_open_plus (*APPEND_NEW,*APPEND_OLD,$filename) ) {
	while (my $line=<APPEND_OLD>) {
	    &B_print(*APPEND_NEW,$line);
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	}
	# Changed != 0 to $pattern so that "" works instead of 0 and perl
	# does not give the annoying
	# Argument "XX" isn't numeric in ne at ...
	if ( $pattern eq "" or ! $found_pattern ) {
	    &B_print(*APPEND_NEW,$line_to_append);
	    &ActionLog("Appended the following line to $filename:\n");
	    &ActionLog("$line_to_append");
	}
	&B_close_plus (*APPEND_NEW,*APPEND_OLD,$filename);
    }
    else {
	$return=0;
	&ErrorLog("# Couldn't append line to $filename, since open failed.");
    }

    $return;

}


###########################################################################
# &B_insert_line ($filename,$pattern,$line_to_insert,$line_to_follow)  
# modifies $filename, inserting $line_to_insert unless one or more lines
# in the file matches $pattern.  The $line_to_insert will be placed
# immediately after $line_to_follow, if it exists.  If said line does not
# exist, the line will not be inserted and this routine will return 0.
#
# B_insert_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here's examples of where you might use this:
#
# You'd like to insert a line in Apache's configuration file, in a 
# particular section.
#
###########################################################################

sub B_insert_line {
  
    my ($filename,$pattern,$line_to_insert,$line_to_follow) = @_;

    my @lines;
    my $found_pattern=0;
    my $found_line_to_follow=0;

    my $return=1;

    if ( &B_open_plus (*INSERT_NEW,*INSERT_OLD,$filename) ) {

	# Read through the file looking for a match both on the $pattern
	# and the line we are supposed to be inserting after...

	my $ctr=1;
	while (my $line=<INSERT_OLD>) {
	    push (@lines,$line);
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	    if ( ($found_line_to_follow < 1) and ($line =~ $line_to_follow)) {
		$found_line_to_follow=$ctr;
	    }
	    $ctr++;
	}

	# Log an error if we never found the line we were to insert after
	unless ($found_line_to_follow ) {
	    $return=0;
	    &ErrorLog("Never found the line that we were supposed to insert after in $filename\n");
	}

	# Now print the file back out, inserting our line if we should...

	$ctr=1;
	while (my $line = shift @lines) {
	    &B_print(*INSERT_NEW,$line);
	    if ( ($ctr == $found_line_to_follow) and ($found_pattern == 0) ) {
		&B_print(*INSERT_NEW,$line_to_insert);
		&ActionLog("Inserted the following line in $filename:\n");
		&ActionLog("$line_to_insert");
	    }
	    $ctr++;
	}

	&B_close_plus (*INSERT_NEW,*INSERT_OLD,$filename);
	
    }
    else {
	$return=0;
	&ErrorLog("# Couldn't prepend line to $filename, since open failed.");
    }

    $return;

}


###########################################################################
# &B_prepend_line ($filename,$pattern,$line_to_prepend)  modifies $filename,
# prepending $line_to_prepend unless one or more lines in the file matches
# $pattern.  This is an enhancement to the prepend_line_if_no_such_line_exists
# idea.  
#
# B_prepend_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here's examples of where you might use this:
#
# You'd like to insert the line "auth   required   pam_deny.so" to the top
# of the PAM stack file /etc/pam.d/rsh to totally deactivate rsh.
#
###########################################################################

sub B_prepend_line {
   
    my ($filename,$pattern,$line_to_prepend) = @_;

    my @lines;
    my $found_pattern=0;
    my $return=1;

    if ( &B_open_plus (*PREPEND_NEW,*PREPEND_OLD,$filename) ) {
	while (my $line=<PREPEND_OLD>) {
	    push (@lines,$line);
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	}
	unless ($found_pattern) {
	    &B_print(*PREPEND_NEW,$line_to_prepend);
	}
	while (my $line = shift @lines) {
	    &B_print(*PREPEND_NEW,$line);
	}

	&B_close_plus (*PREPEND_NEW,*PREPEND_OLD,$filename);
	
	# Log the action
	&ActionLog("Prepended the following line to $filename:\n");
	&ActionLog("$line_to_prepend");
    }
    else {
	$return=0;
	&ErrorLog("# Couldn't prepend line to $filename, since open failed.\n");
    }

    $return;

}


###########################################################################
# &B_replace_line ($filename,$pattern,$line_to_switch_in) modifies $filename,
# replacing any lines matching $pattern with $line_to_switch_in.
#
# It returns the number of lines it replaced (or would have replaced, if
# LOGONLY mode wasn't on...)
#
# B_replace_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here an example of where you might use this:
#
# You'd like to replace any Options lines in Apache's config file with:
#            Options Indexes FollowSymLinks
#
###########################################################################

sub B_replace_line {
   
    my ($filename,$pattern,$line_to_switch_in) = @_;
    my $return=0;

    if ( &B_open_plus (*REPLACE_NEW,*REPLACE_OLD,$filename) ) {
	while (my $line=<REPLACE_OLD>) {
	    unless ($line =~ $pattern) {    
		&B_print(*REPLACE_NEW,$line);
	    }
	    else {
		# Don't replace the line if it's already there.
		unless ($line eq $line_to_switch_in) {
		    &B_print(*REPLACE_NEW,$line_to_switch_in);
		
		    $return++;
		    &ActionLog("# File modification in $filename -- replaced line\n");
		    &ActionLog($line);
		    &ActionLog("with:\n");
		    &ActionLog("$line_to_switch_in");
		}
                # But if it is there, make sure it stays there! (by Paul Allen)
		else {
		    &B_print(*REPLACE_NEW,$line);
                }    
	    }
	}
	&B_close_plus (*REPLACE_NEW,*REPLACE_OLD,$filename);
    }
    else {
	$return=0;
	&ErrorLog("Couldn't replace line(s) in $filename because open failed.\n");
    }

    $return;
}


###########################################################################
# &B_hash_comment_line ($filename,$pattern) modifies $filename, replacing 
# any lines matching $pattern with a "hash-commented" version, like this:
#
#
#        finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
# becomes:
#        #finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
#
#
# B_hash_comment_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
###########################################################################

sub B_hash_comment_line {
   
    my ($filename,$pattern) = @_;
    my $return=1;

    if ( &B_open_plus (*HASH_NEW,*HASH_OLD,$filename) ) {
	while (my $line=<HASH_OLD>) {
	    unless ( ($line =~ $pattern) and ($line !~ /^\s*\#/) ) {    
		&B_print(*HASH_NEW,$line);
	    }
	    else {
		&B_print(*HASH_NEW,"#$line");
		&ActionLog("# File modification in $filename -- hash commented line\n");
		&ActionLog($line);
		&ActionLog("like this:\n");
		&ActionLog("#$line");
	    }
	}
	&B_close_plus (*HASH_NEW,*HASH_OLD,$filename);
    }
    else {
	$return=0;
	&ErrorLog("Couldn't hash-comment line(s) in $filename because open failed.\n");
    }

    $return;
}


###########################################################################
# &B_hash_uncomment_line ($filename,$pattern) modifies $filename, 
# removing any commentting from lines that match $pattern.
#
#        #finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
# becomes:
#        finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
#
#
# B_hash_uncomment_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
###########################################################################

sub B_hash_uncomment_line {
   
    my ($filename,$pattern) = @_;
    my $return=1;

    if ( &B_open_plus (*HASH_NEW,*HASH_OLD,$filename) ) {
	while (my $line=<HASH_OLD>) {
	    unless ( ($line =~ $pattern) and ($line =~ /^\s*\#/) ) {    
		&B_print(*HASH_NEW,$line);
	    }
	    else {
		$line =~ /^\s*\#+(.*)$/;
		$line = "$1\n";

		&B_print(*HASH_NEW,"$line");
		&ActionLog("File modification in $filename -- hash uncommented line\n");
		&ActionLog($line);
	    }
	}
	&B_close_plus (*HASH_NEW,*HASH_OLD,$filename);
    }
    else {
	$return=0;
	&ErrorLog("Couldn't hash-uncomment line(s) in $filename because open failed.\n");
    }

    $return;
}



###########################################################################
# &B_delete_line ($filename,$pattern) modifies $filename, deleting any 
# lines matching $pattern.  It uses B_replace_line to do this.
#
# B_replace_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here an example of where you might use this:
#
# You'd like to remove any timeout=  lines in /etc/lilo.conf, so that your
# delay=1 modification will work.

#
###########################################################################


sub B_delete_line {

    my ($filename,$pattern)=@_;
    my $return=&B_replace_line($filename,$pattern,"");

    $return;
}


###########################################################################
# &B_chunk_replace ($file,$pattern,$replacement) reads $file replacing the
# first occurence of $pattern with $replacement.
# 
###########################################################################

sub B_chunk_replace {

    my ($file,$pattern,$replacement) = @_;

    my @lines;
    my $big_long_line;
    my $return=1;

    &B_open (*OLDFILE,$file);

    # Read all lines into one scalar.
    @lines = <OLDFILE>;
    &B_close (*OLDFILE);
    foreach my $line ( @lines ) {
	$big_long_line .= $line;
    }

    # Substitution routines get weird unless last line is terminated with \n
    chomp $big_long_line;
    $big_long_line .= "\n";

    # Exit if we don't find a match
    unless ($big_long_line =~ $pattern) {
	return 0;
    }
    
    $big_long_line =~ s/$pattern/$replacement/s;

    $return=&B_open_plus (*NEWFILE,*OLDFILE,$file);
    if ($return) {
	&B_print (*NEWFILE,$big_long_line);
	&B_close_plus (*NEWFILE,*OLDFILE,$file);
    }

    $return;
}


###########################################################################
# &B_chunk_append ($file,$pattern,$payload) reads $file adding the $payload
# after the first occurence of $pattern.
# 
###########################################################################

sub B_chunk_append {

    my ($file,$pattern,$payload) = @_;

    my @lines;
    my $big_long_line;
    my $return=1;
    my $position;

    my ($first_half,$second_half);

    &B_open(*OLDFILE,$file);

    # Read all lines into one scalar.
    @lines = <OLDFILE>;
    &B_close(*OLDFILE);
    foreach my $line ( @lines ) {
	$big_long_line .= $line;
    }

    # Substitution routines get weird unless last line is terminated with \n
    chomp $big_long_line;
    $big_long_line .= "\n";

    # Exit if we don't find a match
    unless ($big_long_line =~ $pattern) {
	return 0;
    }

    # We need to figure out which position in the file to insert the text
    
    $position = pos $big_long_line;

    # Insert the payload, unless it's already there.

    $first_half = substr ($big_long_line, 0, ($position-1) );
    $second_half = substr ($big_long_line, $position);

    my $length_to_check = length ($payload);
    unless ($legnth_to_check < 10) {
	$legnth_to_check =10;
    }
    unless ( substr($second_half,0,length ($payload)) eq $payload ) { 
	$big_long_line = $first_half . $payload . $second_half;
    }

    # Write out the result

    $return=&B_open_plus (*NEWFILE,*OLDFILE,$file);
    if ($return) {
	&B_print (*NEWFILE,$big_long_line);
	&B_close_plus (*NEWFILE,*OLDFILE,$file);
    }

    $return;
}



###########################################################################
# &B_delete_file ($file)  deletes the file $file and makes a backup to
# the backup directory.
#
# Here an example of where you might use this:
#
# You'd like to add a   Options Indexes  line to Apache's config. file,
# after you delete all Options lines from said config file.
#
##########################################################################


sub B_delete_file {

    #
    # This API routine deletes the named file, backing it up first to the
    # backup directory.
    # 

    my $filename=shift @_;
    my $return=1;

    # We have to append the prefix ourselves since we don't use B_open_plus

    my $original_filename=$filename;
    $filename = $GLOBAL_PREFIX . $filename;

    &ActionLog("# Deleting (and backing-up) file $original_filename\n");
    &ActionLog("rm $original_filename\n");

    unless ($filename) {
	&ErrorLog("# B_delete_file called with no arguments!\n");
    }
    
    unless ($GLOBAL_LOGONLY) {
	if ( B_backup_file($original_filename) ) {
	    unless ( unlink $filename ) {
		&ErrorLog("# Couldn't unlink file $original_filename");
		$return=0;
	    }
	}
	else {
	    $return=0;
	    &ErrorLog("# B_delete_file did not delete $original_filename since it could not back it up\n");
	}
    }

    $return;

}


###########################################################################
# &B_create_file ($file) creates the file $file, if it doesn't already
# exist.
# It will set a default mode of 0700 and a default uid/gid or 0/0.
#
# &B_create_file, to support Bastille's undo functionality, writes an
# rm $file command to the end of the file &getGlobal('BFILE', "created-files").
#
##########################################################################


sub B_create_file {

    my $file = $_[0];
    my $return=1;

    # We have to create the file ourselves since we don't use B_open_plus

    my $original_file = $file;
    $file = $GLOBAL_PREFIX . $file;

    unless ( -e $file ) {
	unless ($GLOBAL_LOGONLY) {
	    $return=open CREATE_FILE,">$file";
	    
	    if ($return) {
		close CREATE_FILE;
		chmod 0700,$file;
		# Make the undo functionality
		&B_undo_log_entry( &getGlobal('BIN','rm') . " $original_file \n");
	    } else {
		&ErrorLog("# Couldn't create file $original_file even though " . 
			  "it didn't already exist!");
	    }   
	}
	&ActionLog("# Created file $original_file\n");
    } else {
	&ActionLog("# Didn't create file $original_file since it already existed.\n");
	$return=0;
    }

    $return;
}

	    
###########################################################################
# &B_create_dir ($dir) creates the directory $dir, if it doesn't already
# exist.
# It will set a default mode of 0700 and a default uid/gid or 0/0.
#
# &B_create_dir, to support Bastille's undo functionality, writes an
# rmdir $file command to the end of the file &getGlobal('BFILE', "created-files").
#
##########################################################################


sub B_create_dir {

    my $dir = $_[0];
    my $return=1;

    # We have to append the prefix ourselves since we don't use B_open_plus

    my $original_dir=$dir;
    $dir = $GLOBAL_PREFIX . $dir;

    unless ( -d $dir ) {
	unless ($GLOBAL_LOGONLY) {
	    $return=mkdir $dir,0700;
	    
	    if ($return) {
		# Make the undo functionality
		&B_undo_log_entry (&getGlobal('BIN','rmdir') . " $original_dir\n");
	    }
	    else {
		&ErrorLog("# Couldn't create dir $original_dir even though it didn't already exist!");
	    }
	    
	}
	&ActionLog("# Created directory $original_dir\n");
    }
    else {
	&ActionLog("# Didn't create directory $original_dir since it already existed.\n");
	$return=0;
    }

    $return;
}
		

###########################################################################
# &B_print ($handle,@list) prints the items of @list to the filehandle
# $handle.  It logs the action and respects the $GLOBAL_LOGONLY variable.
#
###########################################################################

sub B_print {
   my $handle=shift @_;

   my $result=1;

   unless ($GLOBAL_LOGONLY) {
       $result=print $handle @_;
   }

   ($handle) = "$handle" =~ /[^:]+::[^:]+::([^:]+)/;

   $result;
}


###########################################################################
# &B_remove_suid ($file) removes the suid bit from $file if it
# is set and the file exist. If you would like to remove the suid bit
# from /bin/ping then you need to use:
# 
#                 &B_remove_suid("/bin/ping");
#
# &B_remove_suid respects GLOBAL_LOGONLY and the $GLOBAL_PREFIX.
# &B_remove_suid uses &B_chmod to make the permission changes
# &B_remove_suid allows for globbing.  tyler_e
#
###########################################################################

sub B_remove_suid($) {
    my $file_expr = $_[0];
    &SanitizeEnv;
    my @files = glob($file_expr);

    foreach my $file (@files) {
	# check file existance
	if(-e $file){
	    # stat current file to get raw permissions
	    my $old_perm_raw = (stat $file)[2];
	    # test to see if suidbit is set
	    my $suid_bit = (($old_perm_raw/2048) % 2);
	    if($suid_bit == 1){
		# new permission without the suid bit
		my $new_perm = ((($old_perm_raw/512) % 8 ) - 4) . 
		    (($old_perm_raw/64) % 8 ) . 
			(($old_perm_raw/8) % 8 ) . 
			    (($old_perm_raw) % 8 );
		if(&B_chmod(oct($new_perm), $file)){
		    &ActionLog("Removed SUID bit from \"$file\".");
		}
		else {
		    &ErrorLog("Could not remove SUID bit from \"$file\".");
		}
	    } # No action if SUID bit is not set
	}# No action if file does not exist
    }# Repeat for each file in the file glob
}
    

###########################################################################
# &B_chmod_if_exists ($mode, $file) sets the mode of $file to $mode *if*
# $file exists.  $mode must be stored in octal, so if you want to give 
# mode 700 to /etc/aliases, you need to use:
#
#                 &B_chmod_if_exists ( 0700 , "/etc/aliases");
#
# where the 0700 denotes "octal 7-0-0".
#
# &B_chmod_if_exists respects GLOBAL_LOGONLY and the $GLOBAL_PREFIX and uses 
# &B_undo_log_entry to reset the permissions of the file.
#
# B_chmod_if_exists allow for globbing now, as of 1.2.0.  JJB
#
##########################################################################


sub B_chmod_if_exists {
   my ($new_perm,$file_expr)=@_;
   # If $file_expr has a glob character, pass it on (B_chmod won't complain
   # about nonexistent files if given a glob pattern)
   if ( $file_expr =~ /[\*\[\{]/ ) {   # } just to match open brace for vi
       &ActionLog("Running chmod $new_perm $file_expr"); 
       return(&B_chmod($new_perm,$file_expr));
   }
   # otherwise, test for file existence
   if ( -e $GLOBAL_PREFIX.$file_expr ) { 
       &ActionLog("Running chmod $new_perm $file_expr"); 
       return(&B_chmod($new_perm,$file_expr)); 
   }
}


###########################################################################
# &B_chmod ($mode, $file) sets the mode of $file to $mode.  $mode must
# be stored in octal, so if you want to give mode 700 to /etc/aliases,
# you need to use:
#
#                 &B_chmod ( 0700 , "/etc/aliases");
#
# where the 0700 denotes "octal 7-0-0".
#
# &B_chmod respects GLOBAL_LOGONLY and the $GLOBAL_PREFIX and uses 
# &B_undo_log_entry used to insert a shell command that will return
#         the permissions to the pre-Bastille state.
#
# B_chmod allow for globbing now, as of 1.2.0.  JJB
#
##########################################################################


sub B_chmod {
   my ($new_perm,$file_expr)=@_;
   my $old_perm;
   my $old_perm_raw;
   my $new_perm_formatted;

   my $return=1;

   my $file;

   my @files = glob ($file_expr);

   foreach $file (@files) {

       # Prepend global prefix, but save the original filename for B_backup_file
       my $original_file=$file;
       $file = $GLOBAL_PREFIX . $file;
       
       # Store the old permissions so that we can log them.
       $old_perm_raw=(stat $file)[2];   
       $old_perm= (($old_perm_raw/512) % 8) . 
	   (($old_perm_raw/64) % 8) .
	       (($old_perm_raw/8) % 8) . 
		   ($old_perm_raw % 8);
       
       # formating for simple long octal output of the permissions in string form
       $new_perm_formatted=sprintf "%5lo",$new_perm;
   
       &ActionLog ("# change permissions on $original_file from $old_perm to $new_perm_formatted\n");
       
       &ActionLog( "chmod $new_perm_formatted,\"$original_file\";\n");
       
       # Change the permissions on the file
       
       if ( -e $file ) {
	   unless ($GLOBAL_LOGONLY) {
	       $return=chmod $new_perm,$file;
	       if($return){
		   # if the distrobution is HP-UX then the modifications should
		   # also be made to the IPD (installed product database)
		   if($GLOBAL_DISTRO =~ "^HP-UX"){
		       &B_swmodify($file);
		   }
		   # making changes undo-able
		   &B_undo_log_entry(&getGlobal('BIN', "chmod") . " $old_perm $file\n");
	       }
	   }
	   unless ($return) {
	       &ActionLog("#ERROR: couldn't change permissions on $original_file from $old_perm to $new_perm_formatted\n");
	       &ErrorLog("#ERROR: couldn't change permissions on $original_file from $old_perm to $new_perm_formatted\n");
	       
	       $return=0;
	   }				       
       }
       else {
	   &ActionLog( "#ERROR: chmod: File $original_file doesn't exist!\n");
	   &ErrorLog( "#ERROR: chmod: File $original_file doesn't exist!\n");	 
	   $return=0;
       }
   }

   $return;

}


###########################################################################
# &B_chown ($uid, $file) sets the owner of $file to $uid, like this:
#
#                 &B_chown ( 0 , "/etc/aliases");
#
# &B_chown respects $GLOBAL_LOGONLY and the $GLOBAL_PREFIX and uses 
#
# &B_undo_log_entry used to insert a shell command that will return
#         the file/directory owner to the pre-Bastille state.
#
# Unlike Perl, we've broken the chown function into B_chown/B_chgrp to
# make error checking simpler.
#
# As of 1.2.0, this now supports file globbing. JJB
#
##########################################################################


sub B_chown {
   my ($newown,$file_expr)=@_;
   my $oldown;
   my $oldgown;

   my $return=1;
   
   my $file;
   &SanitizeEnv;
   my @files = glob($file_expr);

   foreach $file (@files) {

       # Prepend prefix, but save original filename 
       my $original_file=$file;
       $file = $GLOBAL_PREFIX . $file;
       
       $oldown=(stat $file)[4];
       $oldgown=(stat $file)[5];
       
       &ActionLog ("# change ownership on $original_file from $oldown to $newown\n");
       &ActionLog ("chown $newown,$oldgown,\"$original_file\";\n");
       if ( -e $file ) {
	   unless ($GLOBAL_LOGONLY) {
	       # changing the files owner using perl chown function
	       $return = chown $newown,$oldgown,$file;
	       if($return){
		   # if the distrobution is HP-UX then the modifications should
		   # also be made to the IPD (installed product database)
		   if($GLOBAL_DISTRO =~ "^HP-UX"){
		       &B_swmodify($file);
		   }
		   # making ownership change undo-able
		   &B_undo_log_entry(&getGlobal('BIN', "chown") . " $oldown $file\n");
	       }	       
	   }
	   unless ($return) {
	       &ActionLog( "#ERROR: couldn't change ownership to $newown on file $original_file\n");
	       &ErrorLog("#ERROR: couldn't change ownership to $newown on file $original_file\n");
	   }
       }
       else {
	   &ActionLog("#ERROR: chown: File $original_file doesn't exist!\n");
	   &ErrorLog("#ERROR: chown: File $original_file doesn't exist!\n");
	   $return=0;
       }
   }
   
   $return;
}




###########################################################################
# &B_chgrp ($gid, $file) sets the group owner of $file to $gid, like this:
#
#                 &B_chgrp ( 0 , "/etc/aliases");
#
# &B_chgrp respects $GLOBAL_LOGONLY and the $GLOBAL_PREFIX and uses 
# &B_undo_log_entry used to insert a shell command that will return
#         the file/directory group to the pre-Bastille state.
#
# Unlike Perl, we've broken the chown function into B_chown/B_chgrp to
# make error checking simpler.
#
# As of 1.2.0, this now supports file globbing.  JJB
#
##########################################################################


sub B_chgrp {
   my ($newgown,$file_expr)=@_;
   my $oldown;
   my $oldgown;

   my $return=1;

   my $file;
   &SanitizeEnv;
   my @files = glob($file_expr);
   
   foreach $file (@files) {
   
       # Prepend global prefix, but save original filename for &B_backup_file
       my $original_file=$file;
       $file = $GLOBAL_PREFIX . $file;
       
       $oldown=(stat $file)[4];
       $oldgown=(stat $file)[5];
       
       &ActionLog( "# change group ownership on $original_file from $oldgown to $newgown\n");
       &ActionLog( "chown $oldown,$newgown,\"$original_file\";\n");
       if ( -e $file ) {
	   unless ($GLOBAL_LOGONLY) {
	       # changing the group for the file/directory
	       $return = chown $oldown,$newgown,$file;
	       if($return){
		   # if the distrobution is HP-UX then the modifications should
		   # also be made to the IPD (installed product database)
		   if($GLOBAL_DISTRO =~ "^HP-UX"){
		       &B_swmodify($file);
		   }
		   &B_undo_log_entry(&getGlobal('BIN', "chgrp") . " $oldgown $file\n");
	       }	       
	   }
	   unless ($return) {
	       &ActionLog("#ERROR: couldn't change ownership to $newgown on file $original_file\n");
	       &ErrorLog("#ERROR: couldn't change ownership to $newgown on file $original_file\n");	    
	   }
       }
       else {
	   &ActionLog( "#ERROR: chgrp: File $original_file doesn't exist!\n");
	   &ErrorLog("#ERROR: chgrp: File $original_file doesn't exist!\n");
	   $return=0;
       }
   }

   $return;
}


###########################################################################
# &B_symlink ($original_file,$new_symlink) creates a symbolic link from
# $original_file to $new_symlink.
#
# &B_symlink respects $GLOBAL_LOGONLY and the $GLOBAL_PREFIX.  It supports
# the undo functionality that you've come to know and love by adding every
# symbolic link it creates to &getGlobal('BFILE', "created-symlinks"), currently set to:
#
#         /root/Bastille/undo/undo-created-symlinks
#
# The undo script, if it works like I think it should, will run this file,
# which should be a script or rm's...
#
##########################################################################

sub B_symlink {
    my ($source_file,$new_symlink)=@_;
    my $return=1;
    my $original_source = $source_file;
    my $original_symlink = $new_symlink;
    
    $source_file = $GLOBAL_PREFIX . $source_file;

    unless ($GLOBAL_LOGONLY) {
	$return=symlink $source_file,$new_symlink;
	if ($return) {
	    &B_undo_log_entry (&getGlobal('BIN',"rm") .  " $original_symlink\n");
	}
    }

    &ActionLog( "# created a symbolic link called $original_symlink from $original_source\n");
    &ActionLog( "symlink \"$original_source\",\"$original_symlink\";\n");
    unless ($return) { 
        &ActionLog("#ERROR: couldn't symlink $original_symlink -> $original_source\n");
	&ErrorLog("#ERROR: couldn't symlink $original_symlink -> $original_source\n");
    }

    $return;

}

###########################################################################
# B_Display was the display function used by IPCHAINS in v1.0.0-v1.0.3.
# Its use should be fully deprecated in 1.1.
#
###########################################################################

sub B_Display {
	# routine to display a long string, automatically wrapping
	# each line to no more than 80 characters
	my ( $text) = @_;
	my ($chunk1, $chunk2, $chunk3);
	while ( length($text) >= 80 ) {
		# pull the first 80 chars off the string
		$text =~ s/^(.{80})//s;
		$chunk1 = $1;
		if ( $chunk1 =~ /^(.*?\n)(.*?)$/s ) {
			$chunk2 = $1; $chunk3 = $2;
			# explicit line endings should be respected
			print $chunk2;
			$text = $chunk3 . $text;
		} elsif ( $chunk1 =~ /\s/ ) {
			# there's a space, grab everything up to the _last_ space
			$chunk1 =~ /^(.*)\s([^\s]*?)$/s;
			$chunk2 = $1; $chunk3 = $2;
			# print the stuff up to the last space
			print "$chunk2\n";
			# put the remaining data back at the front of $text
			$text = $chunk3 . $text;
		} else {
			# no space, just print
			print "$chunk1\n";
		}
	}
	if ( $text =~ /\S/ ) {
		# some non-whitespace data, print it
		print "$text\n";
	}
}


###########################################################################
# &B_chkconfig_on ($daemon_name) creates the symbolic links that are
# named in the "# chkconfig: ___ _ _ " portion of the init.d files.  We
# need this utility, in place of the distro's chkconfig, because of both
# our need to add undo functionality and our need to harden distros that
# are not mounted on /.
#
# It uses the following global variables to find the links and the init
# scripts, respectively:
#
#   &getGlobal('DIR', "rcd")    -- directory where the rc_.d subdirs can be found
#   &getGlobal('DIR', "initd")  -- directory the rc_.d directories link to
#
# Here an example of where you might use this:
#
# You'd like to tell the system to run the firewall at boot:
#       B_chkconfig_on("bastille-firewall")
#
###########################################################################

# PW: Blech. Copied B_chkconfig_off() and changed a few things,
#		then changed a few more things....

sub B_chkconfig_on {

    my $startup_script=$_[0];
    my $return=1;

    my $chkconfig_line;
    my ($runlevelinfo,@runlevels);
    my ($start_order,$stop_order,$filetolink);

    &ActionLog("# chkconfig_on enabling $startup_script\n");
    
    # In Debian system there is no chkconfig script, runlevels are checked
    # one by one (jfs)
    if ($GLOBAL_DISTRO =~/^DB.*/) {
	    $filetolink = &getGlobal('DIR', "initd") . "/$startup_script";
	    if (-x $filetolink)
	    {
		    foreach my $level ("0","1","2","3","4","5","6" ) {
			    my $link = '';
			    $link = &getGlobal('DIR', "rcd") . "/rc" . "$level" . ".d/K50" . "$startup_script";
			    $return=symlink($filetolink,$link);
		    }
	    }
	    return $return;
    }
    # Run through the init script looking for the chkconfig line...
    $return = open CHKCONFIG,&getGlobal('DIR', "initd") . "/$startup_script";
    unless ($return) {
	&ActionLog("# Didn't chkconfig_on $startup_script because we couldn't open " . &getGlobal('DIR', "initd") . "/$startup_script\n");
    }
    else {

      READ_LOOP:
	while (my $line=<CHKCONFIG>) {

	    # We're looking for lines like this one:
	    #      # chkconfig: 2345 10 90
	    # OR this
	    #      # chkconfig: - 10 90
	    
	    if ($line =~ /^#\s*chkconfig:\s*([-\d]+)\s*(\d+)\s*(\d+)/ ) {
		$runlevelinfo = $1;
		$start_order = $2;
		$stop_order = $3;
		# handle a runlevels arg of '-'
		if ( $runlevelinfo eq '-' ) {
		    &ActionLog("# chkconfig_on saw '-' for runlevels for \"$startup_script\", is defaulting to levels 3,4,5\n");
		    $runlevelinfo = '345';
		}
		@runlevels = split(//,$runlevelinfo);
		# make sure the orders have 2 digits
		$start_order =~ s/^(\d)$/0$1/;
		$stop_order =~ s/^(\d)$/0$1/;
		last READ_LOOP;
	    }
	}
	close CHKCONFIG;

	# Do we have what we need?
	if ( (scalar(@runlevels) < 1) || (! $start_order =~ /^\d{2}$/) || (! $stop_order =~ /^\d{2}$/) ) {
		# problem
		&ErrorLog("# B_chkconfig_on $startup_script failed -- no valid runlevel/start/stop info found\n");
		return(-1);
	}

	# Now, run through creating symlinks...
	&ActionLog("# chkconfig_on will use runlevels ".join(",",@runlevels)." for \"$startup_script\" with S order $start_order and K order $stop_order\n");
	
	$return=0;
	# BUG: we really ought to readdir() on &getGlobal('DIR', "rcd") to get all levels
	foreach my $level ( "0","1","2","3","4","5","6" ) {
		my $link = '';
		# we make K links in runlevels not specified in the chkconfig line
	    	$link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/K$stop_order" . $startup_script;
		my $klink = $link;
		# now we see if this is a specified runlevel; if so, make an S link
		foreach my $markedlevel ( @runlevels ) {
			if ( $level == $markedlevel) {
	    			$link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/S$start_order" . $startup_script;
			}
		}
	    	my $target = &getGlobal('DIR', "initd") ."/" . $startup_script;
	    	my $local_return;

		if ( (-e "${GLOBAL_PREFIX}$klink") && ($klink ne $link) ) {
		    # there's a K link, but this level needs an S link
		    unless ($GLOBAL_LOGONLY) {
			$local_return = unlink("${GLOBAL_PREFIX}$klink");
			if ( ! local_return ) {
			    # unlinking old, bad $klink failed
			    &ErrorLog("# Unlinking $klink failed\n");
			} else {
			    &ActionLog("# Removed link $klink\n");
			    # If we removed the link, add a link command to the undo file
			    &B_undo_log_entry (&getGlobal('BIN','ln') . " -s $target $klink\n");
			} # close what to do if unlink works	
		    }	# if not GLOBAL_LOGONLY
		}	# if $klink exists and ne $link
	   	
		# OK, we've disposed of any old K links, make what we need 
	    	if ( (! ( -e "${GLOBAL_PREFIX}$link" )) && ($link ne '') ) {
		    # link doesn't exist and the start/stop number is OK; make it
		    unless ($GLOBAL_LOGONLY) {
			# create the link
			$local_return = &B_symlink($target,$link);
			if ($local_return) {
			    $return++;
			    &ActionLog("# Created link $link\n");
			} else {
			    &ErrorLog("Couldn't create $link when trying to chkconfig on $startup_script\n");
			}
		    }
		    
		} # link doesn't exist
	    } # foreach level
	
    }

    if ($return < @runlevels) {
	$return=0;
    }
    
    $return;

}


###########################################################################
# &B_chkconfig_off ($daemon_name) deletes the symbolic links that are
# named in the "# chkconfig: ___ _ _ " portion of the init.d files.  We
# need this utility, in place of the distro's chkconfig, because of both
# our need to add undo functionality and our need to harden distros that
# are not mounted on /.
#
# chkconfig allows for an UNDO of its work by writing to an executable
# file &getGlobal('BFILE', "removed-symlinks").
#
# It uses the following global variables to find the links and the init
# scripts, respectively:
#
#   &getGlobal('DIR', "rcd")    -- directory where the rc_.d subdirs can be found
#   &getGlobal('DIR', "initd")  -- directory the rc_.d directories link to
#
# Here an example of where you might use this:
#
# You'd like to tell stop running sendmail in daemon mode on boot:
#       B_chkconfig_off("sendmail")
#
###########################################################################



sub B_chkconfig_off {

    my $startup_script=$_[0];
    my $return=1;

    my $chkconfig_line;
    my @runlevels;
    my ($start_order,$stop_order,$filetolink);

    if ($GLOBAL_DISTRO =~/^DB.*/) {
	    $filetolink = &getGlobal('DIR', "initd") . "/$startup_script";
	    if (-x $filetolink)
	    {
		    # Three ways to do this in Debian:
		    # 1.- have the initd script set to 600 mode
		    # 2.- Remove the links in rcd (re-installing the package
		    # will break it)
		    # 3.- Use update-rc.d --remove (same as 2.)
		    # (jfs) 
		    &B_chmod(0600,$filetolink);
		    $return=6;
		    
		    # The second option
		    #foreach my $level ("0","1","2","3","4","5","6" ) {
		    #my $link = '';
		    #$link = &getGlobal('DIR', "rcd") . "/rc" . "$level" . ".d/K50" . "$startup_script"; 
		    #unlink($link);
		    #}
	    }
    }
    else {

	    # Run through the init script looking for the chkconfig line...


	    $return = open CHKCONFIG,&getGlobal('DIR', "initd") . "/$startup_script";
	    unless ($return) {
		    &ActionLog("Didn't chkconfig_off $startup_script because we couldn't open " . &getGlobal('DIR', "initd") . "/$startup_script\n");
	    }
	    else {

		    READ_LOOP:
		    while (my $line=<CHKCONFIG>) {

			    # We're looking for lines like this one:
			    #      # chkconfig: 2345 10 90

			    if ($line =~ /^#\s*chkconfig:\s*([-\d]+)\s*(\d+)\s*(\d+)/ ) {
				    @runlevels=split //,$1;
				    $start_order=$2;
				    $stop_order=$3;


				    # Change single digit runlevels to double digit -- otherwise,
				    # the alphabetic ordering chkconfig depends on fails.
				    if ($start_order =~ /^\d$/ ) {
					    $start_order = "0" . $start_order;
					    &ActionLog("# chkconfig_off converted start order to $start_order\n");
				    }
				    if ($stop_order =~ /^\d$/ ) {
					    $stop_order = "0" . $stop_order;
					    &ActionLog("# chkconfig_off converted stop order to $stop_order\n");
				    }

				    last READ_LOOP;
			    }
		    }
		    close CHKCONFIG;

		    # If we never found a chkconfig line, can we just run through all 5 
		    # rcX.d dirs from 1 to 5...?

		    # unless ( $start_order and $stop_order ) {
		    #	 @runlevels=("1","2","3","4","5");
		    #	 $start_order = "*"; $stop_order="*";
		    # }

		    # Now, run through removing symlinks...



		    $return=0;

		    # Handle the special case that the runlevel specified is solely "-"
		    if ($runlevels[0] =~ /-/) {
			    @runlevels = ( "0","1","2","3","4","5","6" );
		    }

		    foreach $level ( @runlevels ) {
			    my $link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/S$start_order" . $startup_script;
			    my $new_link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/K$stop_order" . $startup_script;
			    my $target = &getGlobal('DIR', "initd") ."/" . $startup_script;
			    my $local_return;


			    # Replace the S__ link in this level with a K__ link.
			    if ( -e $link ) {
				    unless ($GLOBAL_LOGONLY) {
					    $local_return=unlink $link;
					    if ($local_return) {
						    $local_return=symlink $target,$new_link;
						    unless ($local_return) {
							    &ErrorLog("# Linking $target to $new_link failed.\n");
						    }
					    }
					    else {  # unlinking failed
						    &ErrorLog("# Unlinking $link failed\n");
					    }

				    }
				    if ($local_return) {
					    $return++;
					    &ActionLog("# Removed link $link\n");

					    #
					    # If we removed the link, add a link command to the undo file
					    # Write out the undo information for recreating the S__
					    # symlink and deleting the K__ symlink.
					    &B_undo_log_entry(&getGlobal('BIN',"ln") . " -s $target $link\n");
					    &B_undo_log_entry(&getGlobal('BIN',"rm") . " -f $new_link\n");
				    }
				    else {
					    &ErrorLog("# B_chkconfig_off $startup_script failed\n");
				    }

			    }
		    } # foreach

	    } # else-unless

    } # else-DB
    if ($return < @runlevels) {
	    $return=0;
    }

    $return;

}


############################################################################
# &B_cp is the Bastille cp command, which is based on Perl's File::cp.
# &B_cp($source,$target).  It is somewhat strange, to make the Backup and
# undo functions easier to implement, in that:
#
#
#     It can ONLY copy from one file to another! Both $source and
#     $target must be files, not directories!!!!
#
# It respects $GLOBAL_LOGONLY and $GLOBAL_PREFIX, appending the prefix for
# itself.  If $target is an already-existing file, it is backed up.
#
# Undo either appends another  rm $target   to  &getGlobal('BFILE', "created-files")  or
# backs up the file that _was_ there into the &getGlobal('BDIR', "backup").
#
############################################################################

sub B_cp {

    my ($source,$target)=@_;
    my $return=0;

    my $had_to_backup_target=0;

    use File::Copy;

    my $original_source=$source;
    $source = $GLOBAL_PREFIX . $source;
    my $original_target=$target;
    $target = $GLOBAL_PREFIX . $target;

    unless ($GLOBAL_LOGONLY) {
	unless ( -e $target and -f $target ) {
	    &B_backup_file($original_target);
	    &ActionLog("About to copy $original_source to $original_target -- had to backup target\n");
	    $had_to_backup_target=1;
	}

	$return=copy($source,$target);
	if ($return) {
	    &ActionLog("cp $original_source $original_target\n");
	    
	    #
	    # We want to add a line to the &getGlobal('BFILE', "created-files") so that the
	    # file we just put at $original_target gets deleted.
	    #
	    &B_undo_log_entry(&getGlobal('BIN',"rm") . " $original_target\n");
	} else {
	    &ErrorLog("Failed to copy $original_source to $original_target\n");
	}
    }
    $return;
}



############################################################################
# &B_place puts a file in place, using Perl's File::cp.  This file is taken
# from &getGlobal('BDIR', "home") and is used to place a file that came with
# Bastille.  This is a workaround to a problem with using &B_cp for the job:
# &B_cp prepends a GLOBAL_PREFIX to both filenames, while &B_place only 
# prepends to the destination filename.
#
#
# It respects $GLOBAL_LOGONLY and $GLOBAL_PREFIX, appending the prefix for
# itself.  If $target is an already-existing file, it is backed up.
#
# Undo either appends another  rm $target   to  &getGlobal('BFILE', "created-files")  or
# backs up the file that _was_ there into the &getGlobal('BDIR', "backup").
#
# UPDATE:  
#         20010218 - changed the location to /usr/share/Bastille because of
#                    the file location update that MandrakeSoft suggests.
#
############################################################################

sub B_place {

    my ($source,$target)=@_;
    my $return=0;

    my $had_to_backup_target=0;

    use File::Copy;

    my $original_source=$source;
    $source  = &getGlobal('BDIR', "share") . $source;
  #  $source = "/usr/share/Bastille" . $source; #removed hard-coded path
    my $original_target=$target;
    $target = $GLOBAL_PREFIX . $target;

    unless ($GLOBAL_LOGONLY) {
	if ( -e $target and -f $target ) {
	    &B_backup_file($original_target);
	    &ActionLog("About to copy $original_source to $original_target -- had to backup target\n");
	    $had_to_backup_target=1;
	}
	$return=copy($source,$target);
	if ($return) {
	    &ActionLog("placed file $original_source  as  $original_target\n");   
	    #
	    # We want to add a line to the &getGlobal('BFILE', "created-files") so that the
	    # file we just put at $original_target gets deleted.
	    &B_undo_log_entry(&getGlobal('BIN',"rm") . " $original_target\n");
	} else {
	    &ErrorLog("Failed to place $original_source as $original_target\n");
	}
    }

    $return;
}





#############################################################################
#############################################################################
#############################################################################

###########################################################################
# &B_mknod ($file) creates the node $file, if it doesn't already
# exist.  It uses the prefix and suffix, like this:
#
#            mknod $prefix $file $suffix
#
# This is just a wrapper to the mknod program, which tries to introduce
# undo functionality, by writing    rm $file     to the end of the 
# file &getGlobal('BFILE', "created-files").
#
##########################################################################


sub B_mknod {

    my ($prefix,$file,$suffix) = @_;
    my $return=1;

    # We have to create the filename ourselves since we don't use B_open_plus

    my $original_file = $file;
    $file = $GLOBAL_PREFIX . $file;

    unless ( -e $file ) {
	unless ($GLOBAL_LOGONLY) {

	    my $command = &getGlobal("BIN","mknod") . " $prefix $file $suffix";

	    if ( system($command) == 0) {
		# Since system will return 0 on success, invert the error code
		$return=1;
	    }
	    else {
		$return=0;
	    }

	    if ($return) {

		# Make the undo functionality
		&B_undo_log_entry(&getGlobal('BIN',"rm") . " $original_file\n");
	    } else {
		&ErrorLog("# Couldn't mknod $prefix $original_file $suffix even though it didn't already exist!");
	    }
	    
	}
	&ActionLog("# mknod $prefix $original_file $suffix\n");
    }
    else {
	&ActionLog("# Didn't mknod $prefix $original_file $suffix since $original_file already existed.\n");
	$return=0;
    }
    
    $return;
}

###########################################################################
# &B_undo_log_entry ($undo command) prepends a command to a shell script.  This shell
# script is intended to be run by UndoBastille to reverse the changes that 
# Bastille made, returning the system to its original state.
###########################################################################

sub B_undo_log_entry($) {
   
    my $undo_command = $_[0];
    my $undo_actions = &getGlobal('BFILE', "undo-actions");
    my @lines;
    

    if (! (-e $undo_actions)) {
	if (open UNDO_ACTIONS,">" . $undo_actions){ # create undo file
	    close UNDO_ACTIONS; # chown to root, rwx------
	    chmod 0700,$undo_actions;
	    chown 0,0,$undo_actions;
	}
	else { 
	    &ErrorLog ("Can not create undo-actions file");
	}
	
    }
	
    &B_open_plus (*UNDO_NEW, *UNDO_OLD, $undo_actions); 

    while (my $line=<UNDO_OLD>) { #copy file into @lines
	push (@lines,$line);
    }
    print UNDO_NEW $undo_command .  "\n";  #make the undo command first in the new file
    while (my $line = shift @lines) { #write the rest of the lines of the file
	print UNDO_NEW $line;
    }
    close UNDO_OLD;
    close UNDO_NEW;
    if (rename "${undo_actions}.bastille", $undo_actions) { #replace the old file with the new file we
	chmod 0700,$undo_actions;                # just made / mirrors B_close_plus logic
	chown 0,0,$undo_actions;
    } else {
	&ErrorLog("B_undo_log_entry: not able to move ${undo_actions}.bastille to ${undo_actions}!!! $!) !!!\n");
    }
}
    
#This function returns the requested Global Config value, and logs an error if its not found
sub getGlobalConfig ($$) {
  my $module = $_[0];
  my $key = $_[1];
  if (exists $GLOBAL_CONFIG{$module}{$key}) {
    return $GLOBAL_CONFIG{$module}{$key};
  } else {
    # This might flag some errors, but it prints out too many non-errors too, so it should
    # only be reviewed by a developer
    # &ErrorLog("Tried to use \$GLOBAL_CONFIG{$module}{$key} but it does not exist.\n");
    return undef;
  }
}

#This function returns the requested GLOBAL_* hash value, and logs an error if the 
# variable does not exist.

sub getGlobal ($$) {
  my $type = uc($_[0]);
  my $key = $_[1];

  # define a mapping from the first argument to the proper hash
  my %map = ("BIN"   => \%GLOBAL_BIN,
             "FILE"  => \%GLOBAL_FILE,
             "BFILE" => \%GLOBAL_BFILE,
             "DIR"   => \%GLOBAL_DIR,
             "BDIR"  => \%GLOBAL_BDIR,
            );

  # check to see if the desired key is in the desired hash
  if (exists $map{$type}->{$key}) {
    # get the value from the right hash with the key
    return $map{$type}->{$key};
  } else {
    # i.e. Bastille tried to use $GLOBAL_BIN{'cp'} but it does not exist.
    &ErrorLog("Bastille tried to use \$GLOBAL_${type}\{\'$key\'} but it does not exist.");
    return undef;
  }
}
###########################################################################
# &showDisclaimer:
# Print the disclaimer and wait for 2 minutes for acceptance
# Do NOT do so if any of the following conditions hold
# 1. the -d option was used
# 2. the file ~/.spc_disclaimer exists
###########################################################################

sub showDisclaimer {

# Get passwd information on the effective user
    my $nodisclaim = $_[0];
    my $nodisclaim_file = &getGlobal('BFILE', "nodisclaimer");
    my $response;
    my $WAIT_TIME = 120; # we'll wait for 2 minutes
    my $DISCLAIMER =
	"\n" .
	"BASTILLE DISCLAIMER:\n\n" .
        "Use of Bastille can help efficiently optimize\n" .
        "system security, but does not guarantee system security.\n" .
        "Information about security obtained through use of\n" .
        "Bastille is provided on an AS-IS basis only\n" .
        "and is subject to change without notice.  Hewlett-Packard,\n" .
        "Jay Beale, and the Bastille developers DISCLAIM ALL\n" .
        "WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION\n" .
        "THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR\n" .
        "A PARTICULAR PURPOSE.  Customer acknowledges that the customer\n" .
	"is responsible for their system's security. \n";
# once we prompt the user, we'll wait for $WAIT_TIME seconds for a response
# if no response is received, we exit nicely -- we catch the SIGALRM
    $SIG{'ALRM'} = sub {
	&ErrorLog(
		  "\n\nWaited for $WAIT_TIME seconds. No response received.\n" .
		  "Quitting." );
	print STDERR "\n\nWaited for $WAIT_TIME seconds. No response received.\n" .
	          "Quitting.";
	exit 1;
    };

# If the user has specified not to show the disclaimer, or
# the .spc_disclaimer file already exists, then return
    if( ( $nodisclaim ) || -e $nodisclaim_file ) { return 1; }

# otherwise, show the disclaimer
    print ( $DISCLAIMER );
    alarm $WAIT_TIME; # start alarm
    print("You must accept the terms of this disclaimer to use\n" .
	  "Bastille.  Type \"accept\" (without quotes) within 2\n" .
	  "minutes to accept the terms of the above disclaimer\n" .  "> " );

    chop( $response = <STDIN> ); # Script blocks on STDIN here
    alarm 0; # turn off alarm immediately after getting line
 
# there is a response
    if( lc( $response ) =~ "accept" ) {
	my $touch = &getGlobal('BIN', "touch");
	my $retVal = system("$touch $nodisclaim_file");
	if( $retVal != 0 ) {
	    &ErrorLog ( "Unable to touch $nodisclaim_file: $!" .
			"You must use Bastille\'s -d flag (for example:\n" .
			"bastille -i -d) or \'touch $nodisclaim_file \'\n" );
	} # if
	else {
	    print("This disclaimer will not appear again on this machine.\n" .
		  "To suppress the disclaimer on other machines, use Bastille\'s\n" .
		  "-d flag (example: bastille -i -d).\n");
	} # else
    } # outer if
    else { # something besides "accept" was typed
	print("You must accept the terms of the disclaimer before using\n" .
	      "Bastille.  Exiting.\n" );
	exit 0;
    } # else
} # showDisclaimer

1;





