/*
 * Copyright (c) 2000-2001 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Copyright (c) 1988-1990 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


#ifndef ArgusUtil
#define ArgusUtil
#endif

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>

#include <netinet/in.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>

#include <compat.h>
#include <pcap.h>
#include <pcap-int.h>

#include <interface.h>

#include <argus_util.h>
#include <argus_out.h>
#include <argus_parse.h>
#include <argus_filter.h>


int target_flags = 0;

void
ArgusAdjustGlobalTime (struct timeval *now)
{
   struct timeval ArgusTimeDelta;

   ArgusTimeDelta.tv_sec  = ArgusNowTime.tv_sec  - ArgusGlobalTime.tv_sec;
   ArgusTimeDelta.tv_usec = ArgusNowTime.tv_usec - ArgusGlobalTime.tv_usec;
 
   ArgusGlobalTime.tv_sec  = now->tv_sec  - ArgusTimeDelta.tv_sec;
   ArgusGlobalTime.tv_usec = now->tv_usec - ArgusTimeDelta.tv_usec;
 
   if (ArgusGlobalTime.tv_usec < 0) {
      ArgusGlobalTime.tv_sec--;
      ArgusGlobalTime.tv_usec += 1000000;
   } else {
      if (ArgusGlobalTime.tv_usec > 1000000) {
         ArgusGlobalTime.tv_sec++;
         ArgusGlobalTime.tv_usec -= 1000000;
      }
   }
   ArgusNowTime = *now;
}

char *ArgusVersionStr =  "Argus Version ";

int
ArgusConvertInitialWriteStruct (struct WriteStruct *ws, struct ArgusRecord *argus)
{
   int retn = 0;
   char *ptr;
   
   if (ws && argus) {
      bzero ((char *) argus, sizeof (*argus));
      if ((ptr = strstr (ws->ws_init.initString, ArgusVersionStr)) != NULL) {
         ptr = &ptr[strlen(ArgusVersionStr)];
         if (sscanf (ptr, "%d.%d", &major_version, &minor_version)) {
            argus->ahdr.type                 = ARGUS_MAR | ARGUS_WRITESTRUCT;
            argus->ahdr.length               = sizeof (*argus);
            argus->ahdr.length               = htons(argus->ahdr.length);
            argus->ahdr.cause                = ARGUS_START;
            argus->ahdr.status               = htonl(ARGUS_VERSION);
            argus->ahdr.argusid              = htonl(ARGUS_COOKIE);
            argus->argus_mar.startime        = ws->ws_init.startime;
            argus->argus_mar.now             = ws->ws_init.now;
            argus->argus_mar.major_version   = major_version;
            argus->argus_mar.minor_version   = minor_version;
            argus->argus_mar.interfaceType   = ws->ws_init.interfaceType;
            argus->argus_mar.interfaceStatus = ws->ws_init.interfaceStatus;

            argus->argus_mar.reportInterval  = ws->ws_init.reportInterval;
            argus->argus_mar.argusMrInterval = ws->ws_init.dflagInterval;
            argus->argus_mar.record_len      = -1;
            retn = 1;
         }
      }
   }

   return (retn);
}

#include <cons_def.h>

#if !defined(__OpenBSD__) || !defined(_NETINET_IP_ICMP_H_)
#include <netinet/ip_icmp.h>
#define _NETINET_IP_ICMP_H_
#endif

extern int ArgusTotalBytes;
extern int ArgusTotalCount;

extern long long ntohll (long long);

int
ArgusConvertWriteStruct (struct WriteStruct *ws, struct ArgusRecord *argus)
{
   int retn = 0;
   unsigned int status;

   if (ws && argus) {
      bzero ((char *) argus, sizeof (*argus));
      status = ntohl(ws->status);

      if (status & ARGUSCONTROL) {
         argus->ahdr.type                 = ARGUS_MAR | ARGUS_WRITESTRUCT;
         argus->ahdr.length               = sizeof (*argus);
         argus->ahdr.argusid              = 0;
         argus->ahdr.status               = ARGUS_VERSION;

         argus->argus_mar.major_version   = VERSION_MAJOR;
         argus->argus_mar.minor_version   = VERSION_MINOR;
         argus->argus_mar.interfaceType   = ws->ws_stat.interfaceType;
         argus->argus_mar.interfaceStatus = ws->ws_stat.interfaceStatus;
         argus->argus_mar.reportInterval  = ntohs(ws->ws_stat.reportInterval);
         argus->argus_mar.argusMrInterval = ntohs(ws->ws_stat.dflagInterval);

         if (status & CLOSE) {
            argus->ahdr.cause                 = ARGUS_STOP;
            argus->argus_mar.startime.tv_sec  = ntohl(ws->ws_stat.startime.tv_sec);
            argus->argus_mar.startime.tv_usec = ntohl(ws->ws_stat.startime.tv_usec);
            argus->argus_mar.now.tv_sec       = ntohl(ws->ws_stat.now.tv_sec);
            argus->argus_mar.now.tv_usec      = ntohl(ws->ws_stat.now.tv_usec);
            argus->argus_mar.pktsRcvd         = ArgusTotalCount;
            argus->argus_mar.bytesRcvd        = ArgusTotalBytes;
            argus->argus_mar.pktsDrop         = ntohl(ws->ws_stat.pktsDrop);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actTCPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actUDPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actIPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloTCPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloUDPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloIPcons);
            argus->argus_mar.actICMPcons      = ntohl(ws->ws_stat.actICMPcons);
            argus->argus_mar.cloICMPcons      = ntohl(ws->ws_stat.cloICMPcons);
            argus->argus_mar.actFRAGcons      = ntohl(ws->ws_stat.actFRAGcons);
            argus->argus_mar.cloFRAGcons      = ntohl(ws->ws_stat.cloFRAGcons);

         } else {
            argus->ahdr.cause                 = ARGUS_STATUS;
            argus->argus_mar.startime.tv_sec  = ntohl(ws->ws_stat.startime.tv_sec);
            argus->argus_mar.startime.tv_usec = ntohl(ws->ws_stat.startime.tv_usec);
            argus->argus_mar.now.tv_sec       = ntohl(ws->ws_stat.now.tv_sec);
            argus->argus_mar.now.tv_usec      = ntohl(ws->ws_stat.now.tv_usec);

            argus->argus_mar.pktsRcvd         = ntohl(ws->ws_stat.pktsRcvd);
            argus->argus_mar.bytesRcvd        = ntohl(ws->ws_stat.bytesRcvd);

            argus->argus_mar.pktsDrop         = ntohl(ws->ws_stat.pktsDrop);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actTCPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actUDPcons);
            argus->argus_mar.actIPcons       += ntohl(ws->ws_stat.actIPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloTCPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloUDPcons);
            argus->argus_mar.cloIPcons       += ntohl(ws->ws_stat.cloIPcons);
            argus->argus_mar.actICMPcons      = ntohl(ws->ws_stat.actICMPcons);
            argus->argus_mar.cloICMPcons      = ntohl(ws->ws_stat.cloICMPcons);
            argus->argus_mar.actFRAGcons      = ntohl(ws->ws_stat.actFRAGcons);
            argus->argus_mar.cloFRAGcons      = ntohl(ws->ws_stat.cloFRAGcons);
         }

         argus->argus_mar.record_len = -1;
         retn = 1;

      } else {
         struct ArgusMacStruct macbuffer, *mac = &macbuffer;
         unsigned int lasttime = 0;
         unsigned short difftime = 0, ttl = 0;

         argus->ahdr.type                     = ARGUS_FAR | ARGUS_WRITESTRUCT;
         argus->ahdr.length                   = sizeof (argus->ahdr) + sizeof (argus->argus_far);
         argus->ahdr.cause                    = ARGUS_STATUS;
         argus->ahdr.status                   = ARGUS_VERSION;
         argus->ahdr.status                  |= ETHERTYPE_IP;
         argus->ahdr.argusid                  = 0;
         argus->argus_far.type                = ARGUS_FAR;
         argus->argus_far.length              = sizeof(argus->argus_far);
         argus->argus_far.status              = 0;
         argus->argus_far.ArgusTransRefNum    = 0;

         argus->argus_far.flow.ip_flow.ip_src = ntohl(ws->ws_ip.src.s_addr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(ws->ws_ip.dst.s_addr);

         switch (status & (ICMPPROTO | TCPPROTO | UDPPROTO)) {
            case ICMPPROTO: {
               struct icmpWriteStruct *icmp = &ws->ws_ip_icmp;
               struct ArgusICMPFlow *icmpFlow = &argus->argus_far.flow.icmp_flow;

               argus->argus_far.flow.ip_flow.ip_p = IPPROTO_ICMP;

               argus->argus_far.src.count = ntohl(ws->ws_ip_udp.src_count);
               argus->argus_far.src.bytes = ntohl(ws->ws_ip_udp.src_bytes);
               argus->argus_far.dst.count = ntohl(ws->ws_ip_udp.dst_count);
               argus->argus_far.dst.bytes = ntohl(ws->ws_ip_udp.dst_bytes);

               if (!(status & (CON_ESTABLISHED | TIMED_OUT))) {
                  icmpFlow->type   = icmp->type;
                  icmpFlow->code   = icmp->code;
                  icmpFlow->id     = icmp->data;
                  icmpFlow->id     = ntohs(icmpFlow->id);

               } else {
                  icmpFlow->type   = ((unsigned char *) &ws->ws_ip_udp.src_bytes)[0];
                  icmpFlow->code   = ((unsigned char *) &ws->ws_ip_udp.src_bytes)[1];
                  icmpFlow->id     = ((unsigned short *)&ws->ws_ip_udp.src_bytes)[1];
                  icmpFlow->id     = ntohs(icmpFlow->id);

                  argus->argus_far.src.bytes = 0;
                  argus->argus_far.dst.bytes = 0;
               }

               if ((icmpFlow->type == ICMP_UNREACH) && (icmpFlow->code == ICMP_UNREACH_PORT))
                  argus->argus_far.flow.ip_flow.tp_p = ((char *)&ws->ws_ip_icmp.gwaddr.s_addr)[3];

               break;
            }

            case TCPPROTO: {
               struct ArgusTCPObject tcpbuffer, *tcp = &tcpbuffer;

               bzero ((char *) tcp, sizeof (*tcp));
               tcp->type   = ARGUS_TCP_DSR;
               tcp->length = sizeof(*tcp);

               if (status & SAW_SYN)
                  tcp->state |= ARGUS_SAW_SYN;
               if (status & SAW_SYN_SENT)
                  tcp->state |= ARGUS_SAW_SYN_SENT;
               if (status & CON_ESTABLISHED)
                  tcp->state |= ARGUS_CON_ESTABLISHED;
               if (status & SRC_RESET)
                  tcp->state |= ARGUS_SRC_RESET;
               if (status & DST_RESET)
                  tcp->state |= ARGUS_DST_RESET;
               if (status & CLOSE_WAITING)
                  tcp->state |= ARGUS_FIN;
               if (status & NORMAL_CLOSE)
                  tcp->state |= ARGUS_NORMAL_CLOSE;

               if (status & PKTS_RETRANS) {
                  if (status & SRC_PKTS_RETRANS)
                     tcp->state |= ARGUS_SRC_PKTS_RETRANS;
                  if (status & DST_PKTS_RETRANS)
                     tcp->state |= ARGUS_DST_PKTS_RETRANS;
               }

               argus->argus_far.src.count = ntohl(ws->ws_ip_tcp.src_count);
               argus->argus_far.dst.count = ntohl(ws->ws_ip_tcp.dst_count);

               if ((status & SAW_SYN) &&
                        !(status & (SAW_SYN_SENT|CON_ESTABLISHED|RESET))) {
                  tcp->src.seqbase = ntohl(ws->ws_ip_inittcp.seq);
               } else {
                  if ((status & (SAW_SYN | SAW_SYN_SENT)) &&
                                     !(status & (CON_ESTABLISHED))) {
                     tcp->dst.seqbase = ntohl(ws->ws_ip_inittcp.seq);
                  } else {
                     argus->argus_far.src.bytes = ntohl(ws->ws_ip_tcp.src_bytes);
                     argus->argus_far.dst.bytes = ntohl(ws->ws_ip_tcp.dst_bytes);
                  }
               }

               bcopy ((char *)tcp, &((char *)argus)[argus->ahdr.length], sizeof(*tcp));
               argus->ahdr.length += sizeof(*tcp);

               argus->argus_far.flow.ip_flow.ip_p = IPPROTO_TCP;
               argus->argus_far.flow.ip_flow.sport  = ntohs(ws->ws_ip.sport);
               argus->argus_far.flow.ip_flow.dport  = ntohs(ws->ws_ip.dport);
               break;
            }

            case UDPPROTO:
               argus->argus_far.flow.ip_flow.ip_p = IPPROTO_UDP;
               argus->argus_far.flow.ip_flow.sport  = ntohs(ws->ws_ip.sport);
               argus->argus_far.flow.ip_flow.dport  = ntohs(ws->ws_ip.dport);

            default:
               argus->argus_far.src.count = ntohl(ws->ws_ip_udp.src_count);
               argus->argus_far.src.bytes = ntohl(ws->ws_ip_udp.src_bytes);
               argus->argus_far.dst.count = ntohl(ws->ws_ip_udp.dst_count);
               argus->argus_far.dst.bytes = ntohl(ws->ws_ip_udp.dst_bytes);
               break;
         }

         if (status & TIMED_OUT)
            argus->ahdr.cause |= ARGUS_TIMEOUT;

         if (status & FRAGMENTS)
            argus->argus_far.status |= ARGUS_FRAGMENTS;

         if (status & IPOPTIONMASK) {
            argus->argus_far.status |= ARGUS_IPOPTIONS;
            if (status & SSRCROUTE) {
               argus->argus_far.attr_ip.soptions |= ARGUS_SSRCROUTE;
               argus->argus_far.attr_ip.doptions |= ARGUS_SSRCROUTE;
            }
            if (status & LSRCROUTE) {
               argus->argus_far.attr_ip.soptions |= ARGUS_LSRCROUTE;
               argus->argus_far.attr_ip.doptions |= ARGUS_LSRCROUTE;
            }
            if (status & TIMESTAMP) {
               argus->argus_far.attr_ip.soptions |= ARGUS_TIMESTAMP;
               argus->argus_far.attr_ip.doptions |= ARGUS_TIMESTAMP;
            }
            if (status & SECURITY) {
               argus->argus_far.attr_ip.soptions |= ARGUS_SECURITY;
               argus->argus_far.attr_ip.doptions |= ARGUS_SECURITY;
            }
            if (status & RECORDROUTE) {
               argus->argus_far.attr_ip.soptions |= ARGUS_RECORDROUTE;
               argus->argus_far.attr_ip.doptions |= ARGUS_RECORDROUTE;
            }
            if (status & SATNETID) {
               argus->argus_far.attr_ip.soptions |= ARGUS_SATNETID;
               argus->argus_far.attr_ip.doptions |= ARGUS_SATNETID;
            }
         }

         if (status & CON_ESTABLISHED)
            argus->ahdr.status |= ARGUS_CONNECTED;

         argus->argus_far.time.start.tv_sec       = ntohl(ws->ws_ip.startime.tv_sec);
         argus->argus_far.time.start.tv_usec      = ntohl(ws->ws_ip.startime.tv_usec);

         if ((major_version > 1) || (minor_version > 6)) {
            difftime = ntohs(((u_short *)&ws->ws_ip.lasttime.tv_sec)[1]);
            lasttime = ntohl(((arg_uint32)ws->ws_ip.startime.tv_sec)) + difftime;
            ttl = ((u_short *)&ws->ws_ip.lasttime.tv_sec)[0];
            ws->ws_ip.lasttime.tv_sec = lasttime;
         }

         argus->argus_far.time.last.tv_sec    = ws->ws_ip.lasttime.tv_sec;
         argus->argus_far.time.last.tv_usec   = ntohl(ws->ws_ip.lasttime.tv_usec);
         mac->type                            = ARGUS_MAC_DSR;
         mac->length                          = sizeof(*mac);
         bcopy((char *)&ws->ws_ip.ws_phys.ethersrc, (char *)&mac->phys_union.ether.ethersrc, 6);
         bcopy((char *)&ws->ws_ip.ws_phys.etherdst, (char *)&mac->phys_union.ether.etherdst, 6);
 
         bcopy ((char *)mac, &((char *)argus)[argus->ahdr.length], sizeof(*mac));
         argus->ahdr.length += sizeof(*mac);
         retn = 1;
      }

#ifdef _LITTLE_ENDIAN
      ArgusHtoN(argus);
#endif
   }

   return (retn);
}


extern char *ArgusProgramName;


void
print_date(struct ArgusRecord *argus, char *date)
{
   struct ArgusAGRStruct *ArgusThisAgr = NULL;
   struct timeval *start = NULL, *last = NULL;
   struct timeval  buf,  *time = &buf;
   struct timeval zbuf, *ztime = &zbuf;
   struct tm *tm = NULL;
   char *sptr, *iptr, delim = ' ';

   bzero ((char *)ztime, sizeof(zbuf));

   if (argus->ahdr.type & ARGUS_MAR) {
      start = &argus->argus_mar.startime;
      last  = &argus->argus_mar.now;

   } else {
      start = &argus->argus_far.time.start;
      last  = &argus->argus_far.time.last;
   }
   
   if (lflag && !(Gflag))
      *time = *last;
   else
      *time = *start;

   tm = localtime ((time_t *)&time->tv_sec);

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;

   sprintf (date, "                                                   ");
   sprintf (date, "%s%c", print_time(time), delim);

   if (gflag) {
      ArgusThisAgr = (struct ArgusAGRStruct *)ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX];
      if (Hflag && (ArgusThisAgr && (ArgusThisAgr->type == ARGUS_AGR_DSR))) {
         int ArgusThisMultiplier = 1000;

         if (ArgusThisAgr->status & ARGUS_AGR_USECACTTIME)
            ArgusThisMultiplier = 1000000;

         time->tv_sec  = ArgusThisAgr->act.mean / ArgusThisMultiplier;
         time->tv_usec = ArgusThisAgr->act.mean % ArgusThisMultiplier;
         
      } else {
         *time = *last;
         time->tv_sec  -= start->tv_sec;
         time->tv_usec -= start->tv_usec;
 
         if (time->tv_usec < 0) {
            time->tv_sec--; 
            time->tv_usec += 1000000;
         }

         if (time->tv_usec >= 1000000) {
            time->tv_sec++; 
            time->tv_usec -= 1000000;
         }
      }

      iptr = &date[strlen(date)];
      sptr = &date[strlen(date)];

      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
         sprintf(sptr, "%u", (unsigned int) time->tv_sec);
      else
         sprintf(sptr, "%8u", (unsigned int) time->tv_sec);

      if (pflag) {
         sptr = &date[strlen(date)];
         sprintf(sptr, ".%06d", (int) time->tv_usec);
         sptr[pflag + 1] = '\0';
      }

      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
         sprintf(&date[strlen(date)], "%c", delim);

   } else

      if (Gflag) {
         if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
            sprintf(&date[strlen(date)], "%s%c", print_time(last), RaFieldDelimiter);
         else
            sprintf(&date[strlen(date)], " %s", print_time(last));
      }
}

void ArgusGetIndicatorString (struct ArgusRecord *, char *);


void
ArgusGetIndicatorString (struct ArgusRecord *argus, char *buf)
{
   char *ptr = buf;

   bzero (buf, 16);

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      if (argus->ahdr.type & ARGUS_MAR) {

      } else {
         if (argus->ahdr.status & ARGUS_MPLS)
            *ptr++ = 'm';

         if (argus->ahdr.status & ARGUS_PPPoE)
            *ptr++ = 'p';

         if (argus->ahdr.status & ARGUS_VLAN)
            *ptr++ = 'q';

         if ((argus->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
            if (argus->argus_far.status & ARGUS_ICMP_MAPPED)
               *ptr++ = 'I';

            if ((argus->argus_far.attr_ip.soptions & ARGUS_FRAGMENTS) ||
                (argus->argus_far.attr_ip.doptions & ARGUS_FRAGMENTS)) {

               if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_FRAG_FLOWTAG)
                  *ptr++ = 'f';
               else
                  *ptr++ = 'F';

               if (argus->argus_far.attr_ip.soptions & ARGUS_FRAGOVERLAP)
                  *ptr++ = 'V';
            }

            if (argus->ahdr.status & ARGUS_MULTIADDR)
               *ptr++ = 'M';

            if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
               struct ArgusTCPObject *tcp = NULL;
               unsigned int status;

               tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];
               if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                  if (status) {
                     if (status & ARGUS_PKTS_RETRANS) {
                        if ((status & ARGUS_SRC_PKTS_RETRANS) && (status & ARGUS_DST_PKTS_RETRANS))
                           *ptr++ = '*';
                        else {
                           if (status & ARGUS_SRC_PKTS_RETRANS)
                              *ptr++ = 's';
                           if (status & ARGUS_DST_PKTS_RETRANS)
                              *ptr++ = 'd';
                        }
                     }
                     if (status & ARGUS_WINDOW_SHUT) {
                        if ((status & ARGUS_SRC_WINDOW_SHUT) && (status & ARGUS_DST_WINDOW_SHUT))
                           *ptr++ = '@';
                        else {
                           if (status & ARGUS_SRC_WINDOW_SHUT)
                              *ptr++ = 'S';
                           if (status & ARGUS_DST_WINDOW_SHUT)
                              *ptr++ = 'D';
                        }
                     }
                     if (status & ARGUS_ECN_CONGESTED)
                        *ptr++ = 'E';
                  }
               }
            }
            if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS) {
               struct ArgusESPStruct *esp = NULL;

               if ((esp = (struct ArgusESPStruct *)ArgusThisFarHdrs[ARGUS_ESP_DSR_INDEX]) != NULL) {
                  if ((esp->src.lostseq > 0) && (esp->dst.lostseq > 0))
                     *ptr++ = '*';
                  else {
                     if (esp->src.lostseq > 0)
                        *ptr++ = 's';
                     if (esp->dst.lostseq > 0)
                        *ptr++ = 'd';
                  }
               }
            }
         }
      }

      *ptr = RaFieldDelimiter;

   } else {
      int encdone = 0;

      if (Iflag) {
         bcopy ("          ", buf, 9);
         if (argus->ahdr.type & ARGUS_MAR) {
         } else {
            if (argus->ahdr.status & ARGUS_MPLS) {
               buf[1] = 'm';
               encdone++;
            }

            if (argus->ahdr.status & ARGUS_PPPoE) {
               buf[1] = 'p';
               encdone++;
            }

            if (argus->ahdr.status & ARGUS_VLAN) {
               buf[1] = 'q';
               encdone++;
            }

            if (encdone > 1)
               buf[1] = 'E';

            if ((argus->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
               if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
                  struct ArgusTCPObject *tcp = NULL;
                  unsigned int status;

                  tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

                  if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                     if (status) {
                        if (status & ARGUS_WINDOW_SHUT) {
                           if ((status & ARGUS_SRC_WINDOW_SHUT) && (status & ARGUS_DST_WINDOW_SHUT))
                              buf[3] = '@';
                           else {
                              if (status & ARGUS_SRC_WINDOW_SHUT)
                                 buf[3] = 'S';
                              if (status & ARGUS_DST_WINDOW_SHUT)
                                 buf[3] = 'D';
                           }
                        }
                        if (status & ARGUS_PKTS_RETRANS) {
                           if ((status & ARGUS_SRC_PKTS_RETRANS) && (status & ARGUS_DST_PKTS_RETRANS))
                              buf[2] =  '*';
                           else {
                              if (status & ARGUS_SRC_PKTS_RETRANS)
                                 buf[2] = 's';
                              if (status & ARGUS_DST_PKTS_RETRANS)
                                 buf[2] = 'd';
                           }
                        }
                        if (status & ARGUS_ECN_CONGESTED) {
                           if ((status & ARGUS_SRC_CONGESTED) && (status & ARGUS_DST_CONGESTED))
                              buf[3] = 'E';
                           else {
                              if (status & ARGUS_SRC_CONGESTED)
                                 buf[3] = 'e';
                              if (status & ARGUS_DST_CONGESTED)
                                 buf[3] = 'e';
                           }
                        }
                     }
                  }
               }

               if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS) {
                  struct ArgusESPStruct *esp = NULL;

                  if ((esp = (struct ArgusESPStruct *)ArgusThisFarHdrs[ARGUS_ESP_DSR_INDEX]) != NULL) {
                     if ((esp->src.lostseq > 0) && (esp->dst.lostseq > 0))
                        buf[2] = '*';
                     else {
                        if (esp->src.lostseq > 0)
                           buf[2] = 's';
                        if (esp->dst.lostseq > 0)
                           buf[2] = 'd';
                     }
                  }
               }

               if (argus->ahdr.status & ARGUS_MULTIADDR)
                  buf[4] = 'M';

               if (argus->argus_far.status & ARGUS_ICMP_MAPPED)
                  buf[5] = 'I';

               if ((argus->argus_far.attr_ip.soptions & ARGUS_FRAGMENTS) ||
                   (argus->argus_far.attr_ip.doptions & ARGUS_FRAGMENTS)) {

                  if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_FRAG_FLOWTAG)
                     buf[6] = 'f';
                  else
                     buf[6] = 'F';

                  if ((argus->argus_far.attr_ip.soptions & ARGUS_FRAGOVERLAP) ||
                       (argus->argus_far.attr_ip.doptions & ARGUS_FRAGOVERLAP)) {
                     buf[6] = 'V';
                  }
               }

               if ((argus->argus_far.attr_ip.soptions & ARGUS_IPOPTIONS) ||
                   (argus->argus_far.attr_ip.doptions & ARGUS_IPOPTIONS)) {
                  switch ((argus->argus_far.attr_ip.soptions | argus->argus_far.attr_ip.doptions)
                                       & ARGUS_IPOPTIONS) {
                     case SSRCROUTE:    buf[7] = 'S'; break;
                     case LSRCROUTE:    buf[7] = 'L'; break;
                     case TIMESTAMP:    buf[7] = 'T'; break;
                     case SECURITY:     buf[7] = '+'; break;
                     case RECORDROUTE:  buf[7] = 'R'; break;
                     case SATNETID:     buf[7] = 'N'; break;
                     case IPOPTIONMASK: buf[7] = 'E'; break;
                     default:           buf[7] = 'O'; break;
                  }
               }
            }
         }
      }
   }

   return;
}


char RaUserDataStr[MAXSTRLEN];

char *
RaGetUserDataString (struct ArgusRecord *argus)
{
   char *retn = RaUserDataStr;
   char strbuf[MAXSTRLEN], *str = strbuf;
   char delim = ' ';
   int len = 0;

   bzero (RaUserDataStr, MAXSTRLEN);

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;

   if (ArgusSrcUserDataLen > 0) {
      len = 0;
      if (ArgusThisFarStatus & ARGUS_SRCUSRDATA_DSR_STATUS) {
         struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_SRCUSRDATA_DSR_INDEX];

         len = (user->length - 1) * 4;
         len = (len < argus->argus_far.src.appbytes) ? len : argus->argus_far.src.appbytes;
         len = len > ArgusSrcUserDataLen ? ArgusSrcUserDataLen : len;
   
         if ((len = ArgusEncode (&user->data, len, str, sizeof(strbuf))) != 0)
            sprintf (RaUserDataStr, "%cs[%d]=%s", delim, len, str);
   
      } else
         if (delim != ' ')
            sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%c", delim);

      if (delim == ' ')
         sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%*s", (ArgusSrcUserDataLen - len) + 1, " ");
   }


   if (ArgusDstUserDataLen > 0) {
      len = 0;
      if (ArgusThisFarStatus & ARGUS_DSTUSRDATA_DSR_STATUS) {
         struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_DSTUSRDATA_DSR_INDEX];

         len = (user->length - 1) * 4;
         len = (len < argus->argus_far.dst.appbytes) ? len : argus->argus_far.dst.appbytes;
         len = len > ArgusDstUserDataLen ? ArgusDstUserDataLen : len;
   
         if ((len = ArgusEncode (&user->data, len, str, sizeof(strbuf))) != 0)
            sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%cd[%d]=%s", delim, len, str);
      } else
         if (delim != ' ')
            sprintf (&RaUserDataStr[strlen(RaUserDataStr)], "%c", delim);
   }

   return (retn);
}

extern void ArgusLog (int, char *, ...);


static char basis_64[] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????";


int
ArgusEncode (const char *ptr, int len, char *str, int slen)
{
   int retn = 0;

   switch (eflag) {
      case ARGUS_ENCODE_64:
         sprintf (str, "\"");
         retn = ArgusEncode64(ptr, len, &str[strlen(str)], slen - strlen(str));
         strcat (str, "\"");
         break;

      case ARGUS_ENCODE_ASCII:
         sprintf (str, "\"");
         retn = ArgusEncodeAscii(ptr, len, &str[strlen(str)], slen - strlen(str));
         strcat (str, "\"");
         break;

      default:
         ArgusLog (LOG_ERR, "ArgusEncode: error encode method %d unknown", eflag);
         break;
   }

   return (retn);
}

int
ArgusEncode64 (const char *ptr, int len, char *str, int slen)
{
   int retn = 0;
   const unsigned char *in = (const unsigned char *)ptr;
   unsigned char *buf = (unsigned char *) str;
   unsigned char oval;
   unsigned newlen;

   if (ptr && ((newlen = (len + 2) / 3 * 4) < slen)) {
      while (len >= 3) {
          *buf++ = basis_64[in[0] >> 2];
          *buf++ = basis_64[((in[0] << 4) & 0x30) | (in[1] >> 4)];
          *buf++ = basis_64[((in[1] << 2) & 0x3c) | (in[2] >> 6)];
          *buf++ = basis_64[in[2] & 0x3f];
          in += 3;
          len -= 3;
      }
      if (len > 0) {
          *buf++ = basis_64[in[0] >> 2];
          oval = (in[0] << 4) & 0x30;
          if (len > 1) oval |= in[1] >> 4;
          *buf++ = basis_64[oval];
          *buf++ = (len < 2) ? '=' : basis_64[(in[1] << 2) & 0x3c];
          *buf++ = '=';
      }

      if (newlen < slen)
        *buf = '\0';

      retn = newlen;
   }
   
   return (retn);
}

#include <ctype.h>

int
ArgusEncodeAscii (const char *ptr, int len, char *str, int slen)
{
   int retn = 0, newlen = len;
   unsigned char *buf = (unsigned char *) str;

   if (ptr && (len < slen)) {
      while (len > 0) {
         if (isprint((int)*ptr))
            *buf = *ptr;
         else
            *buf = '.';
         buf++;
         ptr++;
         len--;
      }

      if (len < slen)
        *buf = '\0';

      retn = newlen;
   }
   
   return (retn);
}



struct ArgusInterfaceStruct interfacetypes [] = {
{  0, "DLT_NULL", "no link-layer encapsulation"},
{  1, "DLT_EN10MB", "Ethernet (10Mb)"},
{  2, "DLT_EN3MB", "Experimental Ethernet (3Mb)"},
{  3, "DLT_AX25", "Amateur Radio AX.25"},
{  4, "DLT_PRONET", "Proteon ProNET Token Ring"},
{  5, "DLT_CHAOS", "Chaos"},
{  6, "DLT_IEEE802", "IEEE 802 Networks"},
{  7, "DLT_ARCNET", "ARCNET"},
{  8, "DLT_SLIP", "Serial Line IP"},
{  9, "DLT_PPP",  "Point-to-point Protocol"},
{ 10,"DLT_FDDI", "FDDI"},
{ 11, "DLT_ATM_RFC1483", "LLC/SNAP encapsulated atm"},
{ 12, "DLT_LOOP", "loopback"},
{100, "DLT_ATM_RFC1483", "LLC/SNAP encapsulated atm"},
{101, "DLT_RAW", "raw IP"},
{102, "DLT_SLIP_BSDOS", "BSD/OS Serial Line IP"},
{103, "DLT_PPP_BSDOS", "BSD/OS Point-to-point Protocol"},
{104, "DLT_CHDLC", "Cisco HDLC"},
{-1, "Undefined", "Undefined"},
};


char argus_strbuf[MAXSTRLEN];

char *
get_man_string (struct ArgusRecord *ptr)
{
   char protoStr[128], argusIDStrBuf[32], *argusIDStr = argusIDStrBuf;
   char probeIDStrBuf[128], *probeIDStr = probeIDStrBuf;
   char versionStrBuf[128];
   char date [128], fmtstr[MAXSTRLEN], indStr[16];

   bzero (argus_strbuf, MAXSTRLEN);
   bzero (fmtstr, MAXSTRLEN);

   print_date(ptr, date);

   if (mflag) {
      struct ArgusInterfaceStruct *interface = &interfacetypes[0];
      while (interface->value >= 0) {
         if (ptr->argus_mar.interfaceType == interface->value)
            break;
         interface++;
      }
      sprintf(protoStr, " man     InterfaceType     %-*.*s", hfield, hfield, interface->label);
   
   } else
      sprintf(protoStr, "man");

   if (ptr->ahdr.status & ARGUS_ID_IS_IPADDR)
      argusIDStr = strdup (ipaddr_string (&ptr->argus_mar.argusid));
   else
      sprintf (argusIDStr, "%u", ptr->argus_mar.argusid);

   if (idflag)
      sprintf(argus_strbuf, "%-15.15s ", argusIDStr);

   if (!cflag && ((ptr->ahdr.type & ARGUS_RMON) && (ptr->ahdr.status & ARGUS_TOPN)))
      sprintf (probeIDStr, "      ");
   else
      sprintf (probeIDStr, "probeid=%-*.*s %*s", hfield, hfield, argusIDStr, pfield, " ");

   sprintf (versionStrBuf, "version=%d.%d", ptr->argus_mar.major_version,
                                            ptr->argus_mar.minor_version);

   if (ptr->ahdr.cause & ARGUS_START) {

      if (Iflag)
         strcpy (fmtstr, "%s%s%4s %-*.*s %s%*s");
      else
         strcpy (fmtstr, "%s  %4s %-*.*s %s%*s");

      if (cflag) {
         if ((ptr->ahdr.type & ARGUS_RMON) && (ptr->ahdr.status & ARGUS_TOPN))
            strcat (fmtstr, "                   ");
         else
            strcat (fmtstr, "                                            ");
      }

   } else {
      if (Iflag)
#if defined(HAVE_SOLARIS) || defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
         strcpy (fmtstr, "%s%s%4s  pkts %9lld  bytes %12lld  drops %5d  ");
      else
         strcpy (fmtstr, "%s  %4s  pkts %9lld  bytes %12lld  drops %5d  ");
#else
         strcpy (fmtstr, "%s%s%4s  pkts %9Ld  bytes %12Ld  drops %5d  ");
      else
         strcpy (fmtstr, "%s  %4s  pkts %9Ld  bytes %12Ld  drops %5d  ");
#endif
      if (cflag)
         strcat (fmtstr, "flows    %-8u  closed       %-8u    ");
   }

   if (ptr->ahdr.cause & ARGUS_START)              strcat (fmtstr, "STA");
   else if (ptr->ahdr.cause & ARGUS_STATUS)        strcat (fmtstr, "CON");
   else if (ptr->ahdr.cause & ARGUS_STOP)          strcat (fmtstr, "STP");
   else if (ptr->ahdr.cause & ARGUS_SHUTDOWN)      strcat (fmtstr, "SHT");
   else if (ptr->ahdr.cause & ARGUS_ERROR)         strcat (fmtstr, "ERR");
   else if (ptr->ahdr.cause & ARGUS_MAXLISTENEXCD) strcat (fmtstr, "MAX");

   ArgusGetIndicatorString (ptr, indStr);

   if (ptr->ahdr.cause & ARGUS_START) {
      if (Iflag) {
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr, hfield, hfield, versionStrBuf, probeIDStr, pfield, " ");
      } else {
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date,  protoStr, hfield, hfield, versionStrBuf, probeIDStr, pfield, " ");
      }
   } else {
      if (Iflag) {
         if (cflag) {
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  ptr->argus_mar.pktsRcvd, ptr->argus_mar.bytesRcvd, ptr->argus_mar.pktsDrop,
                  ptr->argus_mar.flows, ptr->argus_mar.flowsClosed);
         } else {
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  ptr->argus_mar.pktsRcvd, ptr->argus_mar.bytesRcvd, ptr->argus_mar.pktsDrop);
         }
      } else {
         if (cflag) {
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  ptr->argus_mar.pktsRcvd, ptr->argus_mar.bytesRcvd, ptr->argus_mar.pktsDrop,
                  ptr->argus_mar.flows, ptr->argus_mar.flowsClosed);
         } else {
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  ptr->argus_mar.pktsRcvd, ptr->argus_mar.bytesRcvd, ptr->argus_mar.pktsDrop);
         }
      }
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      char tmpbuf[128], *ptr = tmpbuf, *str = argus_strbuf, lastchr = ' ';
      int len = strlen(date) - 1;
      bzero (tmpbuf, sizeof(tmpbuf));

      bcopy (str, ptr, len);
      str += len;
      ptr += len;

      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }
      bzero (argus_strbuf, MAXSTRLEN);
      bcopy (tmpbuf, argus_strbuf, strlen(tmpbuf));
   }

   return (argus_strbuf);
}


char *ArgusTCPFlags [] = {
   "F", "S", "R", "P", "A", "U", "7", "8"
};

char *
get_tcp_string (argus)
struct ArgusRecord *argus;
{
   struct ArgusFlow *flow;

   unsigned int status, rev = 0;
   int vc = 0, ahdrlen, farhdrlen;
   char *processStr = NULL;
   char statusbuf[MAXSTRLEN], *TCPStatusString = statusbuf;
   char SrcTCPFlagsStr[16], DstTCPFlagsStr[16], delim;
   char argusIDStrBuf[32], *argusIDStr = argusIDStrBuf;
   char *edstString = NULL, *esrcString = NULL;
   char dstString[256], srcString[256];
   char  *protoStr, indStr[16], *blankStr = "                      ";
   char date[128], fmtstr[MAXSTRLEN], protoStrargus_strbuf[16], portbuf[16];
   int src_count = 0, dst_count = 0, src_bytes = 0, dst_bytes = 0;
   u_char proto;
   struct ArgusTCPObject *tcp = NULL;
   
   bzero (protoStrargus_strbuf, 16);
   bzero (argus_strbuf, MAXSTRLEN);
   bzero (statusbuf, MAXSTRLEN);
   bzero (fmtstr, MAXSTRLEN);
   bzero (SrcTCPFlagsStr, 16);
   bzero (DstTCPFlagsStr, 16);
   bzero (argusIDStrBuf, 32);
   bzero (dstString, 256);
   bzero (srcString, 256);
   bzero (portbuf, 16);
   bzero (indStr, 16);
   bzero (date, 128);

   flow = &argus->argus_far.flow;

   if (mflag) {
      if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
         struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];

         esrcString = etheraddr_string ((u_char *)&mac->phys_union.ether.ethersrc);
         edstString = etheraddr_string ((u_char *)&mac->phys_union.ether.etherdst);

         sprintf (srcString, "%17.17s %17.17s   %*.*s",
                      esrcString, edstString, hfield, hfield, ipaddr_string (&flow->ip_flow.ip_src));
      } else
         sprintf (srcString, "%17.17s %17.17s   %*.*s",
                      blankStr, blankStr, hfield, hfield, ipaddr_string (&flow->ip_flow.ip_src));

   } else
      sprintf (srcString, "%*.*s", hfield, hfield, ipaddr_string (&flow->ip_flow.ip_src));

   sprintf (dstString, "%*.*s", hfield, hfield, ipaddr_string (&flow->ip_flow.ip_dst));

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;
   else
      delim = '.';

   if (!((flow->ip_flow.sport == 0xFFFF) && (argus->ahdr.status & ARGUS_MERGED)))
      sprintf (portbuf, "%c%-*.*s", delim, pfield, pfield, tcpport_string(flow->ip_flow.sport));
   else
      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
         sprintf (portbuf, "%c%s%c", RaFieldDelimiter, "*", RaFieldDelimiter);
      } else
         sprintf (portbuf, "%*s ", pfield, "*");

   strcat (srcString, portbuf);

   if (!((flow->ip_flow.dport == 0xFFFF) && (argus->ahdr.status & ARGUS_MERGED)))
      sprintf (portbuf, "%c%-*.*s", delim, pfield, pfield, tcpport_string(flow->ip_flow.dport));
   else
      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
         sprintf (portbuf, "%c%s%c", RaFieldDelimiter, "*", RaFieldDelimiter);
      } else
         sprintf (portbuf, "%*s ", pfield, "*");

   strcat (dstString, portbuf);

   print_date (argus, date);

   if (idflag) {
      if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_ID_IS_IPADDR)
         argusIDStr = strdup (ipaddr_string (&argus->ahdr.argusid));
      else
         sprintf (argusIDStr, "%u", argus->ahdr.argusid);

      sprintf(argus_strbuf, "%-15.15s ", argusIDStr);
   }
 
   if ((argus->ahdr.type & ARGUS_RMON) && (flow->ip_flow.ip_dst == 0)) {
      if (Iflag) {
         strcpy (fmtstr, "%s%s%4s %s  - ");
      } else
         strcpy (fmtstr, "%s  %4s %s  - ");
   } else {
      if (Iflag) {
         strcpy (fmtstr, "%s%s%4s %s  -    %s ");
      } else
         strcpy (fmtstr, "%s  %4s %s  -    %s ");
   }

   ahdrlen   = sizeof(argus->ahdr);
   farhdrlen = sizeof(argus->argus_far);

   if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
      tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

   src_count = argus->argus_far.src.count;
   dst_count = argus->argus_far.dst.count;

   if (Aflag) {
      src_bytes = argus->argus_far.src.appbytes;
      dst_bytes = argus->argus_far.dst.appbytes;
   } else {
      src_bytes = argus->argus_far.src.bytes;
      dst_bytes = argus->argus_far.dst.bytes;
   }

   if ((tcp != NULL) && ((status = tcp->state) != 0)) {
      if (!(status & (ARGUS_SAW_SYN | ARGUS_SAW_SYN_SENT))) {
         fmtstr[12 + vc] = '?';
         if (src_count) fmtstr[13 + vc] = '>';
         if (dst_count) fmtstr[11 + vc] = '<';
      }
      
      if (Rflag && (status & ARGUS_RESET)) {
         if (status & ARGUS_DST_RESET) {
            fmtstr[11 + vc] = '<';
            fmtstr[13 + vc] = ' ';
         }
         if (status & ARGUS_SRC_RESET) {
            fmtstr[11 + vc] = ' ';
            fmtstr[13 + vc] = '>';
         }
         fmtstr[12 + vc] = '|';

      } else 
      if (status & ARGUS_RESET) {
         fmtstr[11 + vc] = ' '; fmtstr[13 + vc] = '>';
         processStr = process_state_strings[5];
      } else
      if (status & ARGUS_NORMAL_CLOSE) {
         fmtstr[11 + vc] = ' '; fmtstr[13 + vc] = '>';
         processStr = process_state_strings[3];
      } else
      if (status & (ARGUS_FIN | ARGUS_FIN_ACK)) {
         fmtstr[11 + vc] = ' '; fmtstr[13 + vc] = '>';
         processStr = process_state_strings[6];
      } else
      if (argus->ahdr.cause & ARGUS_TIMEOUT) {
         if (src_count) fmtstr[13 + vc] = '>';
         if (dst_count) fmtstr[11 + vc] = '<';
         processStr = process_state_strings[4];
      } else
      if (status & ARGUS_CON_ESTABLISHED) {
         fmtstr[11 + vc] = ' '; fmtstr[13 + vc] = '>';
         processStr = process_state_strings[2];
      } else
      if (status & ARGUS_SAW_SYN_SENT) {
         fmtstr[11 + vc] = '<';
         fmtstr[13 + vc] = ' ';
         processStr = process_state_strings[1];
      } else
      if (status & ARGUS_SAW_SYN) {
         fmtstr[13 + vc] = '>';
         fmtstr[11 + vc] = ' ';
         processStr = process_state_strings[0];
      }

      if (status & ARGUS_RESET)
         processStr = process_state_strings[5];

      if (zflag) {
         bzero ((char *)TCPStatusString, sizeof(statusbuf));
         if (status & ARGUS_SAW_SYN)         strcat (TCPStatusString, "s");
         if (status & ARGUS_SAW_SYN_SENT)    strcat (TCPStatusString, "S");
         if (status & ARGUS_CON_ESTABLISHED) strcat (TCPStatusString, "E");
         if (status & ARGUS_FIN)             strcat (TCPStatusString, "f");
         if (status & ARGUS_FIN_ACK)         strcat (TCPStatusString, "F");
         if (status & ARGUS_NORMAL_CLOSE)    strcat (TCPStatusString, "C");
         if (status & ARGUS_RESET)           strcat (TCPStatusString, "R");
     } else

     if (Zflag) {
        int i, index;
        bzero(SrcTCPFlagsStr, sizeof(SrcTCPFlagsStr));
        bzero(DstTCPFlagsStr, sizeof(DstTCPFlagsStr));

        for (i = 0, index = 1; i < 8; i++) {
           if (tcp->src.flags & index) {
              strcat (SrcTCPFlagsStr, ArgusTCPFlags[i]);
           }
           if (tcp->dst.flags & index) {
              strcat (DstTCPFlagsStr, ArgusTCPFlags[i]);
           }
           index <<= 1;
        }
        switch (Zflag) {
           case 'b':
              sprintf(TCPStatusString, "%s_%s", SrcTCPFlagsStr, DstTCPFlagsStr);
              break;
           case 's':
              sprintf(TCPStatusString, "%s", SrcTCPFlagsStr);
              break;
           case 'd':
              sprintf(TCPStatusString, "%s", DstTCPFlagsStr);
              break;
        }
     }
   } else {
      if (argus->ahdr.cause & ARGUS_START)    processStr = "REQ";
      if (argus->ahdr.cause & ARGUS_STATUS)   processStr = "CON";
      if (argus->ahdr.cause & ARGUS_STOP)     processStr = "CLO";
      if (argus->ahdr.cause & ARGUS_TIMEOUT)  processStr = "TIM";
      if (argus->ahdr.cause & ARGUS_ERROR)    processStr = "ERR";

      if (src_count) fmtstr[13 + vc] = '>';
      if (dst_count) fmtstr[11 + vc] = '<';
      fmtstr[12 + vc] = '?';

      if (argus->ahdr.type & ARGUS_CISCO_NETFLOW) {
         if (argus->argus_far.flow.ip_flow.sport < argus->argus_far.flow.ip_flow.dport)
           rev++;
      }
   }

// if (argus->ahdr.cause & ARGUS_SHUTDOWN)
//    processStr = "SHT";

   if (rev) {
      int flag = 0;
      char tmpString[256];
      bcopy (srcString, tmpString, 256);
      bcopy (dstString, srcString, 256);
      bcopy (tmpString, dstString, 256);

      src_count = argus->argus_far.dst.count;
      dst_count = argus->argus_far.src.count;

      if (Aflag) {
         src_bytes = argus->argus_far.dst.appbytes;
         dst_bytes = argus->argus_far.src.appbytes;
      } else {
         src_bytes = argus->argus_far.dst.bytes;
         dst_bytes = argus->argus_far.src.bytes;
      }

      if (fmtstr[13 + vc] == '>')
         flag++;
      
      if (fmtstr[11 + vc] == '<')
         fmtstr[13 + vc] = '>';
      else
         fmtstr[13 + vc] = ' ';

      if (flag)
         fmtstr[11 + vc] = '<';
      else
         fmtstr[11 + vc] = ' ';

   }

   if (cflag)
      strcat (fmtstr, "%-8u %-8u  %-12u %-12u");

   if (processStr == NULL) processStr = "UNK";

   if (zflag || Zflag)
      processStr = TCPStatusString;

   strcat (fmtstr, processStr);

   proto = flow->ip_flow.ip_p;

   sprintf (protoStrargus_strbuf, "%u", proto);

   protoStr = (nflag > 1) ? protoStrargus_strbuf :
             proto >= IPPROTOSTR ? "unas" : ip_proto_string[proto];

   ArgusGetIndicatorString (argus, indStr);

   if ((argus->ahdr.type & ARGUS_RMON) && (flow->ip_flow.ip_dst == 0)) {
      fmtstr[11 + vc] = ' ';
      fmtstr[12 + vc] = ' ';
      fmtstr[13 + vc] = ' ';

      if (Iflag) {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString);
      } else {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString);
      }
   } else {
      if (Iflag) {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, dstString, 
                  src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, dstString);
      } else {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, dstString, 
                  src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, dstString);
      }
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      char tmpbuf[128], *ptr = tmpbuf, *str = argus_strbuf, lastchr = ' ';
      int len = strlen(date) - 1;
      bzero (tmpbuf, sizeof(tmpbuf));

      bcopy (str, ptr, len);
      str += len;
      ptr += len;

      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }
      bzero (argus_strbuf, MAXSTRLEN);
      bcopy (tmpbuf, argus_strbuf, strlen(tmpbuf));
   }

   return (argus_strbuf);
}



char *
get_icmp_string (ptr)
struct ArgusRecord *ptr;
{
   int vc = 0, rev = 0, srccnt = 0, dstcnt = 0;
   int ahdrlen, farhdrlen;
   struct ArgusICMPObject *icmp = NULL;
   char fmtstr[MAXSTRLEN], icmptype[32];
   char extendedstring[32], *blankStr = "                   ";
   char *edstString = NULL, *esrcString = NULL, *protoStr, indStr[16];
   char argusIDStrBuf[32], *argusIDStr = argusIDStrBuf;
   char dstString[256], srcString[256];
   char date[128];

   struct ArgusICMPFlow *icmpFlow = &ptr->argus_far.flow.icmp_flow;
   unsigned char ra_icmp_type = 0, ra_icmp_code = 0;
   unsigned short ra_icmp_data = 0;
   unsigned int  ra_src_addr = 0, ra_dst_addr = 0, ra_gw_addr = 0;
      
   bzero (extendedstring, 32);
   bzero (argusIDStrBuf, 32);
   bzero (fmtstr, MAXSTRLEN);
   bzero (srcString, 256);
   bzero (dstString, 256);
   bzero (indStr, 16);

   if (ptr) {
      ahdrlen   = sizeof(ptr->ahdr);
      farhdrlen = sizeof(ptr->argus_far);
      if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS) {
         icmp = (struct ArgusICMPObject *)ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];
         if (icmp->type != ARGUS_ICMP_DSR) {
            icmp = NULL;
         } else {
            ra_src_addr  = icmp->isrcaddr;
            ra_dst_addr  = icmp->idstaddr;
            ra_gw_addr   = icmp->igwaddr;
            ra_icmp_type = icmp->icmp_type;
            ra_icmp_code = icmpFlow->code;
         }
      } else {
         ra_icmp_type = icmpFlow->type;
         ra_icmp_code = icmpFlow->code;
      }

      ra_icmp_data = icmpFlow->id;
      
      bzero (icmptype, sizeof (icmptype));
      bzero (argus_strbuf, MAXSTRLEN);
      bzero (extendedstring, sizeof (extendedstring));
      bzero (icmptype, sizeof (icmptype));
      bzero (date, 128);

      print_date (ptr, date);

      if (idflag) {
         if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_ID_IS_IPADDR)
            argusIDStr = strdup (ipaddr_string (&ptr->ahdr.argusid));
         else
            sprintf (argusIDStr, "%u", ptr->ahdr.argusid);

         sprintf(argus_strbuf, "%-15.15s ", argusIDStr);
      }

      if (ra_icmp_type < (unsigned char) (ICMP_MAXTYPE + 1))
         strcpy (icmptype, icmptypestr[ra_icmp_type]);
      else
         strcpy (icmptype, "UNK");

      switch (ra_icmp_type) {
         case ICMP_UNREACH:
            switch (ra_icmp_code) {
               case ICMP_UNREACH_NET:
                  strcat (icmptype, "N");
                  if (ra_dst_addr) {
                     u_long addr = ra_dst_addr;
                     sprintf (extendedstring, "net %s", 
                              ipaddr_string (&addr));
                  }
                  break;
               case ICMP_UNREACH_HOST:
                  strcat (icmptype, "H");

                  if (ra_dst_addr)
                     sprintf (extendedstring, "host %s", 
                           ipaddr_string (&ra_dst_addr));
                  break;

               case ICMP_UNREACH_PROTOCOL:
                  strcat (icmptype, "O");
                  if (ra_icmp_data && (ra_icmp_data < IPPROTOSTR))
                     sprintf (extendedstring,"proto %s",
                        ip_proto_string[ra_icmp_data]);
                  break;

               case ICMP_UNREACH_PORT: {
                  int index = icmpFlow->tp_p;
                  strcat (icmptype, "P"); 

                  if ((ra_icmp_data && ((index < IPPROTOSTR)) && (index > 0))) {
                     sprintf (extendedstring, "%s_port     %d", ip_proto_string[index], ra_icmp_data);

                  } else if (ra_icmp_data)
                     sprintf (extendedstring, "port     %d", ra_icmp_data);
                  break;
               }
               case ICMP_UNREACH_NEEDFRAG:
                  strcat (icmptype, "F"); break;
               case ICMP_UNREACH_SRCFAIL:
                  strcat (icmptype, "S"); break;

#ifndef ICMP_UNREACH_NET_UNKNOWN
#define ICMP_UNREACH_NET_UNKNOWN        6
#endif
               case ICMP_UNREACH_NET_UNKNOWN:
                  strcat (icmptype, "NU"); 
                  sprintf (extendedstring, "dst_net unknown"); break;
               
#ifndef ICMP_UNREACH_HOST_UNKNOWN
#define ICMP_UNREACH_HOST_UNKNOWN       7
#endif
               case ICMP_UNREACH_HOST_UNKNOWN:
                  strcat (icmptype, "HU"); 
                  sprintf (extendedstring, "dst_host unknown"); break;

#ifndef ICMP_UNREACH_ISOLATED
#define ICMP_UNREACH_ISOLATED           8
#endif
               case ICMP_UNREACH_ISOLATED:
                  strcat (icmptype, "ISO"); 
                  sprintf (extendedstring, "src_host isolated"); break;

#ifndef ICMP_UNREACH_NET_PROHIB
#define ICMP_UNREACH_NET_PROHIB         9
#endif
               case ICMP_UNREACH_NET_PROHIB:
                  strcat (icmptype, "NPRO"); 
                  sprintf (extendedstring, "admin_net prohib"); break;

#ifndef ICMP_UNREACH_HOST_PROHIB
#define ICMP_UNREACH_HOST_PROHIB        10
#endif
               case ICMP_UNREACH_HOST_PROHIB:
                  strcat (icmptype, "HPRO"); 
                  sprintf (extendedstring, "admin_host prohib"); break;

#ifndef ICMP_UNREACH_TOSNET
#define ICMP_UNREACH_TOSNET             11
#endif
               case ICMP_UNREACH_TOSNET:
                  strcat (icmptype, "NTOS"); 
                  sprintf (extendedstring, "tos_net prohib"); break;

#ifndef ICMP_UNREACH_TOSHOST
#define ICMP_UNREACH_TOSHOST            12
#endif
               case ICMP_UNREACH_TOSHOST:
                  strcat (icmptype, "HTOS"); 
                  sprintf (extendedstring, "tos_host prohib"); break;
 
#ifndef ICMP_UNREACH_FILTER_PROHIB
#define ICMP_UNREACH_FILTER_PROHIB      13
#endif
               case ICMP_UNREACH_FILTER_PROHIB:
                  strcat (icmptype, "FIL"); 
                  sprintf (extendedstring, "admin_filter prohib"); break;

#ifndef ICMP_UNREACH_HOST_PRECEDENCE
#define ICMP_UNREACH_HOST_PRECEDENCE    14
#endif
               case ICMP_UNREACH_HOST_PRECEDENCE:
                  strcat (icmptype, "PRE"); 
                  sprintf (extendedstring, "precedence violation"); break;

#ifndef ICMP_UNREACH_PRECEDENCE_CUTOFF
#define ICMP_UNREACH_PRECEDENCE_CUTOFF  15
#endif
               case ICMP_UNREACH_PRECEDENCE_CUTOFF:
                  strcat (icmptype, "CUT"); 
                  sprintf (extendedstring, "precedence cutoff"); break;

            }
            break;

         case ICMP_MASKREPLY:
            rev = 1;
            if (ra_src_addr)
               sprintf (extendedstring, "mask 0x%08x", ra_src_addr);
            break;

         case ICMP_REDIRECT:
            switch (ra_icmp_code) {
            case ICMP_REDIRECT_NET:
               (void) sprintf (extendedstring, "net %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_HOST:
               (void) sprintf (extendedstring, "host %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_TOSNET:
               (void) sprintf (extendedstring, "tosN %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_TOSHOST:
               (void) sprintf (extendedstring, "tosH %s",
                         ipaddr_string (&ra_gw_addr));
               break;
            }
            break;

#ifndef ICMP_ROUTERADVERT
#define ICMP_ROUTERADVERT               9       /* router advertisement */
#endif
         case ICMP_ROUTERADVERT:
            sprintf (extendedstring, "router advertisement"); break;

#ifndef ICMP_ROUTERSOLICIT
#define ICMP_ROUTERSOLICIT              10      /* router solicitation */
#endif
         case ICMP_ROUTERSOLICIT:
            sprintf (extendedstring, "router solicitation"); break;


         case ICMP_ECHOREPLY:
         case ICMP_TSTAMPREPLY:
         case ICMP_IREQREPLY:
            rev = 1;
            sprintf (extendedstring, "%-6d      %-6d",
                           ptr->argus_far.src.bytes, ptr->argus_far.dst.bytes);
            break;

         case ICMP_TIMXCEED:
               (void) sprintf (extendedstring, "timexceed %s",
                         ra_icmp_code ? "reassembly" : "in-transit");
               break;

         case ICMP_PARAMPROB:
         case ICMP_SOURCEQUENCH:
         case ICMP_ECHO:
         case ICMP_TSTAMP:
         case ICMP_IREQ:
         case ICMP_MASKREQ:
            sprintf (extendedstring, "%-6d       %-6d",
                           ptr->argus_far.src.bytes, ptr->argus_far.dst.bytes);
         default:
            sprintf (extendedstring, "%-6d       %-6d",
                           ptr->argus_far.src.bytes, ptr->argus_far.dst.bytes);
            break;
      }

      if (!(Rflag)) {
         sprintf (extendedstring, "%-6d       %-6d",
            ptr->argus_far.src.bytes, ptr->argus_far.dst.bytes);
      }
   
      protoStr = (nflag > 1) ? "  1" : "icmp";

      if (mflag) {
         if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
            struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];

            esrcString = etheraddr_string ((u_char *)&mac->phys_union.ether.ethersrc);
            edstString = etheraddr_string ((u_char *)&mac->phys_union.ether.etherdst);
            sprintf (srcString, "%17.17s %17.17s   %*.*s",
                         esrcString, edstString, hfield, hfield, ipaddr_string (&icmpFlow->ip_src));

         } else
            sprintf (srcString, "%17.17s %17.17s   %*.*s",
                         blankStr, blankStr, hfield, hfield, ipaddr_string (&icmpFlow->ip_src));
      } else
         sprintf (srcString, "%*.*s", hfield, hfield, ipaddr_string (&icmpFlow->ip_src));

      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
         sprintf (&srcString[strlen(srcString)], "%c%c", RaFieldDelimiter, RaFieldDelimiter);
      else
         sprintf (&srcString[strlen(srcString)] , " %*.*s", pfield, pfield, " ");

      sprintf (dstString, "%*.*s", hfield, hfield, ipaddr_string (&icmpFlow->ip_dst));
      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
         sprintf (&dstString[strlen(dstString)], "%c%c", RaFieldDelimiter, RaFieldDelimiter);
      else
         sprintf (&dstString[strlen(dstString)] , " %*.*s", pfield, pfield, " ");
   
      ArgusGetIndicatorString (ptr, indStr);

      if ((ptr->ahdr.type & ARGUS_RMON) && (icmpFlow->ip_dst == 0)) {
         if (Iflag) {
            strcpy (fmtstr, "%s%s%4s %s  - ");
         } else
            strcpy (fmtstr, "%s  %4s %s  - ");
      } else {
         if (Iflag) {
            strcpy (fmtstr, "%s%s%4s %s  -    %s ");
         } else
            strcpy (fmtstr, "%s  %4s %s  -    %s ");
      }

      if (cflag)
         strcat (fmtstr, "%-8u %-6u    %-24.24s ");

      if (ptr->argus_far.src.count)
         fmtstr[13 + vc] = '>';

      if (ptr->argus_far.dst.count)
         fmtstr[11 + vc] = '<';
   
      srccnt = ptr->argus_far.src.count;
      dstcnt = ptr->argus_far.dst.count;

      strcat (fmtstr, icmptype);

      if ((ptr->ahdr.type & ARGUS_RMON) && (icmpFlow->ip_dst == 0)) {
         fmtstr[11 + vc] = ' ';
         fmtstr[12 + vc] = ' ';
         fmtstr[13 + vc] = ' ';

         if (Iflag) {
            if (cflag) {
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, srccnt, dstcnt, extendedstring);
            } else
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString);
         } else {
            if (cflag) {
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, srccnt, dstcnt, extendedstring);
            } else
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString);
         }
      } else {
         if (Iflag) {
            if (cflag) {
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, dstString, srccnt, dstcnt, extendedstring);
            } else
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, dstString);
         } else {
            if (cflag) {
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, dstString, srccnt, dstcnt, extendedstring);
            } else
               sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, dstString);
         }
      }
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      char tmpbuf[128], *ptr = tmpbuf, *str = argus_strbuf, lastchr = ' ';
      int len = strlen(date) - 1;
      bzero (tmpbuf, sizeof(tmpbuf));

      bcopy (str, ptr, len);
      str += len;
      ptr += len;

      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }
      bzero (argus_strbuf, MAXSTRLEN);
      bcopy (tmpbuf, argus_strbuf, strlen(tmpbuf));
   }

   return (argus_strbuf);
}


char *
get_udp_string (argus)
struct ArgusRecord *argus;
{
   return (get_ip_string (argus));
}


char *
get_ip_string (argus)
struct ArgusRecord *argus;
{
   struct ArgusFlow *flow;

   int vc = 0;
   char *edstString = NULL, *esrcString = NULL;
   char dstString[128], srcString[128], delim;
   char protoStr[32], indStr[16], *blankStr = "                   ";
   char argusIDStrBuf[32], *argusIDStr = argusIDStrBuf;
   char date[128], fmtstr[MAXSTRLEN], protoStrargus_strbuf[16], portbuf[16];
   int src_count, dst_count, src_bytes, dst_bytes;
   u_char proto;
   
   bzero (argus_strbuf, MAXSTRLEN);
   bzero (fmtstr, MAXSTRLEN);
   bzero (srcString, 128);
   bzero (dstString, 128);
   bzero (date, 128);

   flow = &argus->argus_far.flow;

   proto = flow->ip_flow.ip_p;

   if (mflag) {
      if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
         struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];

         esrcString = etheraddr_string ((u_char *)&mac->phys_union.ether.ethersrc);
         edstString = etheraddr_string ((u_char *)&mac->phys_union.ether.etherdst);

         sprintf (srcString, "%17.17s %17.17s   %*.*s",
                      esrcString, edstString, hfield, hfield, ipaddr_string (&flow->ip_flow.ip_src));
      } else
         sprintf (srcString, "%17.17s %17.17s   %*.*s",
                      blankStr, blankStr, hfield, hfield, ipaddr_string (&flow->ip_flow.ip_src));
   } else
      sprintf (srcString, "%*.*s", hfield, hfield, ipaddr_string (&flow->ip_flow.ip_src));

   sprintf (dstString, "%*.*s", hfield, hfield, ipaddr_string (&flow->ip_flow.ip_dst));

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;
   else
      delim = '.';

   switch (proto) {
      case IPPROTO_TCP:
         if (flow->ip_flow.sport != 0xFFFF)
            sprintf (portbuf, "%c%-*s", delim, pfield, tcpport_string(flow->ip_flow.sport));
         else
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
               sprintf (portbuf, "%c%c", RaFieldDelimiter, RaFieldDelimiter);
            } else
               sprintf (portbuf, "%-*s ", pfield, " ");

         strcat (srcString, portbuf);

         if (flow->ip_flow.dport != 0xFFFF)
            sprintf (portbuf, "%c%-*s", delim, pfield, tcpport_string(flow->ip_flow.dport));
         else
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
               sprintf (portbuf, "%c%c", RaFieldDelimiter, RaFieldDelimiter);
            } else
               sprintf (portbuf, "%-*s ", pfield, " ");

         strcat (dstString, portbuf);
         break;

      case IPPROTO_UDP:
         if (flow->ip_flow.sport != 0xFFFF)
            sprintf (portbuf, "%c%-*.*s", delim, pfield, pfield, udpport_string(flow->ip_flow.sport));
         else
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
               sprintf (portbuf, "%c%c", RaFieldDelimiter, RaFieldDelimiter);
            } else
               sprintf (portbuf, "%-*s ", pfield, " ");

         strcat (srcString, portbuf);

         if (flow->ip_flow.dport != 0xFFFF)
            sprintf (portbuf, "%c%-*.*s", delim, pfield, pfield, udpport_string(flow->ip_flow.dport));
         else
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
               sprintf (portbuf, "%c%c", RaFieldDelimiter, RaFieldDelimiter);
            } else
               sprintf (portbuf, "%-*s ", pfield, " ");

         strcat (dstString, portbuf);
         break;

      default:
         if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
            sprintf (portbuf, "%c%c", RaFieldDelimiter, RaFieldDelimiter);
         } else
            sprintf (portbuf, "%*s ", pfield, " ");
         strcat (srcString, portbuf);
         strcat (dstString, portbuf);
         break;
   }

   src_count = argus->argus_far.src.count;
   dst_count = argus->argus_far.dst.count;

   if (Aflag) {
      src_bytes = argus->argus_far.src.appbytes;
      dst_bytes = argus->argus_far.dst.appbytes;
   } else {
      src_bytes = argus->argus_far.src.bytes;
      dst_bytes = argus->argus_far.dst.bytes;
   }

   print_date (argus, date);

   if (idflag) {
      if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_ID_IS_IPADDR)
         argusIDStr = strdup (ipaddr_string (&argus->ahdr.argusid));
      else
         sprintf (argusIDStr, "%u", argus->ahdr.argusid);

      sprintf(argus_strbuf, "%-15.15s ", argusIDStr);
   }

   if ((argus->ahdr.type & ARGUS_RMON) && (flow->ip_flow.ip_dst == 0)) {
      if (Iflag) {
         strcpy (fmtstr, "%s%s%4s %s  - ");
      } else
         strcpy (fmtstr, "%s  %4s %s  - ");
   } else {
      if (Iflag) {
         strcpy (fmtstr, "%s%s%4s %s  -    %s ");
      } else
         strcpy (fmtstr, "%s  %4s %s  -    %s ");
   }

   fmtstr[11 + vc] = (dst_count) ? '<' : ' '; 
   fmtstr[13 + vc] = (src_count) ? '>' : ' '; 

   if (cflag)
      strcat (fmtstr, "%-8u %-8u  %-12u %-12u");

   if ((argus->ahdr.cause & ARGUS_TIMEOUT))
      strcat (fmtstr, "TIM");
   else
      if (argus->argus_far.src.count && argus->argus_far.dst.count) {
         if ((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1))
            strcat (fmtstr, "ACC");
         else
            strcat (fmtstr, "CON");
      } else
         if (argus->ahdr.type & ARGUS_START)
            strcat (fmtstr, "INT");

   sprintf (protoStrargus_strbuf, "%u", proto);

   if ((flow->ip_flow.tp_p == ARGUS_RTP_FLOWTAG) && ((src_count > 3) || (dst_count > 3)))
      sprintf (protoStr, "%s", ((nflag > 1) ? protoStrargus_strbuf : "rtp"));
   else
      sprintf (protoStr, "%s", ((nflag > 1) ? protoStrargus_strbuf :
                proto >= IPPROTOSTR ? "unas" : ip_proto_string[proto]));

   if ((strlen(protoStr) > 4) && !((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')))
      protoStr[4] = '\0';

   ArgusGetIndicatorString (argus, indStr);

   if ((argus->ahdr.type & ARGUS_RMON) && (flow->ip_flow.ip_dst == 0)) {
      fmtstr[11 + vc] = ' ';
      fmtstr[12 + vc] = ' ';
      fmtstr[13 + vc] = ' ';

      if (Iflag) {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, 
                  src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString);
      } else {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString,
                  src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString);
      }
   } else {
      if (Iflag) {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, dstString, 
                  src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
                  srcString, dstString);
      } else {
         if (cflag)
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, dstString, 
                  src_count, dst_count,
                  src_bytes, dst_bytes);
         else
            sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
                  srcString, dstString);
      }
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      char tmpbuf[128], *ptr = tmpbuf, *str = argus_strbuf, lastchr = ' ';
      int len = strlen(date) - 1;
      bzero (tmpbuf, sizeof(tmpbuf));

      bcopy (str, ptr, len);
      str += len;
      ptr += len;

      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }
      bzero (argus_strbuf, MAXSTRLEN);
      bcopy (tmpbuf, argus_strbuf, strlen(tmpbuf));
   }

   return (argus_strbuf);
}


char *
get_arp_string (argus)
struct ArgusRecord *argus;
{
   struct ArgusFlow *flow;

   char *targetString, *sourceString, *protoStr, indStr[16];
   char *esrcString = NULL, *edstString = NULL;
   char srcString[256], dstString[256];
   char argusIDStrBuf[32], *argusIDStr = argusIDStrBuf;
   char date[128], fmtstr[256], *blankStr = "                   ";
   int src_count, dst_count, src_bytes, dst_bytes;
   int afield, xfield;
   unsigned short proto;
   
   bzero (argus_strbuf, MAXSTRLEN);
   bzero (date, 128);

   flow = &argus->argus_far.flow;

   src_count = argus->argus_far.src.count;
   dst_count = argus->argus_far.dst.count;

   if (Aflag) {
      src_bytes = argus->argus_far.src.appbytes;
      dst_bytes = argus->argus_far.dst.appbytes;
   } else {
      src_bytes = argus->argus_far.src.bytes;
      dst_bytes = argus->argus_far.dst.bytes;
   }

   print_date (argus, date);

   if (idflag) {
      if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_ID_IS_IPADDR)
         argusIDStr = strdup (ipaddr_string (&argus->ahdr.argusid));
      else
         sprintf (argusIDStr, "%u", argus->ahdr.argusid);

      sprintf(argus_strbuf, "%-15.15s ", argusIDStr);
   }

   proto = argus->ahdr.status & 0xFFFF;
   protoStr = etherproto_string( proto);

   if (proto == ETHERTYPE_REVARP) {
      if (Rflag) {
         sourceString = etheraddr_string (flow->rarp_flow.tareaddr);
         targetString = ipaddr_string (&flow->rarp_flow.arp_tpa);
   
         if (Iflag) {
            strcpy (fmtstr, "%s%s%4s %s%*.*s is-at  %*.*s %*.*s ");
         } else
            strcpy (fmtstr, "%s  %4s %s%*.*s is-at  %*.*s %*.*s ");
   
      } else {
         sourceString = etheraddr_string (flow->rarp_flow.srceaddr);
         targetString = etheraddr_string (flow->rarp_flow.tareaddr);
   
         if (Iflag) {
            strcpy (fmtstr, "%s%s%4s %s%*.*swho-has %*.*s %*.*s ");
         } else
            strcpy (fmtstr, "%s  %4s %s%*.*swho-has %*.*s %*.*s ");
      }
   } else {
      if (Rflag) {
         sourceString = ipaddr_string (&flow->arp_flow.arp_tpa);
         targetString = etheraddr_string (argus->argus_far.attr_arp.response);
   
         if (Iflag) {
            strcpy (fmtstr, "%s%s%4s %s%*.*s is-at  %*.*s %*.*s ");
         } else
            strcpy (fmtstr, "%s  %4s %s%*.*s is-at  %*.*s %*.*s ");
   
      } else {
         sourceString = ipaddr_string (&flow->arp_flow.arp_spa);
         targetString = ipaddr_string (&flow->arp_flow.arp_tpa);
   
         if (Iflag) {
            strcpy (fmtstr, "%s%s%4s %s%*.*swho-has %*.*s %*.*s ");
         } else
            strcpy (fmtstr, "%s  %4s %s%*.*swho-has %*.*s %*.*s ");
      }
   }

   if (mflag) {
      if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
         struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];

         esrcString = etheraddr_string ((u_char *)&mac->phys_union.ether.ethersrc);
         edstString = etheraddr_string ((u_char *)&mac->phys_union.ether.etherdst);

         sprintf (srcString, "%17.17s %17.17s   %*.*s",
                      esrcString, edstString, hfield, hfield, sourceString);
      } else
         sprintf (srcString, "%17.17s %17.17s   %*.*s",
                      blankStr, blankStr, hfield, hfield, sourceString);

   } else
      sprintf (srcString, "%*.*s", hfield, hfield, sourceString);

   sprintf (dstString, "%s", targetString);

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      sprintf (&srcString[strlen(srcString)], "%c%c", RaFieldDelimiter, RaFieldDelimiter);
      sprintf (&dstString[strlen(dstString)], "%c%c", RaFieldDelimiter, RaFieldDelimiter);
   }

   if (cflag)
      strcat (fmtstr, "%-8u %-8u  %-12u %-12u");

   if ((argus->ahdr.cause & ARGUS_TIMEOUT))
      strcat (fmtstr, "TIM");
   else
      if (argus->argus_far.src.count && argus->argus_far.dst.count) {
         if ((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1))
            strcat (fmtstr, "ACC");
         else
            strcat (fmtstr, "CON");
      } else
         if (argus->ahdr.type & ARGUS_START)
            strcat (fmtstr, "INT");

   ArgusGetIndicatorString (argus, indStr);

   xfield = pfield;
   afield = hfield;

   if (Rflag) {
      afield += 2;
      xfield -= 2;
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      afield += 5;
   }

   if (Iflag) {
      if (cflag)
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
               srcString, pfield, pfield, " ", afield, afield, dstString, xfield, xfield, " ",
               src_count, dst_count,
               src_bytes, dst_bytes);
      else
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
               srcString, pfield, pfield, " ", afield, afield, dstString, xfield, xfield, " ");
   } else {
      if (cflag)
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
               srcString, pfield, pfield, " ", afield, afield, dstString, xfield, xfield, " ",  
               src_count, dst_count,
               src_bytes, dst_bytes);
      else
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
               srcString, pfield, pfield, " ", afield, afield, dstString, xfield, xfield, " ");
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      char tmpbuf[128], *ptr = tmpbuf, *str = argus_strbuf, lastchr = ' ';
      int len = strlen(date) - 1;
      bzero (tmpbuf, sizeof(tmpbuf));

      bcopy (str, ptr, len);
      str += len;
      ptr += len;

      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }
      bzero (argus_strbuf, MAXSTRLEN);
      bcopy (tmpbuf, argus_strbuf, strlen(tmpbuf));
   }

   return (argus_strbuf);
}



char *
get_nonip_string (argus)
struct ArgusRecord *argus;
{
   struct ArgusFlow *flow;

   int vc = 0;
   char srcString[256], dstString[256];
   char protoStr[32], indStr[16];
   char *edstString = NULL, *esrcString = NULL;
   char argusIDStrBuf[32], *argusIDStr = argusIDStrBuf;
   char date[128], fmtstr[MAXSTRLEN], *blankStr = "                      ";
   char sportbuf[16], dportbuf[16], delim;
   int src_count, dst_count, src_bytes, dst_bytes;
   unsigned short proto = 0;
   
   bzero (argus_strbuf, MAXSTRLEN);
   bzero (argusIDStrBuf, 32);
   bzero (fmtstr, MAXSTRLEN);
   bzero (srcString, 256);
   bzero (dstString, 256);
   bzero (protoStr, 32);
   bzero (sportbuf, 16);
   bzero (dportbuf, 16);
   bzero (indStr, 16);
   bzero (date, 128);

   flow = &argus->argus_far.flow;

   sprintf (srcString, "%17.17s", etheraddr_string((unsigned char *)&flow->mac_flow.ehdr.ether_shost));
   sprintf (dstString, "%17.17s", etheraddr_string((unsigned char *)&flow->mac_flow.ehdr.ether_dhost));

   src_count = argus->argus_far.src.count;
   dst_count = argus->argus_far.dst.count;

   if (Aflag) {
      src_bytes = argus->argus_far.src.appbytes;
      dst_bytes = argus->argus_far.dst.appbytes;
   } else {
      src_bytes = argus->argus_far.src.bytes;
      dst_bytes = argus->argus_far.dst.bytes;
   }

   print_date (argus, date);

   if (idflag) {
      if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_ID_IS_IPADDR)
         argusIDStr = strdup (ipaddr_string (&argus->ahdr.argusid));
      else
         sprintf (argusIDStr, "%u", argus->ahdr.argusid);

      sprintf(argus_strbuf, "%-15.15s ", argusIDStr);
   }

   if (mflag) {
      if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
         struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];

         esrcString = etheraddr_string ((u_char *)&mac->phys_union.ether.ethersrc);
         edstString = etheraddr_string ((u_char *)&mac->phys_union.ether.etherdst);

         sprintf (srcString, "%17.17s %17.17s   %*.*s",
                      esrcString, edstString, hfield + 2, hfield + 2,
                      etheraddr_string((unsigned char *)&flow->mac_flow.ehdr.ether_shost));
      } else
         sprintf (srcString, "%17.17s %17.17s    %*.*s",
                      blankStr, blankStr, hfield + 2, hfield + 2, 
                      etheraddr_string((unsigned char *)&flow->mac_flow.ehdr.ether_shost));
   }

   if (Iflag)
      strcpy (fmtstr, "%s%s%4s %s -   %22.22s ");
   else
      strcpy (fmtstr, "%s  %4s %s -   %22.22s ");

   if (cflag)
      strcat (fmtstr, "%-8u %-8u  %-12u %-12u");

   if ((argus->ahdr.cause & ARGUS_TIMEOUT))
      strcat (fmtstr, "TIM");
   else
      if (argus->argus_far.src.count && argus->argus_far.dst.count) {
         if ((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1))
            strcat (fmtstr, "ACC");
         else
            strcat (fmtstr, "CON");
      } else
         if (argus->ahdr.type & ARGUS_START)
            strcat (fmtstr, "INT");

   proto = argus->ahdr.status & 0xFFFF;

   sprintf (protoStr, "%s", etherproto_string(proto));

   if (src_count)
      fmtstr[12 + vc] = '>'; 
   if (dst_count)
      fmtstr[10 + vc] = '<'; 


   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
      delim = RaFieldDelimiter;
   else {
      if (proto == 0) {
         delim = '.';
      } else {
         delim = ' ';
      }
   }

   if (proto == 0) {
      sprintf (sportbuf, "%c%-4.4s", delim, llcsap_string((unsigned char) flow->mac_flow.ssap));
      sprintf (dportbuf, "%c%-4.4s", delim, llcsap_string((unsigned char) flow->mac_flow.dsap));
   } else {
      sprintf (sportbuf, "%c    ", delim);
      sprintf (dportbuf, "%c    ", delim);

      if (dst_count) {
         if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
            fmtstr[9 + vc] = RaFieldDelimiter;
      } else
         if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
            fmtstr[10 + vc] = RaFieldDelimiter;

      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
         fmtstr[22 + vc] = RaFieldDelimiter;
      }
   }

   strcat (srcString, sportbuf);
   strcat (dstString, dportbuf);

   if ((strlen(protoStr) > 4) && !((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')))
      protoStr[4] = '\0';

   ArgusGetIndicatorString (argus, indStr);

   if (Iflag) {
      if (cflag)
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
               srcString, dstString, 
               src_count, dst_count,
               src_bytes, dst_bytes);
      else
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, indStr, protoStr,
               srcString, dstString);
   } else {
      if (cflag)
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
               srcString, dstString, 
               src_count, dst_count,
               src_bytes, dst_bytes);
      else
         sprintf (&argus_strbuf[strlen(argus_strbuf)], fmtstr, date, protoStr,
               srcString, dstString);
   }

   if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
      char tmpbuf[128], *ptr = tmpbuf, *str = argus_strbuf, lastchr = ' ';
      int len = strlen(date) - 1;
      bzero (tmpbuf, sizeof(tmpbuf));

      bcopy (str, ptr, len);
      str += len;
      ptr += len;

      while (*str) {
         if (*str == ' ') {
            if (lastchr != RaFieldDelimiter)
               *ptr++ = RaFieldDelimiter;
            while (isspace((int)*str)) str++;
         }
         lastchr = *str;

         *ptr++ = *str++;
      }
      bzero (argus_strbuf, MAXSTRLEN);
      bcopy (tmpbuf, argus_strbuf, strlen(tmpbuf));
   }

   return (argus_strbuf);
}



#ifdef NOVFPRINTF
/*
 * Stock 4.3 doesn't have vfprintf. 
 * This routine is due to Chris Torek.
 */
vfprintf(f, fmt, args)
   FILE *f;
   char *fmt;
   va_list args;
{
   int ret;

   if ((f->_flag & _IOWRT) == 0) {
      if (f->_flag & _IORW)
         f->_flag |= _IOWRT;
      else
         return EOF;
   }
   ret = _doprnt(fmt, args, f);
   return ferror(f) ? EOF : ret;
}
#endif

