/*
 * Copyright 2006 Ricoh Corporation.
 * 
 * 
 * APACHE LICENSE VERSION 2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 
 * RICOH DEVELOPER PROGRAM SUPPORT:
 * 
 * Support for this software is available only to "Premier Plus" members
 * of the Ricoh Developer Program (RiDP).  You may find out more 
 * information about the Program at
 * 
 *      http://americas.ricoh-developer.com
 * 
 * Premier plus members may find answers and ask questions through the
 * RiDP customer help website at
 * 
 *      https://ridp.custhelp.com
 * 
 * Developers who are not RiDP members may still use this software as
 * stipulated in the license terms given above.
 *
 */ 

import java.lang.*;
import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;

import jp.co.ricoh.dsdk.function.*;
import jp.co.ricoh.dsdk.function.scan.*;
import jp.co.ricoh.dsdk.function.scan.attribute.*;
import jp.co.ricoh.dsdk.function.scan.attribute.standard.*;
import jp.co.ricoh.dsdk.function.scan.event.*;
import jp.co.ricoh.dsdk.panel.*;
import jp.co.ricoh.dsdk.panel.event.*;
import jp.co.ricoh.dsdk.util.image.*;
import jp.co.ricoh.dsdk.xlet.*;

import jp.co.ricoh.dsdk.framework.panel.*;
import jp.co.ricoh.dsdk.framework.panel.scan.*;
import jp.co.ricoh.dsdk.framework.scan.*;
import jp.co.ricoh.dsdk.framework.util.*;


/**
 *	This is a base class that implements Xlet and provides
 *	scanning services, including notifications such as "no paper"
 *	and "jam detected".
 *
 *	This is not a framework, it is a sample; the only purpose
 *	in splitting the functionality like this is to separate
 *	the scanning code from the sample xlet code for the sake
 *	of clarity; there is no need for actual xlets to be
 *	structured like this.
 */
public class ScanningXlet implements Xlet
{
	protected Frame rootContainer;

	protected XletContext context = null;
	protected String homeDir = null;
	protected String scanImageDir = null;
	
	//
	// Scan framework variables
	//
	private ScanSetupWindow scanSetupWindow = null;
	private ScanData scanData = null;
	private ScannerControl	scanner	= null;
	private boolean cancelledByUser = false;
	private ScanInProgressDialog scanInProgressDialog = null;
	
	
	public synchronized void initXlet(XletContext context)
		throws XletStateChangeException
	{
		this.context = context;
		homeDir = (String) context.getXletProperty(XletContext.HOME);
		
		rootContainer = this.getFrame();

		scanImageDir = (String)context.getXletProperty(XletContext.HOME)+ "/ScannedImages";
		File scanImageFile = new File( scanImageDir );
		if( !scanImageFile.exists())
		{
			scanImageFile.mkdir();
		}


		scanData = new ScanData();
		SourceModeInfo sourceModeInfo = new SourceModeInfo();
		scanData.setSourceModeInfo(sourceModeInfo);
	}


	public synchronized void startXlet() throws XletStateChangeException 
	{
	}




	public synchronized void pauseXlet()
	{
	}

	public synchronized void destroyXlet(boolean force)
		throws XletStateChangeException 
	{
		if (true /*force*/) 
		{

			try 
			{
				
				if(scanner != null)
				{
					scanner.destroy();
					scanner = null;
				}
			}
			catch (Exception e) 
			{
			}                       
			
		}
		else 
		{
			throw new XletStateChangeException();
		}
	}
	
	
	/**
	 * Returns a reference to the main window.
	 */
	private Frame getFrame() throws XletStateChangeException 
	{
		// find the frame window
		Container parent = null;

		try {
			parent = context.getContainer();
		} catch (UnavailableContainerException ex) {
			throw new XletStateChangeException(ex.toString());
		}

		while (!(parent instanceof Frame)) {
			parent = parent.getParent();

			if (parent == null) {
				return null;
			}
		}

		return (Frame) parent;
	}
	
	
	/**
	 *	Open a modal dialog with a confirmation message.  All user interaction is blocked until
	 *	this dialog is confirmed and closed via the "OK" button.
	 */
	public void showConfirmationDialog(String message)
	{
		if(scanInProgressDialog != null)
		{
			if(scanInProgressDialog != null)
			{
				scanInProgressDialog.setConfirmMessage(message);
			}
		}
		else
		{
			try 
			{
				final Dialog modalDialog = new Dialog(rootContainer, true);

				int parentWidth = rootContainer.getWidth();
				int parentHeight = rootContainer.getHeight();

				int dialogMargin = 40;
				int dialogWidth = parentWidth - (2 * dialogMargin);
				int dialogHeight = parentHeight / 2;
				int dialogInnerMargin = 16;
				int labelHeight = 16;
				int buttonWidth = 80;
				int buttonHeight = 20;
				
				modalDialog.setLocation(dialogMargin, (parentHeight - dialogHeight) / 2);
				modalDialog.setSize(dialogWidth, dialogHeight);
				
				
				Label theLabel = new Label(message);
				theLabel.setLocation(dialogInnerMargin, dialogInnerMargin + ((dialogHeight - (labelHeight + buttonHeight + (2 * dialogInnerMargin))) / 2));
				theLabel.setSize(dialogWidth - (dialogInnerMargin * 2), labelHeight);
				theLabel.setFont(Font.F12);

				modalDialog.add(theLabel);
				
				
				Button okButton = new Button(new Text("OK"));
				okButton.setLocation(dialogWidth - (buttonWidth + dialogInnerMargin), dialogHeight - (buttonHeight + dialogInnerMargin));
				okButton.setSize(buttonWidth, buttonHeight);
				okButton.setShape(Button.Shape.NOCNR2);

				okButton.addActionListener( new ActionListener()
					{
					    public void actionPerformed(ActionEvent evt)
					    {
				    		// was: modalDialog.dispose();
						modalDialog.getOwner().remove(modalDialog);
					    }
					} );

				modalDialog.add(okButton);

				modalDialog.show();
			} 
			catch (Exception e) 
			{
			}
		}
	}
	
	public class ScanInProgressDialog extends Dialog
	{
		Label scanStatusLabel = null;
		Button cancelButton = null;
		Button okButton = null;
		
		int firstButtonPosition = 0;
		int secondButtonPosition = 0;
		int buttonY = 0;
		boolean continueAfterConfirm = false;
		
		protected ScanInProgressDialog(Window parent, boolean modal)
		{
			super(parent, modal);
		}
		
		public void init(String message)
		{
			try 
			{
				Window owner = this.getOwner();

				int parentWidth = owner.getWidth();
				int parentHeight = owner.getHeight();

				int dialogMargin = 40;
				int dialogWidth = parentWidth - (2 * dialogMargin);
				int dialogHeight = parentHeight / 2;
				int dialogInnerMargin = 16;
				int labelHeight = 16;
				int buttonWidth = 80;
				int buttonHeight = 20;
				
				firstButtonPosition = dialogWidth - (buttonWidth + dialogInnerMargin);
				secondButtonPosition = firstButtonPosition - (buttonWidth + dialogInnerMargin);
				buttonY = dialogHeight - (buttonHeight + dialogInnerMargin);
				
				this.setLocation(dialogMargin, (parentHeight - dialogHeight) / 2);
				this.setSize(dialogWidth, dialogHeight);

				scanStatusLabel = new Label(message);
				scanStatusLabel.setLocation(dialogInnerMargin, dialogInnerMargin + ((dialogHeight - (labelHeight + buttonHeight + (2 * dialogInnerMargin))) / 2));
				scanStatusLabel.setSize(dialogWidth - (dialogInnerMargin * 2), labelHeight);
				scanStatusLabel.setFont(Font.F12);

				this.add(scanStatusLabel);

				cancelButton = new Button(new Text("Cancel"));
				cancelButton.setLocation(firstButtonPosition, buttonY);
				cancelButton.setSize(buttonWidth, buttonHeight);
				cancelButton.setShape(Button.Shape.NOCNR2);

				cancelButton.addActionListener( new ActionListener()
					{
					    public void actionPerformed(ActionEvent evt)
					    {
				    		cancelScanByUser();
					    }
					} );

				this.add(cancelButton);


				okButton = new Button(new Text("OK"));
				okButton.setLocation(secondButtonPosition, buttonY);
				okButton.setSize(buttonWidth, buttonHeight);
				okButton.setShape(Button.Shape.NOCNR2);

				okButton.addActionListener( new ActionListener()
					{
					    public void actionPerformed(ActionEvent evt)
					    {
					    	continueScanAfterConfirm();
					    }
					} );
				
				okButton.setVisible(false);
				this.add(okButton);

			} 
			catch (Exception e) 
			{
			}
		
		}
		
		public void enterContinueUntilCancelMode()
		{
			cancelButton.setVisible(false);
			cancelButton.repaint();
			okButton.setVisible(false);
			okButton.repaint();
			
			cancelButton.setLocation(firstButtonPosition, buttonY);
			cancelButton.setVisible(true);
			cancelButton.repaint();
		}
		
		public void enterConfirmMode()
		{
			continueAfterConfirm = false;
			
			cancelButton.setVisible(false);
			cancelButton.repaint();
			okButton.setVisible(false);
			okButton.repaint();
			
			okButton.setLocation(firstButtonPosition, buttonY);
			okButton.setVisible(true);
			okButton.repaint();
		}
		
		public void enterConfirmOrCancelMode()
		{
			continueAfterConfirm = true;
			
			cancelButton.setVisible(false);
			cancelButton.repaint();
			okButton.setVisible(false);
			okButton.repaint();
			
			cancelButton.setLocation(secondButtonPosition, buttonY);
			cancelButton.setVisible(true);
			cancelButton.repaint();
			
			okButton.setLocation(firstButtonPosition, buttonY);
			okButton.setVisible(true);
			okButton.repaint();
		}
		
		/**
		 *	Change the label on this dialog to show a particular message
		 */
		public void setScanStatusMessage(String message)
		{
			if(scanStatusLabel != null)
			{
				scanStatusLabel.setText(message);
				scanStatusLabel.repaint();
			}		
		}
		
		/**
		 *	Display a message and wait for the user to press
		 *	"OK" or "Cancel".  The scan is either continued
		 *	or cancelled depending on the result.
		 *
		 *	This dialog is used when a recoverable paused
		 *	state is entered.
		 */
		public void setConfirmOrCancelMessage(String message)
		{
			System.err.println("**** Scan confirm or cancel dialog");
			this.setScanStatusMessage(message);
			this.enterConfirmOrCancelMode();
		}
		
		/**
		 *	Display a message where the only option available
		 *	is "OK".
		 *
		 *	This dialog is used in unrecoverable situations.
		 */
		public void setConfirmMessage(String message)
		{
			System.err.println("**** Scan confirm dialog");
			this.setScanStatusMessage(message);
			this.enterConfirmMode();
		}

	}
	
	/**
	 *	Open a modal dialog.  All user interaction is blocked until
	 *	this dialog is closed by the scan process or cancelled with
	 *	the cancel button.
	 */
	public void openStartScanDialog(String message)
	{
		try 
		{
			scanInProgressDialog = new ScanInProgressDialog(rootContainer, true);
			scanInProgressDialog.init(message);
			
			scanInProgressDialog.show();
			
			scanner = new ScannerControl(scanInProgressDialog, scanImageDir + "/", scanData, new ScannerControlEventListener()
				{
					public void actionEvent(ScannerControlEvent event) 
					{
						if (scanner != null) 
						{
							processScanEvent((Result)event.getSource());
						}
					}
				}  );

			// Enable Start key.
			scanner.setStartKey(false);
			// Disable # key.
			scanner.setSharpKey(true);
			// Change Start key green.
			scanner.LedControl(true);

		} 
		catch (Exception e) 
		{
		}

	}
	
	public void setScanStatusMessage(String message)
	{
		if(scanInProgressDialog != null)
		{
			scanInProgressDialog.setScanStatusMessage(message);
		}		
	}

	/**
	 *	Open a modal dialog with a confirmation message. 
	 *	The scan will continue after the user clicks "OK",
	 *	or it will be aborted after the user clicks "Cancel".
	 */
	public void showContinueScanDialog(String message)
	{
		if(scanInProgressDialog != null)
		{
			scanInProgressDialog.setConfirmOrCancelMessage(message);
		}
	}
	
	public void continueScanAfterConfirm()
	{
		scanInProgressDialog.enterContinueUntilCancelMode();
		if (!scanData.isADFMode() && !scanner.isSharpKey()) 
		{
			// If # key is enabled on manual mode or SADF mode,
			// display "Press # key if it is end of scan" on label area.
			setScanStatusMessage("Press '#' to end scan.");
		} 
		else 
		{
			setScanStatusMessage("Press START to continue.");	
		}	
	}

	public void cancelScanByUser()
	{
		if(scanner != null)
		{
			// Disable # key.
			scanner.setSharpKey(true);
			cancelledByUser = true;
	//		scanner.destroy();	// is it okay to call destroy here?
			scanner = null;
			
			closeScanDialog();
		}
	}
	public void closeScanDialogAfterCompletion()
	{
	
		if(scanner != null)
		{
			// Disable Start key.
			scanner.setStartKey(true);
			// Disable # key.
			scanner.setSharpKey(true);

			scanner.reset();
	//		scanner.destroy(); // calling scanner.destroy() here locks up the system.
			scanner = null;
		}
		closeScanDialog();
	}

	public void closeScanDialog()
	{
		if(scanInProgressDialog != null)
		{
			scanInProgressDialog.getOwner().remove(scanInProgressDialog);
			// was: scanInProgressDialog.dispose();
			scanInProgressDialog = null;
		}
	}

	
	/**
	 *	Open the scan settings dialog.  It will take over all input events
	 *	until the "finish" button is pressed.
	 */
	public void openScanSettingsDialog()
	{
		scanSetupWindow = new ScanSetupWindow(new Window(rootContainer),scanData,new ActionListener()
				{
				    public void actionPerformed(ActionEvent e)
				    {
					scanSetupWindow.setEnabledOwnerWindow(true);
					scanSetupWindow.dispose();
					scanSetupWindow.destroyWindow();
					scanSetupWindow = null;
				    }
				} );

		scanSetupWindow.create();
		scanSetupWindow.setEnabledOwnerWindow(false);
		scanSetupWindow.show();	
	}

	protected static boolean deleteDirectoryContents(String pathName) 
	{
		boolean result = true;
		
		try 
		{
			File f = new File(pathName);
			File fileList[] = f.listFiles();
			for (int i = 0; (i < fileList.length) && (result == true); i++) 
			{
				if (!fileList[i].delete()) 
				{
					result = false;
				}
			}
		}
		catch(Exception e)
		{
			result = false;
		}
		
		return result;
	}
	
	/**
	 *	Called by processScanEvent when the scan operation first starts up.
	 */
	protected void setupInitialScanParameters()
	{
		//
		// Clear out our temporary directory
		//	
		deleteDirectoryContents(scanImageDir);
		
		//
		// Setup scanner image file name. (YYYYMMDDHHMMSS-sequential number)
		//
		DecimalFormat f = new DecimalFormat("00");
		DateTimeUtility dt = DateTimeUtility.getRealTime();
		String scannedImageBaseName = dt.getYear().toString() + f.format(dt.getMonth()) + f.format(dt.getDay()) +
				 f.format(dt.getHour()) + f.format(dt.getMinute()) + f.format(dt.getSecond())+"-";
		
		//
		// Set image file name and scan data from the setup dialog
		//
		scanner.setImageFileName(scannedImageBaseName);
		scanner.setScanData((ScanData)scanData.clone());
		
		//
		// Set the flag we use to detect user cancellations.
		//
		cancelledByUser = false; 
	}
	
	/**
	 *	Called by processScanEvent when the scan operation comletes successfully.
	 */
	protected void scanEventCompleted()
	{
	}
			
	/**
	 *	Process scan event callbacks
	 *
	 * 	@see jp.co.ricoh.dsdk.framework.scan.ScannerStateInfo
	 * 	@see jp.co.ricoh.dsdk.framework.util.Result#getEvent()
	 */
	private void processScanEvent(Result result) 
	{
		System.err.println("**** Scan event = " + result.getEvent() + " reason = " + result.getReason());
		try 
		{
			switch( result.getEvent() )
			{
			//
			// The scan start event is generated exactly once at the
			// beginning of a scan operation.
			//
			case	ScannerStateInfo.SCAN_EVENT_START:
				setupInitialScanParameters();
				setScanStatusMessage("Scanning...");
				scanInProgressDialog.enterContinueUntilCancelMode();
				break;

			//
			// Scan continue events are generated whenever the scan job
			// is resumed via the START key after going into an idle state.
			//
			case	ScannerStateInfo.SCAN_EVENT_CONTINUE_START:
				scanInProgressDialog.enterContinueUntilCancelMode();
				setScanStatusMessage("Scan continuing...");
				break;

			//
			// Note that the scan framework does NOT guarentee to notify
			// the client on idle events, so it is best to not do anything
			// here.
			//
			case	ScannerStateInfo.SCAN_EVENT_IDLE:
				break;

			//
			// The scan event completed event is generated when the
			// scan completes successfully.
			//
			case	ScannerStateInfo.SCAN_EVENT_COMPLETED:
				// Disable # key after post-process.
				scanner.setSharpKey(true);
				// Enable Start key.
				scanner.setStartKey(false);	
				// Change Start key green.
				scanner.LedControl(true);
				
				scanEventCompleted();

				break;


			//
			// Scan paused and scan failed events are generated in response to
			// a number of different situations that can come up during scanning,
			// including the user pressing the CLEAR/STOP key, the scan system
			// waiting during a manual or semi-automatic scan operation, or
			// error conditions such as jams and no original found.
			//
			// Processing these events requires determining the message to
			// show to the user and then displaying it.  In some cases, the
			// user is given the chance to either continue or cancel; in other
			// cases the only option is to confirm.
			//
			case	ScannerStateInfo.SCAN_EVENT_PAUSED:
			case	ScannerStateInfo.SCAN_EVENT_FAILED:

				//
				// If the user cancelled the scan, then we abort.
				//
				if (cancelledByUser && result.getEvent() == ScannerStateInfo.SCAN_EVENT_FAILED) 
				{
					showConfirmationDialog("Scan cancelled");
					cancelledByUser = false; 
				}
				//
				// If the reason for the scan paused event is that
				// we are waiting for the user to prepare the paper
				// being scanned, then enable the '#' key.
				//
				else if( 	(result.getReason() == ScannerStateInfo.SCAN_REASON_MANUAL) ||
						(result.getReason() == ScannerStateInfo.SCAN_REASON_SADF) )
				{
					setScanStatusMessage("Press '#' to end scan.");
					// Enable # key.
					scanner.setSharpKey(false);
				}
				else
				{
					String message = lookUpScanEventMessage(result);
					if(message != null)
					{
						//
						// The scan framework provides a method that returns
						// a boolean that is 'true' if the scan operation is
						// continuable, or 'false' if it is not.
						//
						if (result.getResultCode()) 
						{
							//
							// Show the message and allow the user to
							// continue or cancel.
							//
							showContinueScanDialog(message);
						}
						else
						{
							//
							// The scan cannot be continued;
							// Show the message with a single "OK" button.
							//
							showConfirmationDialog(message);
						}
					}
					
					// Enable Start key
					scanner.setStartKey(false);	
					// Change Start key green.
					scanner.LedControl(true);
					

				}
				break;

			default:
				break;
			}
		} 
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}
	
	/**
	 *	Find the message corresponding to the current scan state
	 *	so that the user may be informed.
	 */
	protected String lookUpScanEventMessage(Result result) 
	{

		String message = null;
		
		switch( result.getReason() )
		{
	   	// Paused with paper jam
		case	ScannerStateInfo.SCAN_REASON_JAM:
			message="Paper Jam.  Please correct, then press START to continue.";
			break;

		// Paused with cover open
		case	ScannerStateInfo.SCAN_REASON_COVER_OPEN:
			// "Paused since cover is open. Press Start button when ready."
			message="Cover is open; press START to resume.";
			break;
		// Pause because platen is up
		case	ScannerStateInfo.SCAN_REASON_LIFT_UP:
			// "Platen is up."
			message = "Platen is up.";
			break;

		// Cancel scan service since OTHER error occurred
		case	ScannerStateInfo.SCAN_REASON_OTHER_ERROR:
			// "Error occurred."
			message = "An error occurred while scanning; cannot continue.";
			break;

		// Scanner is busy status
		case	ScannerStateInfo.SCAN_REASON_BUSY:
			// "Scanner is unavailable since other is in-use."
			message="Scanner is in use by another application.";
			break;

		// Paused since document size unknown
		// You may wonder why you sometimes get this message even when there
		// is an original on the platen.  The reason for that is that the
		// SDK/J scanning mechanism will not detect an original on the platen
		// unless you raise and then lower the lid.
		case	ScannerStateInfo.SCAN_REASON_SIZE_UNKNOWN:
			// "Original size unknown."
			message="No document found on ADF or platen.  Please prepare document to scan, then press START to continue.";
			break;

		// Paused since originals are unset on ADF
		case	ScannerStateInfo.SCAN_REASON_ORIGINAL_SET:
			// "Original is not set on ADF. Please set originals."
			message="Please place document to scan on ADF to continue.";
			break;
			
		// Paused since originals are unset on platen.
		case	ScannerStateInfo.SCAN_REASON_ORIGINAL_LEFT:
			// "Originals are unset on platen. Please set originals."
			message="Place place document on platen to continue.";
			break;

		// Scanner service is down
		case	ScannerStateInfo.SCAN_REASON_DOWN:
			// "Error occurred."
			message="Scan service unavailable; cannot continue.";
			break;
			
		// Paused since connecting to device
		case	ScannerStateInfo.SCAN_REASON_CONNECTING_TO_DEVICE:
			message="Cannot connect to scanner device.";
			break;
			
		// Parameter invalid
		// When a Start key is pressed, either one of arguments, other than image file output destination directory,
		// was null from the arguments when ScannerControl instance is generated.	
		case	ScannerStateInfo.SCAN_REASON_ILLEGAL_PARAMETER:
			message="Illegal parameter (programming error).";
			break;
			
		// Invalid parameter
		case	ScannerStateInfo.SCAN_REASON_ILLEGAL_SCAN_PARAMETER:
			message="Invalid scanner parameter; please check scan settings.";
			break;
			
		// Paused with Clear/Stop key
		case	ScannerStateInfo.SCAN_REASON_CLEAR_STOP:
			message="Paused with Clear/Stop key.";
			break;
		
		case	ScannerStateInfo.SCAN_REASON_NONE:
		case	ScannerStateInfo.SCAN_REASON_MANUAL:
		case	ScannerStateInfo.SCAN_REASON_SADF:
		default:
			break;
		}

		System.err.println("**** Scan event reason message is: " + message);
		
		return message;
	}
}
