/*
 *  Copyright (c) 2005, 2006 Imola Informatica.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the LGPL License v2.1
 *  which accompanies this distribution, and is available at
 *  http://www.gnu.org/licenses/lgpl.html
 */


package it.imolinfo.jbi4corba.netbeansplugin.idl2wsdlwizard;

import it.imolinfo.jbi4corba.webservice.generator.WSDLDescriptor;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.Properties;
import java.util.ResourceBundle;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableModel;
import org.openide.util.NbBundle;

/**
 * Panel to insert mandatory informations to create WSDL file from a IDL file.
 *
 * @author <a href="mailto:mcimatti@imolinfo.it">Marco Cimatti</a>
 */
public final class CreateWSDLPanel extends JPanel {

    /**
     * Generated serial version UID.
     */
    private static final long serialVersionUID = -8184972798675560971L;

    /**
     * The name of the property to define ORB host.
     *
     * @see  #ORB_HOST_PROPERTY_VALUE
     */
    private static final String ORB_HOST_PROPERTY_NAME
            = "org.omg.CORBA.ORBInitialHost";

    /**
     * The value of the property to define ORB host.
     *
     * @see  #ORB_HOST_PROPERTY_NAME
     */
    private static final String ORB_HOST_PROPERTY_VALUE = "localhost";

    /**
     * The name of the property to define ORB port.
     *
     * @see  #ORB_PORT_PROPERTY_VALUE
     */
    private static final String ORB_PORT_PROPERTY_NAME
            = "org.omg.CORBA.ORBInitialPort";

    /**
     * The value of the property to define ORB port.
     *
     * @see  #ORB_PORT_PROPERTY_NAME
     */
    private static final String ORB_PORT_PROPERTY_VALUE = "1050";

    /**
     * The size of text fields, expressed in columns.
     */
    private static final int DEFAULT_TEXT_FIELD_SIZE = 32;

    /**
     * The numbers of properties editable.
     */
    private static final int DEFAULT_PROPERTIES_NUMBER = 32;

    /**
     * The column in the <code>javax.swing.table.TableModel</code> where
     * properties names are stored.
     */
    private static final int PROPERTY_NAME_COLUMN = 0;

    /**
     * The column in the <code>javax.swing.table.TableModel</code> where
     * properties values are stored.
     */
    private static final int PROPERTY_VALUE_COLUMN = 1;

    /**
     * Vertical gap to separate graphic components, in pixel.
     */
    private static final int V_GAP = 3;

    /**
     * Horizontal gap to separate graphic components, in pixel.
     */
    private static final int H_GAP = 5;

    /**
     * The CORBA service name text field.
     */
    private JTextField corbaServiceName
            = new JTextField(DEFAULT_TEXT_FIELD_SIZE);

    /**
     * The name space text field.
     */
    private JTextField nameSpace = new JTextField(DEFAULT_TEXT_FIELD_SIZE);

    /**
     * The endpoint name text field.
     */
    private JTextField endpointName = new JTextField(DEFAULT_TEXT_FIELD_SIZE);

    /**
     * The resource bundle to use for internationalization.
     */
    private ResourceBundle bundle = NbBundle.getBundle(getClass());

    /**
     * The <code>javax.swing.table.TableModel</code> used to show optional
     * properties.
     */
    private DefaultTableModel propertiesModel = new DefaultTableModel(
            new String[] { bundle.getString("LBL_PropertyName"),
            bundle.getString("LBL_PropertyValue") }, DEFAULT_PROPERTIES_NUMBER);

    /**
     * The button to confirm WSDL creation.
     */
    private JButton createWSDL
            = new JButton(bundle.getString("CTL_CreateWSDL"));

    /**
     * The button to cancel WSDL creation.
     */
    private JButton cancel = new JButton(bundle.getString("CTL_Cancel"));

    /**
     * The listener that suggests the endpoint name while service or name space
     * are edited by the user.
     */
    private DocumentListener docListener = new DocumentListener() {
        public void changedUpdate(final DocumentEvent e) {
            endpointName.setText(suggestedEndpoint());
        }

        public void insertUpdate(final DocumentEvent e) {
            endpointName.setText(suggestedEndpoint());
        }

        public void removeUpdate(final DocumentEvent e) {
            endpointName.setText(suggestedEndpoint());
        }
    };

    /**
     * Creates a new panel used to edit additional data necessary to create a
     * WSDL file starting from an IDL file.
     */
    public CreateWSDLPanel() {
        super(new BorderLayout());

        JLabel label = new JLabel(bundle.getString("LBL_Instructions"));

        label.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        add(label, BorderLayout.NORTH);
        initCenterPart();
        initSouthPart();
    }

    /**
     * Composes the center part of this panel.
     */
    private void initCenterPart() {
        JPanel panel = new JPanel(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        DocumentListener createWSDLButtonEnabler = new DocumentListener() {
            public void changedUpdate(final DocumentEvent e) {
                checkPanelData();
            }

            public void insertUpdate(final DocumentEvent e) {
                checkPanelData();
            }

            public void removeUpdate(final DocumentEvent e) {
                checkPanelData();
            }
        };
        final JCheckBox useSunOrb
                = new JCheckBox(bundle.getString("LBL_UseSunOrb"));
        final JTable properties = new JTable(propertiesModel);

        // 1st row
        c.insets  = new Insets(V_GAP, H_GAP, V_GAP, H_GAP);
        c.anchor  = GridBagConstraints.WEST;
        c.fill    = GridBagConstraints.NONE;
        c.weightx = 1.0;
        panel.add(new JLabel(bundle.getString("LBL_CorbaServiceName")), c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        corbaServiceName.setToolTipText(
                bundle.getString("TXT_CorbaServiceName"));
        corbaServiceName.getDocument().addDocumentListener(docListener);
        corbaServiceName.getDocument().addDocumentListener(
                createWSDLButtonEnabler);
        panel.add(corbaServiceName, c);

        // 2nd row
        c.gridwidth = 1;
        panel.add(new JLabel(bundle.getString("LBL_NameSpace")), c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        nameSpace.getDocument().addDocumentListener(createWSDLButtonEnabler);
        panel.add(nameSpace, c);

        // 3rd row
        c.gridwidth = 1;
        panel.add(new JLabel(bundle.getString("LBL_EndpointName")), c);
        c.gridwidth = GridBagConstraints.REMAINDER;
        endpointName.addKeyListener(new EndpointNameSuggester());
        endpointName.getDocument().addDocumentListener(createWSDLButtonEnabler);
        panel.add(endpointName, c);

        // 4th row
        c.anchor = GridBagConstraints.WEST;
        useSunOrb.addItemListener(new ItemListener() {
            public void itemStateChanged(final ItemEvent e) {

                // Updates properties only if check has been selected
                if (useSunOrb.isSelected()) {
                    updateProperty(ORB_HOST_PROPERTY_NAME,
                                   ORB_HOST_PROPERTY_VALUE);
                    updateProperty(ORB_PORT_PROPERTY_NAME,
                                   ORB_PORT_PROPERTY_VALUE);
                }
            }
        });
        panel.add(useSunOrb, c);

        // 5th row
        c.anchor = GridBagConstraints.NORTHWEST;
        c.gridwidth = 1;
        panel.add(new JLabel(bundle.getString("LBL_Properties")), c);
        c.anchor = GridBagConstraints.WEST;
        c.gridwidth = GridBagConstraints.REMAINDER;
        properties.setPreferredScrollableViewportSize(new Dimension(480, 100));
        properties.addFocusListener(new FocusListener() {
            public void focusGained(final FocusEvent e) {
            }

            public void focusLost(final FocusEvent e) {

                // Stops editing, writes data to TableModel and validates data
                if (properties.isEditing()) {
                    properties.getCellEditor().stopCellEditing();
                    checkPanelData();
                }
            }
        });
        propertiesModel.addTableModelListener(new TableModelListener() {
            public void tableChanged(final TableModelEvent e) {
                checkPanelData();
            }
        });
        panel.add(new JScrollPane(properties), c);

        add(panel);
    }

    /**
     * Composes the south part of this panel.
     */
    private void initSouthPart() {
        JPanel panel
                = new JPanel(new FlowLayout(FlowLayout.RIGHT, H_GAP, V_GAP));

        panel.add(createWSDL);
        createWSDL.setEnabled(false);
        createWSDL.setMnemonic(bundle.getString("MNEM_CreateWSDL").charAt(0));
        panel.add(cancel);
        cancel.setMnemonic(bundle.getString("MNEM_Cancel").charAt(0));

        add(panel, BorderLayout.SOUTH);
    }

    /**
     * Updates the <code>JTable</code> content adding or modifing the property
     * identified by parameters. If a property called <code>name</code> doesn't
     * exist, it will be created by this method; if the property already exists,
     * its value will be changed to <code>value</code>.
     *
     * @param  name   the property name.
     * @param  value  the property value.
     */
    private void updateProperty(final String name, final String value) {
        int firstEmptyRow = -1;

        for (int row = 0; row < propertiesModel.getRowCount(); ++row) {
            String curName = (String) propertiesModel.getValueAt(
                                                    row, PROPERTY_NAME_COLUMN);

            if ((firstEmptyRow == -1) && isBlank(curName)) {
                firstEmptyRow = row;
            }
            if ((curName != null) && curName.trim().equals(name)) {
                propertiesModel.setValueAt(name, row, PROPERTY_NAME_COLUMN);
                propertiesModel.setValueAt(value, row, PROPERTY_VALUE_COLUMN);
                return;
            }
        }

        // Property not found: add a new property/row to the JTable
        if (firstEmptyRow == -1) {
            propertiesModel.addRow(new Object[] { "", "" });
            firstEmptyRow = propertiesModel.getRowCount() - 1;
        }
        propertiesModel.setValueAt(name, firstEmptyRow, PROPERTY_NAME_COLUMN);
        propertiesModel.setValueAt(value, firstEmptyRow, PROPERTY_VALUE_COLUMN);
    }

    /**
     * Calculates the suggested endpoint name from current value of name
     * service.
     *
     * @return  the endpoint name calculated from current name service.
     */
    private String suggestedEndpoint() {
        String service = corbaServiceName.getText().trim();
        int lastSlashIndex = service.lastIndexOf('/');

        if (lastSlashIndex >= 0) {
            service = service.substring(lastSlashIndex);
        }
        return service;
    }

    /**
     * Validates all data contained in this <code>JPanel</code>, enabling or
     * disabling confirm button. The button is enabled if and only if all data
     * result valid.
     */
    private void checkPanelData() {
        createWSDL.setEnabled(!isBlank(corbaServiceName) && !isBlank(nameSpace)
                && !isBlank(endpointName) && propertiesAreValid());
    }

    /**
     * Tests if the specified text field is blank, containing only spaces.
     *
     * @param   textField  the text field to check. Must be not
     *                     <code>null</code>.
     * @return  <code>true</code> if and only if the text field contains no text
     *          or only space characters (<code>'&#92;u0020'</code>).
     */
    private static boolean isBlank(final JTextField textField) {
        return textField.getText().trim().length() == 0;
    }

    /**
     * Validates current properties values.
     *
     * @return  <code>true</code> if and only if there are not properties with
     *          value but without name. The checks made ignore blank characters.
     */
    private boolean propertiesAreValid() {
        for (int row = propertiesModel.getRowCount() - 1; row >= 0; --row) {
            Object name = propertiesModel.getValueAt(row, PROPERTY_NAME_COLUMN);
            Object value = propertiesModel.getValueAt(row,
                                                      PROPERTY_VALUE_COLUMN);

            if (isBlank((String) name) && !isBlank((String) value)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks if a <code>String</code> is empty (<code>""</code>) or made by
     * blank characters only (<code>'&#92;u0020'</code>).
     *
     * @param   s  the string to check.
     * @return  <code>true</code> if and only if <code>s</code> is
     *          <code>null</code> or contains only space characters.
     */
    private static boolean isBlank(final String s) {
        return (s == null) || (s.trim().length() == 0);
    }

    /**
     * Gets the button to confirm WSDL creation.
     *
     * @return  the button to confirm WSDL creation, contained in this panel.
     */
    JButton getCreateWSDLButton() {
        return createWSDL;
    }

    /**
     * Gets the button to cancel WSDL creation.
     *
     * @return  the button to cancel WSDL creation.
     */
    JButton getCancelButton() {
        return cancel;
    }

    /**
     * Creates the WSDL descriptor from current values contained in this panel.
     * This method must be called when all values inserted by the user are legal
     * and correct.
     *
     * @return  a new <code>WSDLDescriptor</code> describing the WSDL to
     *          generate from current values contained in this panel.
     */
    WSDLDescriptor createWSDLDescriptor() {
        WSDLDescriptor desc = new WSDLDescriptor(corbaServiceName.getText(),
                                nameSpace.getText(), endpointName.getText());
        Properties p = new Properties();

        for (int row = propertiesModel.getRowCount() - 1; row >= 0; --row) {
            String name  = (String) propertiesModel.getValueAt(
                                            row, PROPERTY_NAME_COLUMN);
            String value = (String) propertiesModel.getValueAt(
                                            row, PROPERTY_VALUE_COLUMN);

            if (!isBlank(name)) {
                if (isBlank(value)) {
                    value = "";
                }
                p.setProperty(name, value);
            }
        }
        desc.setOrbProperties(p);
        return desc;
    }

    /**
     * Listener that suggests the endpoint name while name service and name
     * space fields are edited.
     *
     * @author  <a href="mailto:mcimatti@imolinfo.it">Marco Cimatti</a>
     */
    private final class EndpointNameSuggester implements KeyListener, Runnable {

        public void keyPressed(final KeyEvent e) {
        }

        public void keyReleased(final KeyEvent e) {
        }

        public void keyTyped(final KeyEvent e) {

            /*
             * We need to delay the listener execution, so it will be invoked
             * AFTER ALL OTHERS KeyListener have terminated their execution and
             * the JTextField is updated cause key typed
             */
            SwingUtilities.invokeLater(this);
        }

        public void run() {
            String endpoint = endpointName.getText().trim();

            /*
             * If the user modify the suggested endpoint name by-hand, we
             * terminate to automatically suggest the endpoint name
             */
            if (!endpoint.equals(suggestedEndpoint())) {
                corbaServiceName.getDocument().removeDocumentListener(
                        docListener);
                endpointName.removeKeyListener(this);
            }
        }
    }
}
