/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.bpel.model.validation.xpath;

import java.text.MessageFormat;
import java.util.LinkedList;
import java.util.List;
import org.netbeans.modules.bpel.model.api.Activity;

import org.netbeans.modules.bpel.model.api.BooleanExpr;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.model.api.Branches;
import org.netbeans.modules.bpel.model.api.Condition;
import org.netbeans.modules.bpel.model.api.ContentElement;
import org.netbeans.modules.bpel.model.api.DeadlineExpression;
import org.netbeans.modules.bpel.model.api.ExpressionLanguageSpec;
import org.netbeans.modules.bpel.model.api.FinalCounterValue;
import org.netbeans.modules.bpel.model.api.For;
import org.netbeans.modules.bpel.model.api.From;
import org.netbeans.modules.bpel.model.api.OnAlarmEvent;
import org.netbeans.modules.bpel.model.api.RepeatEvery;
import org.netbeans.modules.bpel.model.api.StartCounterValue;
import org.netbeans.modules.bpel.model.api.To;
import org.netbeans.modules.bpel.model.impl.services.ExpressionUpdater.ExpressionException;
import org.netbeans.modules.bpel.model.impl.services.ExpressionUpdater.InvalidExpressionException;
import org.netbeans.modules.bpel.model.impl.services.XPathVisitorAdaptor;
import org.netbeans.modules.bpel.model.validation.SimpleValidationVisitorImpl;
import org.netbeans.modules.xml.xam.spi.Validator;
import org.netbeans.modules.xml.xam.spi.Validator.ResultItem;
import org.netbeans.modules.xml.xam.spi.Validator.ResultType;
import org.netbeans.modules.xml.xpath.AbstractXPathModelHelper;
import org.netbeans.modules.xml.xpath.XPathException;
import org.netbeans.modules.xml.xpath.XPathExpression;
import org.netbeans.modules.xml.xpath.XPathExtensionFunction;
import org.netbeans.modules.xml.xpath.XPathModel;
import org.openide.util.NbBundle;


/**
 *
 * Validates expressions in TEXT child of DOM elements against
 * XPath expression.
 *
 * @author ads
 *
 */
class BpelXpathValidatorVisitor extends SimpleValidationVisitorImpl {
    
    public static final String XPATH_EXPRESSION_TYPE =
            "urn:oasis:names:tc:wsbpel:2.0:sublang:xpath1.0"; // NOI18N
    
    public static final String BAD_XPATH_EXPRESSION = "BAD_XPATH_EXPRESSION"; // NOI18N
    
    public static final String INVALID_FUNCTION_FOUND = "INVALID_FUNCTION_FOUND"; // NOI18N
    
    public static final String INVALID_FUNCTIONS_FOUND = "INVALID_FUNCTIONS_FOUND"; // NOI18N
    
    BpelXpathValidatorVisitor(Validator validator) {
        myValidator = validator;
        init();
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.BooleanExpr)
     */
    @Override
    public void visit( BooleanExpr expr ) {
        checkXPathExpression( expr );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.Branches)
     */
    @Override
    public void visit( Branches branches ) {
        checkXPathExpression( branches );
    }
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.Condition)
     */
    @Override
    public void visit( Condition condition ){
        checkXPathExpression( condition );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.DeadlineExpression)
     */
    @Override
    public void visit( DeadlineExpression expression ) {
        checkXPathExpression( expression );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.FinalCounterValue)
     */
    @Override
    public void visit( FinalCounterValue value ) {
        checkXPathExpression( value );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.For)
     */
    @Override
    public void visit( For fo ) {
        checkXPathExpression( fo );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.From)
     */
    @Override
    public void visit( From from ) {
        checkXPathExpression( from );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.RepeatEvery)
     */
    @Override
    public void visit( RepeatEvery repeatEvery ) {
        checkXPathExpression( repeatEvery );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.StartCounterValue)
     */
    @Override
    public void visit( StartCounterValue value ) {
        checkXPathExpression( value );
    }
    
    /* (non-Javadoc)
     * @see org.netbeans.modules.bpel.model.api.support.BpelModelVisitorAdaptor#visit(org.netbeans.modules.bpel.model.api.To)
     */
    @Override
    public void visit( To to ) {
        checkXPathExpression( to );
    }
    
    public void visit( OnAlarmEvent event) {
        validatedActivity = event;
    }
    
    protected void visit( Activity activity ) {
        validatedActivity = activity;
    }
    
    private void checkXPathExpression( ContentElement element ){
        String content = element.getContent();
        if ( content == null ){
            return;
        }
        content = content.trim();
        if ( content.length() == 0 ) {
            return;
        }
        String expressionType = null;
        if ( element instanceof ExpressionLanguageSpec ) {
            expressionType = ((ExpressionLanguageSpec) element).
                    getExpressionLanguage();
        }
        try {
            checkExpression( expressionType, content, element);
        } catch ( InvalidExpressionException e) {
            String bundleKey = e.getMessage();
            Throwable throwable = getInitialCause( e );
            if ( throwable == null ) {
                addError( bundleKey, (BpelEntity)element, content , "" );
            } else {
                addError( bundleKey, (BpelEntity)element, content , " : "+  // NOI18N
                        throwable.getMessage());
            }
        } catch ( InvalidFunctionException e) {
            String bundleKey = INVALID_FUNCTION_FOUND;
            String[] functions = e.getFunctions();
            if ( functions.length >1 ) {
                bundleKey = INVALID_FUNCTIONS_FOUND;
            }
            StringBuilder builder = new StringBuilder();
            for (String function : functions) {
                builder.append( function );
                builder.append( " ,");                  // NOI18N
            }
            // Just to be sure that buffer have at least two chars
            if ( builder.length() <2 ) {
                builder.append( "  " );
            }
            
            addError( bundleKey , (BpelEntity)element, content ,
                    builder.substring( 0, builder.length() -2 ));
        } catch ( ExpressionException e) {
            assert false;
        }
        
    }
    
    public void checkExpression(String expressionType, 
            String expression, ContentElement element)
            throws ExpressionException {
        // we can handle only xpath expressions.
        if ( expressionType == null ||
                XPATH_EXPRESSION_TYPE.equals( expressionType )) {
            AbstractXPathModelHelper helper= AbstractXPathModelHelper.getInstance();
            XPathModel model = helper.newXPathModel();
            XPathExpression xpath = null;
            try {
                xpath = model.parseExpression(expression);
            } catch (XPathException e) {
                throw new InvalidExpressionException( BAD_XPATH_EXPRESSION , e );
            }
            assert xpath!= null;
            //
            InvalidFunctionsVisitor visitor = new InvalidFunctionsVisitor();
            xpath.accept( visitor );
            String[] functions = visitor.getInvalidFunctions();
            if ( functions.length >0 ) {
                throw new InvalidFunctionException( functions );
            }
            //
            assert validatedActivity != null;
            PathValidationContext context = new PathValidationContext(
                    myValidator, this, validatedActivity, element);
            PathValidatorVisitor pathVVisitor = new PathValidatorVisitor(context);
            xpath.accept(pathVVisitor);
        }
    }
    
    private void addError( String bundleKey , BpelEntity entity, Object... values ){
        String str =
                NbBundle.getMessage( BpelXpathValidator.class ,
                bundleKey );
        if ( values!= null && values.length >0 ) {
            str = MessageFormat.format(str, values );
        }
        ResultItem resultItem = new ResultItem( myValidator,
                ResultType.ERROR, entity, str );
        getResultItems().add( resultItem );
    }
    
    private Throwable getInitialCause( Throwable throwable ) {
        if ( throwable == null ) {
            return null;
        }
        Throwable cause = throwable.getCause();
        if ( cause == null ) {
            return throwable;
        } else {
            return getInitialCause( cause );
        }
    }
    
    class InvalidFunctionsVisitor extends XPathVisitorAdaptor {
        
    /* (non-Javadoc)
     * @see org.netbeans.modules.xml.xpath.visitor.AbstractXPathVisitor#visit(org.netbeans.modules.xml.xpath.XPathExtensionFunction)
     */
        @Override
        public void visit( XPathExtensionFunction function ) {
            if ( !AbstractXPathModelHelper.getInstance().isValidFunction(
                    function.getName() ) ) {
                myInvalidFunctions.add( function.getName() );
            }
            super.visit( function );
        }
        
        String[] getInvalidFunctions() {
            return myInvalidFunctions.toArray( new String[ myInvalidFunctions.size()] );
        }
        
        private List<String> myInvalidFunctions = new LinkedList<String>();
        
    }
    
    public class InvalidFunctionException extends ExpressionException {

        private static final long serialVersionUID = -6951046982294553909L;

        /**
         * {@inheritDoc}
         */
        public InvalidFunctionException( String[] functions ) {
            super();
            myFunctions = functions;
        }
        
        public String[] getFunctions() {
            return myFunctions;
        }
        
        private String[] myFunctions;
        
    }
    
    
    private Validator myValidator;
    
    private BpelEntity validatedActivity; 
}
