/*
** Copyright (c) Massachusetts Institute of Technology 1994-1998.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. Works derived from this code are not distributed for
**        commercial gain without explicit permission from MIT 
**        (for permission contact lclint-request@sds.lcs.mit.edu).
*/
/*
** syntable.c
**
** Larch/C Interface language synonym table
**
**	This table stores synonyms for the Larch/C Interface Language.  It is
**	essentially an array of token-handles indexed by string-handles.
**	Therefore, synonyms (strings) can be converted to the actual token.
**
**  AUTHORS:
**	J.P. Wild
**
**
**  CREATION DATE:  90.08.10
*/

# include "lclintMacros.nf"
# include "basic.h"
# include "lcltokentable.h"
# include "lclsyntable.h"

static long unsigned MaxSyn;	/* size of SynTable[]              */
static /*@only@*/ /*@reldef@*/ /*@null@*/ lsymbol *SynTable;
static void
  AllocSynTable (void)
  /*@globals SynTable, MaxSyn@*/
  /*@modifies *SynTable, MaxSyn@*/;

void
LCLAddSyn (lsymbol ntok, lsymbol otok)
{
  while (otok >= MaxSyn)
    {
      /* No more space available.  Allocate more. */
      AllocSynTable ();
    }

  llassert (SynTable != NULL);

  if (SynTable[ntok] == 0)
    {
     /* Entry is empty. Fill it in. */
      SynTable[ntok] = otok;

      /* Mark oldToken as having a synonym. */
      LCLSetTokenHasSyn (otok, TRUE);
    }
  else
    {
      llbuglit ("LCLAddSyn: invalid argument");
    }
}

/*@exposed@*/ ltoken
LCLGetTokenForSyn (lsymbol ntok)
{
  llassert (SynTable != NULL);

  if (!((ntok < MaxSyn) || (SynTable[ntok] != 0)))
    llbuglit ("LCLGetSyn: bad argument");

  return LCLGetToken (SynTable[ntok]);
}

bool
LCLIsSyn (lsymbol str)
{
  if (MaxSyn == 0)
    {
      return FALSE;
    }
  else
    {
      llassert (SynTable != NULL);

      if (str < MaxSyn)
	{
	  /* Check for synonym entry in table. */
	  return (SynTable[str] != 0);
	}
      else
	{
	  /* No token for synonym.  Return FALSE. */
	  return FALSE;
	}
    }
}
  
static void
AllocSynTable (void) /*@globals SynTable; @*/
{
  long unsigned newSize, oldSize;
  long unsigned int i;

  oldSize = MaxSyn;

  if (oldSize == 0)
    {
      /* First time SynTable allocated.  Set initial size. */
      newSize = INITSYNTABLE;
      SynTable = (lsymbol *) dmalloc (newSize * sizeof (*SynTable));
    }
  else
    {
      lsymbol *oldSynTable = SynTable; 

      llassert (oldSynTable != NULL);

      /* Synonym table already allocated.  Calulate extension size. */
      newSize = (unsigned long) (DELTASYNTABLE * oldSize);
      SynTable = (lsymbol *) dmalloc (newSize * sizeof (*SynTable));

      for (i = 0; i < oldSize; i++)
	{
	  SynTable[i] = oldSynTable[i];
	}
      
      sfree (oldSynTable);
    }

  /* Zero out new allocated space.  Need to detect when cells are empty     */
  /* and do this by checking that SynTable[x] == 0.			    */

  /* ###  Should the "for" loop be replaced with the following?	            */
  /* #if VMS								    */
  /* # include <string.h>;						    */
  /* #else								    */
  /* # include <memory.h>;						    */
  /*									    */
  /* memset (SynTable[oldSize], 0,					    */
  /*	      (newSize - oldSize) * sizeof (*SynTable));		            */
  
  for (i = oldSize; i < newSize; i++)
    {
      SynTable[i] = 0;
    }

  MaxSyn = newSize;
}


void
LCLSynTableInit (void)
{
  MaxSyn = 0;
}

void
LCLSynTableReset (void)
{
}

void
LCLSynTableCleanup (void)
{
  sfree (SynTable);
  SynTable = NULL;
}
