// vw_converter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <sys/soundcard.h>
#include "localdefs.h"
#include "application.h"
#include "vw_converter.h"
#include "typeconvert.h"

VW_Converter::VW_Converter() : ConverterDevice("/dev/dsp"), audioBufferSize(0) {
    BUG("VW_Converter::VW_Converter()");
}

VW_Converter::~VW_Converter() {
	stop();
}

boolean
VW_Converter::isPlayableFormat(DataType type) {
	return (type < IntData && type != SignedCharData);
}

// what is best format to play (if given choice)

DataType
VW_Converter::bestPlayableType() {
	return ShortData;
}

// for future use -- no real way to do this yet

int
VW_Converter::pause() {
	return false;
}

int
VW_Converter::stop() {
	return ioctl(SNDCTL_DSP_RESET, 0) && Super::stop();
}

int
VW_Converter::checkChannels(int chans) {
	int status = false;
	if(chans != 1 && chans != 2)
		Application::alert("Converter only handles monaural and stereo files.");
	else
		status = true;
	return status;
}

int
VW_Converter::doConfigure() {
    BUG("VW_Converter::doConfigure()");
	int status = false;
	if(reOpen()) {
		int dsp_speed = sampleRate();
		int dsp_stereo = (channels() == 2);
		int sampleFormat = 0;
		switch(dataType()) {
			case UnsignedCharData: sampleFormat = AFMT_U8; break;
			case SignedCharData: sampleFormat = AFMT_S8; break;
//			case ALawData: sampleFormat = AFMT_A_LAW; break;
			case MuLawData: sampleFormat = AFMT_MU_LAW; break;
			case ShortData: sampleFormat = AFMT_S16_LE; break;
			default: break;
		};
		int confirmedFormat = sampleFormat;

		int sizeCode = 0x1;
		// desired buffer size is 1/10 of a second's worth of sound
		int bufferSize = type_to_sampsize(dataType()) *
		                 round(float(channels()) * sampleRate() / 10.0);
#ifdef DEBUG
		fprintf(stderr, "data type = %d, data size = %d bytes\n",
			(int) dataType(), type_to_sampsize(dataType()));

		fprintf(stderr, "requesting buffer size %d\n", bufferSize);
#endif
		static const double L10_2 = log10(2.0);
		sizeCode = int(log10(bufferSize) / L10_2);
		sizeCode |= 2 << 16;

		// sizeCode = 0x0002XXXX where XXXX is (int) log2(bufsize)
		// and 0002 is 2 max number of fragments

#ifdef DEBUG
		fprintf(stderr, "setting frag size code to 0x%x\n", sizeCode);
#endif

		if (!ioctl (SNDCTL_DSP_SETFRAGMENT, (char *) &sizeCode))
			error("Unable to set fragment size.");
		else if (!ioctl(SNDCTL_DSP_SETFMT, (char *) &confirmedFormat))
			error("Unable to set sample format.");
		else if(confirmedFormat != sampleFormat)
			error("This sample format not supported by hardware.");
		else if (!ioctl(SNDCTL_DSP_STEREO, (char *) &dsp_stereo))
			error("Unable to set channel attribute.");
		else if (!ioctl (SNDCTL_DSP_SPEED, (char *) &dsp_speed))
			error("Unable to set sample rate.");
		else if(!ioctl(SNDCTL_DSP_GETBLKSIZE, (char *) &audioBufferSize))
			error("Unable to get audio buffer size.");
		else status = true;
#ifdef DEBUG
		fprintf(stderr, "audio buffer size is %d\n", audioBufferSize);
#endif
	}
	return status;
}

// return size of buffer, in bytes, to be written to the device during play

int
VW_Converter::writeSize() {
	return audioBufferSize;
}

// return size of buffer, in bytes, to be read from the device during record

int
VW_Converter::readSize() {
	return audioBufferSize;
}

