/*
Copyright (c) 1998 Peter Zelezny.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "style.h"
#include "xchat.h"
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <time.h>
#include "dcc.h"


static char *dcctypes[]= { "SEND", "RECV", "CHAT", "CHAT", "DRAW", "DRAW"};
static char *dccstat[]= { "Queued", "Active", "Failed", "Done", "Connect", "Aborted" };

struct dccwindow dccrwin; // recv
struct dccwindow dccswin; // send
struct dccwindow dcccwin; // chat

extern GSList *sess_list;


struct DCC *new_dcc(void);
void open_dcc_recv_window(void);
void update_dcc_recv_window(void);
void open_dcc_send_window(void);
void update_dcc_send_window(void);
void open_dcc_chat_window(void);
void update_dcc_chat_window(void);
void dcc_send(struct session *sess, char *tbuf, char *to, char *file);
struct DCC *find_dcc(char *nick, char *file, int type);

extern void private_msg(struct server *serv, char *tbuf, char *from, char *ip, char *text);
extern void PrintText(struct session *sess, char *text);
extern int waitline(int sok, char *buf, int bufsize);
extern void dcc_open_draw_window(struct DCC *dcc);
extern void dcc_close_draw_window(struct DCC *dcc);
extern void dcc_read_draw(struct DCC *dcc, gint sok);

extern struct DCC firstDCC;



char *file_part(char *file)
{
   char *filepart = file;
   while(1)
   {
      switch(*file)
      {
       case 0:
	 return(filepart);
       case '/':
	 filepart = file + 1;
	 break;
      }
      file++;
   }
}

void dcc_filereq_done(GtkWidget *wid, struct dcc_send *ds)
{
   char tbuf[256];
   dcc_send(ds->sess, tbuf, ds->nick,
	    gtk_file_selection_get_filename(GTK_FILE_SELECTION(ds->freq)));
   gtk_widget_destroy(ds->freq);  
   free(ds);
}

void dcc_send_filereq(struct session *sess, char *nick)
{
   struct dcc_send *ds = malloc(sizeof(struct dcc_send));
   char tbuf[128];
   GtkWidget *freq;
   
   ds->sess = sess;
   ds->nick = nick;
   
   sprintf(tbuf, "Send file to %s", nick);
   
   freq = gtk_file_selection_new(tbuf);
   ds->freq = freq;
   gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(freq)->cancel_button),
			     "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(freq));
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(freq)->ok_button),
		      "clicked", (GtkSignalFunc) dcc_filereq_done, (gpointer)ds);
   gtk_widget_show(freq);
}

long net_connect(unsigned long addr, long port)
{
   long sok;
   struct sockaddr_in SAddr;

   sok = socket(AF_INET, SOCK_STREAM, 0);
   if(sok == -1) return -1;

   memset(&SAddr, 0, sizeof(SAddr));
   SAddr.sin_port = htons(port);
   SAddr.sin_family = AF_INET;
   SAddr.sin_addr.s_addr = addr;
   
   fcntl(sok, F_SETFL, O_NONBLOCK);
   connect(sok, (struct sockaddr *)&SAddr, sizeof(SAddr));
   fcntl(sok, F_SETFL, O_NONBLOCK);
   //fcntl(sok, F_SETFL, 0);
   return sok;
}

void update_dcc_window(int type)
{
   switch(type)
   {
    case TYPE_SEND: update_dcc_send_window(); break;
    case TYPE_RECV: update_dcc_recv_window(); break;
    case TYPE_CHATRECV:
    case TYPE_CHATSEND: update_dcc_chat_window(); break;
   }
}

void dcc_close(struct DCC *dcc, int stat, int destroy)
{
   if(dcc->sok != -1)
   {
      gdk_input_remove(dcc->iotag);
      close(dcc->sok);
      dcc->sok = -1;
      switch(dcc->stat)
      {      
       case STAT_ACTIVE:
	 {
	    char buf[128];
	    sprintf(buf, STARTON" DCC %s connection closed to %s\n",
		 dcctypes[(int)dcc->type], dcc->nick);
	    PrintText(dcc->sess, buf);
	    break;
	 }
      }
   }
   if(dcc->fp != -1 )
   {
      close(dcc->fp);
      dcc->fp = -1;
   }
   dcc->stat = stat;
   if(destroy) dcc->used = 0;
   update_dcc_window(dcc->type);
   if(dcc->type == TYPE_DRAWSEND || dcc->type == TYPE_DRAWRECV)
   {
      if(dcc->dccdraw) dcc_close_draw_window(dcc);
   }
}

unsigned long dcc_write_chat(char *nick, char *text)
{
   struct DCC *dcc = find_dcc(nick, "", TYPE_CHATRECV);
   if(!dcc) dcc = find_dcc(nick, "", TYPE_CHATSEND);
   if(dcc && dcc->stat == STAT_ACTIVE)
   {
      int len = strlen(text);
      dcc->size += len;
      send(dcc->sok, text, len, 0);
      send(dcc->sok, "\n", 1, 0);
      update_dcc_chat_window();
      return dcc->addr;
   } else
     return 0;
}

void dcc_read_chat(struct DCC *dcc, gint sok)
{
   long len;
   char tbuf[1226];
   char buf[1026];

   //len = waitline(sok, buf, sizeof buf);
   len = recv(sok, buf, sizeof buf-2, 0);
   if(len < 1)
   {
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }
   if(len > 0)
   { 
      GSList *list = sess_list;
      struct session *sess;
      buf[len] = 0;
      dcc->pos += len;
      while(list)
      {
	 sess = (struct session *)list->data;
	 if(sess->server == dcc->sess->server)
	 {
	    private_msg(sess->server, tbuf, dcc->nick, "", buf);
	    update_dcc_chat_window();
	    return;
	 }
	 list = list->next;
      }
   }
}

void dcc_read(struct DCC *dcc, gint sok)
{
   long n;
   char buf[4098];

   if(dcc->fp == -1)
   {
      if(dcc->resumable)
      {
	 dcc->fp = open(dcc->file, O_WRONLY | O_APPEND);
	 dcc->pos = dcc->resumable;
      } else
	 dcc->fp = open(dcc->file, O_WRONLY | O_CREAT, 0600);
   }
   
   if(dcc->fp == -1)
   {
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }
      
   n = recv(dcc->sok, buf, sizeof buf-2, 0);
   if(n < 0)
   {
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }
   if(n > 0)
   {
      long pos, sec;
      write(dcc->fp, buf, n);
      dcc->pos += n;
      pos = htonl(dcc->pos);
      send(dcc->sok, (char *)&pos, 4, 0);
      
      sec = time(0) - dcc->starttime;
      if(sec < 1) sec = 1;
      dcc->cps = dcc->pos / sec;
	
      if(dcc->pos >= dcc->size)
	dcc_close(dcc, STAT_DONE, FALSE);
      else {
	 if(dcc->pos >= (dcc->oldpos + 4096))
	 {
	    update_dcc_recv_window();
	    dcc->oldpos = dcc->pos;
	 }
      }
   }
}

void dcc_connect_finished(struct DCC *dcc, gint sok)
{ 
   gdk_input_remove(dcc->iotag);

   if(send(dcc->sok, 0, 0, 0) < 0)
      dcc->stat = STAT_FAILED;
   else {
      dcc->stat = STAT_ACTIVE;
      switch(dcc->type)
      {
       case TYPE_RECV: dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			 (GdkInputFunction)dcc_read, dcc); break;
       case TYPE_CHATRECV: dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			 (GdkInputFunction)dcc_read_chat, dcc); break;
       case TYPE_DRAWRECV: dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			 (GdkInputFunction)dcc_read_draw, dcc);
	       dcc_open_draw_window(dcc);
	 break;
      }
   }
   update_dcc_window(dcc->type);
   time(&dcc->starttime);

   {
      char tbuf[256];
      unsigned char *a = (char *)&dcc->addr;
      sprintf(tbuf, STARTON" DCC %s connection established to %s (%d.%d.%d.%d)\n",
	      dcctypes[(int)dcc->type], dcc->nick, a[0], a[1], a[2], a[3]);
      PrintText(dcc->sess, tbuf);
   }
}

void dcc_connect(struct session *sess, struct DCC *dcc)
{
   if(dcc->stat == STAT_CONNECTING) return;
   dcc->stat = STAT_CONNECTING;
   dcc->sok = net_connect(dcc->addr, dcc->port);
   if(dcc->sok == -1)
   {
      dcc->stat = STAT_FAILED;
      if(sess) PrintText(sess, STARTON" Connection failed.\n");
      update_dcc_window(dcc->type);
      return;
   }
   update_dcc_window(dcc->type);
   dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_WRITE,
			 (GdkInputFunction)dcc_connect_finished, dcc);
}

void dcc_send_data(struct DCC *dcc, gint sok)
{
   char buf[1024];

   if(sok != -1)
   {
      recv(dcc->sok, (char *)&dcc->ack, 4, 0);
      dcc->ack = ntohl(dcc->ack);
   }
   
   if(dcc->pos < dcc->size)
   {
      long sec, len = read(dcc->fp, buf, sizeof buf);
      
      if(send(dcc->sok, buf, len, 0) < 0)
      {
	 dcc_close(dcc, STAT_FAILED, FALSE);
	 return;
      }
      
      sec = time(0) - dcc->starttime;
      if(sec < 1) sec = 1;
      dcc->cps = dcc->pos / sec;
      dcc->pos += len;
   }

   if(dcc->pos >= dcc->size  &&  dcc->ack >= dcc->size)
     dcc_close(dcc, STAT_DONE, FALSE);
   else {
      if(dcc->pos >= (dcc->oldpos + 4096))
      {
	 dcc->oldpos = dcc->pos;
	 update_dcc_send_window();
      }
   }
}

void dcc_accept(struct DCC *dcc, gint sokk)
{
   struct sockaddr_in CAddr;
   int len, sok;
   char tbuf[256];
   
   len = sizeof(CAddr);
   sok = accept(dcc->sok, (struct sockaddr *)&CAddr, &len);
   gtk_input_remove(dcc->iotag);
   close(dcc->sok);
   dcc->sok = sok;
   dcc->addr = ntohs(CAddr.sin_addr.s_addr);
   if(sok < 0)
   {
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }
   dcc->stat = STAT_ACTIVE;
   time(&dcc->starttime);
   switch(dcc->type)
   {
    case TYPE_SEND:
      dcc->iotag = gdk_input_add(sok, GDK_INPUT_READ,
			      (GdkInputFunction)dcc_send_data, dcc);
      dcc_send_data(dcc, -1);
      break;
      
    case TYPE_CHATSEND:
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			 (GdkInputFunction)dcc_read_chat, dcc);
      break;

    case TYPE_DRAWSEND:
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			 (GdkInputFunction)dcc_read_draw, dcc);
      dcc_open_draw_window(dcc);
      break;
   }

   sprintf(tbuf, STARTON" DCC %s connection established from %s (%s)\n",
	   dcctypes[(int)dcc->type], dcc->nick, inet_ntoa(CAddr.sin_addr));
   PrintText(dcc->sess, tbuf);
}

int dcc_listen_init(struct DCC *dcc)
{
   int len;
   struct sockaddr_in SAddr;

   dcc->sok = socket(AF_INET, SOCK_STREAM, 0);
   if(dcc->sok == -1) return 0;

   memset(&SAddr, 0, sizeof(struct sockaddr_in));

   len = sizeof(SAddr);
   getsockname(dcc->sess->server->sok, (struct sockaddr *) &SAddr, &len); 

   SAddr.sin_family = AF_INET;
   SAddr.sin_port = 0;

   bind(dcc->sok, (struct sockaddr *)&SAddr, sizeof(SAddr));

   len = sizeof(SAddr);
   getsockname(dcc->sok, (struct sockaddr *) &SAddr, &len); 

   dcc->port = ntohs(SAddr.sin_port);
   dcc->addr = htonl(SAddr.sin_addr.s_addr);
   
   fcntl(dcc->sok, F_SETFL, O_NONBLOCK);
   listen(dcc->sok, 1);
   fcntl(dcc->sok, F_SETFL, 0);

   dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ,
			      (GdkInputFunction)dcc_accept, dcc);

   return TRUE;
}

void dcc_send(struct session *sess, char *tbuf, char *to, char *file)
{
   struct stat st;
   struct DCC *dcc = new_dcc();
   if(!dcc) return;
   
   if(stat(file, &st) != -1) 
   {
      if(*file_part(file) == 0) goto jump;
      if(st.st_size < 1) goto jump;
      dcc->sess = sess;
      dcc->stat = STAT_QUEUED;
      dcc->sok = -1;
      dcc->pos = 0;
      dcc->oldpos = 0;
      dcc->ack = 0;
      dcc->cps = 0;
      dcc->size = st.st_size;
      dcc->type = TYPE_SEND;
      dcc->fp = open(file, O_RDONLY);
      if(dcc->fp != -1)
      {
	 if(dcc_listen_init(dcc))
	 {
	    strcpy(dcc->nick, to);
	    strcpy(dcc->file, file);
	    open_dcc_send_window();
	    sprintf(tbuf, "PRIVMSG %s :\001DCC SEND %s %lu %ld %ld\001\r\n",
		 to, file_part(file), dcc->addr, dcc->port, dcc->size);
	    send(sess->server->sok, tbuf, strlen(tbuf), 0);
	    sprintf(tbuf, STARTON" Offering %s to %s\n", file, to);
	    PrintText(sess, tbuf);
	 } else {
	    dcc_close(dcc, 0, TRUE);
	    PrintText(sess, STARTON" Failed to create listen port\n");
	 }
      }
   } else {
      jump:
	 sprintf(tbuf, STARTON" Cannot access %s\n", file);
	 PrintText(sess, tbuf);
	 dcc_close(dcc, 0, TRUE);
   }
}

struct DCC *find_dcc_from_port(int port, int type)
{
   struct DCC *dcc = &firstDCC;
   do
   {
      if(dcc->used && dcc->port == port &&
	 dcc->stat == STAT_QUEUED &&
	 dcc->type == type)
	return dcc;
      dcc = dcc->next;
   } while(dcc);
   return 0;  
}

struct DCC *find_dcc(char *nick, char *file, int type)
{
   struct DCC *dcc = &firstDCC;
   do
   {
      if(dcc->used)
      {
	 if(!strcasecmp(nick, dcc->nick))
	 {
	    if(!strcasecmp(file, file_part(dcc->file)))
	    {
	       if(type == -1 || dcc->type == type) return dcc;
	    }
	 }
      }
      dcc = dcc->next;
   } while(dcc);
   return 0;
}

void dcc_abort_nick_file(char *nick, char *file)
{
   struct DCC *dcc = find_dcc(nick, file, -1);
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_QUEUED:
       case STAT_CONNECTING:
       case STAT_ACTIVE:
	  dcc_close(dcc, STAT_ABORTED, FALSE);
       	  break;
       default:	
	  dcc_close(dcc, 0, TRUE);
      }
   }
}

void dcc_resume_nick_file(char *nick, char *file)
{
   struct DCC *dcc = find_dcc(nick, file, TYPE_RECV);
   if(dcc)
   {
     if(dcc->stat == STAT_QUEUED && dcc->resumable)
     {
	char tbuf[256];
	sprintf(tbuf, "PRIVMSG %s :\001DCC RESUME %s %ld %lu\001\r\n",
	      nick, file, dcc->port, dcc->resumable);
	send(dcc->sess->server->sok, tbuf, strlen(tbuf), 0);
     }
   }
}

void dcc_get_nick_file(char *nick, char *file)
{
   struct DCC *dcc = find_dcc(nick, file, TYPE_RECV);
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc->resumable = 0;
	   dcc_connect(0, dcc);
	   break;
	 case STAT_DONE:
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
	   break;
      }
   }
}

void dcc_get(struct session *sess, char *nick)
{
   struct DCC *dcc = &firstDCC;
   do
   {
      if(dcc->used)
      {
	 if(!strcasecmp(nick, dcc->nick))
	 {
	   if(dcc->stat == STAT_QUEUED && dcc->type == TYPE_RECV)
	   {
	      dcc->resumable = 0;
	      dcc_connect(sess, dcc);
	      return;
	   }
	 }
      }
      dcc = dcc->next;
   } while(dcc);
   if(sess) PrintText(sess, STARTON" No such DCC offer.\n");
}

void free_dccs(void)
{
   struct DCC *next;
   struct DCC *dcc = firstDCC.next;
   while(dcc)
   {
      next = dcc->next;
      free(dcc);
      dcc = next;
   }
}

struct DCC *new_dcc(void)
{
   struct DCC *dcc = &firstDCC;
   struct DCC *prev;
   while(1)
   {
      if(dcc->used == 0)
      {
	 struct DCC *next = dcc->next;
	 memset(dcc, 0, sizeof(struct DCC));
	 dcc->next = next;
	 dcc->used = 1;
	 return(dcc);
      }
      prev = dcc;
      dcc = dcc->next;
      if(!dcc)
      {
	 dcc = malloc(sizeof(struct DCC));
	 if(dcc)
	 {
	    memset(dcc, 0, sizeof(struct DCC));
	    prev->next = dcc;
	 } else
	   return 0;
      }
   }
}
 
void dcc_chat(struct session *sess, char *nick)
{
   char outbuf[256];
   struct DCC *dcc;
     
   dcc = find_dcc(nick, "", TYPE_CHATSEND);
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_ACTIVE:
       case STAT_QUEUED:
       case STAT_CONNECTING:
	 sprintf(outbuf, STARTON" Already offering CHAT to %s\n", nick);
	 PrintText(sess, outbuf);
	 return;
       case STAT_ABORTED:
       case STAT_FAILED:
	 dcc_close(dcc, 0, TRUE);
      }
   }
   
   dcc = find_dcc(nick, "", TYPE_CHATRECV);
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc_connect(0, dcc);
	   break;
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
      }
      return;
   }
   
   // offer DCC CHAT
   
   dcc = new_dcc();
   if(!dcc) return;
   dcc->sess = sess;
   dcc->stat = STAT_QUEUED;
   dcc->sok = -1;
   dcc->type = TYPE_CHATSEND;
   time(&dcc->starttime);
   strcpy(dcc->nick, nick);
   if(dcc_listen_init(dcc))
   {
      sprintf(outbuf, "PRIVMSG %s :\001DCC CHAT chat %lu %ld\001\r\n",
		 nick, dcc->addr, dcc->port);
      send(dcc->sess->server->sok, outbuf, strlen(outbuf), 0);
      sprintf(outbuf, STARTON" Offering DCC CHAT to %s\n", nick);
      PrintText(sess, outbuf);
   } else {
      dcc_close(dcc, 0, TRUE);
      PrintText(sess, STARTON" Failed to create listen port\n");
   }
}

void handle_dcc(struct session *sess, char *outbuf, char *nick, char *word[], char *word_eol[])
{
   struct DCC *dcc;
   char *type = word[5];
   int port;

   if(!strcasecmp(type, "DRAW"))
   {
      unsigned long addr;
      long port = atol(word[9]);

      sscanf(word[8], "%lu", &addr);
      addr = ntohl(addr);
      if(!addr) return;
      dcc = new_dcc();
      if(dcc)
      {
	 dcc->pos = atoi(word[6]); // x
	 dcc->size = atoi(word[7]); // y
	 dcc->sess = sess;
	 dcc->type = TYPE_DRAWRECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->sok = -1;
	 dcc->addr = addr;
	 dcc->port = port;
	 strcpy(dcc->nick, nick);
	 sprintf(outbuf,
		 STARTON" Received a DCC DRAW offer from %s\n"
		 STARTON" Type \00311/DRAW %s \003 to accept\n",
		 nick, nick);
	 PrintText(sess, outbuf);
	 return;
      }  
   }

   if(!strcasecmp(type, "CHAT"))
   {
      unsigned long addr;
      long port = atol(word[8]);

      sscanf(word[7], "%lu", &addr);
      addr = ntohl(addr);

      if(!addr) return;

      dcc = new_dcc();
      if(dcc)
      {
	 dcc->sess = sess;
	 dcc->type = TYPE_CHATRECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->sok = -1;
	 dcc->addr = addr;
	 dcc->port = port;
	 strcpy(dcc->nick, nick);
	 sprintf(outbuf, STARTON" Received a CHAT offer from %s\n", nick);
	 PrintText(sess, outbuf);
	 return;
      }
   }

   if(!strcasecmp(type, "RESUME"))
   {
      port = atol(word[7]);
      dcc = find_dcc_from_port(port, TYPE_SEND);
      if(dcc && dcc->stat == STAT_QUEUED)
      {
	 sscanf(word[8], "%lu", &dcc->resumable);
	 dcc->pos = dcc->resumable;
	 lseek(dcc->fp, dcc->pos, SEEK_SET);
	 sprintf(outbuf, "PRIVMSG %s :\001DCC ACCEPT %s %ld %lu\001\r\n",
		 dcc->nick, file_part(dcc->file), dcc->port, dcc->resumable);
	 send(dcc->sess->server->sok, outbuf, strlen(outbuf), 0);
      }
   }

   if(!strcasecmp(type, "ACCEPT"))
   {
      port = atol(word[7]);
      dcc = find_dcc_from_port(port, TYPE_RECV);
      if(dcc && dcc->stat == STAT_QUEUED) dcc_connect(0, dcc);
   }

   if(!strcasecmp(type, "SEND"))
   {
      unsigned long addr;
      char *file= file_part(word[6]);
      long port = atol(word[8]);
      long size = atol(word[9]);

      sscanf(word[7], "%lu", &addr);
      addr = ntohl(addr);

      if(!addr || !size) return;

      dcc = new_dcc();
      if(dcc)
      {
	 struct stat st;
	 if(stat(file, &st) != -1)
	 {
	   if(st.st_size < 1)
	   {
	      sprintf(outbuf, STARTON" Cannot accept %s from %s, filesize is zero.\n", file, nick);
	      PrintText(sess, outbuf);
	      dcc_close(dcc, 0, TRUE);
	      return;
	   }
	   dcc->resumable = st.st_size;
	 } else
	   dcc->resumable = 0;
	 if(st.st_size == size) dcc->resumable = 0;
	 dcc->sess = sess;
	 dcc->type = TYPE_RECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->sok = -1;
	 dcc->addr = addr;
	 dcc->port = port;
	 dcc->size = size;
	 dcc->fp = -1;
	 dcc->pos = 0;
	 dcc->oldpos = 0;
	 dcc->ack = 0;
	 dcc->cps = 0;
	 strcpy(dcc->file, file);
	 strcpy(dcc->nick, nick);
	 open_dcc_recv_window();
      }
      sprintf(outbuf, STARTON" %s has offered %s (%ld bytes)\n",
	      nick, file, size);
   } else
     sprintf(outbuf, STARTON" Received '%s' from %s\n", 
	    word_eol[4]+2, nick);
   PrintText(sess, outbuf);
}

void dcc_list(struct session *sess, char *outbuf)
{
   int i = 0;
   struct DCC *dcc = &firstDCC;
   PrintText(sess,
	     STARTON" \00308,02 Type  To/From    Status  Size    Pos     File   \003 \n"
	     STARTON"\002 \00303-------------------------------------------------\002\003 \n"
	     );
   do
   {
      if(dcc->used)
      {
	 i++;
	 sprintf(outbuf, STARTON"  %-5.5s %-10.10s %-7.7s %-7ld %-7ld %s\n",
	      dcctypes[(int)dcc->type], dcc->nick, dccstat[(int)dcc->stat],
		 dcc->size, dcc->pos, dcc->file);
	 PrintText(sess, outbuf);
      }
      dcc = dcc->next;
   } while(dcc);
   if(!i) PrintText(sess, STARTON" No active DCCs\n");
}

void close_dcc_recv_window(void)
{
   if(dccrwin.window)
   {
      gtk_widget_destroy(dccrwin.window);
      dccrwin.window = 0;
   }
}

void update_dcc_recv_window(void)
{
   struct DCC *dcc = &firstDCC;
   gchar *new[1][6];
   char size[16];
   char pos[16];
   char cps[16];
   
   if(!dccrwin.window) return;
   
   gtk_clist_clear((GtkCList*)dccrwin.list);
   new[0][2] = size;
   new[0][3] = pos;
   new[0][4] = cps;
   do
   {
      if(dcc->used && dcc->type == TYPE_RECV)
      {
	 new[0][0] = dccstat[(int)dcc->stat];
	 new[0][1] = dcc->file;
	 new[0][5] = dcc->nick;
	 sprintf(size, "%lu", dcc->size);
	 sprintf(pos, "%lu", dcc->pos);
	 sprintf(cps, "%lu", dcc->cps);
	 gtk_clist_append((GtkCList*)dccrwin.list, new[0]);
      }
      dcc = dcc->next;
   } while(dcc);
   if(dccrwin.selected != -1)
     gtk_clist_select_row((GtkCList*)dccrwin.list, dccrwin.selected, 0);
}

void resume_clicked(GtkWidget *wid, gpointer none)
{
   gchar *file, *nick;
   if(dccrwin.selected == -1) return;
   gtk_clist_get_text(GTK_CLIST(dccrwin.list), dccrwin.selected, 1, &file);
   gtk_clist_get_text(GTK_CLIST(dccrwin.list), dccrwin.selected, 5, &nick);
   dcc_resume_nick_file(nick, file);
}

void abort_clicked(GtkWidget *wid, gpointer none)
{
   gchar *file, *nick;
   if(dccrwin.selected == -1) return;
   gtk_clist_get_text(GTK_CLIST(dccrwin.list), dccrwin.selected, 1, &file);
   gtk_clist_get_text(GTK_CLIST(dccrwin.list), dccrwin.selected, 5, &nick);
   dcc_abort_nick_file(nick, file);
}

void accept_clicked(GtkWidget *wid, gpointer none)
{
   gchar *file, *nick;
   if(dccrwin.selected == -1) return;
   gtk_clist_get_text(GTK_CLIST(dccrwin.list), dccrwin.selected, 1, &file);
   gtk_clist_get_text(GTK_CLIST(dccrwin.list), dccrwin.selected, 5, &nick);
   dcc_get_nick_file(nick, file);
}

void recv_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   if(none == 0)
     dccrwin.selected = row;
   else
     dccrwin.selected = -1;
}

void open_dcc_recv_window(void)
{  
   GtkWidget *vbox, *bbox;
   GtkWidget *accept, *resume, *abort;
   static gchar *titles[] = { "Status","File","Size","Position","CPS","From" };
   
   if(dccrwin.window)
   {
      update_dcc_recv_window();
      return;
   }
   
   dccrwin.selected = -1;
   
   dccrwin.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_widget_set_usize(dccrwin.window, 500, -1);
   gtk_window_set_title(GTK_WINDOW(dccrwin.window), "X-Chat: DCC Receive List");
   gtk_signal_connect(GTK_OBJECT(dccrwin.window),
                             "destroy",
                             GTK_SIGNAL_FUNC(close_dcc_recv_window), 0);
   
   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dccrwin.window), vbox);
   gtk_widget_show(vbox);
   
   dccrwin.list = gtk_clist_new_with_titles(6, titles);
   gtk_signal_connect(GTK_OBJECT(dccrwin.list),
		      "select_row", GTK_SIGNAL_FUNC(recv_row_selected), 0);
   gtk_signal_connect(GTK_OBJECT(dccrwin.list),
		      "unselect_row", GTK_SIGNAL_FUNC(recv_row_selected), (gpointer)1);
   gtk_clist_set_border(GTK_CLIST(dccrwin.list), GTK_SHADOW_OUT);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 3, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 4, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 5, 120);
   gtk_clist_set_policy(GTK_CLIST(dccrwin.list), GTK_POLICY_AUTOMATIC,
			      GTK_POLICY_AUTOMATIC);
   gtk_container_add(GTK_CONTAINER(vbox), dccrwin.list); 
   gtk_widget_show(dccrwin.list);
   
   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);
   
   accept = gtk_button_new_with_label("Accept");
   gtk_signal_connect(GTK_OBJECT(accept), "clicked",
		      GTK_SIGNAL_FUNC(accept_clicked), 0);
   gtk_container_add(GTK_CONTAINER(bbox), accept);
   gtk_widget_show(accept);
   
   resume = gtk_button_new_with_label("Resume");
   gtk_signal_connect(GTK_OBJECT(resume), "clicked",
		      GTK_SIGNAL_FUNC(resume_clicked), 0);
   gtk_container_add(GTK_CONTAINER(bbox), resume); 
   gtk_widget_show(resume);
   
   abort = gtk_button_new_with_label("Abort");
   gtk_signal_connect(GTK_OBJECT(abort), "clicked",
		      GTK_SIGNAL_FUNC(abort_clicked), 0);
   gtk_container_add(GTK_CONTAINER(bbox), abort); 
   gtk_widget_show(abort);
   
   gtk_widget_show(dccrwin.window);
   update_dcc_recv_window();
}

void close_dcc_send_window(void)
{
   if(dccswin.window)
   {
      gtk_widget_destroy(dccswin.window);
      dccswin.window = 0;
   }
}

void update_dcc_send_window(void)
{
   struct DCC *dcc = &firstDCC;
   gchar *new[1][7];
   char size[14];
   char pos[14];
   char cps[14];
   char ack[14];
   
   if(!dccswin.window) return;
   
   gtk_clist_clear((GtkCList*)dccswin.list);
   new[0][2] = size;
   new[0][3] = pos;
   new[0][4] = ack;
   new[0][5] = cps;
   do
   {
      if(dcc->used && dcc->type == TYPE_SEND)
      {
	 new[0][0] = dccstat[(int)dcc->stat];
	 new[0][1] = file_part(dcc->file);
	 new[0][6] = dcc->nick;
	 sprintf(size, "%lu", dcc->size);
	 sprintf(pos, "%lu", dcc->pos);
	 sprintf(cps, "%lu", dcc->cps);
	 sprintf(ack, "%lu", dcc->ack);
	 gtk_clist_append((GtkCList*)dccswin.list, new[0]);
      }
      dcc = dcc->next;
   } while(dcc);
   if(dccswin.selected != -1)
     gtk_clist_select_row((GtkCList*)dccswin.list, dccswin.selected, 0);
}

void send_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   if(none == 0)
     dccswin.selected = row;
   else
     dccswin.selected = -1;
}

void abort_send_clicked(GtkWidget *wid, gpointer none)
{
   gchar *file, *nick;
   if(dccswin.selected == -1) return;
   gtk_clist_get_text(GTK_CLIST(dccswin.list), dccswin.selected, 1, &file);
   gtk_clist_get_text(GTK_CLIST(dccswin.list), dccswin.selected, 6, &nick);
   dcc_abort_nick_file(nick, file);
}

void open_dcc_send_window(void)
{  
   GtkWidget *vbox, *bbox;
   GtkWidget *abort;
   static gchar *titles[] = { "Status","File","Size","Position","Ack","CPS","To" };
   
   if(dccswin.window)
   {
      update_dcc_send_window();
      return;
   }

   dccswin.selected = -1;
   
   dccswin.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_widget_set_usize(dccswin.window, 550, -1);
   gtk_window_set_title(GTK_WINDOW(dccswin.window), "X-Chat: DCC Send List");
   gtk_signal_connect(GTK_OBJECT(dccswin.window),
                             "destroy",
                             GTK_SIGNAL_FUNC(close_dcc_send_window), 0);
   
   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dccswin.window), vbox);
   gtk_widget_show(vbox);
   
   dccswin.list = gtk_clist_new_with_titles(7, titles);
   gtk_signal_connect(GTK_OBJECT(dccswin.list),
		      "select_row", GTK_SIGNAL_FUNC(send_row_selected), 0);
   gtk_signal_connect(GTK_OBJECT(dccswin.list),
		      "unselect_row", GTK_SIGNAL_FUNC(send_row_selected), (gpointer)1);
   gtk_clist_set_border(GTK_CLIST(dccswin.list), GTK_SHADOW_OUT);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 3, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 4, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 5, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 6, 120);
   gtk_clist_set_policy(GTK_CLIST(dccswin.list), GTK_POLICY_AUTOMATIC,
			      GTK_POLICY_AUTOMATIC);
   gtk_container_add(GTK_CONTAINER(vbox), dccswin.list); 
   gtk_widget_show(dccswin.list);
   
   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   abort = gtk_button_new_with_label("Abort");
   gtk_signal_connect(GTK_OBJECT(abort), "clicked",
		      GTK_SIGNAL_FUNC(abort_send_clicked), 0);
   gtk_container_add(GTK_CONTAINER(bbox), abort); 
   gtk_widget_show(abort);
   
   gtk_widget_show(dccswin.window);
   update_dcc_send_window();
}


// DCC CHAT GUIs BELOW


void abort_chat_clicked(GtkWidget *wid, gpointer none)
{
   gchar *nick;
   if(dcccwin.selected == -1) return;
   gtk_clist_get_text(GTK_CLIST(dcccwin.list), dcccwin.selected, 1, &nick);
   dcc_abort_nick_file(nick, "");
}

void update_dcc_chat_window(void)
{
   struct DCC *dcc = &firstDCC;
   gchar *new[1][5];
   char pos[14];
   char siz[14];
   
   if(!dcccwin.window) return;
   
   gtk_clist_clear((GtkCList*)dcccwin.list);
   new[0][2] = pos;
   new[0][3] = siz;
   do
   {
      if(dcc->used && (dcc->type == TYPE_CHATSEND || dcc->type == TYPE_CHATRECV))
      {
	 new[0][0] = dccstat[(int)dcc->stat];
	 new[0][1] = dcc->nick;
	 sprintf(pos, "%lu", dcc->pos);
	 sprintf(siz, "%lu", dcc->size);
	 new[0][4] = ctime(&dcc->starttime);
	 gtk_clist_append((GtkCList*)dcccwin.list, new[0]);
      }
      dcc = dcc->next;
   } while(dcc);
   if(dcccwin.selected != -1)
     gtk_clist_select_row((GtkCList*)dcccwin.list, dcccwin.selected, 0);
}

void chat_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   if(none == 0)
     dcccwin.selected = row;
   else
     dcccwin.selected = -1;
}

void close_dcc_chat_window(void)
{
   if(dcccwin.window)
   {
      gtk_widget_destroy(dcccwin.window);
      dcccwin.window = 0;
   }
}

void open_dcc_chat_window(void)
{  
   GtkWidget *vbox, *bbox;
   GtkWidget *abort;
   static gchar *titles[] = { "Status","To/From","Recv","Sent","StartTime" };
   
   if(dcccwin.window)
   {
      update_dcc_chat_window();
      return;
   }

   dcccwin.selected = -1;
   
   dcccwin.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_widget_set_usize(dcccwin.window, 550, -1);
   gtk_window_set_title(GTK_WINDOW(dcccwin.window), "X-Chat: DCC Chat List");
   gtk_signal_connect(GTK_OBJECT(dcccwin.window),
                             "destroy",
                             GTK_SIGNAL_FUNC(close_dcc_chat_window), 0);
   
   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dcccwin.window), vbox);
   gtk_widget_show(vbox);
   
   dcccwin.list = gtk_clist_new_with_titles(5, titles);
   gtk_signal_connect(GTK_OBJECT(dcccwin.list),
		      "select_row", GTK_SIGNAL_FUNC(chat_row_selected), 0);
   gtk_signal_connect(GTK_OBJECT(dcccwin.list),
		      "unselect_row", GTK_SIGNAL_FUNC(chat_row_selected), (gpointer)1);
   gtk_clist_set_border(GTK_CLIST(dcccwin.list), GTK_SHADOW_OUT);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 2, 65);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 3, 65);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 4, 65);
   gtk_clist_set_policy(GTK_CLIST(dcccwin.list), GTK_POLICY_AUTOMATIC,
			      GTK_POLICY_AUTOMATIC);
   gtk_container_add(GTK_CONTAINER(vbox), dcccwin.list); 
   gtk_widget_show(dcccwin.list);
   
   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   abort = gtk_button_new_with_label("Abort");
   gtk_signal_connect(GTK_OBJECT(abort), "clicked",
		      GTK_SIGNAL_FUNC(abort_chat_clicked), 0);
   gtk_container_add(GTK_CONTAINER(bbox), abort); 
   gtk_widget_show(abort);
   
   gtk_widget_show(dcccwin.window);
   update_dcc_chat_window();
}
/*

Here is a description of the mIRC DCC Resume Protocol.

User1 is sending the file.
User2 is receiving the file.

To initiate a DCC Send, User1 sends:

PRIVMSG User2 :DCC SEND filename ipaddress port filesize

Normally, if User2 accepts the DCC Send request, User2 connects to the
address 
and port number given by User1 and the file transfer begins.

If User2 chooses to resume a file transfer of an existing file, the
following 
negotiation takes place:

User2 sends:

PRIVMSG User1 :DCC RESUME filename port position

filename = the filename sent by User1.
port = the port number sent by User1.
position = the current size of the file that User2 has.

User1 then responds:

PRIVMSG User2 :DCC ACCEPT filename port position

This is simply replying with the same information that User2 sent as 
acknowledgement.

At this point User2 connects to User1 address and port and the transfer
begins 
from the specified position.

NOTE: the newer versions of mIRC actually ignore the filename as it is 
redundant since the port uniquely identifies the connection. However, to
remain 
compatible mIRC still sends a filename as "file.ext" in both RESUME and
ACCEPT.

----

*/
