//  UInput.cpp version 1.1
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998  Gaspar Sinai
// 
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//
//

#include "UInput.h"
#include "UCache.h"
#include "UCommon.h"
#include <iostream.h>
#include <string.h>
#include <sys/types.h>
#include <stdlib.h>

#if HAVE_STRTOL==0
#error This function needs strtol.
#endif

#define 	LINE_LENGTH	1024

static          UCache<UInputCache*>* cache=0;
char*		UInput::EXTENSION = ".kmap";

UInputSearch::UInputSearch()
{
	arraySize = 0;
	size = 0;
	codes = 0;
	keys = 0;
}

UInputSearch::~UInputSearch()
{
	int	i;
	if (codes != 0)
	{
		for (i=0; i< size; i++)
		{
			delete codes[i];
		}
		delete codes;
	}
	if (keys != 0)
	{
		for (i=0; i< size; i++)
		{
			delete keys[i];
		}
		delete keys;
	}
}

//
// line is like something = something
//
int
UInputSearch::addItem (const char *line)
{
	const char	*from;
	const char	*to;
	char *		toOut;
	int		kcIndex;
	unsigned char	key[UINPUT_STATE_SIZE+1];
	UCS2		code[UINPUT_STATE_SIZE+1];
	char		kcSize[2];
	int		value;
	int		base;

	to = line;
	kcIndex =0;
	kcSize[0] = 0;
	kcSize[1] = 0;

	while (to[0] != 0)
	{
		from = to;
		if (*from==' ')
		{
			to++; continue;
		}
		if (*from=='=')
		{
			to++; kcIndex=1; continue;
		}
		if (kcSize[kcIndex] >= UINPUT_STATE_SIZE)
		{
			cerr << "error: kmap size exceeded "
				<< UINPUT_STATE_SIZE << ".\n";
			return 0;
		}

		//
		// adding number
		//
		if (to[0] >= '0' && to[0] <= '9')
		{
			base = 10;
			if (to[0] == '0')
			{
				to++;
				if (to[0] == 'x' || to[0] =='X')
				{
					base = 16;
					to++;
				}
				else
				{
					base = 8;
				}
			}
			value = (int)  strtol ((const char*) to, &toOut, base);
			to = toOut;
			if (kcIndex==0)
			{
				key[kcSize[0]] = (unsigned char) value;
			}
			else
			{
				code[kcSize[1]]  = (UCS2) value;
			}
			kcSize[kcIndex]++;
			continue;
		}

		// straight
		while (to[0] != 0)
		{
			if (to[0]==' ')
			{
				to++; break;
			}
			if (to[0]=='=') break;
			if (kcSize[kcIndex] >= UINPUT_STATE_SIZE)
			{
				cerr << "error: kmap size exceeded "
				<< UINPUT_STATE_SIZE << ".\n";
				return 0;
			}
			value = to[0];
			if (kcIndex==0)
			{
				key[kcSize[0]] = (unsigned char) value;
			}
			else
			{
				code[kcSize[1]]  = (UCS2) value;
			}
			to++;
			kcSize[kcIndex]++;
		}
	}

	if (kcSize[1]==0)
	{
		cerr << "error: kmap map is empty at '" << line << "'\n";
		return 0;
	}
	key[kcSize[0]] = 0;
	code[kcSize[1]] = 0;
	addItem (key, code);
	return 1;
}

void
UInputSearch::addItem (const unsigned char* key, const UCS2* code)
{
	UCS2**			newCodes;
	unsigned char**		newKeys;
	UCS2*			myCode;
	unsigned char*		myKey;
	int			newSize;
	int			index;
	int			i;
	int			keylen;
	int			codelen;
	

	if (arraySize <= size)
	{
		newSize = (arraySize==0) ? 1 : arraySize << 1;
		newCodes = new UCS2* [newSize];
		CHECKNULL (newCodes);
		newKeys = new unsigned char*[newSize];
		CHECKNULL (newKeys);
		if (size !=0)
		{
			memcpy (newCodes, codes, size * sizeof (UCS2*));
			memcpy (newKeys, keys, size *sizeof(unsigned char*));
		}
		if (arraySize != 0)
		{
			delete codes;
			delete keys;
		}
		codes = newCodes;
		keys = newKeys;
		arraySize = newSize;
	}
	if (code ==0 || code[0] ==0)
	{
		// multipart can be empty 
		//cerr << "warn: key is empty:" << key << ".\n";
		//return;
	}
	index = binarySearch (key);

	// do not add duplicate
	if (index<size && strcmp ((const char*)keys[index], 
		(const char*)key) ==0)
	{
		cerr << "warn: duplicate key:" << key << ".\n";
		return;
	}

	keylen = strlen ((const char*)key);
	codelen = UCS2Len (code);
	
	
	myKey = new unsigned char[keylen+1];
	CHECKNULL (myKey);

	myCode = new UCS2 [codelen+1];
	CHECKNULL (myCode);

	memcpy (myKey, key, (keylen +1) * sizeof (unsigned char));

	memcpy (myCode, code, (codelen +1) * sizeof (UCS2));

	// move guys up
	for (i=size; i>index; i--)
	{
		keys[i] = keys[i-1];
		codes[i] = codes[i-1];
	}
	keys[index] = myKey;
	codes[index] = myCode;
	size++;
	return;
	
}

int
UInputSearch::binarySearch (const unsigned char* key)
{
	int	top;
	int	bottom;
	int	mid;
	int	cmp;

	bottom=0;
	top=size;
	mid = 0;
	while (top > bottom)
	{
		mid = (top+bottom)/2;
		cmp = strcmp ((const char*) key, (const char*) keys[mid]);
		if (cmp==0) return mid;
		if (cmp < 0)
		{
			top = mid;
			continue;
		}
		bottom = mid+1;
	}
	return top;
}

UInputCache::UInputCache (int size_)
{
	int		i;

	search = new UInputSearch*[size_];
	for (i=0; i<size_; i++)
	{
		search[i] = new UInputSearch();
	}
	CHECKNULL (search);
	size = size_;
}

UInputCache::~UInputCache ()
{
	int	i;

	for (i=0; i<size; i++)
	{
		delete search [i];
	}
	if (search) delete search;
}

UInput::UInput()
{
	name = 0;
	bucket = 0;
	inputCache = 0;
	stateSize = 0;
	partition[0] = 0;
}

UInput::~UInput()
{
	if (cache!=0 && name!=0)
	{
		cache->unuseItem (name);
	}
	if (name!=0) delete name;
}

void
UInput::resetState ()
{
	stateSize = 0;
	bucket = 0;
	partition[0] = 0;
	accumulated[0][0] = 0;
	accumulated[1][0] = 0;
	accumulated[0][1] = 0;
	accumulated[1][1] = 0;
}
void
UInput::deleteLast ()
{
	int		i;
	unsigned char	str[UINPUT_STATE_SIZE];

	if (stateSize==0) return;
	stateSize--;
	stateString[stateSize] = 0;
	if (inputCache!=0 && inputCache->size > 1)
	{
		strcpy ((char*)str, (const char*) stateString);
		resetState ();
		for (i=0; str[i] != 0; i++)
		{
			translate (str[i]);
		}
	}
	
}

UCS2*
UInput::translate (const int asci)
{
	unsigned char	copyStr[UINPUT_STATE_SIZE+1];
	UCS2*		tmpUcs2;
	UCS2*		ucs2;
	UCS2		ucs2Str[2];
	UMatchType	type;
	int		index;
	UCS2		gccBug;

	// Straight converter
	if (inputCache == 0 || asci==0)
	{
		copyStr[0] = (unsigned char) asci;
		copyStr[1] = 0;
		return UCS2Get (copyStr);
	}
	if (asci==0) return 0;
	stateString [stateSize++] = asci;
	stateString [stateSize] = 0;
	
	if (stateSize >= UINPUT_STATE_SIZE-1)
	{
		cerr << "error: UInput stateSize overflow.\n";
		ucs2= UCS2Get (stateString);
		resetState ();
		return ucs2;
	}

	// Translate
	type = translate (inputCache->search[bucket % inputCache->size],
		&stateString[partition[bucket]], 
		stateSize-partition[bucket], &index);

	switch (type)
	{
	// Easy
	case MAY_MATCH0:
		return 0;

	case MAY_MATCH:
		if (bucket < inputCache->size) return 0;
	case NO_MATCH:
		//
		// get the first character off and try to match the rest.
		//
		if (bucket < inputCache->size)
		{
			strcpy ((char*) copyStr,  
				(const char*) &stateString [1]);
			stateString [1] = 0;
			tmpUcs2= UCS2Get (stateString);

			resetState ();
			ucs2 = translateAdd (tmpUcs2, copyStr);
			return ucs2;
		}
		//
		// Well. It matched till now....
		//
		ucs2Str[0] = accumulated[0][0] + accumulated[0][1];
		ucs2Str[1] = 0;

		// return the matched stuff.
		tmpUcs2 = UCS2Dup (ucs2Str);

		// move new string down.
		strcpy ((char*) copyStr,  
		  (const char*) &stateString [partition [inputCache->size]]);

		resetState ();
		ucs2 = translateAdd (tmpUcs2, copyStr);
		return ucs2;

	//
	// match with zero bucket will not occur. Size 1 and zero should
	// be filtered out
	//
	case MATCH:
		if (inputCache->size==1)
		{
			ucs2= UCS2Dup (inputCache->search[0]->codes[index]);
			resetState ();
			return ucs2;
		}
		
		accumulated[bucket/inputCache->size]
		  [((bucket+1)%inputCache->size == 0) ? 1 : 0] += 
			inputCache->search[bucket%inputCache->size]->codes[index][0];

		// if this is the new stuff, or old but there is no hope
		// to get it matched in the first bucket,
		// copy the remaining stuff to the beginning
		// and return the string. null match will not return here
		if (bucket == inputCache->size 
			|| (bucket==inputCache->size-1 && !canGive ()))
		{
			if (bucket==inputCache->size)
			{
				strcpy ((char*) copyStr, (const char*) 
					&stateString[partition[inputCache->size]]);
				strcpy ((char*) stateString, (const char*) copyStr);
				stateSize = strlen ((const char*) stateString);
				partition[1] = 1;
				bucket = 1;
			}
			else
			{
				stateString[0] = 0;
				stateSize = 0;
				bucket = 0;
			}

			ucs2Str[0] = accumulated[0][0] + accumulated[0][1];
			ucs2Str[1] = 0;
			ucs2 = UCS2Dup (ucs2Str);

			gccBug = accumulated[1][0];
			accumulated[0][1] = 0;
			accumulated[0][0] = gccBug;
			accumulated[1][0] = 0;
			accumulated[1][1] = 0;

			return  ucs2;
		}
		bucket++;
		partition[bucket] = stateSize;
		return 0;

	case DID_MATCH:
	case DID_MATCH0:

		//
		// Single key
		//
		if (inputCache->size==1)
		{
			tmpUcs2= UCS2Dup (inputCache->search[0]->codes[index]);
			strcpy ((char*) copyStr,  (const char*) &stateString [ 
				strlen ((const char*) 
				inputCache->search[0]->keys[index])] );
			resetState ();
			ucs2 = translateAdd (tmpUcs2, copyStr);
			return ucs2;
		}

		accumulated[bucket/inputCache->size][1] = 0;
		accumulated[bucket/inputCache->size]
		  [((bucket+1)%inputCache->size == 0) ? 1 : 0] += 
		  inputCache->search[bucket%inputCache->size]->codes[index][0];
		//
		// Tentative position of next bucket 
		//
		partition[bucket+1] = partition[bucket]
		  + strlen ((const char*) inputCache->
			search [bucket%inputCache->size]->keys[index]);

		//
		// If we can not give away characters return matched.
		//
		if (bucket == inputCache->size &&
		 (inputCache->search[bucket%inputCache->size]->keys[index][0]!=0
			|| !canGive ()))
		{
			strcpy ((char*) copyStr, (const char*) 
			  &stateString [partition[inputCache->size]]);
			ucs2Str[0] = accumulated[0][0] + accumulated[0][1];
			ucs2Str[1] = 0;
			resetState ();
			tmpUcs2 = UCS2Dup (ucs2Str);
			ucs2 = translateAdd (tmpUcs2, copyStr);
			return ucs2;
		}

		if (bucket != inputCache->size || inputCache->search [bucket%inputCache->size]->keys[index][0]!=0)
		{

			strcpy ((char*) copyStr,  
			   (const char*) &stateString [ 
			    partition[bucket+1]] );
			stateSize = partition[bucket+1];
			stateString[stateSize] = 0;
			bucket++;
			ucs2 = translateAdd (0, copyStr);
			return ucs2;
		}

		//
		// give char to next position if 'null' matched.
		//
		canGive (1);
		strcpy ((char*) copyStr, (const char*) 
			&stateString[partition[inputCache->size]]);

		ucs2Str[0] = accumulated[0][0] + accumulated[0][1];
		ucs2Str[1] = 0;
		resetState ();

		tmpUcs2 = UCS2Dup (ucs2Str);
		ucs2 = translateAdd (tmpUcs2, copyStr);
		return ucs2;

	}
	return 0;

}


UCS2*
UInput::translateAdd (UCS2* ucs2_, const unsigned char* str)
{
	int		i;
	UCS2*		ucs2;
	UCS2*		tmpUcs2;

	tmpUcs2 = ucs2_;
	for (i=0; str[i] != 0; i++)
	{
		ucs2 = translate ((const int) str[i]);
		if (ucs2 != 0)
		{
			if (tmpUcs2!=0)
			{
				ucs2 = UCS2Concat (tmpUcs2, ucs2);
				delete tmpUcs2;
			}
			tmpUcs2 = ucs2;
		}
	}
	ucs2 = tmpUcs2; 
	return ucs2;
}

int
UInput::canGive (int doGive_)
{
	int		len;
	UMatchType	typeNew;
	UMatchType	type;
	int		index;
	int		i;

	if (bucket < inputCache->size-1) return 0;

	//
	// It can not borrow 
	//
	if (inputCache->search [0]->keys[0][0] != 0)
	{
		return 0;
	}

	//
	// Theoretical interest. We do not have the next string yet.
	//
	if (bucket == inputCache->size-1)
	{
		// can not lend anything
		if (stateSize - partition [inputCache->size-1] <=0) return 0;

		//
		// Shift to new one
		//
		type = translate (inputCache->search [0],
			&stateString [stateSize-1], 
			1, &index);

		if (type==NO_MATCH || type==MAY_MATCH0 || type==DID_MATCH0) return 0;

		//
		// Remaining
		//
		type = translate (inputCache->search [inputCache->size-1],
			&stateString [partition [inputCache->size-1]], 
			stateSize - partition [inputCache->size-1]-1, &index);

		if (type==NO_MATCH) return 0;
		return 1;
	}


	//
	// New
	//
	typeNew = translate (inputCache->search [0],
		&stateString [partition [inputCache->size] -1], 
		1, &index);

	if (typeNew==NO_MATCH || typeNew==DID_MATCH0)
	{
		return 0;
	}

	i = index;

	len = partition [inputCache->size] - partition[inputCache->size-1];
	//
	// Nothing to borrow.
	//
	if (len <= 0)
	{
		return 0;
	}

	//
	// If there is a second bucket it should really match
	//
	if (stateString [partition [inputCache->size]] != 0)
	{
		type = translate (inputCache->search [1],
			&stateString [partition [inputCache->size]], 
			1, &index);
		if (type==NO_MATCH) return 0; 
	}
		
	//
	// Check if previous string still matches after 
	// repartitioning
	//
	type = translate (inputCache->search [inputCache->size-1],
		&stateString [partition [inputCache->size-1]], 
		len-1, &index);

	if (type==NO_MATCH || type==DID_MATCH)
	{
		return 0;
	}
	if (type==MAY_MATCH && inputCache->search [inputCache->size-1]->keys[index][len-1]!=0)
	{
		return 0;
	}

	// It is real.
	if (doGive_==1)
	{
		partition[inputCache->size]--;
		accumulated[0][1] =  (type == MAY_MATCH0) ? 
		 inputCache->search [inputCache->size-1]->codes[0][0] :
		 inputCache->search [inputCache->size-1]->codes[index][0];
		accumulated[1][0] = inputCache->search [0]->codes[i][0];
	}
	return 1;
}

//
//
UInput::UMatchType
UInput::translate (UInputSearch* search_, unsigned char* in_, 
	int size_, int* index_)
{

	unsigned char	saveChar;
	int		index;
	int		match0;
	int		match1;
	int		i;

	// null terminate in a hacker's way.
	if (size_<=0) 
	{
		if (search_->keys[0][0]==0) return MAY_MATCH0;
		return NO_MATCH;
	}
	saveChar = in_[size_];
	in_[size_] = 0;

	index = search_->binarySearch (in_);

	// match0 is true if this bucket matches.
	match0 = (index < search_->size && strncmp (
		(const char*) search_->keys[index], 
		(const char*) in_, size_)==0) ? 1 : 0;


	// match0 is true if next bucket matches too.
	match1 = (index < search_->size-1 && strncmp (
		(const char*) search_->keys[index+1], 
		(const char*) in_, size_)==0) ? 1 : 0;


	// See if it matched 
	if (match0 == 0)
	{
		in_[size_] = saveChar;
		if (size_==0) return NO_MATCH;

		for (i=0; i<size_; i++)
		{
			saveChar = in_[size_-i];
			in_[size_-i] = 0;

			index = search_->binarySearch (in_);
			match0 = (index < search_->size && strncmp (
				(const char*) search_->keys[index], 
				(const char*) in_, size_-i)==0) ? 1 : 0;
			in_[size_-i] = saveChar;
			if (match0 != 0)
			{
				if (search_->keys[index][size_-i] ==0)
				{
					break;
				}
				match0 = 0;
			}
		}

		// null char is match
		if (match0==0 && (const char*) search_->keys[0][0]==0)
		{
			*index_ = 0;
			return DID_MATCH0;
		}

		*index_ = index;
		return ((match0==0) ? NO_MATCH : DID_MATCH);
	}
	in_[size_] = saveChar;
	*index_ = index;

	// still matches, next char has a chance.
	if (match1 == 1) return MAY_MATCH; 
	if (search_->keys[index][size_] != 0) return MAY_MATCH;

	// this is exact match
	return MATCH;
}

static char* unicodeMap[] = {
 "0x30 =0x%04x", "0x31 =0x%04x", "0x32 =0x%04x", "0x33 =0x%04x", "0x34 =0x%04x",
 "0x35 =0x%04x", "0x36 =0x%0x", "0x37 =0x%04x", "0x38 =0x%04x", "0x39 =0x%04x",
 "a =0x%04x", "b =0x%04x", "c =0x%04x", "d =0x%04x", "e =0x%04x", "f =0x%04x",
 "A =0x%04x", "B =0x%04x", "C =0x%04x", "D =0x%04x", "E =0x%04x", "F =0x%04x",
  0};
static int unicodeIndex[] =
{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 
0xa, 0xb, 0xc, 0xd, 0xe, 0xf,
0xa, 0xb, 0xc, 0xd, 0xe, 0xf, -1 };

//
// Simple: uxxxx - all small letters
//
UInput::UStatus
UInput::makeUnicodeMap()
{
	char	line[32];
	int	i, j;

	inputCache = new UInputCache (5);

	inputCache->search[0]->addItem ("u = 0x0");
	inputCache->search[0]->addItem ("U = 0x0");
	for(i=0;i<4;i++) for (j=0; unicodeMap[j] != 0; j++)
	{
		sprintf (line, unicodeMap[j], unicodeIndex[j] <<(4 * (3-i)));
		inputCache->search[i+1]->addItem (line);
	}

	if (name!=0) delete name;
	name = new char[sizeof "Unicde"];
	CHECKNULL (name);
	strcpy (name, "Unicode");

	cache->addItem (name, inputCache);
	return OK;
}

UInput::UStatus
UInput::rename (const char* nameIn)
{
	resetState ();

	if (cache!=0 && name!=0)
	{
		cache->unuseItem (name);
		inputCache = 0;
	}
	if (name!=0)
	{
		delete name;
		name = 0;
	}
	// straight converter
	if (nameIn==0)
	{
		return OK;
	}

	name = new char[strlen (nameIn) +1];
	CHECKNULL (name);
	strcpy (name, nameIn);

	if (cache == 0)
	{
		cache = new UCache<UInputCache*> (INPUTCACHE_SIZE);
		CHECKNULL (cache);
	}
	
	inputCache = cache->getItem (name);
	if (inputCache!=0)
	{
		cache->useItem (name);
		return OK;
	}

	//
	// straight unicode map.
	//
	if (strcmp (name, "Unicode")==0)
	{
		return makeUnicodeMap ();
	}

	if (loadCache (name) == 0)
	{
		cerr << "error: can not load Input Converter: "
			<< name << ".\n";
		return ERROR;
	}
	return OK;
}

//
// load it from path, put it in cache, return it.
//
int
UInput::loadCache (const char* name_)
{
	char*		pathName;
	char*		fullName;
	ifstream	*file;
	int    result;

	fullName = new char [strlen (name_) + strlen (EXTENSION) +1];
	CHECKNULL (fullName);
	strcpy (fullName, name_);
	strcat (fullName, EXTENSION);
	pathName = UFindFile (UGetMapPath (), fullName);

	if (pathName==0) return 0;
	file = new ifstream (pathName);
	CHECKNULL (file);

	if (file->rdbuf() == 0 || !file->rdbuf()->is_open())
	{
		delete file;
		delete fullName;
		delete pathName;
		return 0;
	}

	result = loadCache (file, pathName);
	delete file;
	delete pathName;
	delete fullName;
	return result;
}

//
//
int
UInput::loadCache (ifstream* file, const char* filename)
{
	char		line [LINE_LENGTH+1];
	char		quoted [LINE_LENGTH+1];
	int		lineNo;
	int		error;
	char		*from, *to;
	int		comment;
	int		empty;
	int		index;
	int		i;
	int		currentSegment;
	int		segmentCount;
	char		segments[UINPUT_MAX_SEGMENTS+1][LINE_LENGTH+1];
	char		segmentId[LINE_LENGTH+1];


	lineNo = 0;
	error = 0;
	comment = 0;
	inputCache = 0;
	currentSegment = 0;

	segmentCount = 0;

	while (file->getline (line, LINE_LENGTH))
	{
		// simple parser, only '//' commmenst are OK at the
		// beginning of the line
		line[LINE_LENGTH]=0;
		lineNo++;
		to = line;
		from= line;
		index = 0;
		while (to[0] != 0)
		{
			if (comment)
			{
				if (to[0]=='*' && to[1] == '/')
				{
					to++; 
					comment = 0;
				}
				to++; 
				continue;
			}
			if (to[0]=='/' && to[1] == '*')
			{
				to++; to++;
				comment = 1;
				continue;
			}
			if (to[0]=='/' && to[1] == '/') break;
			if (from == line && to[0] != '"')
			{
				to++; continue;
			}
			if (from == line && to[0] == '"')
			{
				to++;
				from = to;
				continue;
			}
			if (to[0] == '\\')
			{
				switch (to[1])
				{
				case 't':
					quoted[index++] = '\t';
					to++;
					break;
				case '\\':
					quoted[index++] = '\\';
					to++; 
					break;
				case '"':
					quoted[index++] = '"';
					to++; 
					break;
				case '0':
				case '1':
				case '2':
				case '3':
					if (to[2] < '0' || to[2] >'7'
					 || to[3] < '0' || to[3] > '7')
					{
						quoted[index++] = '\\';
						break;
					}
					quoted[index++] = ((to[1]-'0') << 6)
					        + ((to[2]-'0') << 3)
						+ to[3]-'0';
					to++; to++; to++;
					break;
				}
				to++;
				continue;
			}
			if (to[0] == '"')
			{
				break;
			}
			quoted[index++] = to[0];
			to++;
		}
		if (index == 0)
		{
			if (from != line)
			{
				error=1;
				break;
			}
			continue;
		}
		quoted[index] = 0;

		// Ugly but it works.
		if (UINPUT_MAX_SEGMENTS != 5)
		{
			error=1 ;
			break;
		}

		if (inputCache == 0)
		{
			if (strstr (quoted, "=") == 0)
			{
				segmentCount = sscanf (quoted, 
					"%[^+]+%[^+]+%[^+]+%[^+]+%[^+]+%[^+]",
					segments[0], segments[1],
					segments[2], segments[3],
					segments[4], segments[5]);
				if (segmentCount==0 && segmentCount > 5)
				{
					error=1;
					break;
				}
				currentSegment = -1;
				inputCache = new UInputCache (segmentCount);
				CHECKNULL (inputCache);
				continue;
			}
			else
			{
				currentSegment = 0;
				segmentCount = 0;
				inputCache = new UInputCache (1);
				CHECKNULL (inputCache);
			}
		}
		if (currentSegment==-1)
		{
			for (i=0; i<segmentCount; i++)
			{
				strcpy (segmentId, "begin ");
				strcat (segmentId, segments[i]);

				if (strcmp (quoted, segmentId)==0)
				{
					strcpy (segmentId, "end ");
					strcat (segmentId, segments[i]);
					currentSegment = i;
					break;
				}
			}
			if (currentSegment==-1)
			{
				error=1;
				break;
			}
			continue;
		}
		if (segmentCount != 0)
		{
			if (strcmp (quoted, segmentId) == 0)
			{
				currentSegment=-1;
				continue;
			}
		}
		if (inputCache->search[currentSegment]->addItem (quoted)==0)
		{
cerr << "Here\n";
			error=1;
			break;
		}
	}
	if (error)
	{
		cerr << "error: bad syntax in file '"
			<< (filename==0?"unkown":filename)
			<< "' at line " << lineNo << "\n";
		if (inputCache) delete inputCache;
		inputCache = 0;
		return 0;
	}
	empty = 0;
	for (i=0; i<segmentCount || i==0; i++)
	{
		if (inputCache->search[i]->size==0) 
		{
			if (segmentCount==0)
			{
				cerr << "error: bad syntax in file '"
					<< (filename==0?"unkown":filename)
					<< "' at eof. \n";
			}
			else if (currentSegment!=-1)
			{
				cerr << "error: bad syntax (missing end?) in file '"
					<< (filename==0?"unkown":filename)
					<< "' at eof. \n";
			}
			else
			{
				cerr << "error: bad syntax in file '"
					<< (filename==0?"unkown":filename)
					<< "' at multipart '" << segments[i]
					<< "'\n";
			}
			delete inputCache;
			inputCache = 0;
			return 0;
		}
		else if (inputCache->search[i]->size==1 
			&& inputCache->search[i]->keys[0][0] ==0) 
		{
			cerr << "error: bad syntax (null match) in file '"
				<< (filename==0?"unkown":filename)
				<< "' at multipart '" << segments[i]
				<< "'\n";
			delete inputCache;
			inputCache = 0;
			return 0;
		}
		if (inputCache->search[i]->keys[0][0] == 0)
		{
			if (empty || segmentCount < 2)
			{
				cerr << "error: bad syntax (adjacent null  matches) in file '"
					<< (filename==0?"unkown":filename)
					<< "'\n";
				delete inputCache;
				inputCache = 0;
				return 0;
			}
			empty = 1;
		}
		else
		{
			empty = 0;
		}
			
	}
	//
	// the calling routine should make sure it is not duplicate.
	//
	cache->addItem (name, inputCache);
	return 1;
}


