/*	$Id: funcs.c,v 1.21 1997/10/31 20:35:49 sandro Exp $	*/

/*
 * Copyright (c) 1997
 *	Sandro Sigala, Brescia, Italy.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "zile.h"
#include "extern.h"

int
cancel(void)
{
	minibuf_error("%FCQuit%E");

	return FALSE;
}

DEFUN("suspend-zile", suspend_zile)
/*+
Stop Zile and return to superior process.
+*/
{
	raise(SIGTSTP);

	return TRUE;
}

DEFUN("keyboard-quit", keyboard_quit)
/*+
Cancel current command.
+*/
{
	return cancel();
}

static char *
make_buffer_flags(bufferp bp, int iscurrent)
{
	static char buf[4];

	buf[0] = iscurrent ? '.' : ' ';
	buf[1] = (bp->flags & BFLAG_MODIFIED) ? '*' : ' ';
	/*
	 * Display the readonly flag if it is set or the buffer is
	 * the current buffer, i.e. the `*Buffer List*' buffer.
	 */
	buf[2] = (bp->flags & BFLAG_READONLY || bp == cur_bp) ? '%' : ' ';

	return buf;
}

static char *
make_buffer_mode(bufferp bp)
{
	static char buf[16];

	/*
	 * Only two modes are recognized for now.
	 */
	if (bp->flags & BFLAG_CMODE)
		strcpy(buf, "C");
	else
		strcpy(buf, "Text");

	if (bp->flags & BFLAG_FONTLOCK)
		strcat(buf, " Font");

	if (bp->flags & BFLAG_AUTOFILL)
		strcat(buf, " Fill");

	return buf;
}

static int
calculate_buffer_size(bufferp bp)
{
	linep lp = bp->limitp->next;
	int size = 0;

	for (;;) {
		size += lp->size;
		lp = lp->next;
		if (lp == bp->limitp)
			break;
		size += 1;
	}

	return size;
}

static void
print_buf(bufferp old_bp, bufferp bp)
{
	char buf[80];

	if (bp->name[0] == ' ')
		return;

	bprintf("%3s %-14s %6u  %-13s",
		make_buffer_flags(bp, old_bp == bp),
		bp->name,
		calculate_buffer_size(bp),
		make_buffer_mode(bp));
	if (bp->filename != NULL)
		insert_string(short_string(buf, bp->filename, 40));
#if 0
	bprintf(" [F: %o]", bp->flags);
#endif
	insert_newline();
}

DEFUN("list-buffers", list_buffers)
/*+
Display a list of names of existing buffers.
The list is displayed in a buffer named `*Buffer List*'.
Note that buffers with names starting with spaces are omitted.

The M column contains a * for buffers that are modified.
The R column contains a % for buffers that are read-only.
+*/
{
	windowp wp, old_wp = cur_wp;
	bufferp bp;

	if ((wp = find_window("*Buffer List*")) == NULL) {
		cur_wp = popup_window();
		cur_bp = cur_wp->bp;
		bp = find_buffer("*Buffer List*", TRUE);
		switch_to_buffer(bp);
	} else {
		cur_wp = wp;
		cur_bp = wp->bp;
	}

	zap_buffer_content();
	cur_bp->flags = BFLAG_NEEDNAME | BFLAG_NOSAVE | BFLAG_NOUNDO | BFLAG_NOEOB;
	set_temporary_buffer(cur_bp);

	bprintf(" MR Buffer           Size  Mode         File\n");
	bprintf(" -- ------           ----  ----         ----\n");

	/* Print non-temporary buffers. */
	bp = old_wp->bp;
	do {
		if (!(bp->flags & BFLAG_TEMPORARY))
			print_buf(old_wp->bp, bp);
		if ((bp = bp->next) == NULL)
			bp = head_bp;
	} while (bp != old_wp->bp);

	/* Print temporary buffers. */
	bp = old_wp->bp;
	do {
		if (bp->flags & BFLAG_TEMPORARY)
			print_buf(old_wp->bp, bp);
		if ((bp = bp->next) == NULL)
			bp = head_bp;
	} while (bp != old_wp->bp);

	gotobob();

	cur_bp->flags |= BFLAG_READONLY;

	cur_wp = old_wp;
	cur_bp = old_wp->bp;

	return TRUE;
}

DEFUN("overwrite-mode", overwrite_mode)
/*+
In overwrite mode, printing characters typed in replace existing text
on a one-for-one basis, rather than pushing it to the right.  At the
end of a line, such characters extend the line.
C-q still inserts characters in overwrite mode; this
is supposed to make it easier to insert characters when necessary.
+*/
/*
 * XXX Emacs behaviour:
 * "Before a tab, such characters insert until the tab is filled in."
 */
{
	if (cur_bp->flags & BFLAG_OVERWRITE)
		cur_bp->flags &= ~BFLAG_OVERWRITE;
	else
		cur_bp->flags |= BFLAG_OVERWRITE;

	return TRUE;
}

DEFUN("toggle-read-only", toggle_read_only)
/*+
Change whether this buffer is visiting its file read-only.
+*/
{
	if (cur_bp->flags & BFLAG_READONLY)
		cur_bp->flags &= ~BFLAG_READONLY;
	else
		cur_bp->flags |= BFLAG_READONLY;

	return TRUE;
}

DEFUN("auto-fill-mode", auto_fill_mode)
/*+
Toggle Auto Fill mode.
In Auto Fill mode, inserting a space at a column beyond `current-fill-column'
automatically breaks the line at a previous space.
+*/
{
	if (cur_bp->flags & BFLAG_AUTOFILL)
		cur_bp->flags &= ~BFLAG_AUTOFILL;
	else
		cur_bp->flags |= BFLAG_AUTOFILL;

	return TRUE;
}

DEFUN("text-mode", text_mode)
/*+
Turn on the mode for editing text intended for humans to read.
+*/
{
	if (cur_bp->flags & BFLAG_CMODE)
		cur_bp->flags &= ~BFLAG_CMODE;

	return TRUE;
}

DEFUN("c-mode", c_mode)
/*+
Turn on the mode for editing K&R and ANSI C code.
+*/
{
	if (!(cur_bp->flags & BFLAG_CMODE))
		cur_bp->flags |= BFLAG_CMODE;

	return TRUE;
}

DEFUN("set-mark-command", set_mark_command)
/*+
Set mark at where point is.
+*/
{
	cur_bp->markp = cur_wp->pointp;
	cur_bp->marko = cur_wp->pointo;
 
	minibuf_write("%FCMark set%E");

	return TRUE;
}

DEFUN("exchange-point-and-mark", exchange_point_and_mark)
/*+
Put the mark where point is now, and point where the mark is now.
+*/
{
	struct region r;
	linep swapp;
	int swapo;

	if (warn_if_no_mark())
		return FALSE;

	calculate_region(&r);

	/*
	 * Increment or decrement the current line according to the
	 * mark position.
	 */
	if (r.startp == cur_wp->pointp)
		cur_wp->pointn += r.num_lines;
	else
		cur_wp->pointn -= r.num_lines;

	/*
	 * Swap the point and the mark.
	 */
	swapp = cur_bp->markp;
	swapo = cur_bp->marko;
	cur_bp->markp = cur_wp->pointp;
	cur_bp->marko = cur_wp->pointo;
	cur_wp->pointp = swapp;
	cur_wp->pointo = swapo;

	thisflag |= FLAG_NEED_RESYNC;

	return TRUE;
}

DEFUN("mark-whole-buffer", mark_whole_buffer)
/*+
Put point at beginning and mark at end of buffer.
+*/
{
	gotoeob();
	F_set_mark_command(1);
	gotobob();

	return TRUE;
}

#define CASE_UPPER	1
#define CASE_LOWER	2

/*
 * Set the region case.
 */
static int
setcase_region(int rcase)
{
	struct region r;
	linep lp;
	char *p;
	int size;

	if (warn_if_readonly_buffer())
		return FALSE;

	if (warn_if_no_mark())
		return FALSE;

	calculate_region(&r);
	size = r.size;

	undo_save(UNDO_REPLACE_BLOCK, r.startn, r.starto, size);

	lp = r.startp;
	p = lp->text + r.starto;
	while (size--) {
		if (p < lp->text + lp->size) {
			if (rcase == CASE_UPPER)
				*p = toupper(*p);
			else
				*p = tolower(*p);
			++p;
		} else {
			lp = lp->next;
			p = lp->text;
		}
	}

	cur_bp->flags |= BFLAG_MODIFIED;

	return TRUE;
}

DEFUN("upcase-region", upcase_region)
/*+
Convert the region to upper case.
+*/
{
	return setcase_region(CASE_UPPER);
}

DEFUN("downcase-region", downcase_region)
/*+
Convert the region to lower case.
+*/
{
	return setcase_region(CASE_LOWER);
}

DEFUN("quoted-insert", quoted_insert)
/*+
Read next input character and insert it.
This is useful for inserting control characters.
+*/
{
	int c;

	minibuf_write("%FGC-q-%E");
	c = cur_tp->xgetkey(GETKEY_NONFILTERED, 0);
	minibuf_clear();

	if (c == '\r')
		insert_newline();
	else
		insert_char(c);

	return TRUE;
}

int
universal_argument(int keytype, int xarg)
{
	char buf[128];
	int arg = 4, neg = 1, i = 0, compl = 0;
	int c;

	if (keytype == KBD_META) {
		strcpy(buf, "ESC ");
		arg = xarg;
		++i;
		goto gotesc;
	} else
		strcpy(buf, "C-u-");
	for (;;) {
		c = do_completion(buf, &compl);
		if (c == KBD_CANCEL)
			return cancel();
		else if (c == (KBD_CTL | 'u')) {
			if (arg * 4 < arg)
				arg = 4;
			else
				arg *= 4;
			i = 0;
		}
		else if (c == '-' && i == 0)
			neg = -neg;
		else if (c > 255 || !isdigit(c)) {
			cur_tp->ungetkey(c);
			break;
		} else {
			if (i == 0)
				arg = c - '0';
			else
				arg = arg * 10 + c - '0';
			++i;
		}
	gotesc:
		if (keytype == KBD_META)
			sprintf(buf, "ESC %d", arg * neg);
		else
			sprintf(buf, "C-u %d", arg * neg);
	}

	last_uniarg = arg * neg;

	thisflag |= FLAG_SET_UNIARG;

	minibuf_clear();

	return TRUE;
}

DEFUN("universal-argument", universal_argument)
/*+
Begin a numeric argument for the following command.
Digits or minus sign following C-u make up the numeric argument.
C-u following the digits or minus sign ends the argument.
C-u without digits or minus sign provides 4 as argument.
Repeating C-u without digits or minus sign multiplies the argument
by 4 each time.
+*/
{
	return universal_argument(KBD_CTL | 'u', 0);
}
