/*	$Id: basic.c,v 1.15 1997/10/31 20:35:47 sandro Exp $	*/

/*
 * Copyright (c) 1997
 *	Sandro Sigala, Brescia, Italy.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "zile.h"
#include "extern.h"

DEFUN("beginning-of-line", beginning_of_line)
/*+
Move point to beginning of current line.
+*/
{
	cur_wp->pointo = 0;

	return TRUE;
}

DEFUN("end-of-line", end_of_line)
/*+
Move point to end of current line.
+*/
{
	cur_wp->pointo = cur_wp->pointp->size;

	return TRUE;
}

/*
 * Get the current goal column.  Take care of expanding
 * tabulations.
 */
static int
get_goalc(void)
{
	int col = 0, t = cur_bp->tab_width;
	char *sp = cur_wp->pointp->text, *p = sp;

	while (p < sp + cur_wp->pointo) {
		if (*p == '\t')
			col |= t - 1;
		++col, ++p;
	}

	return col;
}

/*
 * Go to the column `goalc'.  Take care of expanding
 * tabulations.
 */
static void
goto_goalc(int goalc)
{
	int col = 0, t = cur_bp->tab_width, w;
	char *sp = cur_wp->pointp->text, *p = sp;

	while (p < sp + cur_wp->pointp->size) {
		if (col == goalc)
			break;
		else if (*p == '\t') {
			for (w = t - col % t; w > 0; w--)
				if (++col == goalc)
					break;
		} else
			++col;
		++p;
	}

	cur_wp->pointo = p - sp;
}

static int cur_goalc;

int
previous_line(void)
{
	if (cur_wp->pointp->prev != cur_bp->limitp) {
		thisflag |= FLAG_DONE_CPCN | FLAG_NEED_RESYNC;

		if (!(lastflag & FLAG_DONE_CPCN))
			cur_goalc = get_goalc();

		cur_wp->pointp = cur_wp->pointp->prev;
		--cur_wp->pointn;

		goto_goalc(cur_goalc);

		return TRUE;
	}

	return FALSE;
}

DEFUN("previous-line", previous_line)
/*+
Move cursor vertically up one line.
If there is no character in the target line exactly over the current column,
the cursor is positioned after the character in that line which spans this
column, or at the end of the line if it is not long enough.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_next_line(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!previous_line()) {
			minibuf_error("%FCBeginning of buffer%E");
			return FALSE;
		}

	return TRUE;
}

int
next_line(void)
{
	if (cur_wp->pointp->next != cur_bp->limitp) {
		thisflag |= FLAG_DONE_CPCN | FLAG_NEED_RESYNC;

		if (!(lastflag & FLAG_DONE_CPCN))
			cur_goalc = get_goalc();

		cur_wp->pointp = cur_wp->pointp->next;
		++cur_wp->pointn;

		goto_goalc(cur_goalc);

		return TRUE;
	}

	return FALSE;
}

DEFUN("next-line", next_line)
/*+
Move cursor vertically down one line.
If there is no character in the target line exactly under the current column,
the cursor is positioned after the character in that line which spans this
column, or at the end of the line if it is not long enough.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_previous_line(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!next_line()) {
			minibuf_error("%FCEnd of buffer%E");
			return FALSE;
		}

	return TRUE;
}

/*
 * Move point to the beginning of the buffer; do not touch the mark.
 */
void
gotobob(void)
{
	cur_wp->pointp = cur_bp->limitp->next;
	cur_wp->pointo = 0;
	cur_wp->pointn = 0;
	thisflag |= FLAG_DONE_CPCN | FLAG_NEED_RESYNC;
}

DEFUN("beginning-of-buffer", beginning_of_buffer)
/*+
Move point to the beginning of the buffer; leave mark at previous position.
+*/
{
	F_set_mark_command(1);
	gotobob();

	return TRUE;
}

/*
 * Move point to the end of the buffer; do not touch the mark.
 */
void
gotoeob(void)
{
	cur_wp->pointp = cur_bp->limitp->prev;
	cur_wp->pointo = cur_wp->pointp->size;
	cur_wp->pointn = cur_bp->num_lines;
	thisflag |= FLAG_DONE_CPCN | FLAG_NEED_RESYNC;
}

DEFUN("end-of-buffer", end_of_buffer)
/*+
Move point to the end of the buffer; leave mark at previous position.
+*/
{
	F_set_mark_command(1);
	gotoeob();

	return TRUE;
}

int
backward_char(void)
{
	if (cur_wp->pointo > 0) {
		--cur_wp->pointo;

		return TRUE;
	} else if (cur_wp->pointp->prev != cur_bp->limitp) {
		thisflag |= FLAG_NEED_RESYNC;
		cur_wp->pointp = cur_wp->pointp->prev;
		--cur_wp->pointn;
		F_end_of_line(1);

		return TRUE;
	}

	return FALSE;
}

DEFUN("backward-char", backward_char)
/*+
Move point left one character.
On attempt to pass beginning or end of buffer, stop and signal error.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_forward_char(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!backward_char()) {
			minibuf_error("%FCBeginning of buffer%E");
			return FALSE;
		}

	return TRUE;
}

int
forward_char(void)
{
	if (cur_wp->pointo < cur_wp->pointp->size) {
		++cur_wp->pointo;

		return TRUE;
	} else if (cur_wp->pointp->next != cur_bp->limitp) {
		thisflag |= FLAG_NEED_RESYNC;
		cur_wp->pointp = cur_wp->pointp->next;
		++cur_wp->pointn;
		F_beginning_of_line(1);

		return TRUE;
	}

	return FALSE;
}

DEFUN("forward-char", forward_char)
/*+
Move point right one character.
On reaching end of buffer, stop and signal error.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_backward_char(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!forward_char()) {
			minibuf_error("%FCEnd of buffer%E");
			return FALSE;
		}

	return TRUE;
}

int
ngotoup(int n)
{
	for (; n > 0; n--)
		if (cur_wp->pointp->prev != cur_bp->limitp)
			F_previous_line(1);
		else
			return FALSE;

	return TRUE;
}

int
ngotodown(int n)
{
	for (; n > 0; n--)
		if (cur_wp->pointp->next != cur_bp->limitp)
			F_next_line(1);
		else
			return FALSE;

	return TRUE;
}

/* The number of lines to scroll down or up. */
#define SCROLL_LINES	cur_wp->eheight

int
scroll_down(void)
{
	if (cur_wp->pointn > 0) {
		if (ngotoup(SCROLL_LINES)) {
			/* XXX */
			return TRUE;
		} else
			return FALSE;
	} else if (cur_wp->pointo > 0)
		F_beginning_of_line(1);
	else {
		minibuf_error("%FCBeginning of buffer%E");
		return FALSE;
	}

	return TRUE;
}

DEFUN("scroll-down", scroll_down)
/*+
Scroll text of current window downward near full screen.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_scroll_up(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!scroll_down())
			return FALSE;

	return TRUE;
}

int
scroll_up(void)
{
	if (cur_wp->pointn < cur_bp->num_lines) {
		if (ngotodown(SCROLL_LINES)) {
			/* XXX */
			return TRUE;
		} else
			return FALSE;
	} else if (cur_wp->pointo < cur_wp->pointp->size)
		F_end_of_line(1);
	else {
		minibuf_error("%FCEnd of buffer%E");
		return FALSE;
	}

	return TRUE;
}

DEFUN("scroll-up", scroll_up)
/*+
Scroll text of current window upward near full screen.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_scroll_down(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!scroll_up())
			return FALSE;

	return TRUE;
}
