/*
 * Utility log functions.
 *
 * These are linked in if and only if the application doesn't
 * implement its own log functions
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <stdarg.h>
#include <stdio.h>
#include <syslog.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include "utils.h"

static int	use_syslog = 0;

static void	def_fatal(const char *, ...);
static void	def_log(const char *, ...);
static void	def_debug(const char *, ...);
static void	def_clog(int, const char *, ...);
static void	vlog(int, const char *, va_list);

void		(*fatal)(const char *, ...) = def_fatal;
void		(*log)(const char *, ...) = def_log;
void		(*debug)(const char *, ...) = def_debug;
void		(*clog)(int, const char *, ...) = def_clog;

void
log_open_syslog(const char *name, int log_perror)
{
	if (use_syslog)
		closelog();
	openlog(name, LOG_PID | (log_perror? LOG_PERROR : 0), LOG_DAEMON);
	use_syslog = 1;

	/* Reinit logging */
	fatal = def_fatal;
	log   = def_log;
	clog  = def_clog;
}

static void
vlog(int level, const char *fmt, va_list ap)
{
	if (use_syslog) {
		vsyslog(level, fmt, ap);
	} else {
		vfprintf(stderr, fmt, ap);
		if (!strchr(fmt, '\n'))
			fputs("\n", stderr);
	}
}

static void
def_fatal(const char *fmt, ...)
{
	va_list		ap;

	va_start(ap, fmt);
	vlog(LOG_NOTICE, fmt, ap);
	exit(1);
}

static void
def_log(const char *fmt, ...)
{
	va_list		ap;

	va_start(ap, fmt);
	vlog(LOG_INFO, fmt, ap);
	va_end(ap);
}

static void
def_debug(const char *fmt, ...)
{
	va_list		ap;

	va_start(ap, fmt);
	vlog(LOG_DEBUG, fmt, ap);
	va_end(ap);
}

static void
def_clog(int code, const char *fmt, ...)
{
	char		buffer[1024];
	va_list		ap;
	int		n;

	va_start(ap, fmt);
	sprintf(buffer, "%03d ", code);
	n = strlen(buffer);
	vsnprintf(buffer+n, sizeof(buffer)-n, fmt, ap);
	vlog(LOG_INFO, "%s", buffer);
	va_end(ap);
}

int
make_pidfile(const char *pathname, int force)
{
	char	pids[32], tempname[1024], *sp;
	int	fd, okay = 0;
	pid_t	pid;

	if (!force
	 && (pid = read_pidfile(pathname)) > 0
	 && kill(pid, 0) < 0 && errno == ESRCH) {
		log("Removed stale lock %s", pathname);
		unlink(pathname);
	}

	assert(strlen(pathname) + sizeof("fenceXXXXXX") < sizeof(tempname));
	strcpy(tempname, pathname);
	if ((sp = strrchr(tempname, '/')) != 0) {
		sp += 1;
	} else {
		sp = tempname;
	}
	strcpy(sp, "fenceXXXXXX");

	if ((fd = mkstemp(tempname)) < 0)
		return 0;

	fchmod(fd, 0644);

	sprintf(pids, "%u\n", getpid());
	write(fd, pids, strlen(pids));
	close(fd);

	if (force)
		okay = rename(tempname, pathname) >= 0;
	else
		okay = link(tempname, pathname) >= 0;

	if (!okay)
		syslog(LOG_NOTICE, "Failed to lock %s: %m", pathname);
	unlink(tempname);
	return okay;
}

int
read_pidfile(const char *pathname)
{
	char	buffer[32];
	int	n, fd, pid;

	if ((fd = open(pathname, O_RDONLY)) < 0)
		return -1;

	n = read(fd, buffer, sizeof(buffer)-1);
	close(fd);

	if (n > 0) {
		buffer[n] = '\0';
		pid = atoi(buffer);
		if (pid > 0)
			return pid;
	}

	return -1;
}

/*
 * Split line into white space separated words
 */
int
line_split(char *inbuf, char **argv, unsigned int max)
{
	unsigned int	n = 0;
	char		*s;

	inbuf[strcspn(inbuf, "#\r\n")] = '\0';
	s = strtok(inbuf, " \t");
	while (s) {
		if (n < max - 1)
			argv[n++] = s;
		s = strtok(NULL, " \t");
	}
	argv[n] = NULL;
	return n;
}

/*
 * Merge a list
 */
char *
merge_list(int argc, char **argv, const char *sepa)
{
	char	*result = NULL, *s;
	int	total = 0, slen, len;

	slen = strlen(sepa);
	while (argc--) {
		s = *argv++;
		len = strlen(s);
		result = (char *) realloc(result, total + slen + len + 1);
		if (*result)
			strcat(result, sepa);
		strcat(result, s);
		total = strlen(result);
	}

	if (!result)
		result = strdup("");

	return result;
}

/*
 * Get the list of groups for a given user
 */
void
get_groups(const char *username, char **groups, unsigned int max)
{
	struct passwd	*pwd;
	struct group	*grp;
	unsigned int	n = 0, m;

	if ((pwd = getpwnam(username)) != NULL
	 && (grp = getgrgid(pwd->pw_gid)) != NULL) {
	 	groups[n++] = strdup(grp->gr_name);
	}

	setgrent();
	while (n < max && (grp = getgrent()) != NULL) {
		for (m = 0; grp->gr_mem[m]; m++) {
			if (!strcasecmp(grp->gr_mem[m], username)) {
				groups[n++] = strdup(grp->gr_name);
				break;
			}
		}
	}
	groups[n] = NULL;
	endgrent();
}
