/*
 * Resource class handling
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include <sys/stat.h>
#include <stdlib.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include "resmgrd.h"

res_class_t *		res_classes;


static int		lock_stale(res_device_t *);

res_class_t *
res_class_get(const char *name)
{
	res_class_t	*cls;

	for (cls = res_classes; cls; cls = cls->next) {
		if (!strcmp(cls->name, name))
			return cls;
	}
	return NULL;
}

res_class_t *
res_class_create(const char *name)
{
	res_class_t	*cls;

	cls = (res_class_t *) calloc(1, sizeof(*cls));
	cls->name = strdup(name);
	cls->refcnt = 1;

	cls->next = res_classes;
	res_classes = cls;

	return cls;
}

res_device_t *
res_class_add(res_class_t *cls, const char *devname, int flags)
{
	res_device_t	*dev, **p;

	for (p = &cls->devices; (dev = *p) != NULL; p = &dev->next)  {
		if (!strcmp(dev->name, devname)) {
			/* existing device: just update flags */
			dev->flags = flags;
			return dev;
		}
	}

	*p = dev = res_device_create(devname, flags);
	return dev;
}

void
res_class_remove(res_class_t *cls, const char *devname)
{
	res_device_t	*dev, **p;

	p = &cls->devices;
	while ((dev = *p) != NULL) {
		if (!strcmp(dev->name, devname)) {
			*p = dev->next;
			res_device_free(dev);
		} else {
			p = &dev->next;
		}
	}
}

res_device_t *
res_class_get_device(res_class_t *cls, res_name_t *name)
{
	res_device_t	*dev;

	for (dev = cls->devices; dev != NULL; dev = dev->next)  {
		if (res_name_match(name, dev))
			return dev;
	}
	return NULL;
}

void
res_class_revoke(res_class_t *cls, res_user_t *user)
{
	/* XXX to be implemented */
}

void
res_class_free(res_class_t *cls)
{
	res_device_t	*dev;

	cls->refcnt--;
	if (cls->refcnt == 0) {
		res_acl_free(cls->acl);
		while ((dev = cls->devices) != NULL) {
			cls->devices = dev->next;
			res_device_free(dev);
		}
		free(cls->name);
		memset(cls, 0, sizeof(*cls));
		free(cls);
	}
}

/*
 * Check whether a given path name is a sane device name
 * XXX This should go to a per-family routine
 */
int
res_device_sane(const char *path)
{
	char		resolved_path[PATH_MAX+1];
	struct stat	stb;
	int		is_usb;


	is_usb = !strncmp(path,
			_PATH_PROC_BUS_USB,
			sizeof(_PATH_PROC_BUS_USB)-1);

	/* It must start with /dev/ */
	if (!is_usb && strncmp(path, "/dev/", 5))
		return 0;

	/* It must not contain .. */
	if (strstr(path, ".."))
		return 0;

	/* It does not exist - give it the benefit of doubt?
	 * Bad idea in a security service. */
	if (lstat(path, &stb) < 0)
		return 0;

	/* If it's a symlink, make sure the resolved path
	 * points to a proper device */
	if (S_ISLNK(stb.st_mode)) {
		if (!realpath(path, resolved_path))
			return 0;
		return res_device_sane(resolved_path);
	}

	if (is_usb)
		return S_ISREG(stb.st_mode);

	/* Must be a device */
	if (!S_ISBLK(stb.st_mode) && !S_ISCHR(stb.st_mode))
		return 0;

	return 1;
}

/*
 * Create device objects
 */
res_device_t *
res_device_create(const char *name, int flags)
{
	res_device_t	*dev;
	char		path[4096], *s;

	dev = (res_device_t *) calloc(1, sizeof(*dev));
	dev->name = strdup(name);
	dev->flags = flags;

	snprintf(path, sizeof(path), "%s/LCK.", res_config_lock_dir);
	if (!strncmp(name, "/dev/", 5))
		name += 4;

	s = path + strlen(path);
	while (*name && s < path+sizeof(path)-1) {
		if (*name == '/') {
			*s++ = '.';
			while (*name == '/')
				name++;
		} else {
			*s++ = *name++;
		}
	}
	*s++ = '\0';
	dev->lock = strdup(path);

	return dev;
}

/*
 * Lock/unlock a device
 */
int
res_device_lock(res_device_t *dev, uid_t uid, pid_t pid)
{
	char	locktemp[4096], pidbuf[64];
	int	fd;

	sprintf(pidbuf, "%u\n", pid);

	/* First create a temp lock file */
	snprintf(locktemp, sizeof(locktemp),
			"%s.XXXXXX", dev->lock);
	if ((fd = mkstemp(locktemp)) < 0) {
		log("Unable to create lock fencepost: %m");
		return -1;
	}

	write(fd, pidbuf, strlen(pidbuf));
	close(fd);

	if (link(locktemp, dev->lock) >= 0) {
		chmod(dev->lock, 0644);
		chown(dev->lock, uid, 0);
		unlink(locktemp);
		return 0;
	}

	if (errno != EEXIST) {
		unlink(locktemp);
		return -1;
	}
	unlink(locktemp);

	/* There's a conflicting lock - see if it still exists */
	if (!(pid = lock_stale(dev)))
		return -1;

	/* Stale lock file. Return the holder's pid */
	return pid;
}

int
res_device_unlock(res_device_t *dev, uid_t uid)
{
	struct stat	stb;

	if (stat(dev->lock, &stb) < 0)
		return -1;
	if (stb.st_uid != uid && !lock_stale(dev))
		return -1;
	unlink(dev->lock);
	return 0;
}

int
lock_stale(res_device_t *dev)
{
	char	pidbuf[64];
	int	n, fd, locker;

	/* Now try to read the locker's pid from the file */
	fd = open(dev->lock, O_RDONLY);
	if (fd < 0)
		return 0;

	n = read(fd, pidbuf, sizeof(pidbuf)-1);
	close(fd);

	if (n <= 0)
		return 0;
	pidbuf[n] = '\0';

	/* Transparently deal with binary lock files - nobody
	 * should use that locking style anymore, but better be
	 * safe than sorry */
	if (n == sizeof(int)) {
		memcpy(&locker, pidbuf, sizeof(int));
	} else {
		locker = strtol(pidbuf, NULL, 0);
	}
	if (locker <= 0)
		return 0;

	if (kill(locker, 0) < 0 && errno == ESRCH)
		return locker;

	return 0;
}

/*
 * Delete a device object
 */
void
res_device_free(res_device_t *dev)
{
	free(dev->name);
	free(dev->lock);
	memset(dev, 0, sizeof(*dev));
	free(dev);
}
