/* 
 *	HT Editor
 *	infoview.cc
 *
 *	Copyright (C) 2001 Stefan Weyergraf (stefan@weyergraf.de)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 2 as
 *	published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>

#include "htdoc.h"
#include "htiobox.h"
#include "htstring.h"
#include "infoview.h"

extern "C" {
#include "infolex.h"
}

class info_pos: public ht_data {
public:
	UINT line;
	UINT ofs;
	
	info_pos(UINT l, UINT o) {
		line = l;
		ofs = o;
	}
};

int compare_keys_info_pos(ht_data *data_a, ht_data *data_b)
{
	info_pos *a = (info_pos*)data_a;
	info_pos *b = (info_pos*)data_b;
	if (a->line == b->line) {
		return a->ofs - b->ofs;
	}
	return a->line - b->line;
}

class info_xref: public ht_data {
public:
	char *target;
	UINT len;
	
	info_xref(char *t, UINT l) {
		target = strdup(t);
		len = l;
	}
	
	~info_xref() {
		free(target);
	}
};

/*
 *	CLASS info_node
 */

info_node::info_node(FILEOFS s)
{
	xrefs = new ht_stree();
	((ht_stree*)xrefs)->init(compare_keys_info_pos);

	start = s;
	len = 0;
}
	
info_node::~info_node()
{
	xrefs->destroy();
	delete xrefs;
}

/*
 *
 */

void free_info_node_t(info_node_t *n)
{
	if (n->file) free(n->file);
	if (n->node) free(n->node);
	if (n->prev) free(n->prev);
	if (n->next) free(n->next);
	if (n->up) free(n->up);
}

void free_info_xref_t(info_xref_t *x)
{
	if (x->text) free(x->text);
	if (x->target) free(x->target);
}

bool parse(ht_streamfile *s, ht_tree *n)
{
	YYSTYPE r;
	int t;
	info_node *d = NULL;
	UINT dl = 0;
	int om = 0;

	while ((t = infolex(&r))) {
		if ((t>=0) && (t<=255)) {
			char c=t;
			s->write(&c, 1);
			dl++;
		} else switch (t) {
			case INFO_FILE:
				om += r.node.codelen;
				if (d) {
					d->len = dl;
					d->xrefs->balance();
				}
				d = new info_node(r.node.body_ofs - om);
				dl = 0;
				n->insert(new ht_data_string(r.node.node), d);
				free_info_node_t(&r.node);
				break;
			case INFO_TTBL:
				if (d) {
					d->len = dl;
					d->xrefs->balance();
				}
				break;
			case INFO_XREF: {
				int l = strlen(r.xref.text);
				s->write(r.xref.text, l);
				dl += l;
				om += r.xref.codelen - l;
				if (d) {
					int ll = r.xref.start_line;
					UINT oo = r.xref.start_pofs;
					char *e = r.xref.text;
					while (ll <= r.xref.end_line) {
						char *ne = strchr(e, '\n');
						int vl = ne ? ne-e : strlen(e);
						if (ne) ne++;
						if (!vl) vl++;
						d->xrefs->insert(new info_pos(ll,
							oo),
							new info_xref(r.xref.target, vl));
						ll++;
						oo = 0;
						e = ne;
					}
				}
				free_info_xref_t(&r.xref);
				break;
			}				
		}
	}
/* debug */
#if 0
	ht_file *F = new ht_file();
	F->init("./ITEST", FAM_WRITE + FAM_CREATE);
	s->seek(0);
	s->copy_to(F);
	F->done();
	delete F;
#endif
/* /debug */	
	return true;
}

/*
 *	CLASS ht_info_lexer
 */

void ht_info_lexer::init()
{
	node = NULL;
	cx = 0;
	cy = 0;
}

lexer_state ht_info_lexer::getinitstate()
{
	return 1;
}

lexer_token ht_info_lexer::geterrortoken()
{
	return 42;
}

lexer_token ht_info_lexer::gettoken(char *buf, lexer_state *state, text_pos *p, UINT *len, bool start_of_line, bool only_state_changers)
{
	if (*buf) {
		if (node) {
			info_pos q(p->line, p->pofs);
			info_xref *x = (info_xref*)node->xrefs->get(&q);
			if (x) {
				*len=x->len;
				return ((cy == p->line) && (cx >= p->pofs) &&
				(cx < p->pofs + x->len)) ? 3 : 2;
			}
		}			
		*len=1;
		return 1;
	} else {
		*len=0;
		return 0;
	}
}

vcp ht_info_lexer::gettoken_color(lexer_token t)
{
	switch (t) {
		case 1: return VCP(VC_LIGHT(VC_WHITE), VC_TRANSPARENT);
		case 2: return VCP(VC_LIGHT(VC_YELLOW), VC_TRANSPARENT);
		case 3: return VCP(VC_LIGHT(VC_YELLOW), VC_BLUE);
	}
	return VCP(VC_WHITE, VC_RED);
}

void ht_info_lexer::set_cursor(UINT x, UINT y)
{
	cx = x;
	cy = y;
}

void ht_info_lexer::set_node(info_node *n)
{
	node = n;
}

/*
 *	CLASS ht_info_textfile
 */
 
class info_history_entry: public ht_data {
public:
	char *file;
	char *node;
	UINT cursorx;
	UINT cursory;
	UINT xofs;
	UINT top_line;

	info_history_entry(char *f, char *n, UINT x, UINT y, UINT xo, UINT yo)
	{
		file = ht_strdup(f);
		node = ht_strdup(n);
		cursorx = x;
		cursory = y;
		xofs = xo;
		top_line = yo;
	}

	~info_history_entry()
	{
		if (file) free(file);
		if (node) free(node);
	}
};
 
void ht_info_textfile::init(ht_streamfile *s, bool own_s, ht_syntax_lexer *l)
{
	ht_ltextfile::init(s, own_s, l);
	start=0;
	end=0;
}

void ht_info_textfile::done()
{
	ht_ltextfile::done();
}

ht_ltextfile_line *ht_info_textfile::fetch_line(UINT line)
{
	if (line < linecount())
		return ht_ltextfile::fetch_line(start+line);
	return NULL;
}

UINT ht_info_textfile::linecount()
{
	return end-start;
}

void ht_info_textfile::set_node(UINT ofs, UINT len)
{
	UINT s, e, t;
	start=0;
	end=ht_ltextfile::linecount();
	convert_ofs2line(ofs, &s, &t);
	convert_ofs2line(ofs+len, &e, &t);
	start=s;
	end=e;
}

/*
 *	CLASS ht_info_viewer
 */

void ht_info_viewer::init(bounds *b)
{
	ht_streamfile *f = new ht_streamfile();
	f->init();
	ht_textfile *s = new ht_textfile();
	s->init(f, true);
	ht_text_viewer::init(b, true, s, false, NULL);
	file = NULL;
	node = NULL;
	nodes = NULL;
	cnode = NULL;
	history = new ht_clist();
	((ht_clist*)history)->init();
}

void ht_info_viewer::done()
{
	if (history) {
		history->destroy();
		delete history;
	}
	if (nodes) {
		nodes->destroy();
		delete nodes;
	}
	if (node) free(node);
	if (file) free(file);
	ht_text_viewer::done();
}

void ht_info_viewer::draw()
{
	((ht_info_lexer*)lexer)->set_cursor(cursorx+xofs, cursory+top_line);
	ht_text_viewer::draw();
}

bool ht_info_viewer::gotonode(char *f, char *n)
{
	return igotonode(f, n, true);
}

bool ht_info_viewer::igotonode(char *f, char *n, bool add2hist)
{
	ht_info_textfile *infofile = NULL;
	ht_info_lexer *infolexer = NULL;
	bool isnewfile=(file && f) ? (strcmp(file, f)!=0) : (file!=f);
	
	if (add2hist) history->insert(new info_history_entry(isnewfile ? file : NULL, node, cursorx, cursory, xofs, top_line));
	
	select_clear();

	ht_stree *nt = NULL;

	if (isnewfile) {
		bool free_info;
		char *info;
		UINT infolen;
		if (strcmp(f, "hthelp.info")==0) {
			free_info = false;
			info = (char*)htinfo;
			infolen = sizeof htinfo;
		} else {
			/* FIXME: sdf */
			if (add2hist) history->del(history->count()-1);
			return false;
		}
/**/
		infolex_init();

		void *oldbuffer = infolex_current_buffer();
		void *strbuffer = infolex_scan_bytes_buffer(info, infolen);

		ht_mem_file *m = new ht_mem_file();
		m->init();

		nt = new ht_stree();
		nt->init(compare_keys_string);

		parse(m, nt);
		
		infolex_delete_buffer(strbuffer);
		if (oldbuffer) infolex_switch_buffer(oldbuffer);

		ht_info_textfile *t=new ht_info_textfile();
		t->init(m, true, NULL);
		
		infofile = t;

		ht_info_lexer *l = new ht_info_lexer();
		l->init();

		infolexer = l;
	} else {
		infofile = (ht_info_textfile*)textfile;
		infolexer = (ht_info_lexer*)lexer;
	}
	if (isnewfile || ((node && n) ? (strcmp(node, n)!=0) : (node!=n) )) {
		ht_data_string s(n);
		info_node *nn = (info_node*)(nt ? nt : nodes)->get(&s);

		if (!nn) {
		/* file ok, but node not found... */
			if (add2hist) history->del(history->count()-1);
			return false;
		}

		cnode = nn;

		infolexer->set_node(nn);
		infofile->set_node(nn->start, nn->len);
		
		cursorx = 0;
		cursory = 0;
		xofs = 0;
		top_line = 0;
	}

	if (f) {
		char *t = file;
		file = ht_strdup(f);
		if (t) free(t);
	}		

	if (node) free(node);
	node = ht_strdup(n);

	if (nt) {
		if (nodes) {
			nodes->destroy();
			delete nodes;	
		}

		nodes = nt;
	}

	if (isnewfile) {
		set_textfile(infofile, true);
		set_lexer(infolexer, true);
	}		
	return true;
}

void ht_info_viewer::handlemsg(htmsg *msg)
{
	if (msg->msg == msg_keypressed) {
		switch (msg->data1.integer) {
			case K_Space:
			case K_Return: 
				if (cnode) {
					info_pos p(top_line + cursory, xofs + cursorx);
					info_xref *x = (info_xref*)cnode->xrefs->get(&p);
					if (!x) {
						UINT cx = physical_cursorx();
						info_pos *q = (info_pos*)cnode->xrefs->enum_prev((ht_data**)&x, &p);
						if ((q) && ((q->line != top_line+cursory) ||
						(cx < q->ofs) || (cx >= q->ofs + x->len))) {
							x = NULL;
						}
					}
					if (x) {
						char *q = ht_strdup(file);
						if (!igotonode(q, x->target, true))
							errorbox("help topic '(%s)%s' not found", q, x->target);
						if (q) free(q);
					}
					clearmsg(msg);
					dirtyview();
					return;
				}
				break;
			case K_BackSpace: {
				int c;
				if ((c = history->count())) {
					info_history_entry *e = (info_history_entry*)history->get(c-1);
					if (e->node) {
						if (igotonode(e->file ? e->file : file, e->node, false)) {
							cursorx = e->cursorx;
							cursory = e->cursory;
							xofs = e->xofs;
							top_line = e->top_line;
						} else {
							errorbox("help topic '(%s)%s' not found", e->file ? e->file : file, e->node);
						}						
						history->del(c-1);
						clearmsg(msg);
						dirtyview();
						return;
					}						
				}
				break;					
			}
			case K_Tab: {
				if (cnode) {
					info_pos p(top_line + cursory, xofs + cursorx);
					info_xref *r;
					info_pos *q = (info_pos*)cnode->xrefs->enum_next((ht_data**)&r, &p);
					if (q) {
						goto_line(q->line);
						cursor_pput(q->ofs);
					}
				}					
				clearmsg(msg);
				dirtyview();
				return;
			}
			case K_BackTab: {
				if (cnode) {
					info_pos p(top_line + cursory, xofs + cursorx);
					info_xref *r;
					info_pos *q = (info_pos*)cnode->xrefs->enum_prev((ht_data**)&r, &p);
					if (q) {
						goto_line(q->line);
						cursor_pput(q->ofs);
					}
				}					
				clearmsg(msg);
				dirtyview();
				return;
			}
		}
	}
	ht_text_viewer::handlemsg(msg);
}
