/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: theora_codec_info.cpp,v 1.4.4.3 2005/01/21 19:30:13 nhart Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */
#include "theora_codec_info.h"
#include "hlxclib/memory.h"
#include "hxassert.h"
#include "debug.h"

#define D_THEORA_INFO 0 //0x400000

const unsigned char CTheoraCodecInfo::zm_blankComment[] = {
    0x81, 't', 'h', 'e', 'o', 'r', 'a',
    0x00, 0x00, 0x00, 0x00, // vender string length
    0x00, 0x00, 0x00, 0x00  // comment count
};

CTheoraCodecInfo::CTheoraCodecInfo() :
    m_state(Start),
    m_uFPSNum(0),
    m_uFPSDenom(0),
    m_uBitrate(0),
    m_uKeyFrameShift(0),
    m_pIdentBuf(NULL),
    m_uIdentBufSize(0),
    m_bFirstPkt(TRUE),
    m_currentPos(0),
    m_basePos(0),
    m_dropped(0)
{
#ifdef _DEBUG
    debug_level() |= D_THEORA_INFO;
#endif // _DEBUG
    DPRINTF(D_THEORA_INFO, ("CTheoraCodecInfo::CTheoraCodecInfo()\n"));
}

CTheoraCodecInfo::~CTheoraCodecInfo()
{
    DPRINTF(D_THEORA_INFO, ("CTheoraCodecInfo::~CTheoraCodecInfo()\n"));

    HX_VECTOR_DELETE(m_pIdentBuf);
}

COggCodecInfo* 
CTheoraCodecInfo::Build(ogg_packet* pPkt)
{
    COggCodecInfo* pRet = NULL;

    if (isTheoraHeader(pPkt) &&   // Is it a Theora header?
        (pPkt->packet[0] == 0x80)) // Is it the ident header?
    {
        pRet = new CTheoraCodecInfo;
    }

    return pRet;
}

// COggCodecInfo methods
HX_RESULT 
CTheoraCodecInfo::OnPacket(ogg_packet* pPkt)
{
    HX_RESULT res = HXR_INVALID_PARAMETER;

    if (pPkt)
    {
#if 0
        DPRINTF(D_THEORA_INFO, ("CTheoraCodecInfo::OnPacket() : size %u\n",
                                pPkt->bytes));
#endif
        
        switch(m_state) {
        case Start:
            res = handleIdentHdr(pPkt);
                
            if (HXR_OK == res)
            {
                changeState(NeedComment);
            }
            break;
        case NeedComment:
            if (IsHeader(pPkt) && (pPkt->packet[0] == 0x81))
            {
                changeState(NeedCodebook);
                res = HXR_OK;
            }
            break;
        case NeedCodebook:
            if (IsHeader(pPkt) && (pPkt->packet[0] == 0x82))
            {
                changeState(NeedFirstDataPkt);
                res = HXR_OK;
            }
            break;
        case NeedFirstDataPkt:
            if (!IsHeader(pPkt))
            {
                changeState(NeedBaseTime);
            }
            else
            {
                // we don't want to pass through
                // if this isn't a data packet
                break;
            }
            // pass thru intentional
        case NeedBaseTime:
            res = updatePosition(pPkt);

            if ((HXR_OK == res) && pPkt->granulepos != -1)
            {
                ogg_int64_t currentFrames = granuleToFrames(m_currentPos);
                ogg_int64_t pktFrames = granuleToFrames(pPkt->granulepos);
                
                if (pktFrames >= currentFrames)
                {
                    m_basePos = 
                        framesToGranulePos(pktFrames - currentFrames);
                }
                else
                {
                    m_basePos = 0;
                    m_dropped = 
                        framesToGranulePos(currentFrames - pktFrames);
                }
                
                changeState(Ready);
            }
            break;
        case Ready:
            if (pPkt->packet[0] & 0x80)
            {
                // header packet
                res = HXR_OK;
            }
            else
            {
                // data packet
                res = updatePosition(pPkt);
            }
            break;
        default:
            break;
        };
    }

    return res;
}

OggCodecType CTheoraCodecInfo::Type() const
{
    return ctTheora;
}

BOOL CTheoraCodecInfo::HaveHeaders() const
{
    return ((NeedBaseTime == m_state) ||
            (Ready == m_state));
}

BOOL CTheoraCodecInfo::HaveBaseGranulePos() const
{
    return (Ready == m_state);
}

BOOL CTheoraCodecInfo::AllowPacketOutError() const
{
    return (m_state == NeedFirstDataPkt) ? TRUE : FALSE;
}

ogg_int64_t CTheoraCodecInfo::CurrentGranulePos() const
{
    return m_currentPos;
}

HX_RESULT 
CTheoraCodecInfo::SetCurrentGranulePos(ogg_int64_t pos)
{
    HX_RESULT res = HXR_UNEXPECTED;

    HX_ASSERT(Ready == m_state);

    if (Ready == m_state)
    {
        m_currentPos = pos;
        m_bFirstPkt = TRUE;

        res = HXR_OK;
    }

    return res;
}

ogg_int64_t CTheoraCodecInfo::GranulePosDropped() const
{
    return m_dropped;
}

ogg_int64_t CTheoraCodecInfo::GranulePosDifference(ogg_int64_t a, 
                                                   ogg_int64_t b) const
{
    ogg_int64_t aFrames = granuleToFrames(a);
    ogg_int64_t bFrames = granuleToFrames(b);
                
    return framesToGranulePos(aFrames - bFrames);
}

BOOL 
CTheoraCodecInfo::IsHeader(ogg_packet* pPkt) const
{
    return isTheoraHeader(pPkt);
}

BOOL CTheoraCodecInfo::IsCommentHeader(ogg_packet* pPkt) const
{
    return (IsHeader(pPkt) && (pPkt->packet[0] == 0x81));
}

ogg_packet* CTheoraCodecInfo::CreateBlankCommentPacket() const
{
    ogg_packet* pRet = new ogg_packet;

    if (pRet)
    {
        pRet->b_o_s = 0;
        pRet->e_o_s = 0;
        pRet->granulepos = 0;
        pRet->packetno = 1;
        pRet->bytes = sizeof(zm_blankComment);
        pRet->packet = new unsigned char[pRet->bytes];

        if (pRet->packet)
        {
            memcpy(pRet->packet, zm_blankComment, pRet->bytes);
        }
        else
        {
            delete pRet;
            pRet = NULL;
        }
    }

    return pRet;
}

HX_RESULT 
CTheoraCodecInfo::GetTimestamp(ogg_int64_t granulePos, 
                               COggTimestamp& timestamp) const
{
    HX_RESULT res = HXR_UNEXPECTED;

    HX_ASSERT(Ready == m_state);

    if (Ready == m_state)
    {
        if (granulePos == 0)
        {
            timestamp = COggTimestamp(0, m_uFPSNum);
            timestamp += m_startTimestamp;

            res = HXR_OK;
        }
        else 
        {
            ogg_int64_t granuleFrames = granuleToFrames(granulePos);
            ogg_int64_t baseFrames = granuleToFrames(m_basePos);

            if (granuleFrames >= baseFrames)
            {
                ogg_int64_t llDelta = granuleFrames - baseFrames;

                timestamp = COggTimestamp(llDelta * m_uFPSDenom, m_uFPSNum);
                timestamp += m_startTimestamp;

                res = HXR_OK;
            }
        }
    }

    return res;
}

HX_RESULT 
CTheoraCodecInfo::GetBaseTimestamp(COggTimestamp& timestamp) const
{
    timestamp = COggTimestamp(granuleToFrames(m_basePos) * m_uFPSDenom, 
                              m_uFPSNum);
    return HXR_OK;
}

HX_RESULT 
CTheoraCodecInfo::SetBaseTimestamp(const COggTimestamp& timestamp)
{
    ogg_int64_t samples = timestamp.Samples();
    if (timestamp.SampleRate() != m_uFPSNum)
    {
        COggTimestamp tmp = timestamp;
        tmp.SetSampleRate(m_uFPSNum);

        samples = tmp.Samples();
    }

    m_basePos = framesToGranulePos(samples /  m_uFPSDenom);

    return HXR_OK;
}

HX_RESULT 
CTheoraCodecInfo::SetStartTimestamp(const COggTimestamp& timestamp)
{
    m_startTimestamp = timestamp;
    m_startTimestamp.SetSampleRate(m_uFPSNum);

    return HXR_OK;
}

COggCodecInfo* CTheoraCodecInfo::Clone() const
{
    COggCodecInfo* pRet = NULL;

    if (Ready == m_state)
    {
        CTheoraCodecInfo* pInfo = new CTheoraCodecInfo;

        if (pInfo)
        {
            if (HXR_OK == pInfo->copy(*this))
            {
                pRet = pInfo;
            }
            else
            {
                delete pInfo;
                pInfo = NULL;
            }
        }
    }

    return pRet;
}

HX_RESULT 
CTheoraCodecInfo::copy(const CTheoraCodecInfo& rhs)
{
    HX_RESULT res = HXR_OK;
    
    delete [] m_pIdentBuf;
    m_pIdentBuf = NULL;

    m_state = rhs.m_state;
    m_uFPSNum = rhs.m_uFPSNum;
    m_uFPSDenom = rhs.m_uFPSDenom;
    m_uBitrate = rhs.m_uBitrate;
    m_uKeyFrameShift = rhs.m_uKeyFrameShift;
    m_uIdentBufSize = rhs.m_uIdentBufSize;
    m_bFirstPkt = rhs.m_bFirstPkt;
    m_currentPos = rhs.m_currentPos;
    m_basePos = rhs.m_basePos;
    m_dropped = rhs.m_dropped;
    m_startTimestamp = rhs.m_startTimestamp;

    if (m_uIdentBufSize && rhs.m_pIdentBuf)
    {
        m_pIdentBuf = new UINT8[m_uIdentBufSize];
        if (m_pIdentBuf)
        {
            memcpy(m_pIdentBuf, rhs.m_pIdentBuf, m_uIdentBufSize);
        }
        else
        {
            res = HXR_OUTOFMEMORY;
        }
    }


    return res;
}

static const char* z_pStateNames[] = {
    "Start",
    "NeedComment",
    "NeedCodebook",
    "NeedFirstDataPkt",
    "NeedBaseTime",
    "Ready"
};

void CTheoraCodecInfo::changeState(State newState)
{
    DPRINTF(D_THEORA_INFO, ("CTheoraCodecInfo::changeState() : %s -> %s\n",
                            z_pStateNames[m_state],
                            z_pStateNames[newState]));
    m_state = newState;
}

HX_RESULT CTheoraCodecInfo::handleIdentHdr(ogg_packet* pPkt)
{
    HX_RESULT res = HXR_INVALID_PARAMETER;

    if (IsHeader(pPkt) && 
        (pPkt->packet[0] == 0x80) &&
        (pPkt->bytes >= 42))
    {
        const unsigned char* pIdent = (const unsigned char*)pPkt->packet;

        // Ident Header
        m_uFPSNum = ((pIdent[0x16] << 24) | (pIdent[0x17] << 16) |
                      (pIdent[0x18] << 8) | (pIdent[0x19]));
        m_uFPSDenom = ((pIdent[0x1a] << 24) | (pIdent[0x1b] << 16) |
                        (pIdent[0x1c] << 8) | (pIdent[0x1d]));
        
        // Attempt to reduce the numerator and denominator
        // by finding the greatest common denominator.
        UINT32 tmp = gcd(m_uFPSNum, m_uFPSDenom);
        if (tmp > 1)
        {
            m_uFPSNum /= tmp;
            m_uFPSDenom /= tmp;
        }

        m_startTimestamp.SetSampleRate(m_uFPSNum);

        m_uBitrate = ((pIdent[0x25] << 16) | (pIdent[0x26] << 8) |
                       (pIdent[0x27]));
        m_uKeyFrameShift = (((pIdent[0x28] & 0x3) << 3) | 
                            (pIdent[0x29] >> 5));
        
        DPRINTF(D_THEORA_INFO, 
                ("CTheoraCodecInfo::handleIdentHdr : FPSnum %u FPSdenom %u bitrate %u keyframeShift %d\n",
                 m_uFPSNum, m_uFPSDenom, m_uBitrate, m_uKeyFrameShift));

        if (m_uFPSDenom == 0)
        {
            m_uFPSDenom = 1;
        }

        HX_VECTOR_DELETE(m_pIdentBuf);
        m_pIdentBuf = new UINT8[pPkt->bytes];

        if (m_pIdentBuf)
        {
            memcpy(m_pIdentBuf, pPkt->packet, pPkt->bytes);
            m_uIdentBufSize = pPkt->bytes;

            res = HXR_OK;
        }
        else
        {
            res = HXR_OUTOFMEMORY;
        }
    }

    return res;
}

HX_RESULT CTheoraCodecInfo::updatePosition(ogg_packet* pPkt)
{
    HX_RESULT res = HXR_INVALID_PARAMETER;

    if (pPkt && (pPkt->packet[0] & 0x80) == 0)
    {
        if (!m_bFirstPkt)
        {
            if ((pPkt->packet[0] & 0x40) == 0)
            {
                // This is an I Frame
                m_currentPos = 
                    framesToGranulePos(granuleToFrames(m_currentPos) + 1);
            }
            else
            {
                // This is a P Frame
                m_currentPos++;
            }
        }

        m_bFirstPkt = FALSE;
        res = HXR_OK;
    }

    return res;
}

UINT32 CTheoraCodecInfo::gcd(UINT32 a, UINT32 b)
{
    // Compute the greatest common denominator
    while(b != 0)
    {
        UINT32 tmp = b;
        b = a % b;
        a = tmp;
    }

    return a;
}

ogg_int64_t CTheoraCodecInfo::granuleToFrames(ogg_int64_t pos) const
{
    ogg_int64_t mask = (1 << m_uKeyFrameShift) - 1;

    return  (pos & mask) + (pos >> m_uKeyFrameShift);
}

ogg_int64_t CTheoraCodecInfo::framesToGranulePos(ogg_int64_t frames) const
{
    return frames << m_uKeyFrameShift;
}

BOOL CTheoraCodecInfo::isTheoraHeader(ogg_packet* pPkt)
{
    BOOL bRet = FALSE;

    if (pPkt && pPkt->packet && (pPkt->bytes > 7) &&
        ((pPkt->packet[0] & 0x80) == 0x80) &&
        !memcmp(pPkt->packet + 1, "theora", 6))
        
    {
        bRet = TRUE;
    }

    return bRet;
}
