/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: hxnetif.cpp,v 1.6.8.1 2004/07/09 02:06:49 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "hxcom.h"
#include <string.h>
#include <stdio.h>

#ifndef _WINCE
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <winsock2.h>
#include <ws2tcpip.h>
#endif /* _WINCE */

#include <windows.h>
#include "hlxclib/sys/socket.h"
#include "hlxosstr.h"
#include "hxresult.h"
#include "hxslist.h"
#include "netbyte.h"
#include "hxengin.h"
#include "hxnetif.h"

#define CHECK_ADDR_CHANGE_INTERVAL  60000

HXNetInterface::HXNetInterface(IUnknown* pContext)
		: m_lRefCount(0)
		, m_bInitialized(FALSE)
		, m_hIPLib(NULL)
		, m_hWinSockLib(NULL)
		, m_hAddrChangeEvent(NULL)
		, m_pScheduler(NULL)
		, m_pAddrChangeCallback(NULL)
		, m_pNetInterfaceList(NULL)
		, m_pSinkList(NULL)
		, _pGetIfTable(NULL)
		, _pGetIpAddrTable(NULL)
		, _pNotifyAddrChange(NULL)
		, _hxWSAStartup(NULL)
		, _hxWSACleanup(NULL)
#ifdef _WINCE
		, _pInetAddr(NULL)
		,_pGetAdaptersInfo(NULL)
#else
		, _hxsocket(NULL)
		, _hxclosesocket(NULL)
		, _raWSAIoctl(NULL)
#endif /* _WINCE */
		, m_handle(NULL)
{
    m_pContext = pContext;
    HX_ADDREF(m_pContext);
}

HXNetInterface::~HXNetInterface()
{
    Close();
}

STDMETHODIMP
HXNetInterface::QueryInterface(REFIID riid, void**ppvObj)
{
    if (IsEqualIID(riid, IID_IUnknown))
    {
	AddRef();
	*ppvObj = this;
	return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IHXNetInterfaces))
    {
	AddRef();
	*ppvObj = (IHXNetInterfaces*)this;
	return HXR_OK;
    }

    *ppvObj = NULL;
    return HXR_NOINTERFACE;
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IUnknown::AddRef
//  Purpose:
//	Everyone usually implements this the same... feel free to use
//	this implementation.
//
STDMETHODIMP_(ULONG32) 
HXNetInterface::AddRef()
{
    return InterlockedIncrement(&m_lRefCount);
}

/////////////////////////////////////////////////////////////////////////
//  Method:
//	IUnknown::Release
//  Purpose:
//	Everyone usually implements this the same... feel free to use
//	this implementation.
//
STDMETHODIMP_(ULONG32) 
HXNetInterface::Release()
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
    {
        return m_lRefCount;
    }

    delete this;
    return 0;
}

STDMETHODIMP
HXNetInterface::UpdateNetInterfaces(void)
{
    HX_RESULT	    rc = HXR_OK;    

    if (!m_bInitialized)
    {
	m_bInitialized = TRUE;

	rc = RetrieveNetInterface0(m_pNetInterfaceList);
	if (!m_pNetInterfaceList || m_pNetInterfaceList->GetCount() == 0)
	{
	    // try the generic method
	    RetrieveNetInterface1(m_pNetInterfaceList);
	}

	if (_pNotifyAddrChange)
	{
	    m_hAddrChangeEvent = CreateEvent(NULL, TRUE, FALSE, NULL);

#ifndef _WINCE
	    m_overLapped.Internal = NULL;
	    m_overLapped.InternalHigh = NULL;
	    m_overLapped.Offset = 0;
	    m_overLapped.OffsetHigh = 0;
	    m_overLapped.hEvent = m_hAddrChangeEvent;

	    if (ERROR_IO_PENDING == _pNotifyAddrChange(&m_handle, &m_overLapped))
#else
	// Overlapped param is not supported in wince 3.0 in the second param to NotifyAddrChange
	
	if(NO_ERROR == _pNotifyAddrChange(&m_hAddrChangeEvent, NULL))
#endif /* _WINCE */
	    {
		if (!m_pScheduler)
		{
		    m_pContext->QueryInterface(IID_IHXScheduler, (void**)&m_pScheduler);
		}

		if (m_pScheduler)
		{
		    m_pAddrChangeCallback = new AddrChangeCallback;
		    m_pAddrChangeCallback->m_pParent = this;
		    m_pAddrChangeCallback->AddRef();

		    m_pAddrChangeCallback->m_bIsCallbackPending = TRUE;
		    m_pAddrChangeCallback->m_PendingHandle = m_pScheduler->RelativeEnter(m_pAddrChangeCallback, CHECK_ADDR_CHANGE_INTERVAL);
		}
	    }
	}
    }
    else if (IsNetInterfaceChanged())
    {
	if (m_pSinkList)
	{
	    CHXSimpleList::Iterator ndx = m_pSinkList->Begin();
	    for (; ndx != m_pSinkList->End(); ++ndx)
	    {
		IHXNetInterfacesAdviseSink* pSink = (IHXNetInterfacesAdviseSink*) (*ndx);
		pSink->NetInterfacesUpdated();
	    }
	}
    }

    return rc;
}

STDMETHODIMP_(UINT32)
HXNetInterface::GetNumOfNetInterfaces()
{
    if (!m_bInitialized)
    {
	UpdateNetInterfaces();
    }

    return m_pNetInterfaceList ? m_pNetInterfaceList->GetCount() : 0;
}

STDMETHODIMP
HXNetInterface::GetNetInterfaces(UINT16	lIndex,
				  REF(NIInfo*)	pNIInfo)
{
    HX_RESULT	rc = HXR_OK;
    int		i = 0;
    CHXSimpleList::Iterator iter;

    pNIInfo = NULL;

    if (!m_bInitialized)
    {
	UpdateNetInterfaces();
    }

    if (m_pNetInterfaceList)
    {
	iter = m_pNetInterfaceList->Begin();
	for (; iter != m_pNetInterfaceList->End(); ++iter, ++i)
	{
	    NIInfo* pInfo = (NIInfo*)(*iter);
	    if (i == lIndex)
	    {
		pNIInfo = pInfo;
		break;
	    }
	}
    }

    if (!pNIInfo)
    {
	rc = HXR_FAILED;
    }

    return rc;
}

STDMETHODIMP
HXNetInterface::AddAdviseSink(IHXNetInterfacesAdviseSink* pSink)
{
    HX_RESULT	rc = HXR_OK;

    if (!m_pSinkList)
    {
	m_pSinkList = new CHXSimpleList();
    }

    pSink->AddRef();
    m_pSinkList->AddTail(pSink);

    return rc;
}

STDMETHODIMP
HXNetInterface::RemoveAdviseSink(IHXNetInterfacesAdviseSink* pSink)
{
    HX_RESULT	rc = HXR_OK;

    LISTPOSITION lPosition = m_pSinkList->Find(pSink);

    if (!lPosition)
    {
	rc = HXR_UNEXPECTED;
	goto cleanup;
    }

    m_pSinkList->RemoveAt(lPosition);
    pSink->Release();

cleanup:

    return rc;
}

HX_RESULT
HXNetInterface::RetrieveNetInterface0(CHXSimpleList*& pNetInterfaceList)
{
    HX_RESULT		rc = HXR_OK;
    char		temp;
    DWORD		dwFlags = 0;
    UINT32		ulSize = 1;
    UINT32		ulNetAddress = 0;
    UINT32		ulNetMask = 0;
    DWORD		i = 0;
    DWORD		j = 0;
    PMIB_IFTABLE	ifTable = NULL;
    PMIB_IPADDRTABLE	ipAddrTable = NULL;
    NIType		type = UNKNOWN;
    NIInfo*		pNIInfo = NULL;

    if (!m_hIPLib)
    {
	m_hIPLib = LoadLibrary(OS_STRING("IPHLPAPI.DLL"));
	if (m_hIPLib)
	{
	    _pGetIfTable = (GETIFTABLE)GetProcAddress(m_hIPLib, OS_STRING("GetIfTable"));
	    _pGetIpAddrTable = (GETIPADDRTABLE)GetProcAddress(m_hIPLib, OS_STRING("GetIpAddrTable"));
	    _pNotifyAddrChange = (NOTIFYADDRCHANGE)GetProcAddress(m_hIPLib, OS_STRING("NotifyAddrChange"));
	}
    }

    if (!_pGetIfTable	    ||
	!_pGetIpAddrTable   ||
	!_pNotifyAddrChange)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }
    
    // get interface table
    rc = _pGetIfTable(PMIB_IFTABLE(&temp),&ulSize,TRUE);
    if(HXR_OK == rc)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    ifTable = (PMIB_IFTABLE) new char[ulSize];

    rc = _pGetIfTable(ifTable, &ulSize, TRUE);
    if (HXR_OK != rc)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    // get ip addresses table
    ulSize = 1;
  
    rc = _pGetIpAddrTable(PMIB_IPADDRTABLE(&temp), &ulSize, TRUE);
    if (HXR_OK == rc)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    ipAddrTable = (PMIB_IPADDRTABLE) new char[ulSize];

    rc = _pGetIpAddrTable(ipAddrTable, &ulSize, TRUE);
    if (HXR_OK != rc)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    // check if interface exists
    if (0 == ifTable->dwNumEntries)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    // enumerate all interfaces
    for (i = 0; i < ifTable->dwNumEntries; i++) 
    {
	// ignore the LOOPBACK
	if (ifTable->table[i].dwType == MIB_IF_TYPE_LOOPBACK)
	{
	    continue;;
	}
	else if (ifTable->table[i].dwType == MIB_IF_TYPE_PPP &&
		 ifTable->table[i].dwOperStatus)
	{
	    type = POINTTOPOINT;
	}
	else if (ifTable->table[i].dwOperStatus)
	{
	    type = BROADCAST;
	}
	else
	{
	    continue;
	}

	// get corresponding IP
	for(j=0; j < ipAddrTable->dwNumEntries; j++) 
	{
	    if(ipAddrTable->table[j].dwIndex == ifTable->table[i].dwIndex) 
	    {
		ulNetAddress = ipAddrTable->table[j].dwAddr;
		ulNetMask = ipAddrTable->table[j].dwMask;
		break;
	    }
	}

	// ignore net interface with invalid IP/Mask
	if (ulNetAddress == 0 || ulNetMask == 0)
	{
	    continue;
	}

	pNIInfo = new NIInfo;
	pNIInfo->bActive = TRUE;
	pNIInfo->type = type;
	pNIInfo->ulNetAddress = ulNetAddress; 
	pNIInfo->ulNetMask = ulNetMask;

	if (!pNetInterfaceList)
	{
	    pNetInterfaceList = new CHXSimpleList();
	}

	pNetInterfaceList->AddTail(pNIInfo);
    }
    
cleanup:
    
    if (ifTable)
    {
	delete[] ((char*)ifTable);
    }

    if (ipAddrTable)
    {
	delete[] ((char*)ipAddrTable);
    }

    return rc;
}

HX_RESULT
HXNetInterface::RetrieveNetInterface1(CHXSimpleList*& pNetInterfaceList)
{
    HX_RESULT	    rc = HXR_OK;
    int		    iStructures = 0;
    int		    iPosition = 0;
    long	    lFlags = 0;
    UINT32	    ulBytes = 0; 
    UINT32	    ulNetAddress = 0;
    UINT32	    ulNetMask = 0;
    SOCKET	    s = 0;
    WORD	    wVersionRequested = 0;
    NIType	    type = UNKNOWN;
    NIInfo*	    pNIInfo = NULL;
    WSADATA	    wsaData;
#ifdef _WINCE
	PIP_ADAPTER_INFO pAdapterInfo = NULL;
	PIP_ADDR_STRING pAddrList = NULL;
	ULONG ulSize = 0;
	DWORD dwResult = ERROR_NOT_SUPPORTED;
#else
	INTERFACE_INFO* pInfo = NULL;
#endif
    if (!m_hWinSockLib)
    {
#ifdef _WINCE
	m_hWinSockLib = LoadLibrary(OS_STRING("winsock.dll"));
#else
	m_hWinSockLib = LoadLibrary(OS_STRING("ws2_32.dll"));
#endif /* _WINCE */
	if (m_hWinSockLib)
	{
		_hxWSAStartup = (WSASTARTUP)GetProcAddress(m_hWinSockLib, OS_STRING("WSAStartup"));
		_hxWSACleanup = (WSACLEANUP)GetProcAddress(m_hWinSockLib, OS_STRING("WSACleanup"));
#ifdef _WINCE
		_pInetAddr = (INET_ADDR) ::GetProcAddress(m_hWinSockLib, OS_STRING("inet_addr"));
#else
		_hxsocket = (HXSOCKET)GetProcAddress(m_hWinSockLib, OS_STRING("socket"));
		_hxclosesocket = (CLOSESOCKET)GetProcAddress(m_hWinSockLib, OS_STRING("closesocket"));
		_raWSAIoctl = (WSAIOCTL)GetProcAddress(m_hWinSockLib, OS_STRING("WSAIoctl"));
#endif /* _WINCE */
	}
    }

#ifdef _WINCE
	if (!_hxWSAStartup	||
	!_hxWSACleanup	||
	!_pInetAddr)
#else
    if (!_hxsocket	||
	!_hxclosesocket ||
	!_hxWSAStartup  ||
	!_hxWSACleanup  ||
	!_raWSAIoctl)
#endif /* _WINCE */
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    wVersionRequested = (1 << 8) + 1;	
    _hxWSAStartup(wVersionRequested, &wsaData);

#ifndef _WINCE
    s = _hxsocket(AF_INET, SOCK_DGRAM, 0);
    
    if (s == INVALID_SOCKET)
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    pInfo = new INTERFACE_INFO[MAX_INTERFACES];

    // get interfaces list
    if (_raWSAIoctl(s, SIO_GET_INTERFACE_LIST, NULL, NULL, pInfo,
		    sizeof(INTERFACE_INFO)*MAX_INTERFACES, &ulBytes, NULL, NULL))
    {
	rc = HXR_FAILED;
	goto cleanup;
    }

    // caculate # of interfaces we have
    iStructures = ulBytes / sizeof(INTERFACE_INFO);

    // iterate through the interface list
    for (iPosition = 0; iPosition < iStructures; iPosition++)
    {
	lFlags = pInfo[iPosition].iiFlags;

	if (lFlags & IFF_LOOPBACK)
	{
	    continue;
	}
	else if ((lFlags & IFF_POINTTOPOINT) && (lFlags & IFF_UP))
	{
	    type = POINTTOPOINT;
	}
	else if ((lFlags & IFF_BROADCAST) && (lFlags & IFF_UP))
	{
	    type = BROADCAST;
	}
	else
	{
	    continue;
	}

	ulNetAddress = ((sockaddr_in*)&(pInfo[iPosition].iiAddress))->sin_addr.s_addr;
	ulNetMask = ((sockaddr_in*)&(pInfo[iPosition].iiNetmask))->sin_addr.s_addr;

	// ignore the net interface with invalid IP/Mask
	if (ulNetAddress == 0 || ulNetMask == 0)
	{
	    continue;
	}
	
	pNIInfo = new NIInfo;
	pNIInfo->bActive = TRUE;
	pNIInfo->type = type;
	pNIInfo->ulNetAddress = ulNetAddress; 
	pNIInfo->ulNetMask = ulNetMask;

	if (!pNetInterfaceList)
	{
	    pNetInterfaceList = new CHXSimpleList();
	}

	pNetInterfaceList->AddTail(pNIInfo);
    }

cleanup:

    if (_hxclosesocket)
    {
	_hxclosesocket(s);
    }

    if (_hxWSACleanup)
    {
	_hxWSACleanup();
    }

    HX_VECTOR_DELETE(pInfo);
#else

// WinCE does not support Winsock2, yet. Also, it has many variations for registry settings.
// Thus, it is better to use GetAdapterInfo defined in iphlpapi.h.
// Note: This call is supported only in WinCE 3.0 or later.


	if (NULL == m_hIPLib)
		m_hIPLib = LoadLibrary(OS_STRING("IPHLPAPI.DLL"));

	if (m_hIPLib)
		_pGetAdaptersInfo = (GETADAPTERSINFO) ::GetProcAddress(m_hIPLib, OS_STRING("GetAdaptersInfo"));

	if (NULL == _pGetAdaptersInfo)
    {
		rc = HXR_FAILED;
		goto cleanup;
    }

	// first make a call with ulSize = 0 to get the exact size needed
	dwResult = _pGetAdaptersInfo(pAdapterInfo, &ulSize);
	if (ERROR_BUFFER_OVERFLOW == dwResult)
	{
		// allocate right amount of space for adapter info
		pAdapterInfo = (PIP_ADAPTER_INFO)(new char[ulSize]);
		if (!pAdapterInfo)
		{
			rc = HXR_OUTOFMEMORY;
			goto cleanup;
		}

		// fill up adapters info
		dwResult= _pGetAdaptersInfo(pAdapterInfo, &ulSize);
		if (ERROR_SUCCESS != dwResult)
		{
			rc = HXR_FAILED;
			goto cleanup;
		}

		// step thru the adapters list that we received
		while (pAdapterInfo != NULL)
		{
			if (MIB_IF_TYPE_PPP == pAdapterInfo->Type)
				type = POINTTOPOINT;
			else if (MIB_IF_TYPE_ETHERNET == pAdapterInfo->Type)
				type = BROADCAST;
			else
				continue;

			// step thru all IP addresses for each adapter
			pAddrList = &(pAdapterInfo->IpAddressList);
			do
			{
				ulNetAddress = _pInetAddr(pAddrList->IpAddress.String);
				ulNetMask = _pInetAddr(pAddrList->IpMask.String);

				// ignore the net interface with invalid IP/Mask
				if (INADDR_NONE != ulNetAddress && 0 != ulNetAddress && INADDR_NONE != ulNetMask && 0 != ulNetMask)
				{
					pNIInfo = new NIInfo;
					if (!pAdapterInfo)
					{
						rc = HXR_OUTOFMEMORY;
						goto cleanup;
					}
					pNIInfo->bActive = TRUE;
					pNIInfo->type = type;
					pNIInfo->ulNetAddress = ulNetAddress; 
					pNIInfo->ulNetMask = ulNetMask;
				
					if (!pNetInterfaceList)
					{
						pNetInterfaceList = new CHXSimpleList();
						if (!pAdapterInfo)
						{
							rc = HXR_OUTOFMEMORY;
							goto cleanup;
						}
					}
					pNetInterfaceList->AddTail(pNIInfo);
					pAddrList = pAddrList->Next;
				}				
			}while (pAddrList != NULL);

			pAdapterInfo = pAdapterInfo->Next;
		}
	}
	else
	{
		rc = HXR_FAILED;
		goto cleanup;
	}

cleanup:
	
    HX_VECTOR_DELETE(pAdapterInfo);

    if (_hxWSACleanup)
    {
	_hxWSACleanup();
    }


#endif /* _WINCE */

    return rc;
}	



void
HXNetInterface::CheckAddrChange(void)
{
    DWORD dwResult = 0;

    if (WAIT_OBJECT_0 == WaitForSingleObject(m_hAddrChangeEvent, 0))
    {
	Reset(m_pNetInterfaceList);
	RetrieveNetInterface0(m_pNetInterfaceList);

	if (m_pSinkList)
	{
	    CHXSimpleList::Iterator ndx = m_pSinkList->Begin();
	    for (; ndx != m_pSinkList->End(); ++ndx)
	    {
		IHXNetInterfacesAdviseSink* pSink = (IHXNetInterfacesAdviseSink*) (*ndx);
		pSink->NetInterfacesUpdated();
	    }
	}

#ifdef _WINCE
	// Overlapped param is not supported in wince 3.0 in the second param to NotifyAddrChange
			
	if(NO_ERROR == _pNotifyAddrChange(&m_hAddrChangeEvent, NULL))
#else			
	if (ERROR_IO_PENDING == _pNotifyAddrChange(&m_handle, &m_overLapped))
#endif /* _WINCE */
	{
	    m_pAddrChangeCallback->m_bIsCallbackPending = TRUE;
	    m_pAddrChangeCallback->m_PendingHandle = m_pScheduler->RelativeEnter(m_pAddrChangeCallback, CHECK_ADDR_CHANGE_INTERVAL);
	}
    }
    else
    {
	m_pAddrChangeCallback->m_bIsCallbackPending = TRUE;
	m_pAddrChangeCallback->m_PendingHandle = m_pScheduler->RelativeEnter(m_pAddrChangeCallback, CHECK_ADDR_CHANGE_INTERVAL);
    }

    return;
}

BOOL
HXNetInterface::IsNetInterfaceChanged(void)
{
    BOOL	    bResult = FALSE;
    CHXSimpleList*  pTempNetInterfaceList = NULL;

    if (m_pAddrChangeCallback && m_pAddrChangeCallback->m_bIsCallbackPending)
    {
	if (WAIT_OBJECT_0 == WaitForSingleObject(m_hAddrChangeEvent, 0))
	{
	    bResult = TRUE;

	    Reset(m_pNetInterfaceList);
	    RetrieveNetInterface0(m_pNetInterfaceList);

#ifdef _WINCE
		// Overlapped param is not supported in wince 3.0 in the second param to NotifyAddrChange
		
		if(NO_ERROR == _pNotifyAddrChange(&m_hAddrChangeEvent, NULL))
#else			
	    if (ERROR_IO_PENDING == _pNotifyAddrChange(&m_handle, &m_overLapped))
#endif /* _WINCE */
	    {
		m_pScheduler->Remove(m_pAddrChangeCallback->m_PendingHandle);
		m_pAddrChangeCallback->m_PendingHandle = m_pScheduler->RelativeEnter(m_pAddrChangeCallback, CHECK_ADDR_CHANGE_INTERVAL);
	    }
	}
	goto cleanup;
    }
    else
    {
	pTempNetInterfaceList = new CHXSimpleList();

	RetrieveNetInterface0(pTempNetInterfaceList);
	if (!pTempNetInterfaceList || pTempNetInterfaceList->GetCount() == 0)
	{
	    // try the generic method
	    RetrieveNetInterface1(pTempNetInterfaceList);
	}

	if (pTempNetInterfaceList && m_pNetInterfaceList)
	{
	    if (pTempNetInterfaceList->GetCount() != m_pNetInterfaceList->GetCount())
	    {
		bResult = TRUE;
	    }
	    else
	    {
		CHXSimpleList::Iterator ndx0 = pTempNetInterfaceList->Begin();
		CHXSimpleList::Iterator ndx1 = m_pNetInterfaceList->Begin();
		for (; ndx0 != pTempNetInterfaceList->End() && ndx1 != m_pNetInterfaceList->End(); ++ndx0, ++ndx1)
		{
		    NIInfo* pInfo0 = (NIInfo*)(*ndx0);
		    NIInfo* pInfo1 = (NIInfo*)(*ndx1);

		    if (pInfo0->ulNetAddress != pInfo1->ulNetAddress ||
			pInfo0->ulNetMask != pInfo1->ulNetMask)
		    {
			bResult = TRUE;
		    }
		}
	    }
	}
	else if (pTempNetInterfaceList != m_pNetInterfaceList)
	{
	    bResult = TRUE;
	}

	if (bResult)
	{
	    Reset(m_pNetInterfaceList);	
	    HX_DELETE(m_pNetInterfaceList);

	    m_pNetInterfaceList = pTempNetInterfaceList;
	}
	else
	{
	    Reset(pTempNetInterfaceList);
	    HX_DELETE(pTempNetInterfaceList);
	}
    }

cleanup:

    return bResult;
}

void
HXNetInterface::Reset(CHXSimpleList* pNetInterfaceList)
{
    if (pNetInterfaceList)
    {
	while (pNetInterfaceList->GetCount())
	{
	    NIInfo* pNIInfo = (NIInfo*)pNetInterfaceList->RemoveHead();
	    HX_DELETE(pNIInfo);
	}
    }
}

void
HXNetInterface::Close(void)
{
    if (m_pAddrChangeCallback)
    {
	if (m_pAddrChangeCallback->m_bIsCallbackPending)
	{
	    m_pAddrChangeCallback->m_bIsCallbackPending = FALSE;
	    m_pScheduler->Remove(m_pAddrChangeCallback->m_PendingHandle);
	    m_pAddrChangeCallback->m_PendingHandle = 0;
	}

	HX_RELEASE(m_pAddrChangeCallback);
    }

    if (m_hAddrChangeEvent)
    {
	CloseHandle(m_hAddrChangeEvent);
	m_hAddrChangeEvent = NULL;
    }

    if (m_handle)
    {
#ifndef _WINCE
	CancelIo(m_handle);
#else
	CloseHandle(m_handle);
#endif /* _WINCE */
	// XXX HP
	// Confirmed with MS TS that the caller doesn't need to
	// call CloseHandle() on m_handle, it's taken care of by
	// the FreeLibrary()
	m_handle = NULL;
    }

    Reset(m_pNetInterfaceList);
    HX_DELETE(m_pNetInterfaceList);

    if (m_pSinkList)
    {
	HX_ASSERT(m_pSinkList->GetCount() == 0);
	CHXSimpleList::Iterator ndx = m_pSinkList->Begin();
	for (; ndx != m_pSinkList->End(); ++ndx)
	{
	    IHXNetInterfacesAdviseSink* pSink = (IHXNetInterfacesAdviseSink*) (*ndx);
	    HX_RELEASE(pSink);
	}
	HX_DELETE(m_pSinkList);
    }


    if (m_hIPLib)
    {
	FreeLibrary(m_hIPLib);
	m_hIPLib = NULL;
    }

    if (m_hWinSockLib)
    {
	FreeLibrary(m_hWinSockLib);
	m_hWinSockLib = NULL;
    }

    HX_RELEASE(m_pScheduler);
    HX_RELEASE(m_pContext);
}

AddrChangeCallback::AddrChangeCallback() :
     m_lRefCount (0)
    ,m_pParent (0)
    ,m_PendingHandle (0)
    ,m_bIsCallbackPending (FALSE)
{
}

AddrChangeCallback::~AddrChangeCallback()
{
}

/*
 * IUnknown methods
 */

/////////////////////////////////////////////////////////////////////////
//      Method:
//              IUnknown::QueryInterface
//      Purpose:
//              Implement this to export the interfaces supported by your
//              object.
//
STDMETHODIMP AddrChangeCallback::QueryInterface(REFIID riid, void** ppvObj)
{
	QInterfaceList qiList[] =
	{
		{ GET_IIDHANDLE(IID_IUnknown), this },
		{ GET_IIDHANDLE(IID_IHXCallback), (IHXCallback*) this },
		{ GET_IIDHANDLE(IID_IHXInterruptSafe), (IHXInterruptSafe*) this },
	};	
    return QIFind(qiList, QILISTSIZE(qiList), riid, ppvObj);
}

/////////////////////////////////////////////////////////////////////////
//      Method:
//              IUnknown::AddRef
//      Purpose:
//              Everyone usually implements this the same... feel free to use
//              this implementation.
//
STDMETHODIMP_(ULONG32) AddrChangeCallback::AddRef()
{
    return InterlockedIncrement(&m_lRefCount);
}

/////////////////////////////////////////////////////////////////////////
//      Method:
//              IUnknown::Release
//      Purpose:
//              Everyone usually implements this the same... feel free to use
//              this implementation.
//
STDMETHODIMP_(ULONG32) AddrChangeCallback::Release()
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
    {
	return m_lRefCount;
    }

    delete this;
    return 0;
}


STDMETHODIMP AddrChangeCallback::Func(void)
{
    m_PendingHandle         = 0;
    m_bIsCallbackPending    = FALSE;

    if (m_pParent)
    {
	m_pParent->CheckAddrChange();
    }

    return HXR_OK;
}


STDMETHODIMP_(BOOL) AddrChangeCallback::IsInterruptSafe()
{
    return FALSE;
}
